/*
 *  XmNap  A Motif napster client
 *  
 *  Copyright (C) 2000 Mats Peterson
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 *  
 *  Please send any comments/bug reports to
 *  matsp888@yahoo.com  (Mats Peterson)
 */

#include <Xm/Xm.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "main.h"
#include "connect.h"
#include "message.h"
#include "msgbox.h"
#include "banlist.h"
#include "listwin.h"
#include "util.h"

static LISTBAN *banList = NULL;

static Widget listWin = NULL;


void DestroyBanListWin(void)
{
    if (listWin) {
	DestroyWin(XtParent(listWin));
	listWin = NULL;
    }
}


void ListBans(void)
{
    String p;
    
    while (banList) {
	XtFree(banList->target);
	XtFree(banList->setBy);
	XtFree(banList->reason);
	XtFree(banList->time);
	p = (String)banList;
	banList = banList->next;
	XtFree(p);
    }

    banList = NULL;
    
    if (SendMsg(MSG_CLIENT_BANLIST, ""))
	Disconnect(strerror(errno));
}


void AddBan(String s)
{
    LISTBAN *newBan, *banPtr, *prevPtr = NULL;
    String tmp, p;
    char timeStr[80];
    struct tm *tm;
    time_t time;

    tmp = XtNewString(s);
    newBan = XtNew(LISTBAN);
    p = strtok(s, " ");
    newBan->target = XtNewString(p);
    p = strtok(NULL, " ");
    newBan->setBy = XtNewString(p);
    p = strrchr(tmp, '\"');
    *(p++) = 0;
    newBan->reason = XtNewString(strchr(tmp, '\"') + 1);
    time = (time_t)atol(strtok(p, " "));
    tm = gmtime(&time);
    strftime(timeStr, 80, "%Y-%m-%d  %H:%M", tm);
    newBan->time = XtNewString(timeStr);

    for (p = newBan->reason; *p; p++) {
	if (! isprint(*p))
	    *p = ' ';
    }
    
    for (banPtr = banList; banPtr; banPtr = banPtr->next) {
	if (strcasecmp(banPtr->target, newBan->target) > 0)
	    break;
	prevPtr = banPtr;
    }

    if (prevPtr) {
	newBan->next = prevPtr->next;
	prevPtr->next = newBan;
    } else {
	newBan->next = banList;
	banList = newBan;
    }

    XtFree(tmp);
}


static void UpdateBanList(void)
{
    LISTBAN *lb;
    String tmp = XtMalloc(8192);
    XmString *str;
    int i, numBans;

    for (numBans = 0, lb = banList; lb; numBans++, lb = lb->next);
    
    str = (XmString*)XtMalloc(numBans * sizeof(XmString));
    for (i = 0, lb = banList; i < numBans; i++, lb = lb->next) {
	sprintf(tmp, "%-34.32s%-34.32s%-60.58s%-20s",
		lb->target, lb->setBy, lb->reason, lb->time);
	str[i] = XmStringCreateLocalized(tmp);
    }
    
    XtVaSetValues(listWin,
	    XmNlistItems, str,
	    XmNlistItemCount, numBans,
	    NULL);
    
    for (i = 0; i < numBans; i++)
        XmStringFree (str[i]);
    XtFree((String)str);
    XtFree(tmp);
}


static void ListBansCB(Widget w, XtPointer clientData,
	XmSelectionBoxCallbackStruct *cbs)
{
    Widget list;
    int i, *pos;
    LISTBAN *ban, *prevBan = NULL;
    
    switch (cbs->reason) {
        case XmCR_OK:
	    list = XtNameToWidget(w, "ItemsListSW.ItemsList");
	    XtVaGetValues(list, XmNselectedPositions, &pos, NULL);
	    if (! pos)
		return;
	    for (i = 0, ban = banList; i < ((*pos) - 1);
		 i++, ban = ban->next) {
		prevBan = ban;
	    }

	    if (SendMsg(MSG_CLIENT_UNBAN, ban->target)) {
		Disconnect(strerror(errno));
	        return;
            }
	    
	    if (prevBan)
		prevBan->next = ban->next;
	    else
		banList = banList->next;
	    
	    XtFree(ban->target);
	    XtFree(ban->setBy);
	    XtFree(ban->reason);
	    XtFree(ban->time);
	    XtFree((char*)ban);
	    UpdateBanList();
	    break;
	case XmCR_CANCEL:
	case XmCR_PROTOCOLS:
	    DestroyBanListWin();
    }
}


void DoListBans(void)
{
    DestroyBanListWin();
    if (! banList) {
	InfoMsg("No bans exist");
	return;
    }

    listWin = ListWin("banList", (XtCallbackProc)ListBansCB, NULL);
    UpdateBanList();
}
