/*
cdcd - Command Driven CD player
Copyright (C)1998 Tony Arcieri

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <cdaudio.h>
#include <config.h>

#define CD_DEV "/dev/cdrom"

int play_track = 1;
int cd_desc;
char inbuffer[512];

char
*strnlower(char *string, int len)
{
   int index;
   char *outbuffer;
   
   outbuffer = (char *)malloc(len);
   for(index = 0; index < len; index++) {
      outbuffer[index] = tolower(string[index]);
      if(string[index] == '\0')
	return outbuffer;
   }
   
   return outbuffer;
}

void
proc_command(char *command, char *arg)
{
   int index, track, disp_track, disp_disc, cur_disc;
   struct disc_info disc;
   struct disc_data data;
   struct disc_timeval disc_time;
   struct disc_changer changer;
   struct disc_volume vol;
   char dispbuffer[4096];
   
   if(strcmp(command, "help") == 0) {
      if(strlen(arg) == 0) {
	 puts("cdcd commands:");
	 puts(" play\tstop\topen\tclose\tpause\tresume\tff\trew");
	 puts(" next\tprev\tstatus\tgetvol\tsetvol\tseldisc\tlist\tquit");
	 puts(" info\tinfoall\tedit\text\trefresh");
	 return;
      }
      if(strcmp(arg, "play") == 0) {
	 puts("Usage: play [trackname, track #]");
	 puts("\nBy default, play starts at the beginning of the CD.  If you have the track name database setup, you can invoke play followed by the trackname to start a track.\nYou can also specify the track number.");
      } else if(strcmp(arg, "stop") == 0) {
	 puts("Usage: stop");
	 puts("\nstop will stop the CD, if it is playing.");
      } else if(strcmp(arg, "open") == 0) {
	 puts("Usage: open (or eject)");
	 puts("\nopen will stop the CD, if it is playing, and open the CD tray.");
      } else if(strcmp(arg, "close") == 0) {
	 puts("Usage: close");
	 puts("\nclose will close the CD tray.");
      } else if(strcmp(arg, "pause") == 0) {
	 puts("Usage: pause");
	 puts("\npause will pause the CD, if it is playing.");
      } else if(strcmp(arg, "resume") == 0) {
	 puts("Usage: resume");
	 puts("\nresume will cause a paused CD to continue playing.");
      } else if(strcmp(arg, "ff") == 0) {
	 puts("Usage: ff");
	 puts("\nff will advance the current track 15 seconds.");
      } else if(strcmp(arg, "rew") == 0) {
	 puts("Usage: rew");
	 puts("\nrew will go back in the song 15 seconds.");
      } else if(strcmp(arg, "ext") == 0) {
	 puts("Usage: ext [trackname, track#]");
	 puts("\next displays extended information for a track, if it available.\nSpecifying track 0 displays extended information for the CD itself.");
      } else if(strcmp(arg, "refresh") == 0) {
	 puts("Usage: refresh");
	 puts("\nrefresh will update an entry in the local CDDB cache");
      } else if(strcmp(arg, "next") == 0) {
	 puts("Usage: next");
	 puts("\nnext advances to the next song.");
      } else if(strcmp(arg, "prev") == 0) {
	 puts("Usage: prev");
	 puts("\nprev returns you to the previous song.");
      } else if(strcmp(arg, "info") == 0) {
	 puts("Usage: info");
	 puts("\ninfo will display statistics regarding the CD, such as name, artist, number of\ntracks, etc.  When you invoke info for the first time on a particular CD, it\nwill block as it attempts to retrive CD information from the CDDB.");
      } else if(strcmp(arg, "infoall") == 0) {
	 puts("Usage: infoall");
	 puts("\ninfoall will display album, artist, and all track names.");
      } else if(strcmp(arg, "status") == 0) {
	 puts("Usage: status");
	 puts("\nstatus will display the bare-bones information about the status of the CD.\nFor more detailed information, use 'info'.");
      } else if(strcmp(arg, "edit") == 0) {
	 puts("Usage: edit [name, artist, genre, track #]");
	 puts("\nedit allows you to edit the disc name database.  After specifing what\nyou wish to edit, you will be allowed to edit the information.");
      } else if(strcmp(arg, "getvol") == 0) {
	 puts("Usage: getvol");
	 puts("\ngetvol displays the current volume");
      } else if(strcmp(arg, "setvol") == 0) {
	 puts("Usage: setvol volume #");
	 puts("\nsetvol sets the current volume.  Valid volumes: 0 - 255.");
      } else if(strcmp(arg, "seldisc") == 0) {
	 puts("Usage: seldisc disc #");
	 puts("\nseldisc selects a disc from a CD-ROM changer.");
      } else if(strcmp(arg, "list") == 0) {
	 puts("Usage: list");
	 puts("\nlist displays the entire contents of your CD-ROM changer.");
      } else if(strcmp(arg, "quit") == 0) {
	 puts("Usage: quit");
	 puts("\nquit exits cdcd. (CD keeps playing, of course)");
      } else
	printf("No help available on %s.\n", arg);
   } else if(strcmp(command, "version") == 0) {
      cd_version(dispbuffer, 4096);
      printf("%s version %s, Copyright (C)1998 Tony Arcieri\n", PACKAGE, VERSION);
      puts("Distributed under the GNU General Public License. See file COPYING for details.");
      printf("Built with %s\n", dispbuffer);
   } else if(strcmp(command, "play") == 0) {
      cd_stat(cd_desc, &disc);
      
      if(disc.disc_status == CD_ABSENT) {
	 cd_close(cd_desc);
	 cd_stat(cd_desc, &disc);
	 if(disc.disc_status == CD_ABSENT) {
	    puts("No disc in drive");
	    return;
	 }
      }
	 
      if(strlen(arg) > 0)
	play_track = strtol(arg, NULL, 10);
      
      if(play_track == 0) {
	 if(cddb_read_disc_data(cd_desc, &data) < 0) {
	    perror("cddb_read_disc_data");
	    return;
	 }
	 
	 for(track = 0; track < disc.disc_totaltracks; track++)
	   if(strstr(strnlower(data.data_track[track].track_name, 56), strnlower(arg, 512)) != NULL) {
	      play_track = track + 1;
	      break;
	   }
	 
	 if(play_track == 0) play_track = 1;
      }
      
      if(play_track < 0 || play_track > disc.disc_totaltracks) play_track = 1;
      
      cd_play(cd_desc, play_track);
   } else if(strcmp(command, "ext") == 0) {
      cd_stat(cd_desc, &disc);
      if(disc.disc_status == CD_ABSENT) {
	 cd_close(cd_desc);
	 cd_stat(cd_desc, &disc);
	 if(disc.disc_status == CD_ABSENT) {
	    puts("No disc in drive");
	    return;
	 }
      }
      
      if(cddb_read_disc_data(cd_desc, &data) < 0) {
	 perror("cddb_read_disc_data");
	 return;
      }

      if(strlen(arg) > 0) {
	 if((disp_track = strtol(arg, NULL, 10)) == 0) {
	    for(track = 0; track < disc.disc_totaltracks; track++)
	      if(strstr(strnlower(data.data_track[track].track_name, 56), strnlower(arg, 512)) != NULL) {
		 disp_track = track + 1;
		 break;
	      }
	 }
      } else
	disp_track = disc.disc_track;

      if(disp_track < 0 || disp_track > disc.disc_totaltracks) disp_track = 1;

      data_format_extended_info(dispbuffer, &data, disp_track, 4096);

      if(disp_track == 0)
	puts(data.data_title);
      else {
	 if(strstr(strnlower(data.data_track[disp_track - 1].track_extended[0], 96), strnlower(data.data_track[disp_track - 1].track_name, 128)) == NULL)
	   puts(data.data_track[disp_track - 1].track_name);
      }
      	     
      puts(dispbuffer);
   } else if(strcmp(command, "stop") == 0) {
      cd_stop(cd_desc);
   } else if(strcmp(command, "open") == 0 || strcmp(command, "eject") == 0) {
      cd_stop(cd_desc);
      cd_eject(cd_desc);
   } else if(strcmp(command, "close") == 0) {
      cd_close(cd_desc);
   } else if(strcmp(command, "pause") == 0) {
      cd_pause(cd_desc);
   } else if(strcmp(command, "resume") == 0) {
      cd_resume(cd_desc);
   } else if(strcmp(command, "rew") == 0) {
      disc_time.minutes = 0;
      disc_time.seconds = -15;
      cd_track_advance(cd_desc, &disc_time);
   } else if(strcmp(command, "ff") == 0) {
      disc_time.minutes = 0;
      disc_time.seconds = 15;
      cd_track_advance(cd_desc, &disc_time);
   } else if(strcmp(command, "next") == 0) {
      cd_stat(cd_desc, &disc);
      if(disc.disc_track + 1 > disc.disc_totaltracks)
	cd_stop(cd_desc);
      else
	cd_play(cd_desc, disc.disc_track + 1);
   } else if(strcmp(command, "prev") == 0) {
      cd_stat(cd_desc, &disc);
      if(disc.disc_track - 1 < 1)
	cd_play(cd_desc, 1);
      else
	cd_play(cd_desc, disc.disc_track - 1);
   } else if(strcmp(command, "setvol") == 0) {
      if(strlen(arg) > 0) {
	 vol.vol_front.left = strtol(arg, NULL, 10);
	 vol.vol_front.right = strtol(arg, NULL, 10);
	 vol.vol_back.left = strtol(arg, NULL, 10);
	 vol.vol_back.right = strtol(arg, NULL, 10);
	 if(cd_set_volume(cd_desc, &vol) < 0)
	   puts("Invalid volume");
      } else
	puts("To what?");
   } else if(strcmp(command, "getvol") == 0) {
      cd_get_volume(cd_desc, &vol);
      printf("Left: %d\tRight:%d\n", vol.vol_front.left, vol.vol_front.right);
   } else if(strcmp(command, "status") == 0) {
      cd_stat(cd_desc, &disc);
      switch(disc.disc_mode) {
       case CDAUDIO_PLAYING:
	 printf("Playing %d %02d:%02d %02d:%02d\n", disc.disc_track, disc.track_time.minutes, disc.track_time.seconds, disc.disc_time.minutes, disc.disc_time.seconds);
	 break;
       case CDAUDIO_PAUSED:
	 printf("Paused %d %02d:%02d %02d:%02d\n", disc.disc_track, disc.track_time.minutes, disc.track_time.seconds, disc.disc_time.minutes, disc.disc_time.seconds);
	 break;
       case CDAUDIO_COMPLETED:
	 puts("Stopped");
	 break;
       case CDAUDIO_NOSTATUS:
	 puts("Stopped");
	 break;
      }
   } else if(strcmp(command, "info") == 0) {
      cd_stat(cd_desc, &disc);
      if(disc.disc_status == CD_ABSENT) {
	 cd_close(cd_desc);
	 cd_stat(cd_desc, &disc);
	 if(disc.disc_status == CD_ABSENT) {
	    puts("No disc in drive");
	    return;
	 }
      }
      
      if(cddb_read_disc_data(cd_desc, &data) < 0) {
	 perror("cddb_read_disc_data");
	 return;
      }
      
      printf("Album name:   \t%s\n", data.data_title);
      printf("Album artist: \t%s\n", data.data_artist);
      switch(disc.disc_mode) {
       case CDAUDIO_PLAYING:
         printf("Total tracks: \t%d\tDisc playing:\t%02d:%02d of %02d:%02d\n", disc.disc_totaltracks, disc.disc_time.minutes, disc.disc_time.seconds, disc.disc_length.minutes, disc.disc_length.seconds);
	 printf("Playing:      \t%s %02d %02d:%02d of %02d:%02d\n", data.data_track[disc.disc_track - 1].track_name, disc.disc_track, disc.track_time.minutes, disc.track_time.seconds, disc.track[disc.disc_track - 1].track_length.minutes, disc.track[disc.disc_track - 1].track_length.seconds);
	 break;
       case CDAUDIO_PAUSED:
         printf("Total tracks: \t%d\tDisc paused:\t%02d:%02d of %02d:%02d\n", disc.disc_totaltracks, disc.disc_time.minutes, disc.disc_time.seconds, disc.disc_length.minutes, disc.disc_length.seconds);
	 printf("Paused:       \t%s %02d %02d:%02d of %02d:%02d\n", data.data_track[disc.disc_track - 1].track_name, disc.disc_track, disc.track_time.minutes, disc.track_time.seconds, disc.track[disc.disc_track - 1].track_length.minutes, disc.track[disc.disc_track - 1].track_length.seconds);
	 break;
       case CDAUDIO_COMPLETED:
	 printf("Total tracks: \t%d\tDisc length:\t%02d:%02d\n", disc.disc_totaltracks, disc.disc_length.minutes, disc.disc_length.seconds);
	 puts("Stopped");
	 break;
       case CDAUDIO_NOSTATUS:
	 printf("Total tracks: \t%d\tDisc length:\t%02d:%02d\n", disc.disc_totaltracks, disc.disc_length.minutes, disc.disc_length.seconds);
	 puts("Stopped");
	 break;
      }
   } else if(strcmp(command, "infoall") == 0) {
      cd_stat(cd_desc, &disc);
      if(disc.disc_status == CD_ABSENT) {
	 cd_close(cd_desc);
	 cd_stat(cd_desc, &disc);
	 if(disc.disc_status == CD_ABSENT) {
	    puts("No disc in drive");
	    return;
	 }
      }
      
      if(cddb_read_disc_data(cd_desc, &data) < 0) {
	 perror("cddb_read_disc_data");
	 return;
      }
      
      printf("Album name:   \t%s\n", data.data_title);
      printf("Album artist: \t%s\n", data.data_artist);
      printf("Total tracks: \t%d\tDisc length:\t%02d:%02d\n\n", disc.disc_totaltracks, disc.disc_length.minutes, disc.disc_length.seconds);
      for(track = 0; track < disc.disc_totaltracks; track++)
	printf("Track %d:\t%02d:%02d %s\n", track + 1, disc.track[track].track_length.minutes, disc.track[track].track_length.seconds, data.data_track[track].track_name);
   } else if(strcmp(command, "edit") == 0) {
      if(cddb_read_disc_data(cd_desc, &data) < 0) {
	 puts("No disc in drive");
	 return;
      }
      
      if(strlen(arg) > 0)
	strncpy(inbuffer, arg, 512);
      else {
	 printf("Edit what (name, artist, genre, track#): ");
	 fgets(inbuffer, 512, stdin);
	 inbuffer[strlen(inbuffer) - 1] = '\0';
      }
      if(strcmp(inbuffer, "name") == 0) {
	 printf("Current value: %s\n", data.data_title);
	 printf("Enter new value: ");
	 fgets(data.data_title, 128, stdin);
	 data.data_title[strlen(data.data_title) - 1] = '\0';
      } else if(strcmp(inbuffer, "artist") == 0) {
	 printf("Current value: %s\n", data.data_artist);
	 printf("Enter new value: ");
	 fgets(data.data_artist, 128, stdin);
	 data.data_artist[strlen(data.data_artist) - 1] = '\0';
      } else if(strcmp(inbuffer, "genre") == 0) {
	 printf("Current value: %s\n", cddb_genre(data.data_genre));
	 printf("Enter new value: ");
	 fgets(inbuffer, 512, stdin);
	 for(index = 1; index < 12; index++)
	   if(strncasecmp(inbuffer, cddb_genre(index), 4) == 0) {
	      cddb_erase_entry(&data);
	      data.data_genre = index;
	      cddb_write_disc_data(cd_desc, &data);
	      return;
	   }
	 puts("Sorry.  Valid genres are:");
	 for(index = 1; index < 12; index++)
	   puts(cddb_genre(index));
	 
	 return;
      } else if((track = strtol(inbuffer, NULL, 10)) != 0) {
	 printf("Current value: %s\n", data.data_track[track - 1].track_name);
	 printf("Enter new value: ");
	 fgets(data.data_track[track - 1].track_name, 128, stdin);
	 data.data_track[track - 1].track_name[strlen(data.data_track[track - 1].track_name) - 1] = '\0';
      } else
	puts("What?");
      
      cddb_write_disc_data(cd_desc, &data);
   } else if(strcmp(command, "refresh") == 0)
     cddb_refresh_disc_data(cd_desc, &data);
   else if(strcmp(command, "list") == 0) {
      cd_changer_stat(cd_desc, &changer);
      for(disp_disc = 0; disp_disc < changer.changer_slots; disp_disc++)
	if(changer.changer_disc[disp_disc].disc_status == CD_PRESENT)
	  printf("Disc %d: \t%02d:%02d %s\n", disp_disc + 1, changer.changer_disc[disp_disc].disc_length.minutes, changer.changer_disc[disp_disc].disc_length.seconds, changer.changer_disc[disp_disc].disc_info);
   } else if(strcmp(command, "seldisc") == 0) {
      cur_disc = 0;
      
      if(strlen(arg) > 0)
	cur_disc = strtol(arg, NULL, 10);
      
      if(cur_disc == 0) {
	 puts("What disc was that?");
	 return;
      }
      
      cd_changer_select_disc(cd_desc, cur_disc - 1);
   } else if(strcmp(command, "quit") == 0) {
      close(cd_desc);
      exit(0);
   } else if(strcmp(command, "debug") == 0) {
      cd_stat(cd_desc, &disc);
      if(disc.disc_status == CD_ABSENT) {
	 cd_close(cd_desc);
	 cd_stat(cd_desc, &disc);
	 if(disc.disc_status == CD_ABSENT) {
	    puts("No disc in drive");
	    return;
	 }
      }
      
      if(cddb_read_disc_data(cd_desc, &data) < 0) {
	 perror("cddb_read_disc_data");
	 return;
      }

      puts("Debugging information:");
      printf("Disc id:\t%08lx\n", data.data_id);
      printf("Disc genre:\t%s\n", cddb_genre(data.data_genre));
      printf("Disc mode:\t");
      switch(disc.disc_mode) {
       case CDAUDIO_PLAYING:
	 puts("Playing");
	 break;
       case CDAUDIO_PAUSED:
	 puts("Paused");
	 break;
       case CDAUDIO_COMPLETED:
	 puts("Stopped");
	 break;
       case CDAUDIO_NOSTATUS:
	 puts("Stopped");
	 break;
      }
      printf("Track time:\t%02d:%02d\n", disc.track_time.minutes, disc.track_time.seconds);
      printf("Disc time:\t%02d:%02d\n", disc.disc_time.minutes, disc.disc_time.seconds);
      printf("Disc length:\t%02d:%02d\n", disc.disc_length.minutes, disc.disc_length.seconds);
      printf("Disc frame:\t%d\n", disc.disc_frame);
      printf("Disc track:\t%d\n", disc.disc_track);
      printf("Total tracks:\t%d\n", disc.disc_totaltracks);
      for(track = 0; track < disc.disc_totaltracks; track++) {
	 printf("Track %d:\n", track + 1);
	 printf(" Track start:\t%d\n", disc.track[track].track_start);
	 printf(" Track length:\t%02d:%02d\n", disc.track[track].track_length.minutes, disc.track[track].track_length.seconds);
      }
   } else if(strlen(command) != 0)
     fputs("Unknown command\n", stderr);
}

int
main(int argc, char **argv)
{
   char arg[512];
   
   if((cd_desc = cd_init(CD_DEV)) < 0) {
      perror(CD_DEV);
      exit(1);
   }
   
   if(argc > 1) {
      if(argc > 2)
	strncpy(arg, argv[2], 512);
      else
	arg[0] = '\0';
      
      proc_command(argv[1], arg);
      
      exit(0);
   } else {
      while(1) {
	 printf("%s> ", argv[0]);
	 
	 fgets(inbuffer, 512, stdin);
	 inbuffer[strlen(inbuffer) - 1] = '\0';
	 	 
	 if(strchr(inbuffer, ' ') != NULL) {
	    strtok(inbuffer, " ");
	    strncpy(arg, strtok(NULL, " "), 512);
	 } else
	   arg[0] = '\0';

	 proc_command(inbuffer, arg);
      }
   }
}
