/* 
   WebDAV 207 multi-status response handling
   Copyright (C) 1999-2000, Joe Orton <joe@orton.demon.co.uk>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
  
   Alternatively, you can redistribute it and/or modify it under the
   terms of the GNU Library General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at
   your option) any later version.

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   License for more details.

   You should have received copies of the GNU Library General Public
   License and the GNU General Public License along with this software; 
   if not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
   Cambridge, MA 02139, USA.

   $Id: dav_207.c,v 1.11 2000/03/08 19:52:17 joe Exp $ 

*/

/* Generic handling for WebDAV 207 Multi-Status error responses. */

#include <config.h>

#include "http_utils.h"
#include "hip_xml.h"
#include "dav_207.h"

const struct hip_xml_elm dav_207_elms[] = {
    { "DAV:multistatus", DAV_ELM_multistatus, 0 },
    { "DAV:response", DAV_ELM_response, 0 },
    { "DAV:responsedescription", DAV_ELM_responsedescription, 0 },
    { "DAV:href", DAV_ELM_href, HIP_XML_CDATA },
    { "DAV:propstat", DAV_ELM_propstat, 0 },
    { "DAV:status", DAV_ELM_status, HIP_XML_CDATA },
    { NULL, 0, 0 }
};

int dav_207_startelm( void *userdata, const struct hip_xml_state *s,
		      const char *name, const char **atts ) {
    struct dav_207_parser *ctx = userdata;
    struct dav_response *rsp;
    struct dav_propstat *pst;
    switch( s->elm->id ) {
    case DAV_ELM_response:
	/* Create new response */
	rsp = malloc( sizeof(struct dav_response) );
	memset( rsp, 0, sizeof(struct dav_response) );
	ctx->ms.current = rsp;
	break;
    case DAV_ELM_propstat:
	if( !ctx->ms.current ) {
	    DEBUG_FATAL( DEBUG_XML );
	    return -1;
	}
	pst = malloc( sizeof(struct dav_propstat) );
	memset( pst, 0, sizeof(struct dav_propstat) );
	ctx->ms.current->current = pst;
    }
    return 0;
}

int dav_207_getcurrentpropstat( struct dav_207_parser *p, 
				struct dav_propstat **pstat ) {
    if( p->ms.current != NULL && p->ms.current->current != NULL ) {
	*pstat = p->ms.current->current;
	return 0;
    } else {
	return -1;
    }
}

int dav_207_endelm( void *userdata, const struct hip_xml_state *s,
		    const char *cdata ) {
    struct dav_207_parser *ctx = userdata;
    switch( s->elm->id ) {
    case DAV_ELM_responsedescription:
	switch( s->parent->elm->id ) {
	case DAV_ELM_multistatus:
	    ctx->ms.description = strdup(cdata);
	    break;
	case DAV_ELM_propstat:
	    if( !ctx->ms.current || !ctx->ms.current->current ) {
		DEBUG_FATAL( DEBUG_XML );
		return -1;
	    }
	    ctx->ms.current->current->description = strdup(cdata);
	    break;
	default:
	    return -1;
	}
	break;
    case DAV_ELM_href:
	if( !ctx->ms.current ) {
	    DEBUG_FATAL( DEBUG_XML );
	    return -1;
	}
	ctx->ms.current->href = strdup(cdata );
	break;
    case DAV_ELM_status:
	if( !ctx->ms.current ) {
	    DEBUG_FATAL( DEBUG_XML );
	    return -1;
	}
	switch( s->parent->elm->id ) {
	case DAV_ELM_response: {
	    char *tmp = strdup(cdata);
	    ctx->ms.current->status_line = tmp;
	    if( http_parse_statusline( tmp, &ctx->ms.current->status ) ) {
		    snprintf( ctx->parser.error, BUFSIZ,
			      "Invalid HTTP status line in status element at line %d of response", hip_xml_currentline(&ctx->parser) );
		free( tmp );
		return -1;
	    }
	} break;
	case DAV_ELM_propstat:
	    if( !ctx->ms.current->current ) {
		DEBUG_FATAL( DEBUG_XML );
		return -1;
	    } else {
		char *tmp = strdup(cdata);
		ctx->ms.current->current->status_line = tmp;
		if( http_parse_statusline( tmp, &ctx->ms.current->current->status ) ) {
		    snprintf( ctx->parser.error, BUFSIZ,
			      "Invalid HTTP status line in status element at line %d of response", hip_xml_currentline(&ctx->parser) );
		    free( tmp );
		    return -1;
		}
	    }
	    break;
	}
	break;
    case DAV_ELM_propstat:
	if( !ctx->ms.current || !ctx->ms.current->current ) {
	    DEBUG_FATAL( DEBUG_XML );
	    return -1;
	}
	if( ctx->ms.current->last ) {
	    /* Non-empty list case */
	    ctx->ms.current->last->next = ctx->ms.current->current;
	} else {
	    /* Empty list case */
	    ctx->ms.current->first = ctx->ms.current->current;
	}
	ctx->ms.current->last = ctx->ms.current->current;
	ctx->ms.current->current = NULL;
	break;
    case DAV_ELM_response:
	if( !ctx->ms.current ) {
	    DEBUG_FATAL( DEBUG_XML );
	    return -1;
	}
	if( ctx->ms.last ) {
	    /* Non-empty list case */
	    ctx->ms.last->next = ctx->ms.current;
	} else {
	    /* Empty list case */
	    ctx->ms.first = ctx->ms.current;
	}
	ctx->ms.last = ctx->ms.current;
	ctx->ms.current = NULL;
	break;
    }
    return 0;
}

int dav_207_check( hip_xml_elmid parent, hip_xml_elmid child ) {
    DEBUG( DEBUG_XML, "207_check: %d to %d\n", child, parent );
    switch( parent ) {
    case HIP_ELM_root:
	switch( child ) {
	case DAV_ELM_multistatus:
	case DAV_ELM_response:
	    return 0;
	default:
	}
	break;
    case DAV_ELM_multistatus:
	switch( child ) {
	case DAV_ELM_response:
	case DAV_ELM_responsedescription:
	    return 0;
	default:
	}
	break;
    case DAV_ELM_response:
	switch( child ) {
	case DAV_ELM_href:
	case DAV_ELM_propstat:
	case DAV_ELM_responsedescription:
	case DAV_ELM_status:
	    return 0;
	default:
	}
	break;
    case DAV_ELM_propstat:
	switch( child ) {
	case DAV_ELM_prop: 
	case DAV_ELM_status:
	case DAV_ELM_responsedescription:
	    return 0;
	default:
	}
	break;
    default:
    }
    return -1;
}

int dav_207_init( struct dav_207_parser *p, struct hip_xml_elmlist *list ) {
    struct hip_xml_elmlist *newlist, *prevlist = NULL;
    /* Find the end of the list: we let the caller override any of our
     * own elm handling */
    for( newlist = list; newlist !=NULL; newlist = newlist->next ) {
	prevlist = newlist;
    }
    if( prevlist == NULL ) {
	return -1;
    }
    newlist = malloc( sizeof(struct hip_xml_elmlist) );
    if( !newlist ) return -1;
    /* initialize */
    memset( newlist, 0, sizeof(struct hip_xml_elmlist) );
    newlist->elements = dav_207_elms;
    newlist->validate_cb = dav_207_check;
    newlist->startelm_cb = dav_207_startelm;
    newlist->endelm_cb = dav_207_endelm;
    newlist->userdata = p;
    newlist->next = NULL;
    prevlist->next = newlist;
    /* init the context */
    memset( p, 0, sizeof(struct dav_207_parser) );
    p->list_207 = newlist;
    return hip_xml_init( &p->parser, list );
}

void dav_207_free( struct dav_207_parser *p, 
		   void (*freeprop)( void *ud, struct dav_response *rsp,
				     struct dav_propstat *pstat ), 
		   void *userdata ) {
    struct dav_response *rsp, *rsp_next;
    for( rsp=p->ms.first; rsp!=NULL; rsp=rsp_next ) {
	struct dav_propstat *pst, *pst_next;
	rsp_next = rsp->next;
	for( pst = rsp->first; pst != NULL; pst = pst_next ) {
	    pst_next = pst->next;
	    if( freeprop ) (*freeprop)( userdata, rsp, pst );
	    HTTP_FREE( pst->status_line );
	    HTTP_FREE( pst->description );
	    HTTP_FREE( pst );
	}
	HTTP_FREE( rsp->href );
	HTTP_FREE( rsp->status_line );
	HTTP_FREE( rsp );
    }
    HTTP_FREE( p->ms.description );
}

int dav_207_finish( struct dav_207_parser *p ) {
    free( p->list_207 );
    return hip_xml_finish( &p->parser );
}
