/* 
   cadaver, command-line DAV client
   Copyright (C) 1999-2000, Joe Orton <joe@orton.demon.co.uk>, 
   except where otherwise indicated.
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   $Id: cadaver.c,v 1.69 2000/03/15 18:36:57 joe Exp $
*/

#include <config.h>

/* To pick up strndup() */
#define _GNU_SOURCE

#include <sys/types.h>

#include <sys/time.h>
#include <sys/stat.h>

#include <stdio.h>
#include <ctype.h>

#include <signal.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif 
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

/* joe: My man pages say getpass() should be in pwd.h...
 * looks like getpass() is a BSD-ism, what is the POSIX equiv? */
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <errno.h>

#ifndef HAVE_SNPRINTF
#include "snprintf.h"
#endif

#include "getopt.h"

#ifdef ENABLE_NETRC
#include "netrc.h"
#endif

#include "frontend.h"
#include "protocol.h"
#include "httpdav.h"
#include "string_utils.h"
#include "common.h"
#include "uri.h"

#include "cadaver.h"
#include "cmdline.h"
#include "commands.h"
#include "options.h"

#ifdef ENABLE_NETRC
static netrc_entry *netrc_list; /* list of netrc entries */
#endif

struct proto_host server = {0}; /* server details */
struct proto_host proxy = {0}; /* server details */

static char *progname; /* argv[0] */

/* Global options */

char *path, /* current working collection */
    *old_path; /* previous working collection */
/* TODO: use dynamically allocated memory for paths, maybe */

int tolerant; /* tolerate DAV-enabledness failure */

int in_curses = 0;

bool have_connection; /* true when we are connected to the server */
bool dav_collection;

/* Current output state */
static enum out_state {
    out_none, /* not doing anything */
    out_incommand, /* doing a simple command */
    out_transfer_start, /* transferring a file, not yet started */
    out_transfer_plain, /* doing a plain ... transfer */
    out_transfer_pretty /* doing a pretty progress bar transfer */
} out_state;   

/* Protoypes */

static void usage(void);
static void parse_args( int argc, char *argv[], struct proto_host *server );
static char *read_command(void);

static RETSIGTYPE quit_handler( int signal );

static void close_connection( void );
static void open_connection( const char *hostname, const char *path );

static void init_netrc( void );
static void init_rcfile( void );
static void execute_version( void );
static void init_readline( void );
static void init_options( void );

#ifdef HAVE_LIBREADLINE

static char **completion( char *text, int start, int end );

#endif

#define CMD_VARY 9999

/* Separate structures for commands and command names. */
/* DON'T FORGET TO ADD A NEW COMMAND ALIAS WHEN YOU ADD A NEW COMMAND */
const struct command commands[] = {
/* C1: connected, 1-arg function C2: connected, 2-arg function
 * U0: disconnected, 0-arg function. */
#define C1(x,c,h) { cmd_##x, true, 1, 1, parmscope_remote, (void *)execute_##x,c,h }
#define U0(x,h) { cmd_##x, false, 0, 0, parmscope_none, (void *)execute_##x,#x,h }
#define UO1(x,c,h) { cmd_##x, false, 0, 1, parmscope_none, (void *)execute_##x,c,h }
#define C2M(x,c,h) { cmd_##x, true, 2, CMD_VARY, parmscope_remote, (void *)multi_##x,c,h }
#define C1M(x,c,h) { cmd_##x, true, 1, CMD_VARY, parmscope_remote, (void *)multi_##x,c,h }
    { cmd_ls, true, 0, 1, parmscope_none, (void *)execute_ls, 
      "ls [path]", "List contents of current [or other] collection" },
    C1(cd, "cd path", "Change to specified collection"),
    { cmd_pwd, true, 0, 0, parmscope_none, (void *)execute_pwd,
      "pwd", "Display name of current collection" },
    { cmd_put, true, 1, 2, parmscope_none, (void *)execute_put, 
      "put local [remote]", "Upload local file" },
    { cmd_get, true, 1, 2, parmscope_none, (void *)execute_get, 
      "get remote [local]", "Download remote resource" },
    C1M( mget, "mget remote...", "Download many remote resources" ),
    { cmd_mput, true, 1, CMD_VARY, parmscope_local, (void *)multi_mput, 
      "mput local...", "Upload many local files" },
    C1(edit, "edit resource", "Edit given resource" ),
    C1M(less, "less remote...", "Display remote resource through pager"), 
    C1M(mkcol, "mkcol remote...", "Create remote collection(s)"), 
    C1M(cat, "cat remote...", "Display remote resource(s)"), 
    C1M(delete, "delete remote...", "Delete non-collection resource(s)"),
    C1M(rmcol, "rmcol remote...", "Delete remote collections and ALL contents"),
    C2M(copy, "copy source... dest", "Copy resource(s) from source to dest"), 
    C2M(move, "move source... dest", "Move resource(s) from source to dest"),

/* DON'T FORGET TO ADD A NEW COMMAND ALIAS WHEN YOU ADD A NEW COMMAND */
    
    C1(lock, "lock resource", "Lock given resource"),
    C1(unlock, "unlock resource", "Unock given resource"),
    C1(discover, "discover resource", "Display lock information for resource"),
    C1(steal, "steal resource", "Steal lock token for resource"),    
    U0(showlocks, "Display list of owned locks"),

#if 0
    C1(propedit, "propedit resource", "Enter property editor for resource"),
#endif
    C1(propnames, "propnames res", "Names of properties defined on resource" ),

    { cmd_set, false, 0, 2, parmscope_none, (void *)execute_set, 
      "set [option] [value]", "Set an option, or display options" }, 
    { cmd_open, false, 1, 2, parmscope_none, (void *)open_connection, 
      "open host[:port] [/path/]","Open connection to server at path" },
    { cmd_close, true, 0, 0, parmscope_none, (void *)close_connection, 
      "close", "Close current connection" },
    { cmd_echo, false, 1, CMD_VARY, parmscope_remote, (void *)execute_echo, 
      "echo", NULL },
    { cmd_quit, false, 0, 1, parmscope_none, NULL, "quit", "Exit program" },
    /* Unconnected operation, 1 mandatory argument */
    { cmd_unset, false, 1, 2, parmscope_none, (void *)execute_unset, 
      "unset [option] [value]", "Unsets or clears value from option." },
    /* Unconnected operation, 0 arguments */
    UO1(lcd, "lcd [directory]", "Change local working directory"), 
    { cmd_lls, false, 0, CMD_VARY, parmscope_local, (void *)execute_lls, 
      "lls [options]", "Display local directory listing" },
    U0(lpwd, "Print local working directory" ),
    UO1(help, "help [command]", "Display help message" ), 
    U0(version, NULL ),
    { cmd_unknown, 0 } /* end-of-list marker, DO NOT move */

/* DON'T FORGET TO ADD A NEW COMMAND ALIAS WHEN YOU ADD A NEW COMMAND. */

#undef C1
#undef U0
#undef UO1
#undef C2M
#undef C1M
};    

static void usage() {
    printf( 
"Usage: %s [OPTIONS] [hostname[:port] [path]]\n"
"  Port defaults to 80, path defaults to '/'\n"
"Options:\n"
"  -e, --expect100  Enable sending of `Expect: 100-continue' header.\n"
"     *** Warning: For Apache servers, use only with version 1.3.9 and later.\n"
"  -t, --tolerant   Allow cd/open into non-DAV enabled collection.\n"
"  -V, --version    Display version information.\n"
"  -h, --help       Display this help message.\n"
"Please send bug reports and feature requests to <cadaver@webdav.org>\n", progname );
}

static void execute_version( void ) {
    printf( PACKAGE " " VERSION "\n" );
}

static void close_connection( void ) {
    http_finish( );
    have_connection = false;
    SAFE_FREE( path );
    SAFE_FREE( old_path );
    printf( "Connection closed.\n" );
}

/* Sets the current collection to the given path.  Returns zero on
 * success, non-zero if newpath is an untolerated non-WebDAV
 * collection. */
int set_path( const char *newpath ) 
{
    int is_coll;
    is_coll = is_collection( newpath );
    if( is_coll || tolerant ) {
	if( !is_coll ) {
	    dav_collection = false;
	    printf( "Ignored error: %s not WebDAV-enabled:\n%s\n", newpath,
		    http_error );
	} else {
	    dav_collection = true;
	}
	return 0;
    } else {
	printf( "Could not access %s (not WebDAV-enabled?):\n%s\n", newpath,
		http_error );
	return 1;
    }
}

static void open_connection( const char *hostname, const char *open_path ) 
{
    char *pnt, *real_hostname = strdup(hostname);
    char *use_path, *proxy_host = get_option(opt_proxy);
    if( have_connection ) {
	close_connection( );
    } else {
	SAFE_FREE( path );
	SAFE_FREE( old_path );
    }
    if( proxy_host != NULL ) {
	if( get_option(opt_proxy_port) != NULL ) {
	    proxy.port = atoi((char *)get_option(opt_proxy_port));
	} else {
	    proxy.port = 8080;
	}
	proxy.hostname = proxy_host;
    }
    pnt = strchr( real_hostname, ':' );
    if( pnt != NULL && (strlen(pnt+1)>0) ) {
	*pnt = '\0';
	server.port = atoi( pnt+1 );
    } else {
	server.port = 80;
    }
    if( server.hostname ) free( server.hostname );
    server.hostname = real_hostname;
#ifdef ENABLE_NETRC
    {
	netrc_entry *found;
	found = search_netrc( netrc_list, hostname );
	if( found != NULL ) {
	    if( found->account && found->password ) {
		server.username = found->account;
		server.password = found->password;
	    }
	}
    }
#endif /* ENABLE_NETRC */
    have_connection = false;
    /* Default to '/' if no path given */
    if( open_path != NULL ) {
	if( uri_has_trailing_slash(open_path) ) {
	    use_path = strdup(open_path);
	} else if( open_path[0] == '/' ) {
	    CONCAT2( use_path, open_path, "/" );
	} else {
	    CONCAT3( use_path, "/", open_path, "/" );
	}
    } else {
	use_path = strdup("/");
    }
#if 0
    /* ahhhh, I turned this off for a better reason than "there's no need".
     * turning it back on again because it gets http_webdav_server set properly.
     */
    http_init_checks = false; /* we're going to do a PROPFIND, so no worries */
#endif
    switch( http_init( use_path, &server, proxy_host?&proxy:NULL ) ) {
    case PROTO_OK:
	have_connection = true;
	path = NULL;
	if( set_path( use_path ) ) {
	    close_connection( );
	} else {
	    path = strdup(use_path);
	}
	break;
    case PROTO_LOOKUP:
	printf( "Could not resolve hostname.\n" );
	break;
    case PROTO_CONNECT:
	printf( "Could not connect to remote host.\n" );
	break;
    default:
	printf( "Could not open connection:\n%s\n", http_error );
	break;
    }
    free( use_path );
}
       
/* Sets proxy server from hostport argument */    
static void set_proxy( const char *str )
{
    char *hostname = strdup(str), *pnt;

    pnt = strchr( hostname, ':' );
    if( pnt != NULL ) {
	*pnt++ = '\0';
    }
    set_option( opt_proxy, (void *)hostname );
    set_option( opt_proxy_port, pnt );
}

static void parse_args( int argc, char **argv, 
		       struct proto_host *server ) 
{
    static const struct option opts[] = {
	{ "version", no_argument, NULL, 'V' },
	{ "help", no_argument, NULL, 'h' },
	{ "expect100", no_argument, NULL, 'e' },
	{ "proxy", required_argument, NULL, 'p' },
	{ "tolerant", no_argument, NULL, 't' },
	{ 0, 0, 0, 0 }
    };
    int optc;
    char *hostport = NULL, *apath = NULL;
    while( (optc = getopt_long( argc, argv, "ehtpV", opts, NULL )) != -1 ) {
	switch( optc ) {
	case 'h': usage( ); exit( -1 );
	case 'V': execute_version( ); exit( -1 );
	case 'e': http_enable_expect = true; break;
	case 'p': set_proxy( optarg ); break;
	case 't': tolerant = 1; break;
	case '?': 
	default:
	    printf( "Try `%s --help' for more information.\n", progname );
	    exit(-1);
	}
    }
    if( optind == (argc-2) ) {
	hostport = argv[optind];
	apath = argv[optind+1];
    } else if( optind == (argc-1) ) {
	/* Single argument: see whether it is a URL */
	if( strncmp( argv[optind], "http://", 7 ) == 0 ) {
	    char *abspath = strchr( argv[optind]+7, '/' );
	    if( abspath != NULL ) {
		/* Got a path segment.
		 * FIXME: leak */
		apath = strdup(abspath);
		*abspath = '\0';
	    }
	    hostport = argv[optind] + 7;
	} else {
	    hostport = argv[optind];
	}
    } else if( optind == argc ) {
	hostport = NULL;
	apath = NULL;
    } else {
	usage( );
	exit( -1 );
    }
    if( hostport != NULL ) {
	open_connection( hostport, apath );
#ifdef HAVE_ADD_HISTORY
	{ 
	    char *cmd;
	    if( apath == NULL) {
		CONCAT2( cmd, "open ", hostport );
	    } else {
		CONCAT4( cmd, "open ", hostport, " ", apath );
	    }
	    add_history( cmd );
	    free( cmd );
	}
#endif
    }
}

static char * read_command(void) 
{
    static char prompt[BUFSIZ];
    if( path ) {
	snprintf( prompt, BUFSIZ, "dav:%s%c ", path,
		  dav_collection?'>':'?' );
    } else {
	sprintf( prompt, "dav:!> " );
    }
    return readline( prompt ); 
}

static int execute_command( const char *line ) 
{
    const struct command *cmd;
    char **tokens;
    int argcount, ret = 0;
    tokens = parse_command( line, &argcount );
    if( argcount == 0 ) {
	free( tokens );
	return 0;
    }
    argcount--;
    cmd = get_command( tokens[0] );
    if( cmd == NULL ) {
	printf( "Unrecognised command. Type 'help' for a list of commands.\n");
    } else if( argcount < cmd->min_args ) {
	printf( "The `%s' command requires %d argument%s",
		tokens[0], cmd->min_args, cmd->min_args==1?"":"s" );
	if( cmd->short_help ) {
	    printf( ":\n  %s : %s\n", cmd->call, cmd->short_help );
	} else {
	    printf( ".\n" );
	}
    } else if( argcount > cmd->max_args ) {
	printf( "The `%s' command takes ", tokens[0] );
	if( cmd->max_args ) {
	    printf( "at most %d argument%s", 
		    cmd->max_args, cmd->max_args==1?"":"s" );
	} else {
	    printf( "no arguments" );
	}	    
	if( cmd->short_help ) {
	    printf( ":\n" "  %s : %s\n", cmd->call, cmd->short_help );
	} else {
	    printf( ".\n" );
	}
    } else if( !have_connection && cmd->needs_connection ) {
	printf( "The `%s' command can only be used when connected to the server.\n"
		"Try running `open' first (see `help open' for more details).\n", tokens[0] );
    } else if( cmd->id == cmd_quit ) {
	ret = -1;
    } else {
	/* Cast away */
	void (*take0)(void)=cmd->handler, 
	    (*take1)( const char * )=cmd->handler, 
	    (*take2)( const char *, const char * )=cmd->handler,
	    (*takeV)( int, const char ** )=cmd->handler;
	/* with a nod in the general direction of apache */
	switch( cmd->max_args ) {
	case 0: (*take0)(); break;
	case 1: /* tokens[1]==NULL if argcount==0 */
	    (*take1)( tokens[1] ); break; 
	case 2: 
	    if( argcount <=1 ) {
		(*take2)( tokens[1], NULL );
	    } else {
		(*take2)( tokens[1], tokens[2] );
	    }
	    break;
	case CMD_VARY:
	    (*takeV)( argcount, (const char **) &tokens[1] );
	default:
	    break;
	}
    }
    split_string_free( tokens );
    return ret;
}

static RETSIGTYPE quit_handler( int sig ) {
    /* Reinstall handler */
    if( child_running ) {
	/* The child gets the signal anyway... it can deal with it.
	 * Proper way is probably to ignore signals while child is
	 * running? */
	signal( sig, quit_handler );
	return;
    } else {
	printf( "Terminated by signal %d.\n", sig );
	if( have_connection ) {
	    close_connection( );
	}
	exit( -1 );
    }
}

void init_signals( void ) {
    signal( SIGPIPE, SIG_IGN );
    signal( SIGTERM, quit_handler );
    signal( SIGABRT, quit_handler );
    signal( SIGQUIT, quit_handler );
    signal( SIGINT, quit_handler );
}

static void init_netrc( void ) {
#ifdef ENABLE_NETRC
    char *netrc;
    CONCAT2( netrc, getenv("HOME"), "/.netrc" );
    netrc_list = parse_netrc( netrc );
#endif
}

static void init_rcfile( void ) 
{
    char *rcfile, buf[BUFSIZ];
    struct stat st;
    FILE *f;
    CONCAT2( rcfile, getenv("HOME"), "/.cadaverrc" );
    if( stat( rcfile, &st ) != 0 ) {
	DEBUG( DEBUG_FILES, "No rcfile.\n" );
    } else {
	f = fopen( rcfile, "r" );
	if( f == NULL ) {
	    printf( "Could not read rcfile %s: %s\n", rcfile, 
		    strerror(errno) );
	} else {
	    for(;;) {
		if( fgets( buf, BUFSIZ, f ) != NULL ) {
		    STRIP_EOL( buf );
		    execute_command( buf );
		} else {
		    break;
		}
	    }
	    fclose( f );
	}
    }
    free( rcfile );
}

#ifdef HAVE_LIBREADLINE

static char **completion( char *text, int start, int end )
{
    char **matches = NULL;
    
    if( start == 0 ) {
	matches = completion_matches( text, command_generator );
    } else {
	char *sep = strchr( rl_line_buffer, ' ' );
	/* Find command name, so we can work out whether to do local
	 * or remote completion */
	if( sep != NULL ) {
	    char *cname = strndup( rl_line_buffer, sep - rl_line_buffer );
	    const struct command *cmd;
	    cname[sep - rl_line_buffer] = '\0';
	    cmd = get_command( cname );
	    if( cmd != NULL ) { 
		switch( cmd->scope ) {
		case parmscope_none:
		    break;
		case parmscope_local:
		    matches = completion_matches( text, filename_completion_function );
		    break;
		case parmscope_option:
		    /* TODO */
		    break;
		case parmscope_remote:
		    /* TODO */
		    break;
		}
	    }
	}		    
    }
    return matches;
}

#endif

void output( enum output_type t, const char *fmt, ... ) 
{
    va_list params;
    if( t == o_finish ) {
	switch( out_state ) {
	case out_transfer_plain:
	    printf( "] " );
	    break;
	default:
	    putchar( ' ' );
	    break;
	}
    }
    va_start( params, fmt );
    vfprintf( stdout, fmt, params );
    va_end( params );
    fflush( stdout );
    switch( t ) { 
    case o_start:
	out_state = out_incommand;
	break;
    case o_transfer:
	out_state = out_transfer_start;
	break;
    case o_finish:
	out_state = out_none;
	break;
    }
}

static void init_readline() 
{
#ifdef HAVE_LIBREADLINE
    rl_readline_name = "cadaver";
    rl_attempted_completion_function = (CPPFunction *)completion;
#endif
}

#ifndef HAVE_LIBREADLINE
char *readline( const char *prompt )
{
    static char buf[256];
    char *ret;
    printf( "%s", prompt );
    ret = fgets( buf, 256, stdin );
    if( ret ) {
	STRIP_EOL( buf );
	return strdup(buf);
    } else {
	return NULL;
    }
}
#endif

static void init_options()
{
    char lockowner[BUFSIZ];
    
    /* set this here so they can override it */
    snprintf( lockowner, BUFSIZ,
	      "mailto:%s@%s", getenv("USER"), getenv("HOSTNAME") );
    set_option( opt_lockowner, strdup(lockowner) );
    
    set_option( opt_editor, NULL );

}

int main( int argc, char *argv[] ) 
{
    int ret = 0;
    char *lockstore, *home = getenv("HOME");

    path = NULL;
    have_connection = false;
    progname = argv[0];
    debug_mask = 0;

    if( !home ) {
	/* Show me the way to go home... */
	printf( "Environment variable HOME needs to be set!\n" );
	return -1;
    }

    CONCAT2(lockstore, home, "/.davlocks" );

    /* Options before rcfile, so rcfile settings can
     * override defaults */
    init_options();
    init_netrc();

    init_signals();
    init_locking(lockstore);

    init_rcfile();
    
    parse_args( argc, argv, &server );

    init_readline();

    while(ret == 0) {
	char *cmd;
	cmd = read_command();
	if( cmd == NULL ) {
	    /* Is it safe to do this... they just closed stdin, so
	     * is it bad to write to stdout? */
	    putchar( '\n' );
	    ret = 1;
	} else {
#ifdef HAVE_ADD_HISTORY
	    if( strcmp( cmd, "" ) != 0 ) add_history( cmd );
#endif
	    ret = execute_command(cmd);
	    free( cmd );
	}
    }
    if( have_connection ) {
	close_connection( );
    }
    finish_locking(lockstore);

    return 0;
}

void fe_connection( fe_conn_status status ) {
    switch( out_state ) {
    case out_none:
	switch( status ) {
	case fe_namelookup:
	    printf( "Looking up hostname...\n" );
	    break;
	case fe_connecting:
	    printf( "Connecting to server... " );
	    fflush( stdout );
	    break;
	case fe_connected:
	    printf( "connected.\n" );
	    break;
	}
	break;
    case out_incommand:
	/* fall-through */
    case out_transfer_start:
	switch( status ) {
	case fe_namelookup:
	    /* should never happen */
	    break;
	case fe_connecting:
	    printf( " (reconnecting...");
	    break;
	case fe_connected:
	    printf( "done)" );
	    break;
	}
	break;
    case out_transfer_plain:
	switch( status ) {
	case fe_namelookup:
	    break;
	case fe_connecting:
	    printf( "] reconnecting: " );
	    break;
	case fe_connected:
	    printf( "okay [" );
	    break;
	}
	break;
    case out_transfer_pretty:
	switch( status ) {
	case fe_namelookup:
	    break;
	case fe_connecting:
	    printf( "\rTransfer timed out, reconnecting... " );
	    break;
	case fe_connected:
	    printf( "okay." );
	    break;
	}
	break;	
    }
    fflush( stdout );
}

/* From ncftp.
   This function is (C) 1995 Mike Gleason, (mgleason@NcFTP.com)
 */
static void 
sub_timeval( struct timeval *tdiff, struct timeval *t1, struct timeval *t0)
{
    tdiff->tv_sec = t1->tv_sec - t0->tv_sec;
    tdiff->tv_usec = t1->tv_usec - t0->tv_usec;
    if (tdiff->tv_usec < 0) {
	tdiff->tv_sec--;
	tdiff->tv_usec += 1000000;
    }
}

/* Smooth progress bar.
 * Doesn't update the bar more than once every 100ms, since this 
 * might give flicker, and would be bad if we are displaying on
 * a slow link anyway.
 */
static void pretty_progress_bar( size_t progress, size_t total )
{
    int len, n;
    double pc;
    static struct timeval last_call = {0};
    struct timeval this_call;
    if( progress < total && gettimeofday( &this_call, NULL ) == 0 ) {
	struct timeval diff;
	sub_timeval( &diff, &this_call, &last_call );
	if( diff.tv_sec == 0 && diff.tv_usec < 100000 ) {
	    return;
	}
	last_call = this_call;
    }
    if( progress == 0 || total == 0 ) {
	pc = 0;
    } else {
	pc = (double)progress / total;
    }
    len = pc * 30;
    printf( "\rProgress: [" );
    for( n = 0; n<30; n++ ) {
	putchar( (n<len-1)?'=':
		 (n==(len-1)?'>':' ') );
    }
    printf( "] %5.1f%% of %d bytes", pc*100, total );
    fflush( stdout );
}

void fe_transfer_progress( size_t progress, size_t total ) 
{
    if( in_curses ) return;
    switch( out_state ) {
    case out_none:
    case out_incommand:
	/* Do nothing */
	return;
    case out_transfer_start:
	if( isatty( STDOUT_FILENO ) && total > 0 ) {
	    out_state = out_transfer_pretty;
	    putchar( '\n' );
	    pretty_progress_bar( progress, total );
	} else {
	    out_state = out_transfer_plain;
	    printf( " [." );
	}
	break;
    case out_transfer_pretty:
	pretty_progress_bar( progress, total );
	break;
    case out_transfer_plain:
	putchar( '.' );
	fflush( stdout );
	break;
    }
}

int fe_login( fe_login_context ctx, const char *realm, const char *hostname,
	      char **username, char **password ) 
{
    if( in_curses ) return -1;
    switch( out_state ) {
    case out_transfer_pretty:
	putchar( '\n' );
	/*** fall-through ***/
    case out_none:
	break;
    case out_incommand:
	/* fall-through */
    case out_transfer_start:
	putchar( ' ' );
	break;
    case out_transfer_plain:
	printf( "] " );
	break;
    }
    printf( "Authentication required for %s on %s `%s':\n", 
	    realm, ctx==fe_login_server?"server":"proxy", hostname );
    *username = readline( "Username: " );
    if( *username == NULL ) {
	/* TODO: UI is crap here */
	printf( "\rAuthentication aborted!\n" );
	return -1;
    } else {
	*password = strdup(getpass( "Password: " ));
    }
    switch( out_state ) {
    case out_transfer_start:
    case out_incommand:
	printf( "Retrying:" );
	fflush( stdout );
	break;
    case out_transfer_plain:
	printf( "Retrying [" );
	fflush( stdout );
	break;
    default:
	break;
    }
    return 0;
}
