/* 
   cadaver, command-line DAV client
   Copyright (C) 1999-2000, Joe Orton <joe@orton.demon.co.uk>, 
   except where otherwise indicated.
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: commands.c,v 1.46 2000/03/03 23:19:31 joe Exp $
*/

/* Some UI guidelines:
 *  1. Use dispatch, or out_* to do UI. This makes it CONSISTENT.
 *  2. Get some feedback on the screen before making any requests
 *     to the server. Tell them what is going on: remember, on a slow
 *     link or a loaded server, a request can take AGES to return.
 */

#include <config.h>

#include <sys/types.h>

#include <sys/stat.h>
#include <sys/wait.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <fcntl.h>
#include <errno.h>

/* readline requires FILE *, silly thing */
#include <stdio.h>

#ifdef HAVE_READLINE_H
#include <readline.h>
#elif HAVE_READLINE_READLINE_H
#include <readline/readline.h>
#endif

#include <time.h>

#ifndef HAVE_SNPRINTF
#include "snprintf.h"
#endif

#include "string_utils.h"
#include "uri.h"
#include "httpdav.h"
#include "davlocks.h"

#include "dates.h"
#include "basename.h"
#include "dirname.h"

#include "cadaver.h"
#include "commands.h"
#include "davprops.h"
#include "options.h"

int yesno(void);

/* Local variables */
bool child_running; /* true when we have a child running */

static time_t current_time;

/* Command alias mappings */
const static struct {
    enum command_id id;
    const char *name;
} command_names[] = {
    /* The direct mappings */
#define C(n) { cmd_##n, #n }
    C(ls), C(cd), C(quit), C(open), C(close), C(set), C(unset), C(pwd),
    C(help), C(put), C(get), C(mkcol), C(delete), C(move), C(copy),
    C(less), C(cat), C(lpwd), C(lcd), C(lls), C(echo), C(quit),
    C(mget), C(mput), C(rmcol), C(lock), C(unlock), C(discover), C(steal),
    C(showlocks), C(version), 
#if 0
C(propedit), 
#endif
C(propnames), C(edit),
#undef C
    /* And now the real aliases */
    { cmd_less, "more" }, { cmd_mkcol, "mkdir" }, 
    { cmd_delete, "rm" }, { cmd_copy, "cp"}, { cmd_move, "mv" }, 
    { cmd_help, "h" }, { cmd_help, "?" },
    { cmd_quit, "exit" }, { cmd_quit, "bye" },
    { cmd_unknown, NULL }
};    

inline static void out_start( const char *verb, const char *noun )
{
    output( o_start, "%s `%s':", verb, noun);
}

inline static void out_success( void )
{
    output( o_finish, "succeeded.\n" );
}

struct dav_lock *lock_list = NULL;

inline static void out_result( int ret )
{
    switch( ret ) {
    case PROTO_OK:
	out_success();
	break;
    case PROTO_AUTH:
	output( o_finish, "authentication failed.\n" );
	break;
	/* TODO: others */
    default:
	output( o_finish, "failed:\n%s\n", http_error );
	break;
    }
}

inline static int out_handle( int ret )
{
    out_result( ret );
    return (ret == PROTO_OK);
}

/* The actual commands */

void init_locking( const char *lockstore ) {
    DEBUG( DEBUG_LOCKS, "init locking.\n" );
    dav_lock_list = &lock_list;
    /* TODO: read in lock list from ~/.davlocks */
}

void finish_locking( const char *lockstore ) {
    DEBUG( DEBUG_LOCKS, "Finish locking.\n" );
    /* TODO: write out lock list to ~/.davlocks */
}

/* Auxiliaries */

static void 
map_multi( void (*func)( const char * ), int argc, const char *argv[] );

static void simple_copy( const char *from, const char *to );
static void simple_put( const char *local, const char *remote );
static void simple_move( const char *src, const char *dest );

static void do_copymove( int argc, const char *argv[],
		  const char *v1, const char *v2,
		  void (*cb)( const char *, const char *) );

static char *clever_path( const char *path, const char *src, 
			  const char *dest );

static char *choose_pager(void);

static void dispatch( const char *verb, const char *filename, 
		      int (*func)( const char *), const char *pass );
static void dispatch2( const char *verb, const char *fn1, const char *fn2, 
		       int (*func)( const char *, const char *) );

static void display_help_message( void );

static void print_lock( struct dav_lock *lock );

#ifdef HAVE_LIBREADLINE

/* Command name generator for readline.
 * Copied almost verbatim from the info doc */
char *command_generator( char *text, int state )
{
    static int index, len;
    const char *name;

    if( !state ) {
	index = 0;
	len = strlen(text);
    }

    while( (name = command_names[index].name) != NULL ) {
	index++;
	if( strncmp( name, text, len ) == 0 ) {
	    return strdup(name);
	}
    }
    return NULL;
}

#endif

const struct command *get_command( const char *name ) {
    int n, m;
    for( n = 0; command_names[n].name != NULL; n++ ) {
	if( strcasecmp( command_names[n].name, name ) == 0 ) {
	    for( m = 0; commands[m].id != cmd_unknown; m++ ) {
		if( commands[m].id == command_names[n].id )
		    return &commands[m];
	    }
	    return NULL;
	}
    }
    return NULL;
}

static char *format_time( time_t when )
{
    const char *fmt;
    static char ret[256];
    struct tm *local;

    /* from GNU fileutils... this section is 
     *  Copyright (C) 85, 88, 90, 91, 1995-1999 Free Software Foundation, Inc.
     */
    if( current_time > when + 6L * 30L * 24L * 60L * 60L	/* Old. */
	|| current_time < when - 60L * 60L ) {
	/* The file is fairly old or in the future.
	   POSIX says the cutoff is 6 months old;
	   approximate this by 6*30 days.
	   Allow a 1 hour slop factor for what is considered "the future",
	   to allow for NFS server/client clock disagreement.
	   Show the year instead of the time of day.  */
	fmt = "%b %e  %Y";
    } else {
	fmt = "%b %e %H:%M";
    }
    /* end FSF copyrighted section. */
    local = localtime(&when);
    if( local != NULL ) {
	if( strftime( ret, 256, fmt, local ) ) {
	    return ret;
	}
    }
    return "???";
}

static int compare_alpha( const void *f1_p, const void *f2_p )
{
    /* Gotta love C, eh? */
    const struct proto_file *f1 = *(const void **)f1_p, 
	*f2 = *(const void **)f2_p;
    /* Sort collections before files, otherwise alphabetically */
    if( f1->type == proto_dir ) {
	if( f2->type != proto_dir ) {
	    return -1;
	} else {
	    return strcmp( f1->filename, f2->filename );
	}
    } else {
	if( f2->type != proto_dir ) {
	    return strcmp( f1->filename, f2->filename );
	} else {
	    return 1;
	}
    }
}

/* HAACCKAKCKAKCKACKACKAKCKACKA. 
 *
 * POSIX has qsort(), qsort works on an array.  Convert our linked
 * list into an array, sort it, then convert it back into a linked
 * list again. Adds an extra O(3N) to the sort speed. An O(N) since we
 * don't know how long the files list is...  could be optimized.
 */
static void sort_files_list( struct proto_file **files )
{
    struct proto_file **array, *current;
    int count = 0, num;
    for( current = *files; current != NULL; current = current->next ) {
	count++;
    }
    array = malloc( sizeof(struct proto_file *) * (count+1) );
    num = 0;
    for( current = *files; current != NULL; current = current->next ) {
	array[num] = current;
	num++;
    }
    /* TODO: select comparison function to choose what type of sort
     * to do. */
    qsort( array, count, sizeof(struct proto_file *), compare_alpha );
    for( num = 0; num < count-1; num++ )
	array[num]->next = array[num+1];
    array[count-1]->next = NULL;
    *files = array[0];
    free( array );
}

static void display_ls_line( struct proto_file *file )
{
    char *restype;
    switch( file->type ) {
    case proto_file: restype = ""; break;
    case proto_link: restype = "Ref:"; break;
    case proto_dir: restype = "Coll:"; break;
    default: restype = "???"; break;
    }
    printf( "%5s %-30s %10d  %s\n", restype, file->filename, 
	    file->size, format_time(file->modtime) );
}

void execute_ls( const char *remote ) {
    int ret;
    char *real_remote;
    struct proto_file *files, *current, *next;
    if( remote != NULL ) {
	real_remote = resolve_path( path, remote, true );
    } else {
	real_remote = strdup( path );
    }
    out_start( "Listing collection", real_remote );
    ret = dav_fetch( real_remote, 1, 0, &files );
    if( ret == PROTO_OK ) {
	/* Easy this, eh? */
	if( files == NULL ) {
	    output( o_finish, "collection is empty.\n" );
	} else {
	    out_success();
	    current_time = time(NULL);
	    sort_files_list( &files );
	    for( current = files; current!=NULL; current = next ) {
		next = current->next;
		display_ls_line( current );
		free( current->filename );
		free( current );
	    }
	}
    } else {
	out_result( ret );
    }
    free( real_remote );
}

/* A bit like Haskell's map function... applies func to each
 * of the first argc items in argv */
static void 
map_multi( void (*func)( const char * ), int argc, const char *argv[] ) {
    int n;
    for( n = 0; n < argc; n++ ) {
	(*func)( argv[n] );
    }
}

void multi_mkcol( int argc, const char *argv[] ) {
    map_multi( execute_mkcol, argc, argv );
}

void multi_delete( int argc, const char *argv[] ) {
    map_multi( execute_delete, argc, argv );
}

void multi_rmcol( int argc, const char *argv[] ) {
    map_multi( execute_rmcol, argc, argv );
}

void multi_less( int argc, const char *argv[] ) {
    map_multi( execute_less, argc, argv );
}

void multi_cat( int argc, const char *argv[] ) {
    map_multi( execute_cat, argc, argv );
}

static void dispatch( const char *verb, const char *filename, 
		      int (*func)( const char *), const char *arg ) {
    out_start( verb, filename );
    out_result( (*func)( arg ) );
}

static void dispatch2( const char *verb, const char *fn1, const char *fn2,
		       int (*func)(const char *, const char *) ) {
    output( o_start, "%s `%s' to `%s':", verb, fn1, fn2 );
    out_result( (*func)( fn1, fn2 ) );
}

static char * getowner(void) {
#if 0
    /* mod_dav breaks if you submit an owner */
    return strdup("");
#else
    char *ret, *owner = get_option(opt_lockowner);
    if( owner ) {
	CONCAT3( ret, "<href>", owner, "</href>" );
	return ret;
    } else {
	return NULL;
    }
#endif
}

/* Resolve path, appending trailing slash if resource is a
 * collection. This function MAY make a request to the server.
 * TODO: caching, so it MAY not, too.
 */
static char *true_path( const char *res ) {
    char *full;
    full = resolve_path( path, res, false );
    if( is_collection(full) ) {
	if( !uri_has_trailing_slash(full) ) {
	    char *tmp = malloc(strlen(full)+2);
	    strcpy( tmp, full );
	    strcat( tmp, "/" );
	    free( full );
	    full = tmp;
	}
    }
    return full;
}

static const char *get_lockscope( enum dav_lock_scope s ) {
    switch( s ) {
    case dav_lockscope_exclusive: return "exclusive";
    case dav_lockscope_shared: return "shared";
    default: return "unknown";
    }
}

static const char *get_locktype( enum dav_lock_type t ) {
    if( t == dav_locktype_write ) {
	return "write";
    } else {
	return "unknown";
    }
}

static const char *get_timeout( int t ) {
    static char buf[128];
    switch( t ) {
    case DAV_TIMEOUT_INFINITE: return "infinite";
    case DAV_TIMEOUT_INVALID: return "invalid";
    default:
	sprintf( buf, "%d", t );
	return buf;
    }
}

static char *get_depth( int d ) {
    switch( d ) {
    case DAV_DEPTH_INFINITE:
	return "infinity";
    case 0:
	return "0";
    case 1:
	return "1";
    default:
	return "invalid";
    }
}

static void print_lock( struct dav_lock *lock ) {
    printf( "Lock <%s> on %s:\n"
	    "  Scope: %s  Type: %s  Timeout: %s\n"
	    "  Owner: %s  Depth: %s\n", 
	    lock->token, lock->uri?lock->uri:"(unknown)",
	    get_lockscope( lock->scope ),
	    get_locktype( lock->type ), get_timeout( lock->timeout ),
	    lock->owner?lock->owner:"(none)", get_depth(lock->depth) );
}

void execute_discover( const char *res ) {
    struct lock_result *list, *cur;
    char *real_remote;
    int ret;
    real_remote = resolve_path( path, res, false );
    out_start( "Discovering locks on", res );
    ret = dav_lock_discover( real_remote, &list );
    switch( ret ) {
    case PROTO_OK:
	out_success();
	for( cur = list; cur != NULL; cur=cur->next ) {
	    /* TODO: I don't think we can actually find out the URI which
	     * the lock applies to. */
	    print_lock( &cur->lock );
	}
	/* TODO: free the list */
	break;
    case PROTO_NONE:
	output( o_finish, "no locks found.\n" );
	break;
    default:
	out_result( ret );
	break;
    }
    free( real_remote );
}

void execute_steal( const char *res ) {
    struct lock_result *list, *cur;
    char *real_remote;
    int ret;
    real_remote = resolve_path( path, res, false );
    out_start( "Stealing locks on", res );
    ret = dav_lock_discover( real_remote, &list );
    switch( ret ) {
    case PROTO_OK:
	out_success();
	for( cur = list; cur != NULL; cur=cur->next ) {
	    cur->lock.uri = cur->href;
	    printf( "%s: <%s>\n", cur->href, cur->lock.token );
	    dav_lock_add( &lock_list, &cur->lock );
	}
	/* TODO: free the list */
	break;
    case PROTO_NONE:
	output( o_finish, "no locks found.\n" );
	break;
    default:
	out_result( ret );
	break;
    }
}

void execute_showlocks( ) {
    struct dav_lock *lock;
    if( lock_list == NULL ) {
	printf( "No owned locks.\n" );
    } else {
	printf( "List of owned locks:\n" );
	for( lock = lock_list; lock!= NULL; lock = lock->next ) {
	    print_lock( lock );
	}
    }
}

void execute_lock( const char *res ) {
    char *real_remote;
    struct dav_lock *lock;
    lock = malloc( sizeof(struct dav_lock) );
    if( !lock ) return;
    out_start( "Locking", res );
    real_remote = true_path( res );
    lock->type = dav_locktype_write;
    lock->scope = dav_lockscope_exclusive;
    lock->owner = getowner();
    if( uri_has_trailing_slash(real_remote) ) {
	lock->depth = DAV_DEPTH_INFINITE;
    } else {
	lock->depth = 0;
    }
    lock->uri = real_remote;
    if( out_handle( dav_lock(lock) ) ) {
	/* Locktoken: <%s>\n", lock->token ); */
	dav_lock_add( &lock_list, lock );
    } else {
	free( real_remote );
	free( lock );
    }
}

void execute_unlock( const char *res ) {
    struct dav_lock *lock;
    char *real_remote;
    int isdummy = 0;
    out_start( "Unlocking", res );
    real_remote = true_path( res );
    lock = dav_lock_find( lock_list, real_remote );
    if( !lock ) {
	lock = malloc( sizeof(struct dav_lock) );
	lock->token = readline( "Enter locktoken: " );
	if( !lock->token || strlen(lock->token) == 0 ) {
	    free( lock );
	    return;
	}
	lock->uri = real_remote;
	isdummy=1;
    }
    out_result( dav_unlock( lock ) );
    if( isdummy ) {
	free( lock );
    } else {
	dav_lock_remove( &lock_list, lock );
	dav_lock_free( lock );
    }
}

void execute_mkcol( const char *filename ) {
    char *remote;
    remote = resolve_path( path, filename, true );
    dispatch( "Creating", filename, dav_mkcol, remote );
    free( remote );
}

void execute_propnames( const char *res ) {
    char *remote, **names = NULL;
    remote = resolve_path( path, res, false );
    out_start( "Fetching property names", res );
    if( out_handle( dav_prop_getnames( remote, &names ) ) ) { 
	char **name;
	for( name = names; *name != NULL; name++ ) {
	    printf( "Property: %s\n", *name );
	    free( *name );
	}
	free( names );
    }
    free( remote );
}

void execute_delete( const char *filename ) 
{
    char *remote = resolve_path( path, filename, false );
    out_start( "Deleting", filename );
    if( is_collection( remote ) ) {
	output( o_finish, 
"is a collection resource.\n"
"The `rm' command cannot be used to delete a collection.\n"
"Use `rmcol %s' to delete this collection and ALL its contents.\n", 
filename );
    } else {
	out_result( http_delete(remote) );
    }
    free( remote );
}

void execute_rmcol( const char *filename ) {
    char *remote;
    remote = resolve_path( path, filename, true );
    out_start( "Deleting collection", filename );
    if( !is_collection( remote ) ) {
	output( o_finish, "is not a collection.\n"
		"The `rmcol' command can only be used to delete collections.\n"
		"Use `rm %s' to delete this resource.\n", filename );
    } else {
	out_result( http_delete(remote) );
    }
    free( remote );
}

/* Like resolve_path except more intelligent. */
static char *clever_path( const char *path, const char *src, 
			  const char *dest ) 
{
    char *ret;
    bool src_is_coll, dest_is_coll;
    dest_is_coll = (dest[strlen(dest)-1] == '/');
    src_is_coll = (src[strlen(src)-1] == '/');
    if( strcmp( dest, "." ) == 0 ) {
	ret = resolve_path( path, base_name( src ), false );
    } else if( strcmp( dest, ".." ) == 0 ) {
	char *parent;
	parent = uri_parent( path );
	ret = resolve_path( parent, base_name( src ), false );
	free( parent );
    } else if( !src_is_coll && dest_is_coll ) {
	/* Moving a file to a collection... the destination should
	 * be the basename of file concated with the collection. */
	char *tmp;
	tmp = resolve_path( path, dest, true );
	CONCAT2( ret, tmp, base_name( src ) );
	free( tmp );
    } else {
	ret = resolve_path( path, dest, false );
    }
    return ret;
}

static void cat_callback( void *userdata, const char *buf, const size_t len ) {
    FILE *f = userdata;
    if( fwrite( buf, len, 1, f ) < 0 ) {
	perror( "fwrite" );
    }
}

static char *choose_pager(void) {
    struct stat st;
    char *tmp;
    tmp = getenv("PAGER");
    if( tmp != NULL ) {
	return tmp;
    } else if( stat( "/usr/bin/less", &st ) == 0 ) {
	return "/usr/bin/less";
    } else if( stat( "/bin/less", &st ) == 0 ) {
	return "/bin/less";
    } else {
	return "/bin/more";
    }
}

static FILE *spawn_pager( const char *pager ) {
    /* Create a pipe */
    return popen( pager, "w" );
}

static void kill_pager( FILE *p ) {
    /* This blocks until the pager quits. */
    pclose( p );
}

void execute_less( const char *resource ) {
    char *real_res;
    char *pager;
    FILE *p;
    real_res = resolve_path( path, resource, false );
    pager = choose_pager();
    printf( "Displaying `%s':\n", real_res );
    p = spawn_pager(pager);
    if( p == NULL ) {
	printf( "Error! Could not spawn pager `%s':\n%s\n", pager,
		strerror(errno) );
    } else {
	child_running = true;
	http_read_file( real_res, 0, cat_callback, p );
	kill_pager( p ); /* Blocks until the pager quits */
	child_running = false;
    }
}

void execute_cat( const char *resource ) 
{
    char *real_res = resolve_path( path, resource, false );
    printf( "Displaying `%s':\n", real_res );
    if( http_read_file( real_res, 0, cat_callback, stdout ) != PROTO_OK ) {
	printf( "Failed: %s\n", http_error );
    }	
}

/* Returns non-zero if given resource is not a collection resource.
 * This function MAY make a request to the server. */
int is_collection( const char *res ) {
    struct proto_file *files = NULL;
    int ret = 0;
    /* the zero-length filename check here may be unnecessary */
    if( dav_fetch( res, 0, 1, &files ) == PROTO_OK ) {
	if( files != NULL && files->filename[0] == '\0' ) {
	    if( files->type == proto_dir ) {
		ret = 1;
	    } else {
		strcpy( http_error, "Not a collection resource." );
	    }
	} else {
	    /* FIXME: this error occurs when you do open /foo and get
	     * the response for /foo/ back */
	    strcpy( http_error, "Did not find a collection resource." );
	}
    }
    dav_free_fileslist( files );
    return ret;
}

void multi_copy( int argc, const char *argv[] ) {
    do_copymove( argc, argv, "copying", "copy", simple_copy );
}

void multi_move( int argc, const char *argv[] ) {
    do_copymove( argc, argv, "moving", "move", simple_move );
}

static void do_copymove( int argc, const char *argv[],
		  const char *v1, const char *v2,
		  void (*cb)( const char *, const char *) ) {
    /* We are guaranteed that argc > 2... */
    char *dest;

    dest = resolve_path(path, argv[argc-1], true );
    if( is_collection( dest ) ) {
	int n;
	char *real_src, *real_dest;
	for( n = 0; n < argc-1; n++ ) {
	    real_src = resolve_path(path, argv[n], false );
	    real_dest = clever_path(path, argv[n], dest);
	    if( strcmp( real_src, real_dest ) == 0 ) {
		printf( "%s: %s and %s are the same resource.\n", v2,
			real_src, real_dest );
	    } else {
		(*cb)( real_src, real_dest );
	    }
	    free( real_src );
	    free( real_dest );
	}
    } else if( argc > 2 ) {
	printf( "When %s multiple resources, the last argument must be a collection.\n", v1 );
    } else {
	char *rsrc, *rdest;
	rsrc = resolve_path( path, argv[0], false );
	rdest = resolve_path( path, argv[1], false );
	/* Simple */
	(*cb) ( rsrc, rdest );
	free( rsrc );
	free( rdest );
    }
    free( dest );
}

static void simple_move( const char *src, const char *dest ) {
    dispatch2( "Moving", src, dest, dav_move );
}

static void simple_copy( const char *src, const char *dest ) {
    dispatch2( "Copying", src, dest, dav_copy );
}

/* Returns absolute filename which is 'filename' relative to 
 * (absolute filename) 'path'. e.g.
 *    resolve_path( "/dav/", "asda" ) == "/dav/asda"
 *    resolve_path( "/dav/", "/asda" ) == "/asda"
 * Also removes '..' segments, e.g.
 *    resolve_path( "/dav/foobar/", "../fish" ) == "/dav/fish"
 * If isdir is true, ensures the return value has a trailing slash.
 */
char *resolve_path( const char *path, const char *filename,
			   bool isdir ) {
    char *ret, *pnt;
    if( *filename == '/' ) {
	/* It's absolute */
	ret = strdup( filename );
    } else if( strcmp( filename, "." ) == 0 ) {
	ret = strdup( path );
    } else {
	CONCAT2( ret, path, filename );
    }
    if( isdir && ret[strlen(ret)-1] != '/' ) {
	char *newret;
	CONCAT2( newret, ret, "/" );
	free( ret );
	ret = newret;
    }
    /* Sort out '..', etc... */
    do {
	pnt = strstr( ret, "/../" );
	if( pnt != NULL ) {
	    char *last;
	    /* Find the *previous* path segment, to overwrite...
	     *    /foo/../
	     *    ^- last points here */
	    if( pnt > ret ) {
		for( last = pnt-1; (last > ret) && (*last != '/'); last-- );
	    } else {
		last = ret;
	    }
	    memmove( last, pnt + 3, strlen(pnt+2) );
	} else {
	    pnt = strstr( ret, "/./" );
	    if( pnt != NULL ) {
		memmove( pnt, pnt+2, strlen(pnt+1) );
	    }
	}
    } while( pnt != NULL );
    return ret;    
}

void execute_get( const char *remote, const char *local ) {
    char *filename, *real_remote;
    real_remote = resolve_path( path, remote, false );
    if( local == NULL ) {
	struct stat st;
	/* Choose an appropriate local filename */
	if( stat( base_name(remote), &st ) == 0 ) {
	    /* File already exists... don't overwrite */
	    printf( "Enter local filename for `%s': ", real_remote );
	    filename = readline( NULL );
	    if( filename == NULL ) {
		free( real_remote );
		printf( "cancelled.\n" );
		return;
	    }
	} else {
	    filename = strdup( base_name( remote ) );
	}
    } else {
	filename = strdup( local );
    }
    /* Muck around like this because otherwise the connection status
     * messages and username/password prompts would mess up the nice
     * progress dots. */
    output( o_transfer, "Downloading `%s' to %s:", real_remote, filename );
    out_result( http_get( filename, real_remote, 0, O_CREAT, 0644 ) );
    free( real_remote );
    free( filename );
}

static int run_editor( const char *filename ) {
    char editcmd[BUFSIZ], *editor;
    struct stat before_st, after_st;
    editor = get_option( opt_editor );
    if( editor == NULL ) {
	editor = getenv("EDITOR");
	if( editor == NULL ) {
	    editor = "vi";
	}
    }
    snprintf( editcmd, BUFSIZ, "%s %s", editor, filename );
    if( stat( filename, &before_st ) ) {
	printf( "Could not stat file: %s\n", strerror(errno) );
	return -1;
    }
    printf( "Running editor: `%s'...\n", editcmd );
    system( editcmd );
    if( stat( filename, &after_st ) ) {
	printf( "Error! Could not examine temporary file: %s\n", 
		strerror(errno) );
	return -1;
    }
    if( before_st.st_mtime == after_st.st_mtime ) {
	/* File not changed. */
	printf( "No changes were made.\n" );
	return -1;
    } else {
	printf( "Changes were made.\n" );
	return 0;
    }	
}

/* TODO: this is great big heap of steaming trout.  */
void execute_edit( const char *remote ) 
{
    char *real_remote, *tmp;
    unsigned int can_lock; /* can we LOCK it? */
    struct dav_lock *lock;

    real_remote = resolve_path( path, remote, false );
    /* TODO: fetch 'supportedlock' and check whether we can really
     * lock it. */
    if( http_options( real_remote ) != PROTO_OK ) {
	printf( "Could not determining locking capabilities for `%s':\n%s\n", 
		remote, http_error );
	return;
    }
    can_lock = http_webdav_locking;
    /* Get a temporary filename.
     * FIXME: joe: my tmpname(3) man page says it is deprecated */
    tmp = tmpnam(NULL);
    if( tmp == NULL ) {
	printf( "Could not create temporary filename: %s\n", strerror(errno) );
	return;
    }
    if( can_lock ) {
	/* FIXME: copy'n'friggin'paste */
	lock = malloc( sizeof(struct dav_lock) );
	if( !lock ) {
	    printf( "Out of memory.\n" );
	    return;
	}
	lock->type = dav_locktype_write;
	lock->scope = dav_lockscope_exclusive;
	lock->owner = getowner();
	lock->depth = 0;
	lock->uri = strdup(real_remote);
	out_start( "Locking", remote );
	if( out_handle( dav_lock( lock ) ) ) {
	    dav_lock_add( &lock_list, lock );
	} else {
	    dav_lock_free( lock );
	    return;
	}
    } else {
	/* FIXME: HEAD and get the Etag/modtime */
    }
    /* FIXME: copy'n'friggin'paste. */
    output( o_transfer, "Downloading `%s'", real_remote );
    if( out_handle( http_get( tmp, real_remote, 0, O_CREAT | O_EXCL, 0600 ))) {
	if( run_editor( tmp ) ) {
	    /* Edit failed */
	} else {
	    int upload_okay = 0;
	    /* Edit succeeded */
	    do {
		output( o_transfer, "Uploading changes to `%s'", real_remote );
		/* FIXME: conditional PUT using fetched Etag/modtime if
		 * !can_lock */
		if( out_handle( http_put( tmp, real_remote, 0 ) ) ) {
		    upload_okay = 1;
		} else {
		    /* TODO: offer to save locally instead */
		    printf( "Try uploading again (y/n)? " );
		    if( !yesno() ) {
			upload_okay = 1;
		    }
		}
	    } while( !upload_okay );
	}
    }
    if( unlink( tmp ) ) {
	printf( "Could not remove temporary file `%s': %s\n", 
		tmp, strerror(errno) );
    }
    /* UNLOCK it again whether we succeed or failed in our mission */
    if( can_lock ) {
	output( o_start, "Unlocking `%s':", remote );
	out_result( dav_unlock(lock) );
	dav_lock_remove( &lock_list, lock );
	dav_lock_free( lock );
    }

}

static void simple_put( const char *local, const char *remote ) 
{
    output( o_transfer, "Uploading %s to `%s':", local, remote );
    out_result( http_put( local, remote, false ) );
}

void execute_put( const char *local, const char *remote ) {
    char *real_remote;
    if( remote == NULL ) {
	real_remote = resolve_path( path, base_name( local ), false );
    } else {
	real_remote = resolve_path( path, remote, false );
    }
    simple_put( local, real_remote );
    free( real_remote );
}

/* this is getting too easy */
void multi_mput( int argc, const char *argv[] ) {
    for( ; argv[0] != NULL; argv++ ) {
	char *remote;
	remote = clever_path( path, argv[0], path );
	simple_put( argv[0], remote );
	free( remote );
    }    
}

void multi_mget( int argc, const char *argv[] ) {
    for( ; argv[0] != NULL; argv++ ) {
	execute_get( argv[0], NULL );
    }
}

void execute_lpwd( void ) {
    char pwd[BUFSIZ];
    if( getcwd( pwd, BUFSIZ ) == NULL ) {
	perror( "pwd" );
    } else {
	printf( "Local directory: %s\n", pwd );
    }
}

/* Using a hack, we get zero-effort multiple-argument 'lls' */
void execute_lls( int argc, char * const argv[] ) {
    int pid;
    pid = fork();
    switch( pid ) {
    case 0: /* child...  */
	/* The implicit typecast munging might screw things up in
	 * weird and wonderful ways, here? */
	execvp( "ls", &argv[-1] );
	printf( "Error executing ls: %s\n", strerror(errno) );
	exit( -1 );
	break;
    case -1: /* Error */
	printf( "Error forking: %s\n", strerror(errno) );
	break;
    default: /* Parent */
	wait(NULL);
	break;
    }
    return;
}


void execute_lcd( const char *path ) {
    const char *real_path;
    if( path ) {
	real_path = path;
    } else {
	real_path = getenv("HOME");
	if( !real_path ) {
	    printf( "Could not determine home directory from environment.\n" );
	    return;
	}
    }
    if( chdir( real_path ) ) {
	printf( "Could not change local directory:\nchdir: %s\n",
		strerror(errno) );
    }
}

void execute_pwd( void )
{
    printf( "Current collection is `%s', on server `%s'\n", path, 
	    server.hostname );
}

void execute_cd( const char *newpath ) {
    char *real_path;
    bool is_swap = false;
    bool is_coll;
    if( strcmp( newpath, "-" ) == 0 ) {
	is_swap = true;
	real_path = old_path;
    } else if( path == NULL ) {
	/* The first time we are executing a CD. */
	if( uri_has_trailing_slash( newpath ) ) {
	    real_path = strdup( newpath );
	} else {
	    CONCAT2( real_path, newpath, "/" );
	}
    } else {
	real_path = resolve_path( path, newpath, true );
    }
    if( set_path( real_path ) ) {
	/* Error */
	if( !is_swap ) free( real_path );
    } else {
	/* Success */
	if( !is_swap && old_path ) free( old_path );
	old_path = path;
	path = real_path;
    }
}

static void display_help_message( void ) {
    int n;
    printf( "Commands: \n" );
    
    for( n = 0; commands[n].id != cmd_unknown; n++ ) {
	if( commands[n].call && commands[n].short_help ) {
	    printf( " %-26s %s\n", commands[n].call, commands[n].short_help );
	}
    }
    
    printf( 
	"Aliases: rm=delete, mkdir=mkcol, mv=move, cp=copy, more=less, quit=exit=bye\n"
	);
    
}

void execute_help( const char *arg ) {
    if( !arg ) {
	display_help_message();
    } else {
	const struct command *cmd = get_command( arg );
	if( cmd ) {
	    printf( " `%s'   %s\n", cmd->call, cmd->short_help );
	    if( cmd->needs_connection ) {
		printf( "This command can only be used when connected to a server.\n" );
	    }
	} else {
	    printf( "Command name not known: %s\n", arg );
	}
    }
}

void execute_echo( int count, const char **args ) {
    const char **pnt;
    for( pnt = args; *pnt != NULL; pnt++ ) {
	printf( "%s ", *pnt );
    }
    putchar( '\n' );
}

