/* 
   WebDAV fetch
   Copyright (C) 1998-99, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: davfetch.c,v 1.20 2000/03/21 14:04:40 joe Exp $
*/

#include <config.h>

#include <sys/types.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "string_utils.h"
#include "dates.h"
#include "hip_xml.h"
#include "dav_207.h"
#include "protocol.h"
#include "httpdav.h"
#include "http_utils.h"
#include "uri.h"

/* PROPFIND response parsing */

#define DAV_ELM_getlastmodified 8
#define DAV_ELM_getcontentlength 9
#define DAV_ELM_resourcetype 10
#define DAV_ELM_collection 11
#define DAV_ELM_redirectref 12

/* The fetch context is the userdata to the element callback */
struct fetch_context {
    /* Do we want to include the resource identified by the 
     * Request-URI in the fetched list? */
    unsigned int fetch_target;
    /* The collection we did a PROPFIND on */
    const char *fetch_root;
    struct dav_207_parser *context207;
    struct proto_file **files;
};

static inline int get_depth( const char *href );
static char *strip_href( const char *root, const char *href );

#if 0
/* TODO */
static void http_get_content_charset( const char *name, const char *value );

char *http_content_charset;
#endif

static int check_context( hip_xml_elmid parent, hip_xml_elmid child ) 
{
    switch( parent ) {
    case DAV_ELM_prop:
	switch( child ) {
	case DAV_ELM_resourcetype: case DAV_ELM_getcontentlength: 
	case DAV_ELM_getlastmodified: case HIP_ELM_unknown:
	    return 0;
	default:
	}
	break;
    case DAV_ELM_propstat:
	if( child == DAV_ELM_prop ) {
	    return 0;
	}
	break;
    case DAV_ELM_resourcetype:
	switch( child ) {
	case DAV_ELM_collection:
	case DAV_ELM_redirectref:
	    return 0;
	default:
	}
	break;
    case HIP_ELM_unknown:
	if( child == HIP_ELM_unknown ) {
	    return 0;
	}
    default:
	break;
    }
    return -1;
}

/* Returns the depth of the file in the directory heirarchy,
 * i.e. 1 for /foo.html, 2 for /bar/norm.html, 4 for /a/b/c/d.html
 */
static inline int get_depth( const char *href ) {
    const char *pnt;
    int count = 0;
    for( pnt=href; *pnt != '\0'; pnt++ ) /* oneliner */
	if( *pnt == '/' ) count++;
    return count;
}

/* We want an href relative to the Request-URI */
static char *strip_href( const char *root, const char *href ) {
    const char *tmp;
    char *dec, *ret;
    size_t rootlen;
    DEBUG( DEBUG_XML, "Parsing href [%s]\n", href );
    if( strncmp( href, "http://", 7 ) == 0 ) {
	/* Absolute URI.
	 * Look for the path bit */
	DEBUG( DEBUG_XML, "Got absolute URI.\n" );
	tmp = strchr( href+7, '/' );
	if( tmp == NULL ) {
	    DEBUG( DEBUG_XML, "No path segment found.\n" );
	    return NULL;
	}
    } else {
	tmp = href;
    }
    DEBUG( DEBUG_XML, "Using abspath segment: %s\n", tmp );
    dec = uri_unescape( tmp );
    DEBUG( DEBUG_XML, "Decoded is: [%s]. Root is [%s]\n", tmp, root );
    /* Now, dec points to the absPath section of the URI.
     * We check whether this resource is actually in the
     * collection we have done the PROPFIND against.
     */
    rootlen = strlen( root );
    if( strncmp( dec, root, rootlen ) != 0 ) {
	DEBUG( DEBUG_XML, "strip_href failed: root collection not matched." );
	ret = NULL;
    } else {
	/* We're in the right place */
	DEBUG( DEBUG_XML, "strip_href: Got [%s]\n", dec + rootlen );
	/* Filename must be the basename */
	ret = strdup( dec + rootlen );
    }
    free( dec );
    return ret;
}

static int 
startelm_props( void *userdata, const struct hip_xml_state *s,
		const char *elmname, const char **atts ) {
    struct dav_207_parser *ctx207 = userdata;
    struct dav_propstat *pstat;
    struct proto_file *file;
    
    if( s->elm->id != DAV_ELM_prop ) {
	/* ignore it */
	return 0;
    }
    
    if( dav_207_getcurrentpropstat( ctx207, &pstat ) ) {
	return -1;
    }

    pstat->prop = file = malloc( sizeof(struct proto_file) );
    memset( file, 0, sizeof(struct proto_file) );
    file->type = proto_file;

    return 0;
}

/* End-of-element handler */
static int
endelm_props( void *userdata, const struct hip_xml_state *s, const char *cdata ) 
{
    struct dav_207_parser *ctx207 = userdata;
    struct dav_propstat *pstat;
    struct proto_file *file;
    
    if( dav_207_getcurrentpropstat( ctx207, &pstat ) ||
	pstat->prop == NULL ) {
	return -1;
    }

    file = pstat->prop;

    switch( s->elm->id ) {
    case DAV_ELM_getlastmodified:
	DEBUG( DEBUG_XML, "Parsing date [%s]\n", cdata );
	file->modtime = rfc1123_parse( cdata );
	if( file->modtime == (time_t)-1 ) {
	    DEBUG( DEBUG_XML, "Date is not in RFC1123 format.\n" );
	    return -1;
	}
	break;
    case DAV_ELM_getcontentlength:
	file->size = atoi( cdata );
	break;
    case DAV_ELM_collection:
	file->type = proto_dir;
	break;
    case DAV_ELM_redirectref:
	file->type = proto_link;
	break;
    default:
	break;
    }
    return 0;
}

#if 0

static void http_get_content_charset( const char *name, const char *value ) {
    char **pairs;
    int n;
    if( strcasecmp( name, "Content-Type" ) == 0 ) {
	/* Get the charset so we can pass it on to expat */
	pairs = pair_string( value, ';', '=', http_quotes, http_whitespace );
	for( n = 0; pairs[n] != NULL; n+=2 ) {
	    if( (strcasecmp( pairs[n], "charset") == 0) &&
		pairs[n+1] != NULL ) {
		HTTP_FREE( http_content_charset );
		/* Strip off the quotes */
		http_content_charset = shave_string( pairs[n+1], '\"' );
		DEBUG( DEBUG_XML, "Got content type charset: %s\n",
		       http_content_charset );
	    }
	}
	pair_string_free( pairs );
    }
    return;
}

#endif

static void fetch_free( void *userdata, struct dav_response *rsp,
			struct dav_propstat *pst ) 
{
    struct fetch_context *ctx = userdata;
    struct proto_file **files = ctx->files;
    struct proto_file *file = pst->prop, *current, *previous;
    char *tmp;
    if( !files || 
	(rsp->status_line && rsp->status.class != 2) ||
	(pst->status_line && pst->status.class != 2) ) {
	free( pst->prop );
	return;
    }
    tmp = strip_href( ctx->fetch_root, rsp->href );
    if( !tmp ) {
	DEBUG( DEBUG_XML, "strip_href failed.\n" );
	free( pst->prop );
	return;
    }
    if( !ctx->fetch_target && strlen( tmp ) == 0 ) {
	DEBUG( DEBUG_XML, "Resource is root collection, ignoring.\n" );
	free( pst->prop );
	free( tmp );
	return;
    }
    file->filename = tmp;
    if( file->type == proto_dir && (strlen(tmp) > 1) ) {
	file->filename[strlen(file->filename)-1] = '\0';
    }
    DEBUG( DEBUG_XML, "Filename is really: %s\n", file->filename );
    file->depth = get_depth( file->filename );
    previous = NULL;
    current = *files;
    while( current != NULL && current->depth < file->depth ) {
	previous = current;
	current = current->next;
    }
    if( previous == NULL ) {
	*files = file;
    } else {
	previous->next = file;
    }
    file->next = current;
}

void dav_free_fileslist( struct proto_file *files ) {
    struct proto_file *cur, *next;
    for( cur = files; cur != NULL; cur = next ) {
	next = cur->next;
	HTTP_FREE( cur->filename );
	free( cur );
    }
}

/* WebDAV fetch mode handler. */
int dav_fetch( const char *dirname, int depth, int fetch_target,
	       struct proto_file **files ) 
{
    http_req_t req;
    struct fetch_context ctx;
    int ret, pret;
    static const struct hip_xml_elm elms[] = {
	{ "DAV:prop", DAV_ELM_prop, 0 },
	{ "DAV:getlastmodified", DAV_ELM_getlastmodified, HIP_XML_CDATA },
	{ "DAV:getcontentlength", DAV_ELM_getcontentlength, HIP_XML_CDATA },
	{ "DAV:resourcetype", DAV_ELM_resourcetype, HIP_XML_CDATA },
	{ "DAV:collection", DAV_ELM_collection, 0 },
	{ "DAV:redirectref", DAV_ELM_redirectref, 0 },
	{ "", HIP_ELM_unknown, 0 },
	{ NULL, 0, 0 }
    };
    struct dav_207_parser p;
    struct hip_xml_elmlist elmlist = 
    { elms, check_context, startelm_props, endelm_props, &p, NULL };
    const char *propfind_body =
	"<?xml version=\"1.0\"?>" EOL /* should we use encoding=? */
	"<propfind xmlns=\"DAV:\">" EOL
	"  <prop>" EOL
	"    <getcontentlength/>" EOL
	"    <getlastmodified/>" EOL
	"    <resourcetype/>" EOL
	"  </prop>" EOL
	"</propfind>" EOL;
    const char *myheaders =
	"Content-Type: text/xml" EOL; /* should we use charset=? */

    if( dav_207_init( &p, &elmlist ) ) {
	return PROTO_ERROR;
    }

    http_request_init( &req, "PROPFIND", dirname );

    dav_lock_using_resource( &req, dirname, dav_lockusage_read, depth );

    req.body_callback = hip_xml_parse_v;
    req.body_callback_userdata = &p.parser;
    req.body_want = dav_want_207;

    req.body = http_body_buffer;
    req.body_buffer = propfind_body;
    /* Add in the content type header */
    strcat( req.headers, myheaders );
    switch( depth ) {
    case 0:
	strcat( req.headers, "Depth: 0" EOL );
	break;
    case 1:
	strcat( req.headers, "Depth: 1" EOL );
	break;
    default:
	strcat( req.headers, "Depth: infinity" EOL );
    }

    ret = http_request( &req );
    pret = dav_207_finish( &p );

    if( files ) *files = NULL;
    
    ctx.fetch_root = dirname;
    ctx.fetch_target = fetch_target;
    ctx.files = files;

    dav_207_free( &p, fetch_free, &ctx );
    
    if( ret == PROTO_OK ) {
	if( req.status == 207 ) {
	    if( pret ) {
		snprintf( http_error, BUFSIZ, 
			  "Error parsing XML response:\n%s", p.parser.error );
		ret = PROTO_ERROR;
	    }
	} else {
	    ret = PROTO_ERROR;
	}
    }
    http_request_end( &req );
    return ret;
}

