/* 
   WebDAV lock handling
   Copyright (C) 1999-2000, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   $Id: davlocks.c,v 1.24 2000/03/21 13:42:48 joe Exp $
*/

#include <config.h>

#define _GNU_SOURCE
#include <stdio.h>

#ifndef HAVE_SNPRINTF
#include "snprintf.h"
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <ctype.h>

#include "httpdav.h"
#include "davlocks.h"
#include "hip_xml.h"
#include "dav_207.h"
#include "http_utils.h"
#include "string_utils.h"
#include "uri.h"

struct dav_submit_locks {
    struct dav_lock *lock;
    const char *uri;
    struct dav_submit_locks *next;
};

/* Add a lock to the locklist.
 * 0 on okay, non-zero on error */
static int
dav_submit_lock( struct dav_submit_locks **list, struct dav_lock *lock,
		 const char *uri )
{
    struct dav_submit_locks *item = malloc(sizeof(struct dav_submit_locks));
    if( item != NULL ) {
	item->next = *list;
	item->lock = lock;
	item->uri = uri;
	*list = item;
	return 0;
    } else {
	return -1;
    }    
}

void dav_submitlocks_free( struct dav_submit_locks *list )
{
    struct dav_submit_locks *next;
    while( list != NULL ) {
	next = list->next;
	free( list );
	list = next;
    }
}

/* Find lock in list.
 * Returns zero on not found, non-zero if found. */
static int
dav_submitlocks_find( struct dav_submit_locks *list, struct dav_lock *lock )
{
    struct dav_submit_locks *item;
    for( item = list; item != NULL; item = item->next )
	if( item->lock == lock )
	    return 1;
    return 0;
}

/* 
   Find any locktokens that would need to be submitted if we are
   modifying the resource at URI.  This catches three cases:

   1. If the resource that we are modifying is a collection, and we
   are doing a depth-infinity modification of this collection
   (e.g. MOVE), submit the locktokens for any locks on the children of
   the resource.

   2. Submit the locktokens for any locks on the resource itself.

   3. If there are any depth-infinity locks higher up, which cover this
   resource, submit the locktokens for these locks.
   
   The 'uri' string must remain valid for the lifetime of the request.
   
*/

void dav_lock_using_resource( 
    http_req_t *req, const char *uri, enum dav_lock_usage use, int depth ) 
{
    /* FIXME HACK: yes, we're back to using the global lock list again. */
    struct dav_lock *lock;
    if( use == dav_lockusage_read ) {
	/* TODO: we don't know about read locks... yet */
	return;
    }
    DEBUG( DEBUG_LOCKS, "using_resource %s\n", uri );
    for( lock = *dav_lock_list; lock != NULL; lock = lock->next ) {
	int match=0;
	if( dav_submitlocks_find( req->if_locks, lock ) ) {
	    DEBUG( DEBUG_LOCKS, "Skipped duplicate: %s\n", lock->token );
	    continue;
	}
	if( depth == DAV_DEPTH_INFINITE ) {
	    /* Case 1 */
	    if( uri_childof( uri, lock->uri ) ) {
		DEBUG( DEBUG_LOCKS, "Has child: %s\n", lock->token );
		match = 1;
	    }
	} else /* depth == 0 */ {
	    /* Case 2 */
	    if( uri_compare( uri, lock->uri ) == 0 ) {
		DEBUG( DEBUG_LOCKS, "Has direct lock: %s\n", lock->token );
		match = 1;
	    }
	}
	if( !match ) {
	    /* Case 3 */
	    if( lock->depth == DAV_DEPTH_INFINITE && 
		uri_childof( lock->uri, uri ) ) {
		DEBUG( DEBUG_LOCKS, "Is child of: %s\n", lock->token );
		match = 1;
	    }
	}
	if( match ) {
	    dav_submit_lock( &req->if_locks, lock, uri );
	}
    }
}

/* We are modifying the parent collection of the resource at URI.
 * e.g., we are doing a MOVE, DELETE, etc etc. */
void dav_lock_using_parent( http_req_t *req, const char *uri ) 
{
    char *parent = uri_parent( uri );
    if( parent != NULL ) {
	struct dav_lock *lock;
	lock = dav_lock_find( *dav_lock_list, parent );
	if( lock && !dav_submitlocks_find( req->if_locks, lock ) ) {
	    DEBUG( DEBUG_LOCKS, "Locked parent, %s on %s\n", 
		   lock->token, lock->uri );
	    dav_submit_lock( &req->if_locks, lock, uri );
	} 
	free( parent );
    }
}

/* Return the ifheader */
char *dav_lock_ifheader( http_req_t *req ) {
    if( req->if_locks != NULL ) {
	struct dav_submit_locks *item;
	sbuffer ret = sbuffer_create();
	if( ret == NULL ) return NULL;
	sbuffer_zappend( ret, "If:" );
	for( item = req->if_locks; item != NULL; item = item->next ) {
	    sbuffer_concat( ret, " <", item->uri, "> (<",
			    item->lock->token, ">)", NULL );
	}
	DEBUG( DEBUG_LOCKS, "Using if-header: %s\n", sbuffer_data(ret) );
	sbuffer_zappend( ret, EOL );
	return sbuffer_finish(ret);
    } else {
	return NULL;
    }
}

/* Finds a lock directly on URI */
struct dav_lock *
dav_lock_find( struct dav_lock *list, const char *uri ) {
    struct dav_lock *cur;
    for( cur = list; cur != NULL; cur = cur->next ) {
	if( uri_compare( uri, cur->uri ) == 0 ) return cur;
    }
    return NULL;
}

void dav_lock_add( struct dav_lock **list, struct dav_lock *lock ) {
    if( *list ) {
	(*list)->prev = lock;
    }
    lock->prev = NULL;
    lock->next = *list;
    *list = lock;
}

void dav_lock_remove( struct dav_lock **list, struct dav_lock *lock ) {
    if( lock->prev ) {
	lock->prev->next = lock->next;
    } else {
	*list = lock->next;
    }
    if( lock->next ) {
	lock->next->prev = lock->prev;
    }
}

void dav_lock_free( struct dav_lock *lock ) {
    if( lock->uri ) {
	free( lock->uri );
    }
    free( lock );
}

int dav_unlock( struct dav_lock *lock ) {
    http_req_t req;
    int ret;

    http_request_init( &req, "UNLOCK", lock->uri );
    
    strcat( req.headers, "Lock-Token: <" );
    strcat( req.headers, lock->token );
    strcat( req.headers, ">" EOL );
    
    /* TODO: need this or not?
     * it definitely goes away when lock-null resources go away */
    dav_lock_using_parent( &req, lock->uri );

    ret = http_request( &req );
    
    if( ret == PROTO_OK && req.class == 2 ) {
	ret = PROTO_OK;    
    } else {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );
    
    return ret;
}

/* bit random, these */
#define DAV_ELM_lockdiscovery 2
#define DAV_ELM_activelock 3
#define DAV_ELM_lockscope 4
#define DAV_ELM_locktype 5
#define DAV_ELM_depth 6
#define DAV_ELM_owner 7
#define DAV_ELM_timeout 9
#define DAV_ELM_locktoken 10
#define DAV_ELM_lockinfo 11
#define DAV_ELM_write 14
#define DAV_ELM_exclusive 15
#define DAV_ELM_shared 16

static const struct hip_xml_elm lock_elms[] = {
#define A(x) { "DAV:" #x, DAV_ELM_ ## x, 0 /* DAV_COLLECT */ } /* ANY */
#define D(x) { "DAV:" #x, DAV_ELM_ ## x, 0 }                   /* normal */
#define C(x) { "DAV:" #x, DAV_ELM_ ## x, HIP_XML_CDATA }           /* (#PCDATA) */
#define E(x) { "DAV:" #x, DAV_ELM_ ## x, 0 /* DAV_LEAF */ }    /* EMPTY */
    D(lockdiscovery),
    D(activelock),
    D(lockscope), D(locktype), C(depth), A(owner), C(timeout), D(locktoken),
    /* no lockentry */
    D(lockinfo), D(lockscope), D(locktype),
    E(write), E(exclusive), E(shared),
    E(prop), C(href),
#undef A
#undef D
#undef C
#undef E
    { NULL, 0, 0 }
};

struct lock_context {
    struct lock_result *list;
    struct lock_result *current;
    struct dav_207_parser *context_207; /* FIXME: hack */
};

static int check_context( hip_xml_elmid parent, hip_xml_elmid child) {
    /* TODO: validate. */
    return 0;
}

static int parse_depth( const char *depth ) {
    if( strcasecmp( depth, "infinity" ) == 0 ) {
	return DAV_DEPTH_INFINITE;
    } else if( isdigit( depth[0] ) ) {
	return atoi( depth );
    } else {
	return -1;
    }
}

static int parse_timeout( const char *timeout ) {
    if( strcasecmp( timeout, "infinite" ) == 0 ) {
	return DAV_TIMEOUT_INFINITE;
    } else if( strncasecmp( timeout, "Second-", 7 ) == 0 ) {
        return atoi( timeout+7 );
    } else {
	return DAV_TIMEOUT_INVALID;
    }
}

static void got_lockresult( struct lock_context *ctx ) {
    ctx->current->next = ctx->list;
    ctx->list = ctx->current;
    ctx->current = malloc( sizeof(struct lock_result) );
    memset( ctx->current, 0, sizeof(struct lock_result) );
}

static int 
endelm_locks( void *userdata, const struct hip_xml_state *s, 
	     const char *cdata ) 
{
    struct lock_context *ctx = userdata;
    struct lock_result *l = ctx->current;
    switch( s->elm->id ){ 
    case DAV_ELM_write:
	l->lock.type = dav_locktype_write;
	break;
    case DAV_ELM_exclusive:
	l->lock.scope = dav_lockscope_exclusive;
	break;
    case DAV_ELM_shared:
	l->lock.scope = dav_lockscope_shared;
	break;
    case DAV_ELM_depth:
	DEBUG( DEBUG_LOCKS, "Got depth: %s\n", cdata );
	l->lock.depth = parse_depth( cdata );
	if( l->lock.depth == -1 ) {
	    return -1;
	}
	break;
    case DAV_ELM_timeout:
	DEBUG( DEBUG_LOCKS, "Got timeout: %s\n", cdata );
	l->lock.timeout = parse_timeout( cdata );
	if( l->lock.timeout == DAV_TIMEOUT_INVALID ) {
	    return -1;
	}
	break;
    case DAV_ELM_href:
	switch( s->parent->elm->id ) {
	case DAV_ELM_locktoken:
	    l->lock.token = strdup( cdata );
	    break;
	case DAV_ELM_response:
	    l->href = strdup( cdata );
	    /* FIXME: this is a HACK, because we over-ride the
	     * DAV:href handling from the 207 code. Best way to fix:
	     * have hip_xml find alternate handlers if the first
	     * refuses to handle. */
	    ctx->context_207->ms.current->href = strdup(cdata );
	    break;
	default:
	}
	break;
    case DAV_ELM_status:
	l->reason = strdup( cdata );
	break;
    case DAV_ELM_activelock:
	got_lockresult( ctx );
	break;
    }
    return 0;
}

int dav_lock( struct dav_lock *lock ) 
{
    http_req_t req;
    sbuffer body;
    struct dav_207_parser p;
    struct lock_context ctx = {0};
    struct hip_xml_elmlist 
	elmlist1 = 
    { lock_elms, check_context, NULL, endelm_locks, &ctx, NULL };
    int ret, parse_failed;

    if( dav_207_init( &p, &elmlist1 ) )
	return PROTO_ERROR;
    
    ctx.current = malloc( sizeof(struct lock_result) );
    if( !ctx.current ) 
	return PROTO_ERROR;
    ctx.context_207 = &p;

    body = sbuffer_create();
    if( !body ) return PROTO_ERROR;

    /* Create the body */
    if( sbuffer_concat( body,
			    "<?xml version=\"1.0\" encoding=\"utf-8\"?>" EOL
			    "<lockinfo xmlns='DAV:'>" EOL " <lockscope>",
			    lock->scope==dav_lockscope_exclusive?
			    "<exclusive/>":"<shared/>",
			    "</lockscope>" EOL
			    "<locktype><write/></locktype>", NULL ) )
	return PROTO_ERROR;
    if( lock->owner ) {
	if( sbuffer_concat( body, "<owner>", lock->owner, 
				"</owner>" EOL, NULL ) )
	    return PROTO_ERROR;
    }
    if( sbuffer_zappend( body, "</lockinfo>" EOL ) )
	return PROTO_ERROR;

    http_request_init( &req, "LOCK", lock->uri );
    req.body = http_body_buffer;
    req.body_buffer = sbuffer_data(body);
    req.body_callback = hip_xml_parse_v;
    req.body_callback_userdata = &p.parser;
    strcat( req.headers, "Content-Type: text/xml" EOL );

    /* TODO: 
     * By 2518, we need this only if we are creating a lock-null resource.
     * Since we don't KNOW whether the lock we're given is a lock-null
     * or not, we cover our bases.
     */
    dav_lock_using_parent( &req, lock->uri );
    /* This one is clearer from 2518 sec 8.10.4. */
    dav_lock_using_resource( &req, lock->uri, dav_lockusage_write,
			     lock->depth );

    switch( lock->depth ) {
    case 0:
	strcat( req.headers, "Depth: 0" EOL );
	break;
    default:
	strcat( req.headers, "Depth: infinity" EOL );
	break;
    }

    ret = http_request( &req );

    sbuffer_destroy( body );
    free( ctx.current );
    parse_failed = dav_207_finish( &p );
    
    if( ret == PROTO_OK && req.class == 2 ) {
	if( parse_failed ) {
	    snprintf( http_error, BUFSIZ, 
		      "Error parsing XML response:\n%s", p.parser.error );
	    ret = PROTO_ERROR;
	} else {
	    if( ctx.list != NULL ) {
		/* TODO: Spec problem. If this is a shared lock,
		 * then they might give us back N locks: we don't know
		 * which one is ours. */
		lock->token = ctx.list->lock.token;
		lock->timeout = ctx.list->lock.timeout;
		lock->depth = ctx.list->lock.depth;
		ret = PROTO_OK;
	    } else {
		dav_207_write_errors( &req, &p );
		ret = PROTO_ERROR;
	    }
	}
    } else {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );

    /* TODO: free the list */
    return ret;
}

/* Discover all locks on URI */
int dav_lock_discover( const char *uri, struct lock_result **locks ) 
{
    http_req_t req;
    struct dav_207_parser p;
    struct lock_context ctx = {0};
    static const char *body = 
	"<?xml version=\"1.0\" encoding=\"utf-8\"?>"
	"<propfind xmlns='DAV:'><prop><lockdiscovery/></prop></propfind>" EOL;
    struct hip_xml_elmlist elmlist =
    { lock_elms, check_context, NULL, endelm_locks, &ctx, NULL };
    int ret, parse_failed;

    if( dav_207_init( &p, &elmlist ) )
	return PROTO_ERROR;
    
    ctx.current = malloc( sizeof(struct lock_result) );
    if( !ctx.current ) 
	return PROTO_ERROR;
    memset( ctx.current, 0, sizeof(struct lock_result) );
    ctx.context_207 = &p;

    http_request_init( &req, "PROPFIND", uri );
    req.body = http_body_buffer;
    req.body_buffer = body;
    req.body_callback = hip_xml_parse_v;
    req.body_callback_userdata = &p.parser;

    dav_lock_using_resource( &req, uri, dav_lockusage_read, 0 );

    strcat( req.headers, "Depth: 0" EOL );
    strcat( req.headers, "Content-Type: text/xml" EOL );

    ret = http_request( &req );
    parse_failed = dav_207_finish( &p );

    free( ctx.current );

    if( ret == PROTO_OK && req.class == 2 ) {
	if( parse_failed ) {
	    snprintf( http_error, BUFSIZ, 
		      "Error parsing XML response:\n%s", p.parser.error );
	    ret = PROTO_ERROR;
	} else if( ctx.list != NULL ) {
	    *locks = ctx.list;
	    ret = PROTO_OK;
	} else {
	    ret = PROTO_NONE;
	}
    } else {
	ret = PROTO_ERROR;
    }

    if( ret == PROTO_ERROR ) {
	/* free the list */
    }

    http_request_end( &req );

    return ret;
}

