/* 
   WebDAV property handling
   Copyright (C) 2000, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   $Id: davprops.c,v 1.5 2000/01/24 21:22:27 joe Exp $
*/

#include <config.h>

#define _GNU_SOURCE
#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "httpdav.h"
#include "hip_xml.h"
#include "dav_207.h"
#include "davprops.h"

/* This is a good example of how nice it is to use the new 207
 * handling code.
 * check_getnames validates the XML structure.
 * We store 'struct names_prop' as the prop data.
 */

struct names_prop {
    char **array;
    int size;
};

/* Our elements */
#define DAV_ELM_propname (1)

static int check_getnames( hip_xml_elmid parent, hip_xml_elmid child );
static int startelm_getnames( void *userdata, const struct hip_xml_state *s,
			      const char *name, const char **atts );

static int check_getnames( hip_xml_elmid parent, hip_xml_elmid child ) {
    switch( child ) {
    case HIP_ELM_unknown:
	if( parent == DAV_ELM_prop ) {
	    return 0;
	}
	break;
    case DAV_ELM_prop:
	if( parent == DAV_ELM_propstat ) {
	    return 0;
	}
	break;
    }
    return -1;
}

static int startelm_getnames( void *userdata, const struct hip_xml_state *s,
		       const char *name, const char **atts ) {
    struct dav_207_parser *ctx = userdata;
    struct dav_propstat *pstat;
    struct names_prop *prop;

    if( dav_207_getcurrentpropstat( ctx, &pstat ) ) {
	return -1;
    }

    switch( s->elm->id ) {
    case DAV_ELM_prop:
	pstat->prop = prop = malloc(sizeof(struct names_prop));
	prop->array = NULL;
	prop->size = 0;
	break;
    case HIP_ELM_unknown:
	prop = pstat->prop;
	prop->size++;
	prop->array = realloc( prop->array, (prop->size) * sizeof(char *) );
	prop->array[prop->size-1] = strdup(name);
	DEBUG( DEBUG_FILES, "Got: %s\n", name );
	break;
    }
    return 0;
}

static void getnames_free( void *ud, struct dav_response *rsp, 
			   struct dav_propstat *pstat ) {
    /* FIXME: the status should have been parsed, so we know which
     * one of the DAV:response's is the 200, which is other.
     * We make the nasty assumption that the first is the 200.
     * This will break, for you, which is why you are here reading
     * this comment. */
    char ***names = ud;
    struct names_prop *prop;

    if( !pstat->prop || *names != NULL ) {
	return;
    }
    
    prop = pstat->prop;
    *names = realloc( prop->array, (prop->size + 1) * sizeof(char *));
    (*names)[prop->size] = NULL;	
    free( prop );
}

/* A nice simple use of the generic 207 code */
int dav_prop_getnames( const char *uri, char ***names ) {
    http_req_t req;
    struct dav_207_parser p;
    int ret, parse_failed;
    static const char *propfind_body = 
	"<?xml version=\"1.0\"?>"
	"<propfind xmlns=\"DAV:\"><propname/></propfind>" EOL,
	*headers = 
	"Content-Type: text/xml" EOL
	"Depth: 0" EOL; /* should we use charset=? */
    static const struct hip_xml_elm elms[] = {
	{ "DAV:prop", DAV_ELM_prop, 0 },
	{ "", HIP_ELM_unknown, 0 },
	{ NULL, 0, 0 }	
    };
    struct hip_xml_elmlist elmlist = 
    { elms, check_getnames, startelm_getnames, NULL, &p, NULL };

    if( dav_207_init( &p, &elmlist ) ) {
	/* FIXME: error msg */
	return PROTO_ERROR;
    }

    http_request_init( &req, "PROPFIND", uri );
    req.body_callback = hip_xml_parse_v;
    req.body_callback_userdata = &p.parser;
    req.body_want = dav_want_207;
    
    req.body = http_body_buffer;
    req.body_buffer = propfind_body;
    strcat( req.headers, headers );
    
    ret = http_request( &req );
    parse_failed = dav_207_finish( &p );
    
    if( ret == PROTO_OK && req.status == 207 ) {
	if( parse_failed ) {
	    ret = PROTO_ERROR;
	} else {
	    *names = NULL;
	    dav_207_free( &p, getnames_free, names );
	    ret = PROTO_OK;
	}
    } else {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );

    return ret;

}

    
