/* 
   sitecopy, for managing remote web sites. WebDAV client routines.
   Copyright (C) 1998-99, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: httpdav.c,v 1.43 2000/03/21 13:46:08 joe Exp $
*/

/* This file is a collection of routines to implement a basic WebDAV
 * client, including an HTTP/1.1 client. 
 * Transparently supports basic and digest authentication.
 */

/* HTTP method implementation:
 *   Call, in this order:
 *     1.  http_request_init()  - set up the request
 *     2.  http_request()       - make the request
 *     3.  http_request_end()   - clean up the request
 */

/* TODO:
 *  - Tidy up parse_status, generally ensure that even if the 
 *    server decides to feed us /dev/random instead of a syntactically-
 *    correct response, we can cope.
 *  - Support proxy authentication
 */

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#ifdef __EMX__
#include <sys/select.h>
#endif

#include <netinet/in.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifndef HAVE_SNPRINTF
#include <snprintf.h>
#endif

#include "dates.h"
#include "basename.h"
#include "dirname.h"
#include "string_utils.h"
#include "http_utils.h"
#include "http_auth.h"
#include "uri.h"

#include "frontend.h"
#include "protocol.h"
#include "httpdav.h"
#include "common.h"
#include "socket.h"
#include "dav_207.h"

/* Connection information... */
int http_sock;
bool http_connected;

bool http_use_expect = false;

/* Time in seconds to wait for the server to give us a 100 Continue
 * after submitting a PUT with an "Expect: 100-continue" header.
 */
#define HTTP_EXPECT_TIMEOUT 15
/* 100-continue only used if size > HTTP_EXPECT_MINSIZ */
#define HTTP_EXPECT_MINSIZE 512

/* Whether the current server respects the Expect: 100-continue header */
int http_expect_works; /* == 0 if it we don't know, 1 if it does,
			    -1 if it doesn't support 100-continue */

bool http_enable_expect = false;
bool http_webdav_server = false;
bool http_webdav_locking = false;
bool http_init_checks = true;
bool http_conn_limit = false;

/* Warning given out on spoof attack */
const char *http_warn_spoof = 
  "The server has switched to using basic authentication from digest\n"
  "authenticaion, which may be an attempt to discover your password.\n"
  "Basic auth will NOT be used.";

http_auth_session http_server_auth, http_proxy_auth;

static int give_creds( void *udata, const char *realm,
		       char **username, char **password );

unsigned int http_version_major, http_version_minor;

#ifdef USE_DAV_LOCKS
struct dav_lock **dav_lock_list;
#endif

const char *http_quotes = "\"'";
const char *http_whitespace = " \r\n\t";

proto_read_block http_read_file_callback;

static void 
http_read_file_block( void *userdata, const char *s, const size_t len );


/* Handy macro to free things. */
#define DOFREE(x) if( x!=NULL ) free( x )

#define EOL "\r\n"
#define HTTP_PORT 80

const char *http_useragent = PACKAGE "/" VERSION;

/* We need to remember the remote host and port even after connecting
 * the socket e.g. so we can form the Destination: header */
struct proto_host http_server_host, http_proxy_host;

/* This, to store the address of the server we CONNECT to - i.e.,
 * the proxy if we have one, else the server */
struct in_addr http_remoteaddr;
int http_remoteport;

bool http_use_proxy; /* whether we are using the proxy or not */

static int http_response_read( http_req_t *req, char *buffer, size_t buflen );

/* Sets up the body size for the given request */
static int http_req_bodysize( http_req_t *req );

/* The callback for GET requests (to allow signalling progress to the FE) */
static void http_get_callback( void *user, const char *buffer, const size_t len );

/* Holds the error message to be passed up */
char http_error[BUFSIZ];

/* Put the fixed headers into the request */
static void http_req_fixedheaders( http_req_t *req );

/* Concatenates the remote server name with :port if port!=80 on to 
 * the end of str. */
static void append_hostport( struct proto_host *host, char *str );
static void append_hostport_sbuf( struct proto_host *host, sbuffer buf );

/* Header parser for OPTIONS requests. */
static void http_options_parsehdr( const char *name, const char *value );

static int http_parse_status( http_req_t *req, char *status_line );

static void http_get_lastmod( const char *header, const char *value );

/* Sends the body down the wire */
static int http_req_sendbody( http_req_t *req );

/* Opens the connection to the remote server.
 * Returns:
 *  PROTO_OK       on success
 *  PROTO_CONNECT  if the socket couldn't be connected
 */
static int http_open( void );

/* Closes the connection.
 * Always returns PROTO_OK
 */
static int http_close( void );

static int init_207errors( http_req_t *req );
static void finish_207errors( http_req_t *req );

/* This doesn't really connect to the server.
 * Returns
 *  PROTO_LOOKUP if the hostname of the server could not be resolved
 *  PROTO_LOCALHOST if the hostname of the local machine could not be
 *    found
 *  PROTO_CONNECT if the socket could not be connected
 *  PROTO_OK if the connection was made successfully.
 */
int http_init( const char *remote_root,
	       struct proto_host *server, struct proto_host *proxy ) {
    int ret;
    /* Take a copy of the server information */
    memcpy( &http_server_host, server, sizeof(struct proto_host) );
    fe_connection( fe_namelookup );
    if( proxy != NULL ) {
	memcpy( &http_proxy_host, proxy, sizeof(struct proto_host) );
	http_remoteport = proxy->port;
	http_use_proxy = true;
	if( host_lookup( http_proxy_host.hostname, &http_remoteaddr ) )
	    return PROTO_LOOKUP;
    } else {
	http_remoteport = server->port;
	http_use_proxy = false;
	if( host_lookup( http_server_host.hostname, &http_remoteaddr ) )
	    return PROTO_LOOKUP;
    }
    http_connected = false;
    /* Set expect_works to "we don't know" or "we don't want to try it" */
    http_expect_works = http_enable_expect?0:-1;
    if( http_use_proxy ) {
	http_auth_init( &http_proxy_auth );
    }
    ret = http_open();
    /* Drop out if that failed, or they don't want the OPTIONS */
    if( (!http_init_checks) || (ret != PROTO_OK) )
	return ret;
    /* Capability discovery... we don't care whether this
     * actually works or not, we just want http_webdav_server
     * set appropriately. */
    (void) http_options( remote_root );
    return PROTO_OK;
}

/* Give authentication credentials */
static int give_creds( void *udata, const char *realm,
		       char **username, char **password ) 
{ 
    http_req_t *req = udata;
    if( req->status == 407 ) {
	if( http_proxy_host.username == NULL ) {
	    return fe_login( fe_login_proxy, realm, http_proxy_host.hostname,
			     username, password );
	} else {
	    *username = strdup(http_proxy_host.username);
	    *password = strdup(http_proxy_host.password);
	}
    } else {
	if( http_server_host.username == NULL ) {
	    return fe_login( fe_login_server, realm, http_server_host.hostname,
			     username, password );
	} else {
	    *username = strdup(http_server_host.username);
	    *password = strdup(http_server_host.password);
	}
    }
    return 0;
}

int http_finish( void ) {
    DEBUG( DEBUG_HTTP, "http_finish called.\n" );
    http_auth_finish( &http_server_auth );
    if( http_connected ) http_close( );
    return PROTO_OK;
}

/* Parse the HTTP-Version and Status-Code segments of the
 * given status_line. Sets http_version_* and req->class,status.
 * Returns: PROTO_OK on success, PROTO_ERROR otherwise */
int http_parse_status( http_req_t *req, char *status_line ) {
    char *part;
    DEBUG( DEBUG_HTTP, "HTTP response line: %s", status_line );
    /* Check they're speaking the right language */
    if( strncmp( status_line, "HTTP/", 5 ) != 0 ) {
	strcpy( http_error, "Invalid HTTP response line." );
	return PROTO_ERROR;
    }
    /* And find out which dialect of this peculiar language
     * they can talk... */
    http_version_major = 0;
    http_version_minor = 0; 
    /* Note, we're good children, and accept leading zero's on the
     * version numbers */
    for( part = status_line + 5; *part != '\0' && isdigit(*part); part++ ) {
	http_version_major = http_version_major*10 + (*part-'0');
    }
    if( *part != '.' ) { 
	strcpy( http_error, "Could not parse HTTP major version." );
	return PROTO_ERROR;
    }
    for( part++ ; *part != '\0' && isdigit(*part); part++ ) {
	http_version_minor = http_version_minor*10 + (*part-'0');
    }
    DEBUG( DEBUG_HTTP, "HTTP Version Major: %d, Minor: %d\n", 
	   http_version_major, http_version_minor );
    /* Skip any spaces */
    for( ; *part == ' ' ; part++ ) /* noop */;
    /* Now for the Status-Code. part now points at the space
     * between "HTTP/x.x YYY". We want YYY... could use atoi, but
     * probably quicker this way. */
    if( !isdigit(part[0]) || !isdigit(part[1]) || !isdigit(part[2]) ) {
	strcpy( http_error, "Invalid character in status code." );
	return PROTO_ERROR;
    }
    req->status = 100*(part[0]-'0') + 10*(part[1]-'0') + (part[2]-'0');
    req->class = part[0]-'0';
    /* ...and we can ignore the Reason-Phrase */
    /* Save the line for error codes later */
    memset( http_error, 0, BUFSIZ );
    strncpy( http_error, status_line, BUFSIZ );
    /* Strip off the CRLF for the status line */
    STRIP_EOL( http_error );
    DEBUG( DEBUG_HTTP, "HTTP status code: %d\n", req->status );
    return PROTO_OK;
}

/* Sends the body down the socket.
 * Returns PROTO_OK on success, PROTO_ERROR otherwise */
int http_req_sendbody( http_req_t *req ) {
    int ret;
    switch( req->body ) {
    case http_body_file:
	ret = transfer( fileno(req->body_file), http_sock, req->body_size );
	DEBUG( DEBUG_HTTP, "Sent %d bytes.\n", ret );
	rewind( req->body_file ); /* since we may have to send it again */
	break;
    case http_body_buffer:
	DEBUG( DEBUG_HTTP, "Sending body:\n%s\n", req->body_buffer );
	ret = send_string( http_sock, req->body_buffer );
	DEBUG( DEBUG_HTTP, "send_string returns: %d\n", ret );
	break;
    default:
	DEBUG( DEBUG_HTTP, "Argh in http_req_sendbody!" );
	ret = -1;
    }
    if( ret == -1 ) { 
	/* transfer failed */
	DEBUG( DEBUG_HTTP, "Body send failed.\n" );
	return PROTO_ERROR;
    } else {
	return PROTO_OK;
    }
}

/* Deal with the body size */
int http_req_bodysize( http_req_t *req ) {
    struct stat bodyst;
    /* Do extra stuff if we have a body */
    switch( req->body ) {
    case http_body_file:
	/* Get file length */
	if( fstat( fileno(req->body_file), &bodyst ) < 0 ) {
	    /* Stat failed */
	    DEBUG( DEBUG_HTTP, "Stat failed: %s\n", strerror( errno ) );
	    return PROTO_ERROR;
	}
	req->body_size = bodyst.st_size;
	break;
    case http_body_buffer:
	req->body_size = strlen( req->body_buffer );
	break;
    default:
	/* No body, so no size. */
	return PROTO_OK;
    }
    if( req->body != http_body_none ) {
	char tmp[BUFSIZ];
	/* Add the body length header */
	snprintf( tmp, BUFSIZ, "Content-Length: %d" EOL, req->body_size );
	strcat( req->headers, tmp );
    }
    return PROTO_OK;
}

void append_hostport( struct proto_host *host, char *str ) {
    strcat( str, host->hostname );
    /* Only add the port if it isn't 80 */
    if( host->port != HTTP_PORT ) {
	char buffer[128];
	snprintf( buffer, 128, ":%d", host->port );
	strcat( str, buffer );
    }
}

void append_hostport_sbuf( struct proto_host *host, sbuffer buf ) {
    sbuffer_zappend( buf, host->hostname );
    /* Only add the port if it isn't 80 */
    if( host->port != HTTP_PORT ) {
	char buffer[128];
	snprintf( buffer, 128, ":%d", host->port );
	sbuffer_zappend( buf, buffer );
    }
}


/* Lob the User-Agent, connection and host headers in to the request
 * headers */
void http_req_fixedheaders( http_req_t *req ) {
    strcat( req->headers, "User-Agent: " );
    strcat( req->headers, http_useragent );
    strcat( req->headers, EOL );
    if( http_version_major > 0 && http_version_minor > 0 ) {
	strcat( req->headers, "Connection: Keep-Alive" EOL );
    }
    strcat( req->headers, "Host: " );
    append_hostport( &http_server_host, req->headers );
    strcat( req->headers, EOL );
}

/* Initializes the request with given method and URI.
 * URI must be abs_path - i.e., NO scheme+hostname. It will BREAK 
 * otherwise. */
void http_request_init( http_req_t *req, 
			const char *method, const char *uri ) {
    /* Clear it out */
    sbuffer real_uri;
    memset( req, 0, sizeof( http_req_t ) );

    DEBUG( DEBUG_HTTP, "Request starts.\n" );

    /* Add in the fixed headers */
    http_req_fixedheaders( req );

    http_auth_set_creds_cb( &http_server_auth, give_creds, (void *)req );
    http_auth_set_creds_cb( &http_proxy_auth, give_creds, (void *)req );

    req->rbuf = sbuffer_create_sized(BUFSIZ);

    /* Set the standard stuff */
    req->method = method;
    
    real_uri = sbuffer_create();
    if( real_uri ) {
	char *tmp = uri_abspath_escape( uri );
	if( http_use_proxy ) {
	    sbuffer_zappend( real_uri, "http://" );
	    append_hostport_sbuf( &http_server_host, real_uri );
	}
	sbuffer_zappend( real_uri, tmp );
	free( tmp );
	req->uri = sbuffer_finish(real_uri);
    } else {
	/* FIXME */
	return;
    }
    /* TODO: abs_path is offset into req->uri, const it and
     * use a reference */
    req->abs_path = strdup(uri);
    req->body_callback = NULL;
    req->body = http_body_none;
    req->body_want = dav_want_2xx;

#ifdef USE_DAV_LOCKS
    req->if_locks = NULL;
#endif

}

void http_request_end( http_req_t *req ) {
    HTTP_FREE( req->uri );
    HTTP_FREE( req->abs_path );
    sbuffer_destroy( req->rbuf );

#ifdef USE_DAV_LOCKS
    dav_submitlocks_free( req->if_locks );
#endif
    DEBUG( DEBUG_HTTP, "Request ends.\n" );
}

/* Reads a block of the response into buffer, which is of size buflen.
 * Returns number of bytes read, 0 on end-of-response, or -1 on error.
 */
int http_response_read( http_req_t *req, char *buffer, size_t buflen ) 
{
    int willread, readlen;
    if( req->resp_te==http_te_chunked ) {
	/* We are doing a chunked transfer-encoding.
	 * It goes:  `SIZE CRLF CHUNK CRLF SIZE CRLF CHUNK CRLF ...'
	 * ended by a `CHUNK CRLF 0 CRLF', a 0-sized chunk.
	 * The slight complication is that we have to cope with
	 * partial reads of chunks.
	 * For this reason, resp_chunk_left contains the number of
	 * bytes left to read in the current chunk.
	 */
	if( req->resp_chunk_left == 0 ) {
	    long int chunk_len;
	    /* We are at the start of a new chunk. */
	    DEBUG( DEBUG_HTTP, "New chunk.\n" );
	    if( read_line( http_sock, buffer, buflen ) < 0 ) {
		DEBUG( DEBUG_HTTP, "Could not read chunk size.\n" );
		return -1;
	    }
	    DEBUG( DEBUG_HTTP, "[Chunk Size] < %s", buffer );
	    chunk_len = strtol( buffer, NULL, 16 );
	    if( chunk_len == LONG_MIN || chunk_len == LONG_MAX ) {
		DEBUG( DEBUG_HTTP, "Couldn't read chunk size.\n" );
		return -1;
	    }
	    DEBUG( DEBUG_HTTP, "Got chunk size: %ld\n", chunk_len );
	    if( chunk_len == 0 ) {
		/* Zero-size chunk */
		DEBUG( DEBUG_HTTP, "Zero-size chunk.\n" );
		return 0;
	    }
	    req->resp_chunk_left = chunk_len;
	}
	willread = min( buflen - 1, req->resp_chunk_left );
    } else if( req->resp_length > 0 ) {
	/* Have we finished reading the body? */
	if( req->resp_left == 0 )
	    return 0;
	willread = min( buflen - 1, req->resp_left );
    } else {
	/* Read until socket-close */
	willread = buflen - 1;
    }
    DEBUG( DEBUG_HTTP, "Reading %d bytes of response body.\n", willread );
    readlen = sock_read( http_sock, buffer, willread );
    DEBUG( DEBUG_HTTP, "Got %d bytes.\n", readlen );
    if( readlen < 0 ) {
	/* It broke */
	DEBUG( DEBUG_HTTP, "Could not read block.\n" );
	return -1;
    } else if( (readlen == 0) && 
	       ( (req->resp_length > 0) ||
		 (req->resp_te==http_te_chunked) )) {
	/* Premature close before read all of body, or during chunk read. */
	DEBUG( DEBUG_HTTP, "Socket closed before end of body.\n" );
	return -1;
    }
    buffer[readlen] = '\0';
    DEBUG( DEBUG_HTTPBODY, "Read block:\n%s\n", buffer );
    if( req->resp_te==http_te_chunked ) {
	req->resp_chunk_left -= readlen;
	if( req->resp_chunk_left == 0 ) {
	    char crlfbuf[2];
	    /* If we've read a whole chunk, read a CRLF */
	    if( read_data( http_sock, crlfbuf, 2 ) == 0 ) {
		DEBUG( DEBUG_HTTP, "Read CRLF bytes.\n" );
		if( strncmp( crlfbuf, EOL, 2 ) != 0 ) {
		    DEBUG( DEBUG_HTTP, "CRLF bytes didn't contain CRLF!\n" );
		    return -1;
		}
	    } else {
		return -1;
	    }
	}
    } else if( req->resp_length > 0 ) {
	req->resp_left -= readlen;
    }
    return readlen;
}

/* The HTTP/1.x request/response mechanism 
 *
 * Returns:
 *   PROTO_OK if the request was made (not related to status code)
 *   PROTO_ERROR if the request could not be made
 * The STATUS CODE is placed in req->status. The error string is
 * placed in http_error.
 * 
 * TODO: This should be chopped up into smaller chunks, and get rid of
 * the horrid goto's.  
 */
int http_request( http_req_t *req ) 
{
    char buffer[BUFSIZ]; /* For reading from the socket */
    sbuffer buf, request;
    int ret, attempt, proxy_attempt, con_attempt;
    bool using_expect, /* whether we have sent a Expect: 100 header */
	close_connection,
	keep_alive, /* true if the server said keep-alive */
	dead_connection,
	wants_body; /* whether the caller wants the response body
		     * callbacks */

#define HTTP_FATAL_ERROR(a) do {				\
	DEBUG( DEBUG_HTTP, a );					\
	ret = PROTO_ERROR;					\
	close_connection = true;				\
	goto http_request_finish_proc;				\
    } while(0)

#define HTTP_HANDLE_ERROR(a) 			\
do {						\
  DEBUG( DEBUG_HTTP, a "\n" );			\
  strcpy( http_error, a );			\
  ret = PROTO_ERROR;				\
  close_connection = true;			\
  goto http_request_finish_proc;		\
} while(0)

    /* Initialization... */
    DEBUG( DEBUG_HTTP, "Request started...\n" );
    strcpy( http_error, "Unknown error." );
    ret = PROTO_OK;

    if( http_req_bodysize( req ) != PROTO_OK )
	return PROTO_ERROR;

    buf = req->rbuf;
    sbuffer_clear(buf);
    request = sbuffer_create();

    /* I shall try this only twice...
     * First time, with default authentication stuff (either, what we
     * did last time, or none at all), then with up-to-the-minute
     * what-the-server-requested authentication stuff. */

    proxy_attempt = attempt = con_attempt = 1;
    
    do {
	char *authinfo = NULL, *www_auth = NULL, 
	    *proxy_auth = NULL, *proxy_authinfo = NULL,
	    *tmp;

	keep_alive = 0;

	/* Note that we pass the abs_path here... */
	http_auth_new_request( &http_server_auth, req->method, req->uri,
			       req->body_buffer, req->body_file );
	if( http_use_proxy ) {
	    /* ...and absoluteURI here. */
	    http_auth_new_request( &http_proxy_auth, req->method, req->uri,
				   req->body_buffer, req->body_file );
	}

	/* Empty the buffer for another time round */
	sbuffer_clear(request);
	/* If we are talking to a proxy, we send them the absoluteURI
	 * as the Request-URI. If we are talking to a server, we just 
	 * send abs_path. */
	sbuffer_concat( request, req->method, " ", 
			http_use_proxy?req->uri:req->abs_path, " HTTP/1.1" EOL,
			req->headers, NULL );

	/* Add the authorization headers in */
	tmp = http_auth_request( &http_server_auth );
	if( tmp != NULL ) {
	    sbuffer_concat( request, "Authorization: ", tmp, NULL );
	    free( tmp );
	}
	tmp = http_auth_request( &http_proxy_auth );
	if( tmp != NULL ) {
	    sbuffer_concat( request, "Proxy-Authorization: ", tmp, NULL );
	    free( tmp );
	}

	/* Now handle the body. */
	using_expect = false;
	if( req->body!=http_body_none ) {
	    if( (http_expect_works > -1) &&
		(req->body_size > HTTP_EXPECT_MINSIZE) 
		) {
		/* Add Expect: 100-continue. */
		sbuffer_zappend( request, "Expect: 100-continue" EOL );
		using_expect = true;
	    }
	}

#ifdef USE_DAV_LOCKS
	{
	    char *tmp = dav_lock_ifheader( req );
	    if( tmp != NULL ) {
		sbuffer_zappend( request, tmp );
		free( tmp );
		if( http_version_major >= 1 ) {
		    if( http_version_minor < 1 ) {
			/* HTTP/1.0 */
			sbuffer_zappend( request, "Pragma: no-cache" EOL );
		    } else {
			/* HTTP/1.1 and above */
			sbuffer_zappend( request, "Cache-Control: no-cache" EOL );
		    }
		} else {
		    /* TODO: Go to a psychiatrist. */
		}
	    }
	}	
#endif /* USE_DAV_LOCKS */

	/* Final CRLF */
	sbuffer_zappend( request, EOL );
	
	/* Now send the request */

	/* Open the connection if necessary */
	if( !http_connected ) {
	    if( (ret = http_open()) != PROTO_OK ) {
		strcpy( http_error, "Could not connect to server" );
		return ret;
	    }
	}

	dead_connection = false;

	/* Send the headers */
#ifdef DEBUGGING
	if( (DEBUG_HTTPPLAIN&debug_mask) == DEBUG_HTTPPLAIN ) { 
	    /* Display everything mode */
	    DEBUG( DEBUG_HTTP, "Sending request headers:\n%s", 
		   sbuffer_data(request) );
	} else {
	    /* Blank out the Authorization paramaters */
	    char *reqdebug = strdup(sbuffer_data(request)), *pnt = reqdebug;
	    while( (pnt = strstr( pnt, "Authorization: ")) != NULL ) {
		for( pnt += 15; *pnt != '\r' && *pnt != '\0'; pnt++ ) {
		    *pnt = 'x';
		}
	    }
	    DEBUG( DEBUG_HTTP, "Sending request headers:\n%s", reqdebug );
	    free( reqdebug );
	}
#endif /* DEBUGGING */
	if( send_string( http_sock, sbuffer_data(request) ) < 0 ) {
	    dead_connection = true;
	    HTTP_HANDLE_ERROR( "Could not send request." );
	}

	DEBUG( DEBUG_FILES, "Request size: %d\n", sbuffer_size(request) );
	DEBUG( DEBUG_HTTP, "Request sent\n" );
	
	/* Now, if we are doing a Expect: 100, hang around for a short
	 * amount of time, to see if the server actually cares about the 
	 * Expect and sends us a 100 Continue response if the request
	 * is valid, else an error code if it's not. This saves sending
	 * big files to the server when they will be rejected.
	 */
	
	if( using_expect ) {
	    DEBUG( DEBUG_HTTP, "Waiting for response...\n" );
	    ret = sock_block( http_sock, HTTP_EXPECT_TIMEOUT );
	    switch( ret ) {
	    case -1: /* error */
		HTTP_FATAL_ERROR( "Wait (select) failed.\n" );
		break;
	    case 1: /* we got us a response! */
		DEBUG( DEBUG_HTTP, "Wait got data.\n" );
		http_expect_works = 1; /* it works - use it again */
		break;
	    case 0: 
		/* Timed out - i.e. Expect: ignored. There is a danger
		 * here that the server DOES respect the Expect: header,
		 * but was going SO slowly that it didn't get time to
		 * respond within HTTP_EXPECT_TIMEOUT.
		 * TODO: while sending the body, check to see if the
		 * server has sent anything back - if it HAS, then
		 * stop sending - this is a spec compliance SHOULD */
		DEBUG( DEBUG_HTTP, "Wait timed out.\n" );
		http_expect_works = -1; /* don't try that again */
		/* and give them the body */
		if( http_req_sendbody( req ) != PROTO_OK ) {
		    dead_connection = true;
		    HTTP_HANDLE_ERROR( "Could not send request body" );
		}
		break;
	    }
	} else if( req->body != http_body_none ) {
	    /* Just chuck the file down the socket */
	    DEBUG( DEBUG_HTTP, "Sending body...\n" );
	    if( http_req_sendbody( req ) == PROTO_ERROR ) {
		dead_connection = true;
		HTTP_HANDLE_ERROR( "Could not send request body to server." );
	    }
	    DEBUG( DEBUG_HTTP, "Body sent.\n" );
	}
	
	/* Now, we have either:
	 *   - Sent the header and body, or
	 *   - Sent the header incl. Expect: line, and got some response.
	 * In any case, we get the status line of the response.
	 */
	
	/* HTTP/1.1 says that the server MAY emit any number of
	 * interim 100 (Continue) responses prior to the normal
	 * response.  So loop while we get them.  */
	
	do {
	    if( read_line( http_sock, sbuffer_data(buf), BUFSIZ ) < 0 ) {
		dead_connection = true;
		HTTP_HANDLE_ERROR( "Could not read response status line." );
	    }

	    DEBUG( DEBUG_HTTP, "[Status Line] < %s", sbuffer_data(buf) );
	    
	    /* Got the status line - parse it */
	    if( http_parse_status( req, sbuffer_data(buf) ) == PROTO_ERROR )
		HTTP_HANDLE_ERROR( "Could not parse response status line." );

	    if( req->class == 1 ) {
		DEBUG( DEBUG_HTTP, "Got 1xx-class.\n" );
		/* Skip any headers, we don't need them */
		do {
		    if( read_line( http_sock, sbuffer_data(buf), BUFSIZ ) < 0 ) {
			dead_connection = true;
			HTTP_HANDLE_ERROR( "Could not read response header" );
		    }
		    DEBUG( DEBUG_HTTP, "[Ignored header] < %s", 
			   sbuffer_data(buf) );
		} while( strcmp( sbuffer_data(buf), EOL ) != 0 );
	
		if( using_expect && (req->status == 100) ) {
		    /* We are using Expect: 100, and we got a 100-continue 
		     * return code... send the request body */
		    DEBUG( DEBUG_HTTP, "Got continue... sending body now.\n" );
		    if( http_req_sendbody( req ) != PROTO_OK )
			HTTP_FATAL_ERROR( "Could not send body.\n" );
		    DEBUG( DEBUG_HTTP, "Body sent.\n" );
		}
	    }
	} while( req->class == 1 );
	
	/* We've got the real status line... now get the headers */
	
	req->resp_length = -1;
	req->resp_te = http_te_unknown;
	close_connection = false;
	
	/* Now read the rest of the header... up to the next blank line */
	while( read_line( http_sock, sbuffer_data(buf), BUFSIZ ) > 0 ) {
	    char extra[BUFSIZ], *pnt;
	    DEBUG( DEBUG_HTTP, "[Header:%d] < %s", 
		   strlen(sbuffer_data(buf)), sbuffer_data(buf) );
	    if( strcmp( sbuffer_data(buf), EOL ) == 0 ) {
		DEBUG( DEBUG_HTTP, "CRLF: End of headers.\n" );
		break;
	    }
	    while(true) {
		/* Collect any extra lines into buffer */
		ret = sock_recv( http_sock, extra, 1, MSG_PEEK);
		if( ret <= 0 ) {
		    HTTP_FATAL_ERROR( "Couldn't peek at next line.\n" );
		}
		if( extra[0] != ' ' && extra[0] != '\t' ) {
		    /* No more headers */
		    break;
		}
		ret = read_line( http_sock, extra, BUFSIZ );
		if( ret == -2 ) {
		    /* No newline within BUFSIZ bytes. This is a 
		     * loooong header. */
		    DEBUG( DEBUG_HTTP, 
			   "Header line longer than buffer, skipped.\n" );
		    break;
		} else if( ret <= 0 ) { 
		    HTTP_FATAL_ERROR( "Couldn't read next header line.\n" );
		} else {
		    DEBUG( DEBUG_HTTP, "[Cont:%d] < %s", strlen(extra), extra);
		}
		/* Append a space to the end of the last header, in
		 * place of the CRLF. */
		pnt = strchr( sbuffer_data(buf), '\r' );
		pnt[0] = ' '; pnt[1] = '\0';
		/* Skip leading whitespace off next line */
		for( pnt = extra; *pnt!='\0' && 
			 ( *pnt == ' ' || *pnt =='\t' ); pnt++ ) /*oneliner*/;
		DEBUG( DEBUG_HTTP, "[Continued] < %s", pnt );
		if( sbuffer_zappend( buf, pnt ) ) {
		    HTTP_FATAL_ERROR( "Out of memory" );
		}
	    }
	    /* Now parse the header line. This is all a bit noddy. */
	    pnt = strchr( sbuffer_data(buf), ':' );
	    if( pnt != NULL ) {
		char *name, *value;
		/* Null-term name at the : */
		*pnt = '\0';
		name = sbuffer_data(buf);
		/* Strip leading whitespace from the value */		
		for( value = pnt+1; *value!='\0' && *value==' '; value++ )
		    /* nullop */;
		STRIP_EOL( value );
		DEBUG( DEBUG_HTTP, "Header Name: [%s], Value: [%s]\n",
		       name, value );
		if( strcasecmp( name, "Content-Length" ) == 0 ) {
		    /* TODO: 2068 says we MUST notify the user if this
		     * is not a real number. */
		    req->resp_length = atoi( value );
		} else if( strcasecmp( name, "Transfer-Encoding" ) == 0 ) {
		    if( strcasecmp( value, "chunked" ) == 0 ) {
			req->resp_te = http_te_chunked;
		    } else {
			req->resp_te = http_te_unknown;
		    }
		} else if( strcasecmp( name, "Connection" ) == 0 ) {
		    if( strcasecmp( value, "close" ) == 0 ) {
			close_connection = true;
		    } else if( strcasecmp( value, "Keep-Alive" ) == 0 ) {
			keep_alive = 1;
		    }
		} else if( strcasecmp( name, "WWW-Authenticate" ) == 0 ) {
		    /* Parse the authentication challenge */
		    www_auth = strdup( value );
		} else if( strcasecmp( name, "Proxy-Authenticate" ) == 0 ) {
		    proxy_auth = strdup( value );
		} else if( strcasecmp( name, "Authentication-Info" ) == 0 ) {
		    authinfo = strdup( value );
		} else if( strcasecmp( name, "Proxy-Authentication-Info" ) == 0 ) {
		    proxy_authinfo = strdup( value );
		} else if( req->hdrs_callback != NULL ) {
		    (*req->hdrs_callback)( name, value );
		}
	    }
	}

	/* Body length calculation, bit icky.
	 * Here, we set:
	 * length==-1 if we DO NOT know the exact body length
	 * length>=0 if we DO know the body length.
	 *
	 * RFC2068, section 4.3: 
	 * NO body is returned if the method is HEAD, or the resp status
	 * is 204 or 304
	 */
	if( (strcmp( req->method, "HEAD" ) == 0 ) ||
	    req->status==204 ||
	    req->status==304 ) {
	    req->resp_length = 0;
	} else {
	    /* RFC2068, section 4.4: if we have a transfer encoding
	     * and a content-length, then ignore the content-length. */
	    if( (req->resp_length>-1) && 
		(req->resp_te!=http_te_unknown) ) {
		req->resp_length = -1;
	    }
	}

	/* Do they want the body? */
	wants_body = (*req->body_want)( req );

	if( req->resp_length != 0 ) {
	    /* Now, read the body */
	    int readlen;
	    req->resp_left = req->resp_length;
	    req->resp_chunk_left = 0;
	    do {
		/* Read a block */
		readlen = http_response_read( req, buffer, BUFSIZ );
		DEBUG( DEBUG_HTTP, "Read %d bytes.\n", readlen );
		if( readlen > -1 ) {
		    /* What to do with the body block */
		    if( req->body_callback && wants_body ) 
			(*req->body_callback)( req->body_callback_userdata,
					       buffer, readlen );
		    http_auth_response_body( &http_server_auth, 
					     buffer, readlen );
		} else {
		    dead_connection = true;
		    HTTP_HANDLE_ERROR( "Error reading response from server" );
		}
	    } while( readlen > 0 );
	    if( (readlen == 0) && (req->resp_te == http_te_chunked) ) {
		char *pnt;
		/* Read till CRLF - handle trailing headers */
		do {
		    if( read_line( http_sock, buffer, BUFSIZ ) < 0 ) {
			/* It broke */
			HTTP_FATAL_ERROR( "Could not read trailer.\n" );
		    }
		    DEBUG( DEBUG_HTTP, "[Chunk trailer] %s", buffer );
		    /* Now parse the header line. */
		    pnt = strchr( buffer, ':' );
		    if( pnt != NULL ) {
			char *name, *value;
			/* Null-term name at the : */
			*pnt = '\0';
			name = buffer;
			/* Strip leading whitespace from the value */ 
			for( value = pnt+1; 
			     *value!='\0' && *value==' '; value++ )
			    /* nullop */;
			STRIP_EOL( value );
			DEBUG( DEBUG_HTTP, "Header Name: [%s], Value: [%s]\n",
			       name, value );
			if( strcasecmp( name, "Authentication-Info" ) == 0 && 
			    authinfo == NULL ) {
			    authinfo = strdup( value );
			} else if( 
			    strcasecmp( name, "Proxy-Authentication-Info" ) == 0 && 
			    proxy_authinfo == NULL ) {
			    proxy_authinfo = strdup( value );
			}
		    }
		} while( strcmp( buffer, EOL ) != 0 );
	    }
	}

	if( proxy_authinfo != NULL && 
	    http_auth_verify_response( &http_proxy_auth, proxy_authinfo ) ) {
	    DEBUG( DEBUG_HTTP, "Proxy response authentication invalid.\n" );
	    ret = PROTO_ERROR;
	    sprintf( http_error, "Proxy was not authenticated correctly." );
	} else if( authinfo != NULL &&
		   http_auth_verify_response( &http_server_auth, authinfo ) ) {
	    DEBUG( DEBUG_HTTP, "Response authenticated as invalid.\n" );
	    ret = PROTO_ERROR;
	    sprintf( http_error, "Server was not authenticated correctly." );
	} else if( req->status == 401 && 
		   www_auth != NULL && 
		   attempt == 1) {
	    /* Authenticate me, only once though 
	     * FIXME: what to do with this return value */
	    (void) http_auth_challenge( &http_server_auth, www_auth );
	} else if( req->status == 407 && 
		   proxy_auth != NULL && 
		   proxy_attempt == 1 ) {
	    /* FIXME as above */
	    (void) http_auth_challenge( &http_proxy_auth, proxy_auth );
	} else {
	    ret = PROTO_OK;
	}
	DOFREE( www_auth );
	DOFREE( proxy_auth );
	DOFREE( authinfo );
	DOFREE( proxy_authinfo );
	
http_request_finish_proc:
	if( (http_version_major < 1 || 
	    (http_version_major == 1 && http_version_minor < 1)) &&
	    !keep_alive ) {
	    DEBUG( DEBUG_HTTP, "Persistent connection not probable.\n" );
	    close_connection = true;
	}
	/* Now, do we close the connection? */
	if( close_connection ) {
	    DEBUG( DEBUG_HTTP, "Forced connection close.\n" );
	    http_close( );
	} else if( http_conn_limit ) {
	    DEBUG( DEBUG_HTTP, "Limited connection close.\n" );
	    http_close( );
	}
    
	/* Now, do that all *again* if it didn't work.
	 * Otherwise, give up */

    } while( 
	(dead_connection && (++con_attempt<4)) || 
	((++attempt<3) && (req->status == 401)) || 
	((++proxy_attempt<3) && (req->status == 407)) );
    
    sbuffer_destroy(request);
    
    DEBUG( DEBUG_HTTP, "Req ends, status %d class %dxx, status line:\n%s\n", 
	   req->status, req->class, http_error );

    if( req->status == 401 ) {
	return PROTO_AUTH;
    } else {
	return ret;
    }
}

time_t http_getmodtime_time;

/* Header parser to retrieve Last-Modified date */
static void http_get_lastmod( const char *header, const char *value ) {
    if( strcasecmp( header, "Last-Modified" ) == 0 ) {
	http_getmodtime_time = http_dateparse( value );
    }
}

int http_getmodtime( const char *remote, time_t *modtime ) {
    http_req_t req;
    int ret;

    http_request_init( &req, "HEAD", remote );
    req.hdrs_callback = http_get_lastmod;
    dav_lock_using_resource( &req, remote, dav_lockusage_read, 0 );

    http_getmodtime_time = -1;
    
    ret = http_request( &req );

    if( ret == PROTO_OK && req.class == 2 ) {
	*modtime = http_getmodtime_time;
    } else {
	*modtime = -1;
	ret = PROTO_ERROR;
    }

    return ret;
}

/* Simple HTTP put. 
 * local is the local filename. Remote is the destination URI (URI?)
 * Make it proper.
 * Returns:
 *   PROTO_FILE if no local file
 *   PROTO_ERROR if something general goes wrong
 *   PROTO_OK if it all works fine
 */
int http_put( const char *local, const char *remote, const bool ascii ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "PUT", remote );

    dav_lock_using_resource( &req, remote, dav_lockusage_write, 0 );
    dav_lock_using_parent( &req, remote );

    /* joe: ANSI C says the "b" will be ignored by platforms which
     * should ignore it... but, we'll play safe: */
#if defined (__EMX__) || defined(__CYGWIN__)
    req.body_file = fopen( local, "rb" );
#else
    req.body_file = fopen( local, "r" );
#endif
    if( req.body_file == NULL ) {
	strcpy( http_error, "Could not open file." );
	ret = PROTO_FILE;
    } else {
	req.body = http_body_file;
	
	ret = http_request( &req );
	fclose( req.body_file );
	
	if( ret == PROTO_OK && req.class != 2 )
	    ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

/* Conditional HTTP put. 
 * local is the local filename. Remote is the destination URI (URI?)
 * Make it proper.
 * Returns:
 *   PROTO_FILE if no local file
 *   PROTO_ERROR if something general goes wrong
 *   PROTO_OK if it all works fine
 */
int http_put_if_unmodified( const char *local, const char *remote, 
			    const bool ascii, const time_t since ) {
    http_req_t req;
    char *date;
    time_t modtime;
    int ret;
    
    if( (http_version_major < 1) || 
	((http_version_major == 1) && (http_version_minor < 1))
	) {
	/* Server is not minimally HTTP/1.1 compliant.
	 * Do a HEAD to check the remote mod time */
	if( http_getmodtime( remote, &modtime ) != PROTO_ERROR )
	    return PROTO_ERROR;
	if( modtime != since )
	    return PROTO_FAILED;
    }

    http_request_init( &req, "PUT", remote );
    date = rfc1123_date( since );
    /* Add in the conditionals */
    strcat( req.headers, "If-Unmodified-Since: " );
    strcat( req.headers, date );
    strcat( req.headers, EOL );
    free( date );

    dav_lock_using_resource( &req, remote, dav_lockusage_write, 0 );
    /* NB: this CANNOT modify the parent, since the resource must
     * already exist */

    req.body_file = fopen( local, "r" );
    if( req.body_file == NULL ) {
	strcpy( http_error, "Could not open file." );
	ret = PROTO_FILE;
    } else {
	req.body = http_body_file;
	
	ret = http_request( &req );
	fclose( req.body_file );
	
	if( ret == PROTO_OK ) {
	    if( req.status == 412 ) {
		ret = PROTO_FAILED;
	    } else if( req.class != 2 ) {
		ret = PROTO_ERROR;
	    }
	}
    }

    http_request_end( &req );
    return ret;
}

int http_get_fd;
bool http_get_working;
size_t http_get_total, http_get_progress;

void http_get_callback( void *user, const char *buffer, const size_t len ) {
    http_req_t *req = user;
    if( !http_get_working ) return;
    DEBUG( DEBUG_HTTP, "Got progress: %d out of %d\n", len, http_get_total );
    if( send_data( http_get_fd, buffer, len ) < 0 ) {
	http_get_working = false;
    } else {
	http_get_progress += len;
	if( http_get_progress > http_get_total ) {
	    /* Reset the counter if we're uploading it again */
	    http_get_progress -= http_get_total;
	}
	fe_transfer_progress( http_get_progress, req->resp_length ); 
    }
}

static void 
http_read_file_block( void *userdata, const char *s, const size_t len ) {
    fe_transfer_progress( http_get_progress, http_get_total );
    (*http_read_file_callback)( userdata, s, len );
}

int 
http_read_file( const char *remote, const size_t remotesize, 
		proto_read_block reader, void *userdata ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "GET", remote );
    req.body_callback = http_read_file_block;
    req.body_callback_userdata = userdata;
    http_get_total = remotesize;
    http_read_file_callback = reader;

    dav_lock_using_resource( &req, remote, dav_lockusage_read, 0 );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;

    http_request_end( &req );

    return ret;
}

int http_get( const char *local, const char *remote, const size_t remotesize,
	      int flags, mode_t mode ) 
{
    http_req_t req;
    int ret;

#if defined (__EMX__) || defined (__CYGWIN__)
    /* We have to set O_BINARY, thus need open(). Otherwise it should be
       equivalent to creat(). */
    http_get_fd = open( local, flags | O_TRUNC | O_WRONLY | O_BINARY, 0644 );
#else
    http_get_fd = open( local, flags | O_TRUNC | O_WRONLY, mode );
#endif
    if( http_get_fd < 0 ) {
	snprintf( http_error, BUFSIZ, "Could not open local file: %s", 
		  strerror( errno ) );
	return PROTO_ERROR;
    }

    http_request_init( &req, "GET", remote );
    req.body_callback = http_get_callback;
    req.body_callback_userdata = &req;

    dav_lock_using_resource( &req, remote, dav_lockusage_read, 0 );

    http_get_working = true;
    http_get_progress = 0;
    http_get_total = remotesize;

    DEBUG( DEBUG_HTTP, "Total remote size: %d\n", remotesize );

    ret = http_request( &req );
    
    if( close( http_get_fd ) < 0 ) {
	snprintf( http_error, BUFSIZ, "Error closing local file: %s",
		  strerror( errno ) );
	ret = PROTO_ERROR;
    } else if( ret == PROTO_OK && req.class != 2 ) {
	ret = PROTO_ERROR;
    }
    http_request_end( &req );
    return ret;
}

int dav_copy( const char *from, const char *to ) {
    http_req_t req;
    int ret;

    http_request_init( &req, "COPY", from );

    if( init_207errors( &req ) ) return PROTO_ERROR;

    /* Copy needs to read the source, and write to the destination */    
    dav_lock_using_resource( &req, from, dav_lockusage_read, DAV_DEPTH_INFINITE );
    dav_lock_using_resource( &req, to, dav_lockusage_write, DAV_DEPTH_INFINITE );
    /* And we need to be able to add members to the destination's parent */
    dav_lock_using_parent( &req, to );

    strcat( req.headers, "Destination: http://" );
    append_hostport( &http_server_host, req.headers );
    strcat( req.headers, to );
    strcat( req.headers, EOL );
    strcat( req.headers, "Overwrite: F" EOL );

    if( ! http_webdav_server ) {
	/* For non-WebDAV servers */
	strcat( req.headers, "New-URI: " );
	strcat( req.headers, to );
	strcat( req.headers, EOL );
    }
    
    ret = http_request( &req );
    
    finish_207errors( &req );

    if( ret == PROTO_OK && (req.status == 207 ||  req.class != 2) ) {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

/* Perform the file operations */
int dav_move( const char *from, const char *to ) {
    http_req_t req;
    int ret;

    http_request_init( &req, "MOVE", from );

    if( init_207errors( &req ) ) return PROTO_ERROR;

    dav_lock_using_resource( &req, from, dav_lockusage_write, DAV_DEPTH_INFINITE );
    dav_lock_using_parent( &req, from );
    dav_lock_using_resource( &req, to, dav_lockusage_write, DAV_DEPTH_INFINITE );
    dav_lock_using_parent( &req, to );

    strcat( req.headers, "Destination: http://" );
    append_hostport( &http_server_host, req.headers );
    strcat( req.headers, to );
    strcat( req.headers, EOL );
    strcat( req.headers, "Overwrite: F" EOL );

    if( ! http_webdav_server ) {
	/* For non-WebDAV servers */
	strcat( req.headers, "New-URI: " );
	strcat( req.headers, to );
	strcat( req.headers, EOL );
    }

    ret = http_request( &req );

    finish_207errors( &req );
    
    if( ret == PROTO_OK && (req.status == 207 || req.class != 2) ) {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

/* Deletes the specified resource on the server */
int http_delete( const char *filename ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "DELETE", filename );
    if( init_207errors( &req ) ) return PROTO_ERROR;
    
    dav_lock_using_resource( &req, filename, dav_lockusage_write, DAV_DEPTH_INFINITE );
    dav_lock_using_parent( &req, filename );
    
    ret = http_request( &req );
    finish_207errors( &req );
    
    if( ret == PROTO_OK && (req.status==207 || req.class != 2) )
	ret = PROTO_ERROR;

    http_request_end( &req );
    return ret;
}

/* Deletes the specified resource on the server.
 * I wish we could do Depth: 0 with DELETE. */
int dav_rmdir( const char *filename ) {
    http_req_t req;
    int ret;
    char *dirname;
    
    if( strlen( filename ) < 1 ) {
	return PROTO_ERROR;
    }

    if( *(filename+strlen(filename)) != '/' ) {
	CONCAT2( dirname, filename, "/" );
    } else {
	dirname = strdup( filename );
    }

    http_request_init( &req, "DELETE", dirname );
    
    ret = http_request( &req );

    /* If we get a 207, something has gone wrong */
    if( ret == PROTO_OK && ( (req.status == 207) || (req.class != 2) ) )
	ret = PROTO_ERROR;

    free( dirname );
    http_request_end( &req );
    return ret;

}

int dav_mkcol( const char *dirname ) 
{
    http_req_t req;
    int ret;
    char *realdir;

    if( strlen( dirname ) < 1 ) {
	strcpy( http_error, "Invalid directory name." );
	return PROTO_ERROR;
    }
    if( *(dirname+strlen(dirname)-1) == '/' ) {
	realdir = strdup( dirname );
    } else {
	/* +2 since one for \0, one for / */
	CONCAT2( realdir, dirname, "/" );
    }
    
    http_request_init( &req, "MKCOL", realdir );
    
    dav_lock_using_resource( &req, realdir, dav_lockusage_write, 0 );
    dav_lock_using_parent( &req, realdir );
    if( init_207errors( &req ) ) return PROTO_ERROR;
    
    ret = http_request( &req );
    
    finish_207errors( &req );
    
    if( ret == PROTO_OK && req.class != 2 ) {
	ret = PROTO_ERROR;
    }

    free( realdir );
    http_request_end( &req );
    return ret;
}

static int dummy_check( hip_xml_elmid p, hip_xml_elmid c ) {
    switch( p ) {
    case DAV_ELM_prop:
	if( c != HIP_ELM_unknown ) {
	    return -1;
	}
	break;
    case DAV_ELM_response:
	if( c != DAV_ELM_prop ) {
	    return -1;
	}
	break;
    }
    return 0;
}

int dav_want_207( http_req_t *req ) {
    return (req->status == 207);
}

int dav_want_2xx( http_req_t *req ) {
    return (req->class == 2);
}

static int init_207errors( http_req_t *req ) {
    static const struct hip_xml_elm unk[] = {
	{ "@<unknown>@", HIP_ELM_unknown, 0 },
	{ NULL, 0 }
    };
    static const struct hip_xml_elmlist dummy = 
    { unk, dummy_check, NULL, NULL, NULL };
    req->elmlist_207 = dummy;   

    if( dav_207_init( &req->parser_207, &req->elmlist_207 ) ) {
	strcpy( http_error, "Could not initialize parser." );
	return -1;
    }
    req->body_callback = hip_xml_parse_v;
    req->body_callback_userdata = &req->parser_207;
    req->body_want = dav_want_207;
    return 0;
}

/* Returns zero on success or negative on error */
static void finish_207errors( http_req_t *req ) {
    int invalid = dav_207_finish( &req->parser_207 );
    if( req->status != 207 ) {
	return;
    } else if( invalid ) {
	DEBUG( DEBUG_HTTP, "207 error.\n" );
	strcpy( http_error, req->parser_207.parser.error );
    } else {
	dav_207_write_errors( req, &req->parser_207 );
	dav_207_free( &req->parser_207, NULL, NULL );
    }
}

int dav_207_write_errors( http_req_t *req, struct dav_207_parser *p ) 
{
    /* write the error */
    sbuffer buf;
    struct dav_response *rsp;
    buf = sbuffer_create();
    if( !buf ) {
	strcpy( http_error, "Out of memory." );
	return -1;
    }
    for( rsp = p->ms.first; rsp != NULL; rsp = rsp->next ) {
	if( rsp->status_line && rsp->href ) {
	    sbuffer_concat( buf, rsp->href, ": ", rsp->status_line, "\n",
			    NULL );
	} else {
	    struct dav_propstat *pst;
	    /* TODO: put something meaningful here... this is quite
	     * tricky. */
	    for( pst = rsp->first; pst != NULL; pst = pst->next ) {
		sbuffer_concat( buf, rsp->href, ": ", pst->status_line, "\n",
				NULL );
	    }
	}
    }
    snprintf( http_error, BUFSIZ, sbuffer_finish(buf) );
    STRIP_EOL( http_error );
    return 0;
}

int http_open( void ) {
    if( http_use_proxy ) {
	DEBUG( DEBUG_SOCKET, "Connecting to proxy at %s:%d...\n", 
	       http_proxy_host.hostname, http_proxy_host.port );
    } else {
	DEBUG( DEBUG_SOCKET, "Connecting to server at %s:%d...\n", 
	       http_server_host.hostname, http_server_host.port );
    }
    fe_connection( fe_connecting );
    http_sock = socket_connect( http_remoteaddr, http_remoteport);
    if( http_sock < 0 ) {
	DEBUG( DEBUG_SOCKET, "Could not connect: %s\n", strerror( errno ) );
	return PROTO_CONNECT;
    }
    DEBUG( DEBUG_SOCKET, "Connected.\n" );
    fe_connection( fe_connected );
    http_connected = true;
    return PROTO_OK;
}

int http_close( void ) {
    DEBUG( DEBUG_SOCKET, "Closing socket.\n" );
    socket_close( http_sock );
    http_connected = false;
    DEBUG( DEBUG_SOCKET, "Socket closed.\n" );
    return PROTO_OK;
}

int http_head( const char *collection ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "HEAD", collection );
    dav_lock_using_resource( &req, collection, dav_lockusage_read, 0 );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}

static void http_options_parsehdr( const char *name, const char *value ) {
    char **classes, **class;
    if( strcasecmp( name, "DAV" ) == 0 ) {
	DEBUG( DEBUG_HTTP, "Got OPTIONS header with value: %s\n", value );
	classes = split_string( value, ',', http_quotes, http_whitespace );
	for( class = classes; *class!=NULL; class++ ) {
	    DEBUG( DEBUG_HTTP, "Got compliance class: [%s]\n", *class );
	    if( strcmp( *class, "1" ) == 0 ) {
		DEBUG( DEBUG_HTTP, "Class 1 compliant server.\n" );
		http_webdav_server = true;
	    } else if( strcmp( *class, "2" ) == 0 ) {
		DEBUG( DEBUG_HTTP, "Class 2 compliant server.\n" );
		http_webdav_locking = true;
	    }
	}
	split_string_free( classes );
    }
}

/* Performs an OPTIONS request.
 * Sets http_webdav_server appropriately.
 */
int http_options( const char *collection ) {
    http_req_t req;
    int ret;
    
    http_webdav_server = false;

    http_request_init( &req, "OPTIONS", collection );
    req.hdrs_callback = http_options_parsehdr;
    dav_lock_using_resource( &req, collection, dav_lockusage_read, 0 );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}
