/* 
   WebDAV / HTTP authoring client routines,
   Copyright (C) 1998, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef HTTPDAV_H
#define HTTPDAV_H

/* For fetch_list */
#include "protocol.h"
#include "dav_207.h"

#ifdef USE_DAV_LOCKS

#include "string_utils.h" /* for sbuffer */

#define DAV_TIMEOUT_INFINITE -1
#define DAV_TIMEOUT_INVALID -2

struct dav_lock;
struct dav_submit_locks;

extern struct dav_lock **dav_lock_list;

enum dav_lock_usage {
    dav_lockusage_write,
    dav_lockusage_read
};

#endif /* USE_DAV_LOCKS */

extern char http_error[];

/* Set by http_options to true if server is a WebDAV class 1 server */
extern bool http_webdav_server;
/* Set by http_options to true if server is a WebDAV class 2 server */
extern bool http_webdav_locking;
/* Set to false to bypass http_init OPTIONS check */
extern bool http_init_checks;
/* Set to enable "Expect: 100-continue" support: NOT for Apache 1.3.x
 * where x <= 6. */
extern bool http_enable_expect;
/* Set to limit the number of requests per connection */
extern bool http_conn_limit;

#define REQSIZ 2048

typedef enum {
    http_body_buffer,
    http_body_file,
    http_body_none
} http_body_t;

/* The transfer encoding types */
enum http_te_t {
    /* TODO: Support other types */
    http_te_none,
    http_te_chunked,
    http_te_unknown
};

/* This is called with each of the headers in the response */
typedef void (*http_request_hdrs_read_t) (
    const char *name,
    const char *value );

struct http_req_s;
typedef struct http_req_s http_req_t;

typedef int (*http_want_body) ( http_req_t * );

/* HTTP request information */
struct http_req_s {
    /* Fill in these with http_req_init */
    const char *method;
    char *uri;
    char *abs_path;
    /* Fill in these yourself */
    /* TODO: abstract this... linked-list of key/value pairs will do */
    char headers[REQSIZ];
    /* Set these if you want to send a request body */
    http_body_t body;
    FILE *body_file;
    const char *body_buffer;
    /* Set this is you want to read the response headers */
    http_request_hdrs_read_t hdrs_callback;
    /* Set this if you want to read the response body */
    proto_read_block body_callback;
    http_want_body body_want;
    /* Response entity-body transfer-encoding */
    enum http_te_t resp_te;
    /* Response entity-body content-length */
    int resp_length;
    /* Bytes left to read */
    int resp_left;
    long int resp_chunk_left;
    /* If non-NULL, this will be set to the total size of the
     * response body */
    size_t *resp_body_size;
    /* If non-NULL, passed as first parameter to body_callback */
    void *body_callback_userdata;
    /* We fill in these for you */
    size_t body_size;
    /* Buffer for reading into */
    sbuffer rbuf;
    /* You use this for the status response code */
    int status;
    int class; /* == status/100 */
#ifdef USE_DAV_LOCKS
    /* List of locks to submit */
    struct dav_submit_locks *if_locks;
#endif /* USE_DAV_LOCKS */
    struct dav_207_parser parser_207;
    struct hip_xml_elmlist elmlist_207;
};

#if 0
/* Session support... threadsafeness */
typedef struct {
    int socket;
    struct proto_host server;
    struct proto_host proxy;
    http_auth_session server_auth;
    http_auth_session proxy_auth;
    bool use_proxy;
    /* The last HTTP-Version returned by the server */
    int version_major;
    int version_minor;
    http_req_t current_request; /* or have a ->session in http_req */
    char error[BUFSIZ];
} http_session;
#endif

/* Public Interface:
 * Call:
 *    http_init( "/", "www.myhttpserver.com", 80, "myusername", "mypassword" );
 * then any of the method functions, in any order
 *    http_put, http_delete, dav_move, dav_rmdir, dav_mkcol.
 * lastly:
 *    http_finish( );
 */

extern const char *http_quotes;
extern const char *http_whitespace;

#define DAV_DEPTH_INFINITE 2

int http_init( const char *remote_root,
	       struct proto_host *server, struct proto_host *proxy );
int http_finish( void );

int http_head( const char *directory );
int http_options( const char *directory );

int http_put( const char *local, const char *remote, const bool ascii );
int http_put_if_unmodified( const char *local, const char *remote, 
			    const bool ascii, const time_t time );
int http_getmodtime( const char *remote, time_t *modtime );
int http_delete( const char *filename );
int http_get( const char *local, const char *remote, const size_t remotesize,
	      int flags, mode_t mode );
int 
http_read_file( const char *remote, const size_t remotesize,
		proto_read_block reader, void *userdata );

int dav_move( const char *from, const char *to );
int dav_copy( const char *from, const char *to );
int dav_rmdir( const char *dirname );
int dav_mkcol( const char *dirname );

/* PROPFIND of given depth on a collection. 
 *  If 'include_target' is non-zero, include the target URI in the list.
 *  Depth of PROPFIND is given 'depth'.
 * Returns:
 *   PROTO_OK: Success: list of resources in 'reslist'.
 *   PROTO_FAILED: Method failed; not a DAV-enabled collection?
 *   PROTO_ERROR: Error in response.
 */

int dav_fetch( const char *uri, int depth, int include_target,
	       struct proto_file **reslist );
void dav_free_fileslist( struct proto_file *files );

/* The request mechanism */
int http_request( http_req_t *req );
void http_request_init( http_req_t *req, const char *method, const char *uri );
void http_request_end( http_req_t *req );

int dav_want_207( http_req_t *req );
int dav_want_2xx( http_req_t *req );

int dav_207_write_errors( http_req_t *req, struct dav_207_parser *ctx );

#ifdef USE_DAV_LOCKS
char *dav_lock_ifheader( http_req_t *req );
/* Called to indicate we are going to write to a resource or 
 * read from it */
void dav_lock_using_resource( http_req_t *req, const char *uri, enum dav_lock_usage use, int depth );
/* Called to indicate we are going to manipulate the collection contents
 * of the parent of the given resource */
void dav_lock_using_parent( http_req_t *req, const char *uri );

void dav_submitlocks_free( struct dav_submit_locks *list );

#endif

#endif /* HTTPDAV_H */
