/* 
   cadaver, command-line DAV client
   Copyright (C) 1999-2000, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   $Id: options.c,v 1.12.2.1 2000/03/30 19:43:53 joe Exp $
*/

/* Options handling */

#include <config.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <stdio.h>

#include "common.h"
#include "httpdav.h"
#include "string_utils.h"

#include "cadaver.h"
#include "options.h"

static void set_debug( const char *new );
static void unset_debug( const char *new );
static void disp_debug( void );

/* Option holders */

static struct {
    const char *name;
    enum option_id id;
    void *holder; /* for bool + string options */
    enum {
	opt_bool,
	opt_string,
	opt_handled
    } type;
    /* for handled options */
    void (*set)( const char * );
    void (*unset)( const char * );
    void (*display)( void );
    /* for all options */
    const char *help;
    /* for handled options */
    const char *handle_help;
} options[] = {
#define B(x,v,h) { #x, opt_##x, v, opt_bool, NULL, NULL, NULL, h, NULL }
    /* Booleans */
    B( tolerant, &tolerant, "Tolerate non-WebDAV collections" ),
    B( expect100, &http_enable_expect, "Enable use of 'Expect: 100-continue' header" ),
#undef B
#define S(x,h) { #x, opt_##x, NULL, opt_string, NULL, NULL, NULL, h, NULL }
    S( lockowner, "Lock owner URI" ),
    S( editor, "Editor to use with `edit' command" ),
    S( proxy, "Hostname of proxy server" ),
    { "proxy-port", opt_proxy_port, NULL, opt_string, NULL, NULL, NULL,
      "Port to use on proxy server", NULL },
#undef S
    { "debug", opt_debug, NULL, opt_handled,
      set_debug, unset_debug, disp_debug, "Debugging options",
      "The debug value is a list of comma-separated keywords.\n"
      "Valid keywords are: socket, http, xml, httpauth, cleartext."
    },
    { NULL, 0 }
};

static const struct {
    const char *name;
    int val;
} debug_map[] = {
    { "xml", DEBUG_XML },
    { "xmlparse", DEBUG_XMLPARSE },
    { "http", DEBUG_HTTP },
    { "socket", DEBUG_SOCKET },
    { "httpauth", DEBUG_HTTPAUTH },
    { "httpbody", DEBUG_HTTPBODY },
    { "cleartext", DEBUG_HTTPPLAIN },
    { "files", DEBUG_FILES },
    { "locks", DEBUG_LOCKS },
    { NULL, 0 }
};

static void display_options( void ) 
{
    int n;
    printf( "Options:\n" );
    for( n = 0; options[n].name != NULL; n++ ) {
	int *val = (int *)options[n].holder;
	switch( options[n].type ) {
	case opt_bool:
	    printf( " %s is %s\n", options[n].name, *val?"set":"unset" );
	    break;
	case opt_string:
	    if( options[n].holder == NULL ) {
		printf( " %s is unset\n", options[n].name );
	    } else {
		printf( " %s is %s\n", options[n].name, 
			(char *)options[n].holder );
	    }
	    break;
	case opt_handled:
	    printf( " %s is ", options[n].name );
	    (*options[n].display)( );
	    printf( "\n" );
	    break;
	}
    }
}

static void do_debug( const char *set, int setit ) {
    int n;
    char **vs;
    if( !setit && !set ) {
	/* FIXME: what does this mean? */
	return;
    }
    vs = split_string( set, ',', NULL, NULL );
    for( n = 0; vs[n] != NULL; n++ ) {
	int d, got = 0;
	for( d = 0; debug_map[d].name != NULL; d++ ) {
	    if( strcasecmp( vs[n], debug_map[d].name ) == 0 ) {
		if( setit ) {
		    debug_mask |= debug_map[d].val ;
		} else {
		    debug_mask &= ~debug_map[d].val;
		}
		got = 1;
	    }
	}
	if( !got ) {
	    printf( "Debug option %s unknown.\n", vs[n] );
	}
    }
    split_string_free( vs );
}

static void set_debug( const char *set ) {
    do_debug( set, 1 );
}

static void unset_debug( const char *s ) {
    do_debug( s, 0 );
}

static void disp_debug( void ) {
    int n, flag=0;
    putchar( '{' );
    for( n = 0; debug_map[n].name != NULL; n++ ) {
	if( debug_mask & debug_map[n].val ) {
	    printf( "%s%s", flag++?",":"", debug_map[n].name );
	}
    }
    putchar( '}' );
}

void execute_set( const char *opt, const char *newv ) {
    if( opt == NULL ) {
	display_options();
    } else {
	int n;
	for( n = 0; options[n].name != NULL; n++ ) {
	    if( strcasecmp( options[n].name, opt ) == 0 ) {
		switch( options[n].type ) {
		case opt_bool:
		    if( newv ) {
			printf( "%s is a boolean option.\n", opt );
		    } else {
			*(int *)options[n].holder = 1;
		    }
		    break;
		case opt_string:
		    if( newv == NULL ) {
			printf( "You must give a new value for %s\n", opt );
		    } else {
			char *val = options[n].holder;
			if( val != NULL ) {
			    free( val );
			}
			options[n].holder = strdup(newv);
		    }
		    break;
		case opt_handled:
		    if( !newv ) {
			printf( "%s must be given a value:\n%s\n", opt,
				options[n].handle_help );
		    } else {
			(*options[n].set)( newv );
		    }
		    break;
		}
		return;
	    }
	}
	printf( "Unknown option: %s.\n", opt );
    }
}

void execute_unset( const char *opt, const char *newv ) {
    int n;
    for( n = 0; options[n].name != NULL; n++ ) {
	if( strcasecmp( options[n].name, opt ) == 0 ) {
	    switch( options[n].type ) {
	    case opt_bool:
		if( newv != NULL ) {
		    printf( "%s ia a boolean option.\n", opt );
		} else {
		    *(int *)options[n].holder = 0;
		}
		break;
	    case opt_string:
		/* FIXME: This is bad UI */
		if( newv != NULL ) {
		    printf( "%s cannot take a value to unset.\n", opt );
		} else {
		    char *v = options[n].holder;
		    free( v );
		    options[n].holder = NULL;
		}
		break;	       
	    case opt_handled:
		(*options[n].unset)( newv );
		break;
	    }
	    return;
	}
    }
    printf( "Unknown option: %s.\n", opt );
}

void *get_option( enum option_id id ) {
    int n;
    for( n = 0; options[n].name != NULL; n++ ) {
	if( options[n].id == id ) {
	    return options[n].holder;
	}
    }
    return NULL;
}

void set_option( enum option_id id, void *newval ) {
    int n;
    for( n = 0; options[n].name != NULL; n++ ) {
	if( options[n].id == id ) {
	    options[n].holder = newval;
	    return;
	}
    }
}
