/* 
   socket handling routines
   Copyright (C) 1998, Joe Orton <joe@orton.demon.co.uk>, except where
   otherwise indicated.
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef SOCKET_H
#define SOCKET_H

#include <sys/socket.h>
#include <sys/types.h>

#include <netinet/in.h>

#include "protocol.h"

/* Socket read timeout */
#define SOCKET_TIMEOUT 30

/* sock_read is read() with a timeout of SOCKET_TIMEOUT.
 * Returns:
 *   -1 on error,
 *    0 on no data to read (due to timeout or EOF),
 *   >0 length of data read into buffer.
 */
ssize_t sock_read( int sock, void *buffer, size_t count );

/* sock_recv is recv() with a timeout of SOCKET_TIMEOUT.
 * Returns:
 *   -1 on error,
 *    0 on no data to read (due to timeout or EOF),
 *   >0 length of data read into buffer.
 */
ssize_t sock_recv( int sock, void *buffer, size_t count, unsigned int flags);

/* Blocks waiting for data on the given socket for the given time.
 * Returns:
 *  -1 on error,
 *   0 on no data within timeout,
 *   1 if data arrived on the socket.
 */
int sock_block( int sock, int timeout );

/* Dump the given filename down the given socket.
 * Returns 0 on success, non-zero on error */
int send_file_binary( int sock, const char *filename ); 

/* Dump the given filename down the given socket, performing
 * CR -> CRLF conversion.
 * Returns 0 on success, non-zero on error */
int send_file_ascii( int sock, const char *filename ); 

/* Dump from given socket into given file. Reads only filesize bytes,
 * or until EOF if filesize == -1.
 * Returns number of bytes written on success, or -1 on error */
ssize_t recv_file( int sock, const char *filename, ssize_t filesize ); 

/* Reads readlen bytes from srcfd and writes to destfd.
 * (Not all in one go, obviously).
 * If readlen == -1, then it reads from srcfd until EOF.
 * Returns number of bytes written to destfd, or -1 on error.
 * Calls fe_transfer_progress( a, b ) during transfers, where
 *  a = bytes transferred so far, and b = readlen
 */
ssize_t transfer( int srcfd, int destfd, ssize_t readlen );

/* Sends the given line to given socket, CRLF appended */
int send_line( int sock, const char *line ); 
/* Sends the given block of data down the socket */
int send_data( int sock, const char *data, size_t length ); 
/* Sends the null-terminated string down the given socket */
int send_string( int sock, const char *string ); 

/* Reads a line from given socket */
int read_line( int sock, char *line, int len ); 
/* Reads a chunk of data. */
int read_data( int sock, char *buffer, int buflen );

/* Creates and connects a socket */
int socket_connect( const struct in_addr host, int portnum );

void socket_close( int socket );

/* Do a name lookup on given hostname, writes the address into
 * given address buffer. Return -1 on failure.
 */
int host_lookup( const char *hostname, struct in_addr *addr );

/* Returns the standard TCP port for the given service */
int get_tcp_port( const char *name );

int read_file_blocks( int fd, size_t length,
		      proto_read_block reader, void *userdata );

#endif /* SOCKET_H */
