/*
  httpdget.c  --  get a file from a web server
  Copyright (C) 1999, 2000 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *	$Id: httpdget.c,v 1.18 2000/04/26 22:11:08 solyga Exp $
 */

#include	"httpdget.h"


char none[]= "(none)";


int
display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Get resource from web server.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [url]\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -h\t write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -H\t print hash marks\n" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level on %s (max %d)\n", VERBOSE_CHANNEL==stdout?"stdout":"stderr", VERBOSE_LEVEL_MAX );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation info to %s and exit sucessfully\n", VERSION_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n" );
  fprintf( HELP_CHANNEL, "  -c COOKIE\t cookie to send to server\n" );
  fprintf( HELP_CHANNEL, "  -n BYTES\t number of bytes to get (default all)\n" );
  fprintf( HELP_CHANNEL, "  -o OFFSET\t offset to start get at (default 0)\n" );
  return( 0 );
}


int
display_version( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  DEFAULT_HOST      :  %s\n", DEFAULT_HOST );
  fprintf( VERSION_CHANNEL, "  DEFAULT_PORT      :  %s\n", DEFAULT_PORT_STR );
  fprintf( VERSION_CHANNEL, "  DEFAULT_PATH      :  /%s\n", DEFAULT_PATH );
  fprintf( VERSION_CHANNEL, "  DEFAULT_COOKIE    :  %s\n", DEFAULT_COOKIE );
  fprintf( VERSION_CHANNEL, "  HASH_MARKS        :  %d\n", HASH_MARKS );
  fprintf( VERSION_CHANNEL, "  HASH_SIZE         :  %d\n", HASH_SIZE );
  fprintf( VERSION_CHANNEL, "  HEADER_BASE_SIZE  :  %d\n", HEADER_BASE_SIZE );
  fprintf( VERSION_CHANNEL, "  HEADER_MAX_SIZE   :  %d\n", HEADER_MAX_SIZE );
  fprintf( VERSION_CHANNEL, "  ANSWER_BUFFER_SIZE:  %d\n", ANSWER_BUFFER_SIZE );
  return( 0 );
}


char*
str_dsc( char* str1, char* str2) {
/*
returns pointer to first char of str1 not contained in str2
started 1997-02-09
*/
  char *p1, *p2;
  for ( p1=str1; ; p1++ ) {
    if ( *p1 == '\0' ) return( p1 );
    for ( p2=str2; ; p2++ ) {
      if ( *p2 == '\0' ) return( p1 );
      if ( *p2 == *p1 ) break;
    }
  }
}


int
str_stp( char* str1, char* str2 ) {
/*
string stop
writes '\0' to first char in str1 matching one of the chars from str2,
returns number of deleted characters
started 1997-03-28
*/
  char *p1, *p2;
  int n= 0;
  for( p1=str1; ; p1++ ) {
    if ( *p1 == '\0' ) return( n );
    for( p2=str2; ; p2++ ) {
      if( *p2 == '\0' ) break;
      if( *p2 == *p1 ) {
        *p1= '\0';
        while( *(p1+++n) != '\0' );
        return( n );
      }
    }
  }
}


ulong
get_ipaddr_n_by_name( char* hostname ) {
/* return binary address in network byte order */
  struct hostent* ent= gethostbyname( hostname );
  if( ent == NULL ) return( -1 );
  if( ent->h_addr_list[0] == NULL ) return( -1 );
  return( *( (ulong*)ent->h_addr_list[0] ) );
}


int
getoptarg( char opt, char* arg, int min, int max, char* pn, int* flag ) {
  int val;
  char* p;
  *flag= 1;
  val= (int) strtol( arg, &p, 0 );
  if( p == arg  ||  *str_dsc(p," ") != '\0' ) {
    fprintf( ERROR_CHANNEL,
             "%s: Argument of option %c (`%s') must be a number.\n",
             pn, opt, arg );
    return( val );
  }
  if( val < min  ||  val > max ) { /* out of range */
    fprintf( ERROR_CHANNEL,
             "%s: Argument value of option %c (%d) must be in [%d,%d].\n",
             pn, opt, val, min, max );
    return( val );
  }
  *flag= 0;
  return( val );
}


ssize_t
my_write( int fd, void *buf, size_t count ) {
/* write exactly count bytes to fd (else error) */
  ssize_t nbw;
  ssize_t tnbw= 0;
  size_t rem= count;
  unsigned char *p= (unsigned char*)buf;
  do {
    if( (nbw= write( fd, p+tnbw, rem )) == -1 ) return( -1 );
    tnbw+= nbw;
    rem-= nbw;
  } while( nbw>0 && rem>0 );
  return( tnbw );
}


ssize_t
my_read( int fd, void* buf, size_t count ) {
/* read exactly count bytes from fd (else EOF or error) */
  unsigned char* p= buf;
  ssize_t nbr;
  ssize_t tnbr= 0;
  size_t rem= count;
  do {
    if( (nbr= read( fd, p+tnbr, rem )) == -1 ) return( -1 );
    tnbr+= nbr;
    rem-= nbr;
  } while( nbr>0 && rem>0 );
  return( tnbr );
}


char*
http_head_value( char* header, char* key ) {
/* return value belonging to key or NULL */
  char* line;
  char* p= none;
  if( header == NULL  ||  key == NULL ) return( NULL );
  for( line=header; *line!='\0'; ) {
    if( strncasecmp(line,key,strlen(key)) == 0 ) {
      if( (p=strchr(line,':')) != NULL )
        p= str_dsc( p, ": \t" );
    }
    while( *line++ != '\0' );	/* next line */
  }
  return( p );
}


char*
basename( char* name ) {
/* strip directory from name */
/* basename("/usr/bin/") == "" -- basename(1) would return "bin" !! */
  char* p= name;
  while( *p != '\0' ) p++;
  while( p > name ) {
    if( *(p-1) == '/' ) break;
    else p--;
  }
  return( p );
}


int
main( int argc, char** argv ) {
/*
SOLYGA --------------------
SOLYGA main() httpdget

SOLYGA started      : Wed Oct 6 19:45:30 CEST 1999 @beast
*/
  char* fpn= *argv;
  char* pn= basename( fpn );		/* prg_name is Makefile constant */
  int retval= RETVAL_OK;
  int c;
  int verbose= 0;			/* switch: be verbose */
  int hash_marks= 0;			/* switch: print hash marks */
  int hash_marks_printed= 0;
  char default_cookie[]= DEFAULT_COOKIE;
  char* cookie= default_cookie;
  char default_protocol[]= DEFAULT_PROTOCOL;
  char default_hostname[]= DEFAULT_HOST;
  char default_port_str[]= DEFAULT_PORT_STR;
  char default_path[]= DEFAULT_PATH;
  char* protocol= none;
  char* hostname= none;
  char* port_str= none;
  char* path= none;
  int port= -1;
  int sockfd= 0;
  int out_fd= STDOUT_FILENO;
  char default_out_filename[]= "(none)";
  char* out_fn= default_out_filename;
  struct sockaddr_in host;
  ulong ipaddr_n;
  UCHAR* req= NULL;			/* http request buffer start */
  size_t req_sze;			/* request buffer size */
  int req_len;				/* request length (negl. '\r') */
  UCHAR ans[ANSWER_BUFFER_SIZE];	/* receive buffer, header & body */
  char* hed= NULL;			/* http header buffer start */
  size_t hed_sze= 0;			/* header buffer size */
  char* hp= NULL;
  int hed_len= 0;			/* header length (real size) */
  int nbs;				/* number of bytes sent */
  int nbc;				/* number of bytes received */
  int nbtc;				/* number of bytes to receive */
  int tnbc;				/* total number of bytes received */
  ssize_t nbw;				/* number of bytes written */
  int i;
  int binary_answer;			/* flag: non-ascii answer */
  int header_complete;			/* flag: header completely received */
  int warning_printed;			/* flag: too many bytes warning given */
  int prefix_printed;			/* flag: HASH_PREFIX printed */
  /* http_* are substrings taken from header send by the server */
  char* http_status_str= none;		/* HTTP status message from header */
  char* http_last_modified= none;
  char* http_location= none;
  char* http_content_length= none;
  char* http_content_range= none;
  char* http_connection= none;
  char* http_content_type= none;
  /* converted values from header */
  int status_code;			/* status value from header */
  int content_length;			/* number of body bytes to receive */
  int content_range_from;		/* start offset */
  int content_range_to;			/* end offset (*_length-*_from) */
  int content_range_size;		/* file size */
  /* misc */
  time_t sec0, sec1;			/* speed measurement */
  int bytes= -1;			/* n.o. bytes to get, -1 == all */
  /* hint: GET for 0 bytes is forbidden in HTTP/1.1 */
  int offset= 0;			/* offset to start get at */
#if defined OS_TYPE_AIX
  extern int errno;
  extern char* sys_errlist[];
#endif


/* process options */
  *argv= pn;				/* give getop() the cutted name */
  while( (c=getopt(argc,argv,"c:hHn:o:vV")) != EOF ) {
    int flag;
    switch( c ) {
      case 'c': /* get cookies to send */
        cookie= optarg;
        break;
      case 'h': /* display help and exit sucessfully */
        display_help( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'H': /* print hash marks */
        hash_marks= 1;
        break;
      case 'n': /* number of bytes to get */
        bytes= getoptarg( 'n', optarg, 0, INT_MAX, pn, &flag );
        if( flag ) {
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        break;
      case 'o': /* offset to start get at */
        offset= getoptarg( 'o', optarg, 0, INT_MAX, pn, &flag );
        if( flag ) {
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        break;
      case 'v': /* raise verbosity level */
        verbose++;
        break;
      case 'V': /* display version to VERSION_CHANNEL and exit sucessfully */
        display_version( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case '?': /* refer to -h and exit unsucessfully */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 pn, pn );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Options bug! E-mail me at %s.\n",
                 pn, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }
  }


/* catch trivial solution */
if( bytes == 0 ) {
  tnbc=0; content_length= 0;
  sec0= 0; sec1= 0;
  goto SUMMARY;
}


/* set protocol, hostname, port_str and path */
  if( optind<argc ) { /* url specified */
    char* url= argv[optind];
    char* up;
    /* protocol */
    protocol= default_protocol;
    if( ((up=strstr(url,"://")) != NULL )  &&
        (strchr(url,'/') > up ) ) {	/* protocol specified */
      *up= '\0';
      protocol= url;
      url= up + 1;
    }
    /* hostname & port_str */
    hostname= default_hostname;
    port_str= default_port_str;
    if( url[0]=='/' && url[1]=='/' ) {		/* host specified */
      url+= 2;
      hostname= url;
      if( (up=strchr(url,'/')) != NULL ) {
        *up= '\0';
        url= up + 1;
      }
      else {
        url= none;		/* flag: no path spec. */
      }
      if( (up=strchr(hostname,':')) != NULL ) {	/* port specified */
        *up= '\0';
        port_str= up + 1;
      }
    }
    /* path */
    path= ( url==none ? default_path : url );
    /* replace empty strings be defaults */
    if( strlen(protocol) == 0 ) protocol= default_protocol;
    if( strlen(hostname) == 0 ) hostname= default_hostname;
    if( strlen(port_str) == 0 ) port_str= default_port_str;
  }
  else { /* no url specified, use defaults for all */
    protocol= default_protocol;
    hostname= default_hostname;
    port_str= default_port_str;
    path= default_path;
  }
  path= str_dsc( path, "/" );


/* set port from port_str */
  {
    char* p;
    long int val= strtol( port_str, &p, 0 );
    if( p==port_str || *p!='\0' || val>=INT_MAX || val<=INT_MIN ) {
      fprintf( ERROR_CHANNEL, "%s: Invalid port number `%s'.\n",
               pn, port_str );
      retval= RETVAL_PORT; goto DIE_NOW;
    }
    port= (int) val;
  }


/* check protocol */
  if( strcmp(protocol,"http") ) {
    fprintf( ERROR_CHANNEL,
             "%s: Unknown protocol `%s'. Only `http' supported.\n",
             pn, protocol );
    retval= RETVAL_PROTOCOL; goto DIE_NOW;
  }
#ifdef	DEBUG
  fprintf( DEBUG_CHANNEL, "DEBUG: protocol= `%s'\n", protocol );
  fprintf( DEBUG_CHANNEL, "DEBUG: hostname= `%s'\n", hostname );
  fprintf( DEBUG_CHANNEL, "DEBUG: port    = %d\n", port     );
  fprintf( DEBUG_CHANNEL, "DEBUG: path    = `/%s'\n", path     );
#endif


/* set up request */
  {
    char off_str_from[32]= "";
    char off_str_to[32]= "";
    sprintf( off_str_from, "%d", offset );
    if( bytes > 0 ) sprintf( off_str_to, "%d", offset + bytes - 1 );
    req_sze= strlen( "GET /" ) + 
             strlen( path ) +
             strlen( " HTTP/1.0\r\n" ) +
             strlen( "Host: " ) +
             strlen( hostname ) +
             strlen( "\r\n" ) +
             strlen( "Cookie: " ) +
             strlen( cookie ) +
             strlen( "\r\n" ) +
             strlen( "Range: bytes=" ) +
             strlen( off_str_from ) +
             strlen( "-" ) +
             strlen( off_str_to ) +
             strlen( "\r\n" ) +
             strlen( "User-Agent: " ) +
             strlen( prg_name ) +
             strlen( "-" )+
             strlen( VERSION_NUMBER ) +
             strlen( "\r\n\r\n" );
    if( (req=malloc(req_sze+1)) == NULL ) {
      fprintf( ERROR_CHANNEL, "%s: Couldn't allocate %d bytes for request.\n",
               pn, req_sze+1 );
      retval= RETVAL_MALLOC; goto DIE_NOW;
    }
    sprintf( req, "GET /%s HTTP/1.0\r\nHost: %s\r\nCookie: %s\r\nRange: bytes=%s-%s\r\nUser-Agent: %s-%s\r\n\r\n",
             path, hostname, cookie, off_str_from, off_str_to, prg_name, VERSION_NUMBER );
    if( strlen(req) != req_sze ) {
      fprintf( ERROR_CHANNEL, "%s: Bug detected! E-mail me at %s.\n",
               pn, MY_EMAIL_ADDRESS );
      retval= RETVAL_BUG; goto DIE_NOW;
    }
  }


/* prepare connexion */
  if( verbose >= 3 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Trying to get `%s://%s:%d/%s'.\n",
             pn, protocol, hostname, port, path );
  }
  if( (sockfd=socket(AF_INET,SOCK_STREAM,0)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't create socket. %s.\n",
             pn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( verbose >= 3 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Resolving hostname `%s'.\n",
             pn, hostname );
  }
  if( (ipaddr_n=get_ipaddr_n_by_name( hostname )) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't resolve hostname `%s'. %s.\n",
             pn, hostname, h_errlist[h_errno] );
    retval= RETVAL_HOSTNAME; goto DIE_NOW;
  }
  if( verbose >= 4 ) {
    char str_ipaddr[]= "000.000.000.000";
    UCHAR* p= (UCHAR*)&ipaddr_n;
    sprintf( str_ipaddr, "%d.%d.%d.%d",
              p[0], p[1], p[2], p[3] );
    fprintf( VERBOSE_CHANNEL, "%s: IP address of `%s' is %s.\n",
             pn, hostname, str_ipaddr );
  }


/* open connexion */
  host.sin_family= PF_INET;
  host.sin_port= htons( port );
  host.sin_addr.s_addr= ipaddr_n;
  if( verbose >= 3 ) {
    fprintf( VERBOSE_CHANNEL,
             "%s: Connecting to host `%s' at port %d.\n",
              pn, hostname, port );
  }
  if( connect(sockfd,(struct sockaddr*)&host,sizeof(host)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't connect to host `%s' port %d. %s.\n",
             pn, hostname, port, sys_errlist[errno] );
    retval= RETVAL_CONNECT; goto DIE_NOW;
  }


/* send request */
  if( verbose >= 3 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Sending request (%d bytes).\n",
             pn, req_sze );
  }
  if( (nbs=write(sockfd,req,req_sze)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't send request. %s.\n",
             pn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( verbose >= 4 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Request successfully sent (%d bytes).\n",
             pn, nbs );
  }
  /* convert request to (terminated) strings */
  /* for printing only */
  for( i=0,req_len=0; i<req_sze; i++ )
    if( req[i] != '\r' )
      req[req_len++]= req[i]=='\n' ? '\0' : req[i];
  /* print request */
  if( verbose >= 2 ) {
    for( hp=req; *hp!='\0'; ) {
      fprintf( VERBOSE_CHANNEL, "%s: request: %s\n", pn, hp );
      while( *hp++ != '\0' );		/* next line */
    }
  }


/* receive answer */
  if( verbose >= 3 )
    fprintf( VERBOSE_CHANNEL, "%s: Waiting for header.\n", pn );
  tnbc= 0;
  hed_sze= 0;
  hed_len= 0;
  header_complete= 0;
  /* receive http header */
  while( !header_complete ) {
    if( (nbc=my_read(sockfd,ans,1)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Couldn't receive header. %s.\n",
               pn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    tnbc+= nbc;
    if( nbc < 1 ) {
      fprintf( ERROR_CHANNEL, "%s: No response from server.\n",
               pn );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    if( hed_sze < tnbc ) {
      hed_sze+= HEADER_BASE_SIZE;
      if( hed_sze > HEADER_MAX_SIZE ) {
        fprintf( ERROR_CHANNEL,
                 "%s: Http header becomes too long (>%d).\n",
                 pn, HEADER_MAX_SIZE );
        retval= RETVAL_MALLOC; goto DIE_NOW;
      }
      if( (hed=realloc(hed,hed_sze)) == NULL ) {
        fprintf( ERROR_CHANNEL,
                 "%s: Couldn't allocate %d bytes for http header.\n",
                 pn, hed_sze );
        retval= RETVAL_MALLOC; goto DIE_NOW;
      }
    }
    /* header termination by "\n\n" is OK */
    if( *ans != '\r' )
      hed[hed_len++]= *ans;
    if( hed_len > 2  &&  hed[hed_len-1] == '\n'  &&   hed[hed_len-2] == '\n' )
      header_complete= 1;
  }
  if( verbose >= 3 ) {
    fprintf( VERBOSE_CHANNEL,
             "%s: Header successfully received (%d bytes).\n",
             pn, tnbc );
  }
  /* convert header to (terminated) strings */
  /* this is not only for printing, but for getting the values later !! */
  for( i=0; i<hed_len; i++ )
     if( hed[i] == '\n' ) hed[i]= '\0';
  /* print header */
  /* hed[hed_len-2]=='\0' && hed[hed_len-1]=='\0' */
  if( verbose >= 2 ) {
    for( hp=hed; *hp!='\0'; ) {
      fprintf( VERBOSE_CHANNEL, "%s: header: %s\n", pn, hp );
      while( *hp++ != '\0' );		/* next line */
    }
  }
  /* check protocol */
  if( strncmp(hed,"HTTP",4) ) {
    fprintf( ERROR_CHANNEL, "%s: No http header.\n", pn );
    retval= RETVAL_SERVER_PROTOCOL; goto DIE_NOW;
  }
  /* get status code and message */
  for( hp=hed; *hp!=' '; hp++ );
  {
    char* p;
    hp= str_dsc( hp, " \t" );
    status_code= strtol( hp, &p, 0 );
    if( p == hp ) {
      str_stp( hp, " \t" );
      fprintf( ERROR_CHANNEL,
               "%s: No http header. Invalid status code `%s'.\n", pn, hp );
      retval= RETVAL_SERVER_PROTOCOL; goto DIE_NOW;
    }
    http_status_str= str_dsc( p, " \t" );
  }
  /* get header values */
  http_last_modified= http_head_value( hed, "Last-Modified" );
  http_location= http_head_value( hed, "Location" );
  http_content_length= http_head_value( hed, "Content-Length" );
  http_content_range= http_head_value( hed, "Content-Range" );
  http_connection= http_head_value( hed, "Connection" );
  http_content_type= http_head_value( hed, "Content-Type" );
  /* convert content length */
  content_length= -1;
  if( strcmp(http_content_length,none) ) {
    char* p0= http_content_length;
    char* p;
    int val= (int)strtol( http_content_length, &p, 0 );
    if( p != p0  &&  val < INT_MAX  &&  val >= 0 )
      content_length= val;
  }
  /* convert content range */
  content_range_from= -1;
  content_range_to= -1;
  content_range_size= -1;
  if( strcmp(http_content_range,none) ) {
    char* p0= http_content_length;
    char* p;
    int val;
    val= (int)strtol( p0, &p, 0 );
    if( p != p0  &&  val < INT_MAX  &&  val >= 0 ) {
      content_range_from= val;
      if( *p == '-' ) {
        p0= p+1;
        val= (int)strtol( p0, &p, 0 );
        if( p != p0  &&  val < INT_MAX-1  &&  val >= 0 ) {
          content_range_to= val + 1;	/* using c-convention */
          if( *p == '/' ) {
            p0= p+1;
            if( p != p0  &&  val < INT_MAX  &&  val >= 0 )
              content_range_size= val;
          }
        }
      }
    }
  }
  /* react on server status code */
  switch( status_code/100 ) {
    case 2:	/* success */
      /* Content-Length may differ from the requested one ==> warning */
      if( bytes > -1 ) {		/* explicitely specified */
        if( content_length == -1 ) {	/* Content length unknown */
          fprintf( ERROR_CHANNEL, "%s: Warning: Content length unknown.\n",
                   pn );
        }
        else {				/* Content length known */
          if( content_length != bytes ) {
            fprintf( ERROR_CHANNEL, "%s: Warning: Content length is %d but %d bytes requested.\n", pn, content_length, bytes );
          }
        }
      }
      switch( status_code ) {
        case HTTP_OK:
          /*
           * OK to a GET with offset >0 must be considererd as error
           * because it indicates that full entity is sent
           * Known cases are:
           * 1. apache-1.3.3 sends OK and full file if offset too large
           * 2. NCSA/1.4.2 ignores Range field -- not known in HTTP/1.0
           */
          if( offset > 0 ) {
            fprintf( ERROR_CHANNEL,
                     "%s: Offset cannot be satisfied by server.", pn );
            if( content_range_from != -1 )	/* should never be true */
              fprintf( ERROR_CHANNEL, " Requested/served offsets %d/%d.",
                       offset, content_range_from );
            if( content_length != -1  &&  offset > content_length )
              fprintf( ERROR_CHANNEL,
                       " Requested offset exceeds resource size by %d.\n",
                       offset-content_length );
            if( !strncmp(hed,"HTTP/1.0",8) )
              fprintf( ERROR_CHANNEL, " Server speaks HTTP/1.0 only." );
            fprintf( ERROR_CHANNEL, "\n" );
            retval= RETVAL_OFFSET; goto DIE_NOW;
          }
          break;
        case HTTP_PARTIAL_CONTENT:
          /* check offsets/sizes */
          /* I'm so lazy... */
          break;
      }
      break;
    case 3:	/* redirect */
      if( strcmp(http_location,none) )
        fprintf( ERROR_CHANNEL, "%s: Check %s.\n", pn, http_location );
    case 1:	/* information */
    case 4:	/* client error */
    case 5:	/* server error */
    default:	/* actually protocol error, see RFC 2616 */
      fprintf( ERROR_CHANNEL, "%s: Server objected: %d %s.\n",
               pn, status_code, http_status_str );
      retval= RETVAL_NO_SUCCESS; goto DIE_NOW;
  }

  /* receive and store http body */
  if( verbose >= 3 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Receiving body (", pn );
    if( content_length == -1 )
      fprintf( VERBOSE_CHANNEL, "unknown size).\n" );
    else
      fprintf( VERBOSE_CHANNEL, "%d bytes expected).\n", content_length );
  }
  tnbc= 0;
  nbtc= ANSWER_BUFFER_SIZE;
  binary_answer= 0;
  prefix_printed= 0;
  warning_printed= 0;
  sec0= time( NULL );
  while( 1 ) {
    if( (nbc=my_read(sockfd,ans,nbtc)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Couldn't receive answer. %s.\n", 
               pn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    tnbc+= nbc;
    for( i=0; i<nbc; i++ ) if( NON_ASCII(ans[i]) ) binary_answer= 1;
    if( (nbw=my_write(out_fd,ans,nbc)) < nbc ) {
      fprintf( ERROR_CHANNEL, "%s: Couldn't write to ", pn );
      if( out_fd == STDOUT_FILENO )
        fprintf( ERROR_CHANNEL, "stdout" );
      else
        fprintf( ERROR_CHANNEL, "`%s'", out_fn );
      fprintf( ERROR_CHANNEL, ". %s.\n", sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    if( hash_marks ) {
      int hash_marks_to_print;
      if( content_length==-1 && !prefix_printed ) {
        fprintf( VERBOSE_CHANNEL, HASH_PREFIX );
        prefix_printed= 1;
      }
      if( content_length != -1 ) {
        hash_marks_to_print= content_length==0 ? HASH_MARKS :
                             (tnbc*HASH_MARKS)/content_length -
                             hash_marks_printed;
      }
      else {
        hash_marks_to_print= tnbc/HASH_SIZE -
                             hash_marks_printed;
      }
      for( i=0; i<hash_marks_to_print; i++ ) fprintf( VERBOSE_CHANNEL, "#" );
      hash_marks_printed+= hash_marks_to_print;
    }
    if( content_length!=-1 && !warning_printed && tnbc>content_length ) {
      fprintf( ERROR_CHANNEL,
               "%s: Warning: Receiving more bytes than expected!\n", pn );
      warning_printed= 1;
    }
    if( nbc < nbtc ) break;
  }
  sec1= time( NULL );
  if( hash_marks ) fprintf( VERBOSE_CHANNEL, "\n" );

SUMMARY:
  if( verbose >= 1 ) {
    double speed;
    char unit[]= " "; *unit= '\0';
    fprintf( VERBOSE_CHANNEL,
             "%s: Body successfully received (%d bytes, %s).\n",
             pn, tnbc, binary_answer?"binary":"ascii" );
    speed= (double)(sec1==sec0 ? 1.048565e8 : tnbc/(double)(sec1-sec0));
    if( speed > 1024.0 ) { speed /= 1024.0; unit[0]= 'k'; }
    if( speed > 1024.0 ) { speed /= 1024.0; unit[0]= 'M'; }
    fprintf( VERBOSE_CHANNEL,
             "%s: Wall clock speed %5.3f %sBytes/s\n", pn, speed, unit );
  }
  if( content_length != -1  &&  tnbc != content_length ) {
    fprintf( ERROR_CHANNEL, "%s: Reveived %d bytes but %d bytes expected.\n",
             pn, tnbc, content_length );
    retval= RETVAL_LENGTH; goto DIE_NOW;
  }


/* close connexion */
  if( close( sockfd ) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't close socket. %s.\n",
             pn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }


DIE_NOW:
  free( hed );
  free( req );
  close( sockfd );
  exit( retval );
}
