/*
  httpdtalk.c  --  talk with a web server
  Copyright (C) 1999,2000 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include	"httpdtalk.h"


int retval= RETVAL_OK;
pid_t pid= 1;			/* process id (!= 0 identifies parent) */
int sockfd;
char* line= NULL;		/* user's input (allocated buffer) */
int out_fd= STDOUT_FILENO;	/* output */
char* hist_fn= NULL;		/* input history file */
char* pn= NULL;			/* prg_name is Makefile constant */
int verbose= 0;			/* switch: be verbose */
int use_readline= 1;		/* switch: use readline(3) if linked */
unsigned char* ans;		/* receive (answer) buffer */
ssize_t tnbc;			/* total number of bytes received */
ssize_t tnbw;			/* total number of bytes written */
ssize_t tnbr;			/* total number of bytes read */
ssize_t tnbs;			/* total number of bytes sent */


int
display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Talk with a web server.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [host]\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n");
  fprintf( HELP_CHANNEL, "  -h\t help, write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout?"stdout":"stderr" );
#ifdef	HAVE_READLINE
  fprintf( HELP_CHANNEL, "  -r\t don't use readline(3)\n" );
#endif
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level for %s\n", VERBOSE_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation info to %s and exit sucessfully\n", VERSION_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n");
  fprintf( HELP_CHANNEL, "  -p port\t set port (default is %d)\n", DEFAULT_PORT );
  return( 0 );
}


int
display_version( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  DEFAULT_HOST      : %s\n", DEFAULT_HOST );
  fprintf( VERSION_CHANNEL, "  DEFAULT_PORT      : %d\n", DEFAULT_PORT );
  fprintf( VERSION_CHANNEL, "  ANSWER_BUFFER_SIZE: %d\n", ANSWER_BUFFER_SIZE );
  fprintf( VERSION_CHANNEL, "  HAVE_READLINE     : " );
#ifdef	HAVE_READLINE
  fprintf( VERSION_CHANNEL, "defined\n" );
#else
  fprintf( VERSION_CHANNEL, "undefined\n" );
#endif
  return( 0 );
}


void
handler( int sig ) {
  switch( sig ) {	/* parent and child */
    case SIGINT:
#ifdef	DEBUG
      fprintf( DEBUG_CHANNEL, "DEBUG: SIGINT catched by %s\n",
               pid == 0 ? "child" : "parent" );
#endif
      if( pid == 0 ) {	/* child only */
        /* extern signal, usually Ctrl-C */
        retval= RETVAL_SIGINT;
        kill( getpid(), SIGUSR1 );	/* suicidal tendencies */
        while( 1 ) sleep( 1 );
      }
      else {		/* parent only */
        /* this is the normal termination of the parent */
        int status;
#ifdef	HAVE_READLINE
        write_history( hist_fn );
        free( hist_fn );
#endif
        if( verbose >= 1 ) {
          fprintf( VERBOSE_CHANNEL, "%s: %d/%d bytes read/sent.\n",
                   pn, tnbr, tnbs );
        }
        free( line );
        close( sockfd );
        if( verbose >= 2 ) {
          fprintf( VERBOSE_CHANNEL, "%s: Waiting for child (pid %d).\n",
                   pn, pid );
        }
        if( waitpid(-1,&status,0) > 0 ) {
          /* neccessary if parent catches Ctrl-C before child */
          retval= WEXITSTATUS( status );
        }
#ifdef  DEBUG
        fprintf( DEBUG_CHANNEL, "DEBUG: Parent exits with %d.\n", retval );
#endif
        exit( retval );
      }
      break;
    case SIGCHLD: {	/* parent only */
      int status;
#ifdef	DEBUG
      fprintf( DEBUG_CHANNEL, "DEBUG: SIGCHLD catched by %s\n",
               pid == 0 ? "child" : "parent" );
#endif
      if( waitpid(-1,&status,WNOHANG) > 0 ) {	/* get child status */
        retval= WEXITSTATUS( status );
        kill( getpid(), SIGINT );		/* suicidal tendencies */
      }
      break;
    }
    case SIGUSR1: {	/* child only */
      /* this is the normal termination of the child */
#ifdef	DEBUG
      fprintf( DEBUG_CHANNEL, "DEBUG: SIGUSR1 catched by %s\n",
               pid == 0 ? "child" : "parent" );
#endif
      if( verbose ) {
        fprintf( VERBOSE_CHANNEL, "%s: %d/%d bytes received/written.\n",
                 pn, tnbc, tnbw );
        fprintf( VERBOSE_CHANNEL, "%s: Closing connection.\n", pn );
      }
      close( out_fd );
      close( sockfd );
      free( ans );
      exit( retval );
      break;
    }
    case SIGPIPE:
#ifdef	DEBUG
      fprintf( DEBUG_CHANNEL, "DEBUG: SIGPIPE catched by %s\n",
               pid == 0 ? "child" : "parent" );
#endif
      break;
  }
  return;
}
/* Hint on detection of foreign ^C (SIGINT to both parent & child):
 * since the parent sends itself SIGINT when terminating normally, the only
 * way to detect a foreign SIGINT is by the child which informs the parent
 * via its retval -- thus SIGUSR1 cannot be handled simply by sending SIGINT
 * to itself
 */


uint32_t
get_ipaddr_n_by_name( char* hostname ) {
/* return IPv4 address in network byte order */
  struct hostent* ent= gethostbyname( hostname );
  if( ent == NULL ) return( -1 );
  if( ent->h_addr_list[0] == NULL ) return( -1 );
  if( ent->h_addrtype != AF_INET ) {
    h_errno= 5;				/* no IPv4 address, my extension */
    return( -1 );
  }
  return( *((uint32_t*)ent->h_addr_list[0]) );
}


ssize_t
my_write( int fd, void* buf, size_t count ) {
/*
 * (attempts to) write exactly count bytes from buf to fd
 * returns number of bytes written or -1 on error
 * retval < count indicates nothing special (see write(2))
 *   or EAGAIN when non-blocking
 * started 1998-01-01
 */ 
  unsigned char* p= buf;
  ssize_t nbw;
  ssize_t tnbw= 0;
  size_t rem= count;
  do {
    if( (nbw= write( fd, p+tnbw, rem )) == -1 ) {
      if( errno == EAGAIN ) return( tnbr );
      else                  return( -1 );
    }
    tnbw+= nbw;
    rem-= nbw;
  } while( nbw>0 && rem>0 );
  return( tnbw );
}


char*
my_readline( char* prompt ) {
/* readline(3) wrapper -- don't forget to free the returned address */
  static char* line;
  int call_readline= 0;
#ifdef  HAVE_READLINE
  if( use_readline ) call_readline= 1;
#endif
  if( call_readline ) {
    if( (line=readline(prompt)) != NULL ) add_history( line );
  }
  else {
    size_t size= 0;
    size_t len= 0;
    ssize_t nbr;
    if( prompt != NULL ) { 
      char* p= prompt;
      while( *p != '\0' ) p++;
      write( STDOUT_FILENO, prompt, (size_t)(p-prompt) );
    }
    line= NULL;
    while( 1 ) {
      if( len >= size ) {
        size+= 256;
        if( (line=realloc(line,size)) == NULL ) return( NULL );
      }
      if( (nbr=read(STDIN_FILENO,line+len,1)) == -1 ) return( NULL );
      /* 0x04 is a more(1)/less(1) hack -- terminal attributes are modified */
      if( len == 0  && (nbr==0 || (nbr==1&&*line==0x04)) ) return( NULL );
      if( line[len] == '\n' ) {
        line[len]= '\0';
        break;
      }
      len++;
    }
  }
  return( line );
}


char*
basename( char* name ) {
/*
 * strip directory from name
 * returns pointer to stripped name
 * hint: basename("/usr/bin/") == ""
 *   basename(1) would return "bin" !!
 */
  char* p= name;
  while( *p != '\0' ) p++;
  while( p > name ) {
    if( *(p-1) == '/' ) break;
    else p--;
  }
  return( p );
}


int
my_isfdtype( int fd, int type ) {
/*
 * check whether fd is of type type
 * returns 1 if true, 0 if false, -1 on error
 * allowed types are
 * S_IFSOCK, S_IFLNK, S_IFREG, S_IFBLK, S_IFDIR, S_IFCHR, S_IFIFO
 * started: 2000-02-17
 */
  struct stat buf;
  if( fstat(fd,&buf) < 0 ) return( -1 );
  if( (buf.st_mode&S_IFMT) == type ) return( 1 );
  else return( 0 );
}



int
main( int argc, char** argv ) {
/*
 * main() httpdtalk
 * started      : Wed Aug 25 10:06:20 CEST 1999 @beast
 */
  char* fpn= *argv;
  int c;
  char default_hostname[]= DEFAULT_HOST;
  char* hostname= default_hostname;
  uint16_t port= DEFAULT_PORT;
  char default_out_filename[]= "(none)";
  char* out_fn= default_out_filename;
  struct sockaddr_in server;
  uint32_t ipaddr_n;
#if defined OS_TYPE_AIX
  extern int errno;
  extern char* sys_errlist[];
#endif


/* process options */
  pn= basename( fpn );
  *argv= pn;				/* give getop() the cut name */
  while( (c=getopt(argc,argv,"hp:rvV")) != EOF ) {
    switch( c ) {
      case 'p': /* set port */ {
        char* p;
        port= (uint16_t)strtol( optarg, &p, 0 );
        if( p == optarg ) {
          fprintf( ERROR_CHANNEL,
                   "%s: Invalid port number `%s'.\n",
                   pn, optarg );
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        break;
      }
      case 'h': /* display help and exit sucessfully */
        display_help( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'r': /* don't use readline(3) */
        use_readline= 0;
#ifndef	HAVE_READLINE
        fprintf( ERROR_CHANNEL, "%s: Readline library is not linked, option r without any effect.\n", pn );
#endif
        break;
      case 'v': /* raise verbosity level */
        verbose++;
        break;
      case 'V': /* display version to VERSION_CHANNEL and exit sucessfully */
        display_version( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case '?': /* refer to -h and exit unsucessfully */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 pn, pn );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Options bug! E-mail me at %s.\n",
                 pn, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }
  }

/* prepare connection */
  if( optind<argc ) hostname= argv[optind];
  if( (sockfd=socket(AF_INET,SOCK_STREAM,0)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot create socket. %s.\n",
             pn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( verbose >= 1 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Resolving hostname `%s'.\n",
             pn, hostname );
  }
  if( (ipaddr_n=get_ipaddr_n_by_name( hostname )) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot resolve hostname `%s'. %s.\n",
             pn, hostname, h_errlist[h_errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( verbose >= 2 ) {
    char str[INET_ADDRSTRLEN]= "";
    if( inet_ntop(AF_INET,&ipaddr_n,str,INET_ADDRSTRLEN) == NULL ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot convert IPv4 address. %s.\n",
               pn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    fprintf( VERBOSE_CHANNEL, "%s: IP address of `%s' is %s.\n",
             pn, hostname, str );
  }

/* open connection */
  memset( &server, 0, sizeof(server) );
  server.sin_family= AF_INET;
  server.sin_port= htons( port );
  server.sin_addr.s_addr= ipaddr_n;
  if( verbose >= 1 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Connecting to `%s' port %d.\n",
             pn, hostname, port );
  }
  if( connect(sockfd,(struct sockaddr*)&server,sizeof(server)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot connect to `%s' port %d. %s.\n",
             pn, hostname, port, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( verbose >= 1 ) {
    fprintf( VERBOSE_CHANNEL, "%s: Connection established.\n", pn );
  }

/* talk and listen to server */
  tnbc= 0;
  tnbw= 0;
  tnbr= 0;
  tnbs= 0;
  /* SIGINT : parent: normal exit
   *          child : informs parent on foreign SIGINT + exit via SIGUSR1
   * SIGCHLD: parent: waits for child and exits normally
   *          child : -
   * SIGUSR1: parent: -
   *          child : normal exit caused by parent or server EOF
   * SIGPIPE: parent: just catched
   *          child : just catched
   */
  signal( SIGINT, handler );
  signal( SIGCHLD, handler );
  signal( SIGUSR1, handler );
  signal( SIGPIPE, handler );

  switch( pid=fork() ) {
    case -1:	/* error */
      fprintf( ERROR_CHANNEL, "%s: fork() error\n", pn );
      retval= RETVAL_ERROR; goto DIE_NOW;
    case  0:	/* child -- receive slave */ {
      /* normal termination due to SIGUSR1 */
      int receive= 1;			/* main loop flag */
      int isfifo;			/* flag: out_fd may block */
      ssize_t nbc;			/* number of bytes received */
      ssize_t nbw;			/* number of bytes written */
      off_t ans_off= 0;			/* write-to-buffer-offset */
      size_t ans_size= 0;
      signal( SIGCHLD, SIG_IGN );	/* active: SIGINT, SIGUSR1, SIGPIPE */
      isfifo= my_isfdtype( out_fd, S_IFIFO );
      isfifo= 0;
      while( receive ) {
        /* allocate memory for server's answer */
        if( ans_off+ANSWER_BUFFER_SIZE > ans_size ) {
          unsigned char* p;
          ans_size+= ANSWER_BUFFER_SIZE;
#ifdef	DEBUG
          fprintf( DEBUG_CHANNEL, "DEBUG: Allocating %d bytes for answer\n",
                   ans_size );
#endif
          if( (p=realloc(ans,ans_size)) == NULL ) {
            fprintf( ERROR_CHANNEL, "%s: Cannot allocate memory. %s.\n",
                     pn, sys_errlist[errno] );
            retval= RETVAL_MALLOC;
            kill( getpid(), SIGUSR1 );
            while( 1 ) sleep( 1 );
          }
          ans= p;
        }
        /* receive server's answer (blocking) */
        if( (nbc=read(sockfd,ans+ans_off,ANSWER_BUFFER_SIZE)) == -1 ) {
          fprintf( ERROR_CHANNEL, "%s: Cannot read from socket. %s.\n",
                   pn, sys_errlist[errno] );
          retval= RETVAL_ERROR;
          kill( getpid(), SIGUSR1 );
          while( 1 ) sleep( 1 );
        }
        tnbc+= nbc;
        /* check for server EOF */
        if( nbc == 0 ) {
          if( verbose >= 1 ) {
            fprintf( VERBOSE_CHANNEL,
                     "%s: Connection closed by foreign host.\n", pn );
          }
          isfifo= 0;
          receive= 0;
        }
        /* output server's answer when not buffering */
        if( isfifo ) {
          nbw= 0;
          ans_off+= nbc;
        }
        else {
          if( (nbw=my_write(out_fd,ans,ans_off+nbc)) == -1 ) {
            fprintf( ERROR_CHANNEL, "%s: Cannot write to ", pn );
            if( out_fd == STDOUT_FILENO )
              fprintf( ERROR_CHANNEL, "stdout. " );
            else
              fprintf( ERROR_CHANNEL, "%s. ", out_fn );
            fprintf( ERROR_CHANNEL, "%s.\n", sys_errlist[errno] );
            retval= RETVAL_ERROR;
            kill( getpid(), SIGUSR1 );
            while( 1 ) sleep( 1 );
          }
          ans_off= 0;
        }
        tnbw+= nbw;
      }
      kill( getpid(), SIGUSR1 );		/* exit */
      while( 1 ) sleep( 1 );			/* broken by SIGUSR1 */
      break;
    }	/* child end */
    default:	/* parent -- send slave */ {
      /* normal termination due to SIGINT */
      ssize_t nbr;			/* number of bytes read */
      ssize_t nbs;			/* number of bytes sent */
      char prompt[]= DEFAULT_PROMPT;
      signal( SIGUSR1, SIG_IGN );	/* active: SIGINT, SIGCHLD, SIGPIPE */
      if( my_isfdtype(STDIN_FILENO,S_IFCHR) ) {
        /* stdin is terminal */
#ifdef	HAVE_READLINE
        {
          char* home;
          if( (home=getenv("HOME")) != NULL ) {
            if( (hist_fn=malloc(strlen(home)+1+strlen(HIST_FN)+1)) != NULL ) {
              char* p= hist_fn;
              char* s;
              s= home;	strcpy( p, s ); p+= strlen( s );
              s= "/";	strcpy( p, s ); p+= strlen( s );
              s= HIST_FN;	strcpy( p, s ); p+= strlen( s );
              read_history( hist_fn );
            }
          }
        }
#endif
        while( 1 ) {
          if( (line=my_readline(prompt)) != NULL ) {
            nbr= strlen( line );
            tnbr+= nbr;
            if( (nbs=my_write(sockfd,line,nbr)) == -1 ) {
              fprintf( ERROR_CHANNEL, "%s: Cannot write to socket. %s.\n",
                       pn, sys_errlist[errno] );
              retval= RETVAL_ERROR;
              kill( getpid(), SIGINT );
              while( 1 ) sleep( 1 );
            }
            tnbs+= nbs;
            nbr= 1;	/* EOL */
            tnbr+= nbr;
            if( (nbs=my_write(sockfd,EOL,strlen(EOL))) == -1 ) {
              fprintf( ERROR_CHANNEL, "%s: Cannot write to socket. %s.\n",
                       pn, sys_errlist[errno] );
              retval= RETVAL_ERROR;
              kill( getpid(), SIGINT );
              while( 1 ) sleep( 1 );
            }
            tnbs+= nbs;
            free( line );
          }
          else {
            break;
          }
        }
        /* EOF means: terminate program */
        kill( pid, SIGUSR1 );		/* signal child to exit */
      }
      else {
        /* stdin is pipe */
        unsigned char buf[INPUT_BUFFER_SIZE];
        while( 1 ) {
          if( (nbr=read(STDIN_FILENO,buf,INPUT_BUFFER_SIZE)) == -1 ) {
            fprintf( ERROR_CHANNEL, "%s: Cannot read from stdin. %s.\n",
                     pn, sys_errlist[errno] );
            retval= RETVAL_ERROR;
            kill( getpid(), SIGINT );
            while( 1 ) sleep( 1 );
          }
          if( nbr == 0 ) break;
          tnbr+= nbr;
          if( (nbs=my_write(sockfd,buf,nbr)) == -1 ) {
            fprintf( ERROR_CHANNEL, "%s: Cannot write to socket. %s.\n",
                     pn, sys_errlist[errno] );
            retval= RETVAL_ERROR;
            kill( getpid(), SIGINT );
            while( 1 ) sleep( 1 );
          }
          tnbs+= nbs;
        }
        /* EOF means: wait for server's answer */
      }
      while( 1 ) sleep( 1 );		/* broken by SIGCHLD (or Ctrl-C) */
      break;
    }	/* parent end */
  }

/* this code is reached only when exiting before fork() */
DIE_NOW:
  close( out_fd );
  close( sockfd );
  exit( retval );
}
