/*
  httpdtype.c  --  ask web server for its type
  Copyright (C) 1998, 1999 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include	"httpdtype.h"

int
dsphlp( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Determine type of web server running on host.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [host]\n", pn );
  fprintf( HELP_CHANNEL, "  -h\t help, write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level for %s\n", VERBOSE_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation info to %s and exit sucessfully\n", VERSION_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n");
  fprintf( HELP_CHANNEL, "  -p port\t set port (default is %d)\n", DEFAULT_PORT );
  return( 0 );
}


int
dspver( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  DEFAULT_HOST:  %s\n", DEFAULT_HOST );
  fprintf( VERSION_CHANNEL, "  DEFAULT_PORT:  %d\n", DEFAULT_PORT );
  return( 0 );
}


ulong
get_ipaddr_n_by_name( char* hostname ) {
/* return binary address in network byte order */
  struct hostent* ent= gethostbyname( hostname );
  if( ent == NULL ) return( -1 );
  if( ent->h_addr_list[0] == NULL ) return( -1 );
  return( *( (ulong*)ent->h_addr_list[0] ) );
}


int
hexdump( FILE* channel, UCHAR* buf, int n ) {
#define	BPL	16		/* bytes per line */
  int nbw= 0;
  int i,j;			/* j - line, i - byte in buffer */
  for( j=0; j*BPL<n; j++ ) {
    nbw+= fprintf( channel, "%04x:", j*BPL );
    for( i=j*BPL; i<(j+1)*BPL; i++ ) {
      if( i%4 == 0 ) nbw+= fprintf( channel, " " );
      if( i<n ) nbw+= fprintf( channel, " %02x", buf[i] );
      else      nbw+= fprintf( channel, "   " );
    }
    nbw+= fprintf( channel, "  | " );
    for( i=j*BPL; i<MIN(n,(j+1)*BPL); i++ ) {
      nbw+= fprintf( channel, "%c", IS_PRINTABLE(buf[i])?buf[i]:'.' );
    }
    nbw+= fprintf( channel, "\n" );
  }
  return( nbw );
}


int
main( int argc, char** argv ) {
/*
SOLYGA --------------------
SOLYGA main() httpservertype

SOLYGA started      : Sat Feb 27 23:42:42 CET 1999 @beast
*/
  int retval= RETVAL_OK;
  int c;
  int verbose= 0;
  char default_hostname[128]= DEFAULT_HOST;
  char* hostname= default_hostname;
  int port= DEFAULT_PORT;
  int sockfd= 0;
  struct sockaddr_in host;
  UCHAR buffer[BUFFER_SIZE];
  UCHAR* bp= buffer;
  ulong ipaddr_n;
  int nbts;			/* number of bytes to send */
  int nbs;			/* number of bytes sent */
  int nbr;			/* number of bytes received */
  int i;
  int binary_answer= 0;		/* flag: non-ascii answer */
#if defined OS_TYPE_AIX
  extern int errno;
  extern char* sys_errlist[];
#endif
/***** process options *****/
  while( (c=getopt(argc,argv,"hvVp:s:")) != EOF ) {
    switch( c ) {
      case 'p': /* set port */ {
        char* p;
        port= (int)strtol( optarg, &p, 0 );
        if( p == optarg ) {
          fprintf( ERROR_CHANNEL,
                   "%s: Invalid port number `%s'.\n",
                   *argv, optarg );
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        break;
      }
      case 'h': /* display help and exit sucessfully */
        dsphlp(*argv);
        retval= RETVAL_OK; goto DIE_NOW;
      case 'v': /* raise verbosity level */
        verbose++;
        break;
      case 'V': /* display version to VERSION_CHANNEL and exit sucessfully */
        dspver( *argv );
        retval= RETVAL_OK; goto DIE_NOW;
      case '?': /* refer to -h and exit unsucessfully */
        fprintf( ERROR_CHANNEL, "Try `%s -h' for more information.\n", *argv );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Options bug! E-mail me at %s.\n",
                 *argv, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }
  }

/* get hostname */
  if( optind<argc ) hostname= argv[optind];

/* open connexion */
  if( (sockfd=socket(AF_INET,SOCK_STREAM,0)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't create socket. %s.\n",
             *argv, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( verbose >= 1 ) {
    fprintf( VERBOSE_CHANNEL, "Resolving hostname `%s'.\n", hostname );
  }
  if( (ipaddr_n=get_ipaddr_n_by_name( hostname )) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't resolve hostname `%s'. %s.\n",
             *argv, hostname, h_errlist[h_errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( verbose >= 1 ) {
    char str_ipaddr[]= "000.000.000.000";
    UCHAR* p= (UCHAR*)&ipaddr_n;
    sprintf( str_ipaddr, "%d.%d.%d.%d",
              p[0], p[1], p[2], p[3] );
    fprintf( VERBOSE_CHANNEL, "IP address of `%s' is %s.\n",
             hostname, str_ipaddr );
  }

  host.sin_family= AF_INET;
  host.sin_port= htons(port);
  host.sin_addr.s_addr= ipaddr_n;
  if( verbose >= 1 ) {
    fprintf( VERBOSE_CHANNEL,
             "Connecting to host `%s' at port %d.\n",
              hostname, port );
  }
  if( connect(sockfd,(struct sockaddr*)&host,sizeof(host)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't connect to host `%s' port %d. %s.\n",
             *argv, hostname, port, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }

/* send query */
  nbts= sprintf( (char*)buffer, "%s\n%s\n%s%s\n\n",
                  "HEAD / HTTP/1.0",
                  "Accept: text/html",
                  "User-Agent: ",
                  *argv );
  if( nbts > BUFFER_SIZE ) {
    fprintf( ERROR_CHANNEL,
             "%s: Congratulations! Bug detected. E-mail me at %s.\n",
             *argv, MY_EMAIL_ADDRESS );
    retval= RETVAL_BUG; goto DIE_NOW;
  }
  if( verbose >= 2 ) {
    fprintf( VERBOSE_CHANNEL, "Sending query (%d bytes):\n%s", nbts, buffer );
  }
  if( (nbs=send(sockfd,buffer,nbts,0)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't send query. %s.\n",
             *argv, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }

/* receive answer */
  if( (nbr=recv(sockfd,buffer,BUFFER_SIZE-1,0)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't receive answer. %s.\n", 
             *argv, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  for( i=0; i<nbr; i++ ) if( NON_ASCII(buffer[i]) ) binary_answer= 1;
  buffer[i]= '\0';
  if( verbose >= 2 ) {
    fprintf( VERBOSE_CHANNEL, "Received answer (%d bytes, %s):\n",
             nbr, binary_answer?"binary":"ascii" );
    if( binary_answer ) hexdump( VERBOSE_CHANNEL, buffer, nbr );
    else                fprintf( VERBOSE_CHANNEL, "%s", buffer );
  }

/* close connexion */
  if( close( sockfd ) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't close socket. %s.\n",
             *argv, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }

/* pick out the Server type */
  if( (bp=strstr(buffer,SEARCH_STRING)) == NULL ) {
    fprintf( ERROR_CHANNEL,
             "%s: Unidentified server at `%s' port %d.\n",
             *argv, hostname, port );
    retval= RETVAL_QUIET; goto DIE_NOW;
  }
  bp+= strlen(SEARCH_STRING);
  { /* replace first '\n' or '\r' by '\0' */
    char* p= bp;
    while( *p != '\n' && *p != '\r' ) p++;
    *p= '\0';
  }
  if( verbose ) {
    fprintf( STD_CHANNEL, "Server answered: `%s'.\n", bp );
  }
  else {
    fprintf( STD_CHANNEL, "%s\n", bp );
  }

DIE_NOW:
  close( sockfd );
  exit( retval );
}
