#include <expat-wrapper.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>


#ifdef _DEBUG
#define READ_SIZE 16
#else
#define READ_SIZE (1024*8)
#endif

#ifndef O_BINARY
#ifdef _O_BINARY
#define O_BINARY _O_BINARY
#else
#define O_BINARY 0
#endif
#endif


/*=======================================================================*/

static void processingInstruction(void *userData, const char *target, const char *data)
{
  ((Expat_Wrapper *) userData)->ProcessingInstruction (target, data);
}


static void endElement(void *userData, const char *name)
{
  ((Expat_Wrapper *) userData)->EndElement (name);
}


static void startElement(void *userData, const char *name, const char **atts)
{
  ((Expat_Wrapper *) userData)->StartElement (name, atts);
}


static void characterData(void *userData, const char *s, int len)
{
  ((Expat_Wrapper *) userData)->CharacterData (s, len);
}


/* Lexicographically comparing UTF-8 encoded attribute values,
is equivalent to lexicographically comparing based on the character number. */

static int attcmp(const void *att1, const void *att2)
{
  return strcmp(*(const char **)att1, *(const char **)att2);
}

/*=======================================================================*/


void Expat_Wrapper::CharacterData(const char *s, int len)
{
  for (; len > 0; --len, ++s) {
    switch (*s) {
    case '&':
      fputs("&amp;", outfp);
      break;
    case '<':
      fputs("&lt;", outfp);
      break;
    case '>':
      fputs("&gt;", outfp);
      break;
    case '"':
      fputs("&quot;", outfp);
      break;
    case 9:
    case 10:
    case 13:
      fprintf(outfp, "&#%d;", *s);
      break;
    default:
      putc(*s, outfp);
      break;
    }
  }
}


void Expat_Wrapper::StartElement(const char *name, const char **atts)
{
  int nAtts;
  const char **p;
  putc('<', outfp);
  fputs(name, outfp);

  p = atts;
  while (*p)
    ++p;
  nAtts = (p - atts) >> 1;
  if (nAtts > 1)
    qsort((void *)atts, nAtts, sizeof(char *) * 2, attcmp);
  while (*atts) {
    putc(' ', outfp);
    fputs(*atts++, outfp);
    putc('=', outfp);
    putc('"', outfp);
    CharacterData(*atts, strlen(*atts));
    putc('"', outfp);
    atts++;
  }
  putc('>', outfp);
}


void Expat_Wrapper::EndElement(const char *name)
{
  putc('<', outfp);
  putc('/', outfp);
  fputs(name, outfp);
  putc('>', outfp);
}


void Expat_Wrapper::ProcessingInstruction(const char *target, const char *data)
{
  fprintf(outfp, "<?%s %s?>", target, data);
}


void Expat_Wrapper::ReportError(const char *filename, const char *errmsg,
                                int errcode, int errline, int errcol)
{
  if (errmsg)
    fprintf(stdout, "%s:%d:%ld: %s\n",
	    filename,
	    errline,
	    errcol,
            errmsg);
  else
    fprintf(stderr, "%s: (unknown message %d)\n", filename, errcode);
}


void Expat_Wrapper::FileError(const char *filename)
{
  int code = XML_GetErrorCode(parser);
  const char *message = XML_ErrorString(code);
  ReportError (filename, message, code,
               XML_GetErrorLineNumber(parser),
               XML_GetErrorColumnNumber(parser));
}


/*
void Expat_Wrapper::ProcessFileMAP(const void *data, size_t size, const char *filename, void *args)
{
  XML_Parser parser = ((PROCESS_ARGS *)args)->parser;
  int *retPtr = ((PROCESS_ARGS *)args)->retPtr;
  if (!XML_Parse(parser, data, size, 1)) {
    reportError(parser, filename);
    *retPtr = 0;
  }
  else
    *retPtr = 1;
}
*/


int Expat_Wrapper::ProcessFile (const char *filename)
{

  int fd = open(filename, O_BINARY|O_RDONLY);
  if (fd < 0) {
    perror(filename);
    return 0;
  }
  for (;;) {
    int nread;
    char *buf = (char*) XML_GetBuffer(parser, READ_SIZE);
    if (!buf) {
      close(fd);
      fprintf(stderr, "%s: out of memory\n", filename);
      return 0;
    }
    nread = read(fd, buf, READ_SIZE);
    if (nread < 0) {
      perror(filename);
      close(fd);
      return 0;
    }
    if (!XML_ParseBuffer(parser, nread, nread == 0)) {
      FileError(filename);
      close(fd);
      return 0;
    }
    if (nread == 0) {
      close(fd);
      break;;
    }
  }
  return 1;
}


Expat_Wrapper::Expat_Wrapper(const char *encoding, FILE *outstream)
    : outfp(outstream)
{
  Init(encoding);
}

Expat_Wrapper::~Expat_Wrapper()
{
    XML_ParserFree(parser);
    parser = 0;
}

void Expat_Wrapper::Init(const char *encoding)
{
  XML_Parser parser = XML_ParserCreate(encoding);
  XML_SetUserData(parser, this);
  XML_SetElementHandler(parser, startElement, endElement);
  XML_SetCharacterDataHandler(parser, characterData);
  XML_SetProcessingInstructionHandler(parser, processingInstruction);
}

