
/* if you are a programmer, please have a look at the
   line containing REQUEST FOR ADVICE
*/

/****h* knsbookmark/knsbookmark.c [0.2.3] ************************************
 * NAME
 *   knsbookmark -- convert netscape bookmarkfile into kfm tree structure
 *
 * COPYRIGHT
 *   (c) 1998  Ewald Arnold <earnold@w-4.de>  (home)
 *                          <earnold@rafi.de> (work)
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * NOTES
 *   This file is derived from a sample program shipped with James Clark's
 *   xml parser EXPAT (http://www.jclark.com/xml/expat.html).
 *
 * SEE ALSO
 *   knsbokmark.h, expat-library
 *
 *****************************************************************************
 */

#include "../config.h"
#include "knsbookmark.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stddef.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#ifdef SIM_SED
#include <signal.h>
#endif
    
#ifdef __IBMC__
#include <io.h>
#endif

#ifdef _MSC_VER
#include <io.h>
#endif

#ifndef O_BINARY
#ifdef _O_BINARY
#define O_BINARY _O_BINARY
#else
#define O_BINARY 0
#endif
#endif

#ifdef _MSC_VER
#include <crtdbg.h>
#endif


/****** knsbookmark.c/var/globals [0.2.1] ************************************
 * NOTES
 *   Adapt this block to modifiy the replace mechanism when the structure
 *   of the bookmark file changes. Be aware that XML wants all tags to be
 *   correctly closed: e.g. every <DT> must have its </DT> !
 *   Every attribute MUST have a value: FOLDED="1" is OK.
 *   Unimportant tags like <HR> are just removed.
 *
 * SOURCE
 */

struct SearchReplace substitutes[] = {
    {"<p>",            ""},
    {"<P>",            ""},
    {"<hr>",           ""},
    {"<HR>",           ""},
    {"<br>",           " "},  /* BR in DD */
    {"<BR>",           " "},

    {"</H3>",          "</H3></DT>"},
    {"</A>",           "</A></DT>"},

    {"&ouml;",         ""},
    {"&Ouml;",         ""},
    {"&auml;",         ""},
    {"&Auml;",         ""},
    {"&uuml;",         ""},
    {"&Uuml;",         ""},
    {"&szlig;",        ""},

    {"&",              "&amp;"},

    {"FOLDED",         "FOLDED=\"1\""},
    {0,                0}
};

struct Element act_ref = { 0, 0, 0, 0, 0, 0, 0, 0, 0};
char *paths [MAX_DEPTH];
struct StringList *aliasid = 0;
struct StringList *aliasof = 0;

int   depth = 0;
char *path_name = 0;
char *inter_file = 0;
char *start_dir = "./";
int   verbose = 0;
int   create_alias = 1;
char *last_kdelnk = 0;

/******/

/****** knsbookmark.c/addList [0.2.1] ****************************************
 * NAME
 *   addlist -- add new element to a StringList
 *
 * SYNOPSIS
 *   int addList (root, str, id);
 *
 * INPUTS
 *   - root  StringList**   root of StringList
 *   - str   char*          name of new element
 *   - id    int            id of element
 *
 * NOTES
 *   "str" is duplicated onto heap.
 *
 * RESULT
 *   0 if error occured
 *
 *****************************************************************************
 */

int addList (struct StringList **root, char *str, int id)
{
  struct StringList *ps;

  if (!root) return 0;

  ps = *root;
  if (!ps) {
    if (!(*root = (struct StringList*) malloc (sizeof(struct StringList)))) {
      fprintf(stderr, i18n("out of memory [addList 1]\n"));
      exit (1);
    }
    ps = *root;
  }
  else {
    while (ps->next)  /* goto end of chain */
      ps = ps->next;

    if (!(ps->next = (struct StringList*) malloc (sizeof(struct StringList)))) {
      fprintf(stderr, i18n("out of memory [addList 2]\n"));
      exit (1);
    }
    ps = ps->next;
  }
  
  ps->next = 0;
  if (!(ps->str = strdup(str))) {
    fprintf(stderr, i18n("out of memory [addList 3]\n"));
    exit (1);
  }

  strcpy (ps->str, str);
  ps->id = id;

  return 1;
}

/****** knsbookmark.c/freeList [0.2.1] ***************************************
 * NAME
 *   freelist -- free list and all subelements
 *
 * SYNOPSIS
 *   void freeList (ps);
 *
 * INPUTS
 *   - ps  StringList*   root of StringList
 *
 *****************************************************************************
 */

void freeList (struct StringList *ps)
{
    if (!ps) return;
    
    if (!ps->next)
      freeList (ps->next);  /* start from the end */

    free (ps->str);
    free (ps);
}

/****** knsbookmark.c/getList [0.2.1] ****************************************
 * NAME
 *   getlist -- get pointer to name of an element
 *
 * SYNOPSIS
 *   char* getList (ps, index):
 *
 * INPUTS
 *   - ps     StringList*   root of list
 *   - index  index         index of desired element
 *
 * RESULT
 *   pointer to name or 0 if index too high
 *
 *****************************************************************************
 */

char *getList (struct StringList *ps, int index)
{
    if (!ps) return 0;
    
    if (index == ps->id) return ps->str;
    return getList (ps->next, index);
}

/****** knsbookmark.c/utf8_to_latin1 [0.2.1] *********************************
 * NAME
 *   utf8_to_latin1 --  convert utf8-string to one byte latin1 chars
 *
 * SYNOPSIS
 *   void utf8_to_latin1 (s);
 *
 * INPUTS
 *   - s     char*       pointer to string, contains result afterwards !
 *
 * NOTES
 *   Only latin1-chars are allowed.
 *   Area of result is same as original, because converted string is
 *   shorter than utf8 original.
 *   
 *   The UTF-8 encoding method:
 *   --------------------------
 *
 *   00000000-0000007F 0xxxxxxx
 *   00000080-000007FF 110xxxxx 10xxxxxx
 *   00000800-0000FFFF 1110xxxx 10xxxxxx 10xxxxxx
 *   00010000-001FFFFF 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
 *   00200000-03FFFFFF 111110xx 10xxxxxx ...
 *   04000000-7FFFFFFF 1111110x 10xxxxxx 10xxxxxx ...
 *
 *   Leading bytes start with 00, 01, or 11.
 *   Trailing bytes start with 10
 *   11111111 (FF) does not appear
 *   11111110 (FE) does not appear
 *   
 *   UTF-8 is till Latin/English/US chauvinistic, as only ASCII takes
 *   one byte per character, rest of Europe takes two bytes per character,
 *   and the Far East takes three bytes per character.
 *
 * SEE ALSO
 *   rfc2044
 *
 *****************************************************************************
 */

static void utf8_to_latin1 (char *s)
{
    int c;
    char *p = s;
    while (*p) {
        if (*p & 0x80) {
            c = *p & 0x3F;           /* just assume range of latin1, */
            c <<= 6;
            c |= *(p+1) & 0x3F;      /* nothing else it possible     */
            p += 2;
            *s++ = c;
        }
        else {
            *s++ = *p++;
        }
    }
    *s = '\0';
}

/****** knsbookmark.c/characterData [0.2.1] **********************************
 * NAME
 *   characterData -- process character data (area between tags)
 *
 * SYNOPSIS
 *   void characterData (userData, s, len);
 *
 * INPUTS
 *   - userData     void*       dummy here
 *   - s            char*       string to process
 *   - len          int         valid len of s
 *
 * NOTES
 *   Callback from EXPAT, do not use directly !
 *
 *   Some special characters are transformed into their %-equivalents.
 *
 *****************************************************************************
 */

static void characterData(void *userData, const char *s, int len)
{
  char *p;
  int i = 0;
  free (act_ref.data); act_ref.data = 0;

  p = (char*) s;
  while ((p = memchr (p, '/',len)))   /* count additional bytes */
  {                                   /* for replacements       */
      i++;
      p++;
  }

  p = (char*) s;
  while ((p = memchr (p, '&',len)))   /* count additional bytes */
  {                                   /* for replacements       */
      i++;
      p++;
  }

  if (!(act_ref.data  = (char*) malloc (len+1+i*2)) ) {
    fprintf(stderr, i18n("out of memory [characterData]\n"));
    exit (1);
  }

  p = act_ref.data;
  while (len--) {               /* now replace / by %2F */
   if (*s == '/') {
     *p++ = '%';
     *p++ = '2';
     *p++ = 'F';
     s++;
   }
   else if (*s == '&') {
     *p++ = '%';
     *p++ = '2';
     *p++ = '6';
     s++;
   }
   else
     *p++ = *s++;
  }

  *p = '\0';
     
  utf8_to_latin1 (act_ref.data);
}

/****** knsbookmark.c/attcmp [0.2.1] *****************************************
 * NAME
 *   attcmp -- compare two attribute names
 *
 * SYNOPSIS
 *   int attcmp (att1, att2);
 *
 * INPUTS
 *   - att1     void*       pointer to attribute name 1
 *   - att2     void*       pointer to attribute name 2
 *
 * NOTES
 *   Lexicographically comparing UTF-8 encoded attribute values,
 *   is equivalent to lexicographically comparing based on the character
 *   number.
 *
 * RESULT
 *   0 if names are equal
 *
 *****************************************************************************
 */

static int attcmp(const void *att1, const void *att2)
{
  return strcmp(*(const char **)att1, *(const char **)att2);
}

/****** knsbookmark.c/startElement [0.2.1] **********************************
 * NAME
 *   startElement -- process start tag of xml element
 *
 * SYNOPSIS
 *   startElement (userData, name, atts);
 *
 * INPUTS
 *   - userData     void*       dummy here
 *   - name         char*       name of tag
 *   - atts         char**      tag attributes
 *
 * NOTES
 *   Callback from EXPAT, do not use directly !
 *
 *****************************************************************************
 */

static void startElement(void *userData, const char *name, const char **atts)
{
  int nAtts;
  const char **p;

  const char *elem  = 0,
             *content = 0;

  act_ref.elem = strdup (name);

  p = atts;
  while (*p) ++p;
  nAtts = (p - atts) >> 1;
  if (nAtts > 1)
    qsort((void *)atts, nAtts, sizeof(char *) * 2, attcmp);
  while (*atts) {
    elem = *atts++;     /* Name    */
    content = *atts++;  /* content */

    if (!strcasecmp (elem, "HREF")) {
      act_ref.href = strdup (content);
    }
    else if (!strcasecmp (elem, "FOLDED")) {
      sscanf (content, "%d", &act_ref.folded);
    }
    else if (!strcasecmp (elem, "ADD_DATE")) {
      sscanf (content, "%ld", &act_ref.add_date);
    }
    else if (!strcasecmp (elem, "LAST_MODIFIED")) {
      sscanf (content, "%ld", &act_ref.last_mod);
    }
    else if (!strcasecmp (elem, "LAST_VISIT")) {
      sscanf (content, "%ld", &act_ref.last_visit);
    }
    else if (!strcasecmp (elem, "ALIASID")) {
      act_ref.isalias = ALIASID;
      sscanf (content, "%d", &act_ref.aliasid);
      if (verbose)
        fprintf (stderr, i18n("found alias id %d\n"), act_ref.aliasid);
    }
    else if (!strcasecmp (elem, "ALIASOF")) {
      act_ref.isalias = ALIASOF;
      sscanf (content, "%d", &act_ref.aliasid);
      if (verbose)
        fprintf (stderr, i18n("found alias of %d\n"), act_ref.aliasid);
    }
  }
  act_ref.elem = strdup (name);
  if (!act_ref.elem) {
    fprintf(stderr, i18n("out of memory [startElement]\n"));
    exit (1);
  }

}

/****** knsbookmark.c/getactpath [0.2.1] *************************************
 * NAME
 *   getactpath -- build string with actual path in kfm tree
 *
 * SYNOPSIS
 *   char *getactpath (end_slash);
 *
 * INPUTS
 *   - end_slash         int    flag, add ending slash
 *
 * RESULT
 *   pointer to internal buffer with complete path
 *
 *****************************************************************************
 */

char *getactpath (int end_slash)
{
  static char actpath[2000] = "\0";
  int i, l = 0;

  strcpy (actpath, start_dir);
  if (actpath[strlen(actpath)] != '/')
    strcat (actpath, "/");
  
  for (i = 0; i < depth; i++)
    l += strlen (paths[i]);
  
  if (sizeof(actpath) < strlen(actpath) + l + depth+2) {  /* should't happen .. */
    fprintf (stderr, i18n("internal path buffer too small [getactpath]\n"));
    return 0;
  }

  i = 0;
  while (i < depth) {
    if (i)
      strcat (actpath, "/");
    strcat (actpath, paths[i++]);
  }
  if (end_slash)
    strcat (actpath, "/");
  return actpath;
}

/****** knsbookmark.c/endElement [0.2.3] *************************************
 * NAME
 *   endElement -- process end tag of xml element
 *
 * SYNOPSIS
 *   endElement (userData, name);
 *
 * INPUTS
 *   - userData     void*       dummy here
 *   - name         char*       name of tag
 *
 * NOTES
 *   Callback from EXPAT, do not use directly !
 *
 *****************************************************************************
 */

static void endElement(void *userData, const char *name)
{
  char *p   = 0;
  FILE *lfp = 0;
  
  if (!strcasecmp (name, "H3")) {
    paths[depth++] = strdup (act_ref.data);
    if (verbose)
      fprintf (stderr, i18n("generating Dir : %s  \n"), getactpath(0));
    if (mkdir ((p = getactpath(0)), S_IRWXU ) )
      fprintf (stderr, i18n("could\'t mkdir %s\n"), p);
  }
  else if (!strcasecmp (name, "DL")) {
    if (depth-- < -1)
      fprintf (stderr, i18n("mismatch in dir recursion\n"));
  }
  else if (!strcasecmp (name, "DD")) {
    if (last_kdelnk) { /* append description */
      FILE *fp;
      if ((fp = fopen (last_kdelnk, "aw"))) {
        fprintf (fp, "comment=%s\n", act_ref.data);
        fclose (fp);
      }
      else {
        fprintf (stderr, i18n("couldn\'t open %s, %s\n"), last_kdelnk, strerror(errno));
        return;
      }
      last_kdelnk = 0;
    }
  }
  else if (!strcasecmp (name, "A")) {
    char *bxpm,
         *sxpm;
    
    char *lnk = (char *) malloc (strlen (getactpath(1)) + strlen(act_ref.data) + strlen (KDE_LINK)+1);
    if (lnk) {
      strcpy (lnk, getactpath(1) );
      strcat (lnk, act_ref.data);
      strcat (lnk, KDE_LINK);

      free (last_kdelnk);
      last_kdelnk = strdup (lnk);
      if (!last_kdelnk) {
        fprintf(stderr, i18n("out of memory [endElement]\n"));
        exit (1);
      }
      
      if (act_ref.isalias == ALIASOF && create_alias) {
        addList (&aliasof, lnk, act_ref.aliasid);
      }
      else {
        bxpm = sxpm = "www.xpm";
        if (!strncasecmp ("ftp:", act_ref.href, 4))  /* change icon for ftp */
          bxpm = sxpm = "ftp.xpm";
        if (!strncasecmp ("news:", act_ref.href, 5)) /* change icon for news */
          bxpm = sxpm = "krn.xpm";
  
        if (verbose)
          fprintf (stderr, i18n("generating %s\n"), lnk);

        /* write all info we know, too short to waste diskspace */
        
        if ((lfp = fopen (lnk, "w"))) {
          fprintf (lfp, "# KDE Config File\n"
                        "[KDE Desktop Entry]\n"
                        "URL=%s\n"
                        "Icon=%s\n"
                        "MiniIcon=%s\n"
                        "Type=Link\n"
                        "[Netscape Data]\n",  act_ref.href, bxpm, sxpm);
          fprintf (lfp, "Folded=%d\n", act_ref.folded);
          fprintf (lfp, "Add date=%ld # %s", act_ref.add_date, ctime (&act_ref.add_date) );
          fprintf (lfp, "Last modified=%ld # %s", act_ref.last_mod, ctime (&act_ref.last_mod) );
          fprintf (lfp, "Last visit=%ld # %s", act_ref.last_visit, ctime (&act_ref.last_visit) );
  
          if (act_ref.isalias == ALIASID) {
            addList (&aliasid, lnk, act_ref.aliasid);
            fprintf (lfp, "Alias ID=%d\n", act_ref.aliasid);
          }
          fclose (lfp);
        }
        else {
          fprintf (stderr, i18n("couldn\'t open %s, %s\n"), lnk, strerror(errno));
          return;
        }
      }
      free (lnk);
    }
    else {
      fprintf(stderr, i18n("out of memory [endElement]\n"));
      exit (1);
    }
  }

  free (act_ref.elem);
  free (act_ref.data);
  free (act_ref.href);
  memset (&act_ref, 0, sizeof(act_ref) );
}


/****** knsbookmark.c/processingInstruction [0.2.1] **************************
 * NAME
 *   processingInstruction -- process xml processing instruction
 *
 * SYNOPSIS
 *   void processingInstruction (userData, target, data);
 *
 * INPUTS
 *   - userData     void*       
 *   - target       char*       
 *   - data         char*       
 *
 * NOTES
 *   Callback from EXPAT, do not use directly !
 *
 *****************************************************************************
 */

static void processingInstruction(void *userData, const char *target, const char *data)
{
}

/****** knsbookmark.c/reportError [0.2.1] ************************************
 * NAME
 *   reportError -- show error message from expat parser
 *
 * SYNOPSIS
 *   void reportError (parser, filename);
 *
 * INPUTS
 *   - parser    XML_Parser
 *   - filename  char*         name of file to process
 *
 *****************************************************************************
 */

static
void reportError(XML_Parser parser, const char *filename)
{
  int code = XML_GetErrorCode(parser);
  const char *message = XML_ErrorString(code);
  if (message)
    fprintf(stderr, "%s:%d:%d: %s\n",
            filename,
            XML_GetErrorLineNumber(parser),
            XML_GetErrorColumnNumber(parser),
            message);
  else
    fprintf(stderr, i18n("%s: (unknown message %d)\n"), filename, code);
}

/****** knsbookmark.c/processStream [0.2.1] **********************************
 * NAME
 *    processStream -- parse an xml data stream
 *
 * SYNOPSIS
 *   int  processStream (fd, filename, parser);
 *
 * INPUTS
 *   - fd        int           input stream
 *   - filename  char*         name of file to process
 *   - parser    XML_Parser
 *
 * RESULT
 *   1 if successful
 *
 *****************************************************************************
 */

static
int processStream(int fd, const char *filename, XML_Parser parser)
{
  int   eof_found = 0;
  char *p;
  
  for (;;) {
    int nread;
    char *buf = XML_GetBuffer(parser, READ_SIZE);
    if (!buf) {
      fprintf(stderr, i18n("out of memory [processStream]\n"));
      return 0;
    }
    nread = read(fd, buf, READ_SIZE);
    
    if (nread < 0) {
      perror(filename);
      return 0;
    }

#ifdef SIM_SED

    /* REQUEST FOR ADVICE
     
       If someone knows why there is no EOF in a closed pipe, let me know.

       Is the following really necessary: looking for
       the "terminating" 0 to stop reading ?
    */

    if ((p = memchr (buf, 0, nread) )) {
      eof_found = 1;
      nread = p-buf;                        /* do not feed our eof to parser */
    }

#endif

    if (!XML_ParseBuffer(parser, nread, nread == 0)) {
      fprintf (stderr, i18n("error, last block fed to parser:\n"));
      fwrite (buf, 1, nread, stderr);       /* show last block fed to parser */
      reportError(parser, filename);
      return 0;
    }

    if (nread == 0 || eof_found) {
      break;
    }

  }
  return 1;
}

/****** knsbookmark.c/usage [0.2.1] ******************************************
 * NAME
 *    usage -- show info about this program
 *
 * SYNOPSIS
 *   void usage(prog);
 *
 * INPUTS
 *   - prog       char*        name of program
 *
 *****************************************************************************
 */

static
void usage(const char *prog)
{
    fprintf(stderr, "%s", prog);
    fprintf(stderr, i18n(" " VERSION " -- convert Netscape bookmark files into kfm bookmark tree\n"));
    fprintf(stderr, i18n("  usage: %s [-h] [-v] [-r] [-d outputdir] [-w xml-file] bookmark-file\n"
                               "          h       show help\n"
                               "          a       don't link to aliases, create .kdelnk files instead\n"
                               "          v       show info\n"
                               "          w file  dump intermediate (well formed) bookmark file\n"
                               "          o dir   output directory\n"), prog);
  exit(1);
}

#ifdef SIM_SED

/****** knsbookmark.c/substitute [0.2.1] *************************************
 * NAME
 *    substitute -- substitute parts in netscape bookmarkfile to get real xml
 *
 * SYNOPSIS
 *   int  substitute (buff, maxlen, old, new);
 *
 * INPUTS
 *   - buff      char**        string to process
 *   - maxlen    int*          max len of buff
 *   - old       char*         string to search
 *   - new       char*         replacement for "old"
 *
 * RESULT
 *   actual length of buffered string
 *
 *****************************************************************************
 */

int substitute (char **buff, int *maxlen, char *old, char* new)
{
   int    offset = 0;
   char  *tmp,
         *p;
   int    n = strlen(new),
          o = strlen(old),
          l = strlen (*buff);
         
   while ((p = strstr ((*buff)+offset, old))) {
     if (n - o + l >= *maxlen) {                    /* need more buff space ?     */
       *maxlen = l+n+100;                           /* get a bit more than needed */
       if (!(tmp  = (char*) malloc (*maxlen) )) {
         fprintf(stderr, i18n("out of memory [substitute]\n"));
         exit (1);
       }
       strcpy (tmp, *buff);
       free (*buff);
       *buff = tmp;
       p = strstr ((*buff)+offset, old);
     }
     offset = p - *buff + n;

     if (n != o)
       memmove (p+n, p+o, l - (p-(*buff))-o+1);

     if (n)
       memmove (p, new, n);

     l += n-o;
   }
   return l;
}

/****** knsbookmark.c/my_getdelim [0.2.3] ************************************
 * NAME
 *    my_getdelim -- get string from file up to stop char
 *
 * SYNOPSIS
 *   int  my_getdelim (lineptr, n, delimiter, fp);
 *
 * INPUTS
 *   - lineptr    char**        buffer
 *   - n          size_t        size of buffer "lineptr" points to
 *   - delimiter  int           char designating line end
 *   - fp         FILE*         stream to read from
 *
 * RESULT
 *   length of current buffer, -1 if error occured
 *
 *****************************************************************************
 */

/* Text from original GNU package:

   Read up to (and including) a TERMINATOR from FP into *LINEPTR
   (and null-terminate it).  *LINEPTR is a pointer returned from malloc (or
   NULL), pointing to *N characters of space.  It is realloc'ed as
   necessary.  Returns the number of characters read (not including the
   null terminator), or -1 on error or EOF.  */

/* #undef HAVE_GETDELIM */

#ifndef HAVE_GETDELIM
long my_getdelim (char ** lineptr, size_t *n, int delimiter, FILE *fp)
{
   char  *tmp,
         *p;
   int    l       = 0,
          reading = 1,
          c;

   if (!lineptr || !*lineptr || !*n)
     return -1;

   p = *lineptr;
   while (reading) {
     if ((c = fgetc (fp)) < 0)
       return -1;

     if (l >= *n-1) {                          /* need more buff space ?     */
       *n = l+100;                             /* get a bit more than needed */
       if (!(tmp  = (char*) malloc (*n) )) {
         fprintf(stderr, i18n("out of memory [my_getdelim]\n"));
         exit (1);
       }
       strcpy (tmp, *lineptr);
       free (*lineptr);
       *lineptr = tmp;
       l = strlen (*lineptr);
     }
     p[l++] = c;
     p[l] = '\0';
     
     if (c == delimiter)
       reading = 0;
   }
   return l;
}
#endif

/****** knsbookmark.c/make_wellformed [0.2.3] ********************************
 * NAME
 *    make_wellformed -- generate a well formed xml document
 *
 * SYNOPSIS
 *   int  make_wellformed (ifd, ofd);
 *
 * INPUTS
 *   - ifd        int           input data stream
 *   - ofd        int           output data stream
 *
 * RESULT
 *   0 if successful
 *
 *****************************************************************************
 */

int make_wellformed (int ifd, int ofd)
{
   char *start_tag  = "<!DOCTYPE NETSCAPE-BOOKMARKFILE> <NETSCAPE-BOOKMARKFILE>\n";
   char *end_tag    = "</NETSCAPE-BOOKMARKFILE>\n\0"; /* 0 = terminator to processStream ! */

   char *buff;
   int  buffsize = 1000;
   int  well = 0;
   int  raw;
   FILE *ifs = fdopen (ifd, "r");
   FILE *interfd = 0;
   int  xread = 0;
   struct SearchReplace *snr;

   if (!(buff  = (char*) malloc (buffsize) )) {  /* start with 1000 bytes for buff,
                                                    should always be enough           */
     fprintf(stderr, i18n("out of memory [make_wellformed]\n"));
     exit(1);
   }

   /* dump intermediate stuff for debugging purposes */

   if (inter_file) {
     if (!(interfd = fopen (inter_file, "w"))) {
       fprintf (stderr, i18n("couldn\'t open %s, %s\n"), inter_file, strerror(errno));
       exit (1);
     }
   }

   /* read line by line, make it well formed xml and feed it to the parser
    * through the pipe
    */
   
#ifdef HAVE_GETDELIM
   while ((raw = getdelim (&buff, (size_t *)&buffsize, '\n', ifs)) > 0) {
#else
   while ((raw = my_getdelim (&buff, (size_t *)&buffsize, '\n', ifs)) > 0) {
#endif
     xread += raw;
     if (strstr (buff, "<!DOCTYPE")) {
       if (interfd) {
         if (fwrite (start_tag, 1, strlen(start_tag), interfd) == 0) {
           fprintf (stderr, i18n("couldn\'t write to %s, %s\n"), inter_file, strerror(errno));
           exit (1);
         }
       }
       else
         write (ofd, start_tag, strlen(start_tag));
     }
     else {
       char *p;
       /* special case: <DD> describes link in previous line */
       if (   (p = strstr(buff, "<DD>"))
           || (p = strstr(buff, "<dd>"))) {
         /* just append missing /DD */
         well = substitute (&buff, &buffsize, "\n", "</DD>\n");
       }
       snr = &substitutes[0];
       while (snr->search) {
         well = substitute (&buff, &buffsize, snr->search, snr->replace);
         snr ++;
       }

       if (interfd) {
         if (well)
           if (fwrite (buff, 1, well, interfd) == 0) {
             fprintf (stderr, i18n("couldn\'t write to %s, %s\n"), inter_file, strerror(errno));
             exit (1);
           }
       }
       else
         write (ofd, buff, well);
     }
   }

   if (interfd) {
     if (fwrite (end_tag, 1, strlen(end_tag), interfd) == 0) {
       fprintf (stderr, i18n("couldn\'t write to %s, %s\n"), inter_file, strerror(errno));
       exit (1);
     }
     fclose (interfd);
   }
   else {
     write (ofd, end_tag, strlen(end_tag)+1); /* also send terminator 0 */
   }
   
   return 0;
}

#endif



int main(int argc, char **argv)
{

#ifdef SIM_SED

  pid_t child_pid;
  int sed_pipe[2];

#endif

  int i;
  const char *encoding = "ISO-8859-1";
  char *filename = 0;
  FILE *fp = stdout;
  int fd;
  int result = 0;
  XML_Parser parser;
  struct StringList *ps;
  
  i = 1;
  while (i < argc && argv[i][0] == '-') {
    int j;
    if (argv[i][1] == '-' && argv[i][2] == '\0') {
      i++;
      break;
    }
    j = 1;
    if (argv[i][j] == 'v') {
      verbose = 1;
      if (argv[i][j+1] == 'v')
        verbose = 2;
      i++;
    }
    else if (argv[i][j] == 'h') {
      usage(argv[0]);
    }
    else if (argv[i][j] == 'a') {
      create_alias = 0;
    }
    else if (argv[i][j] == 'o') {
      if (argv[i][j + 1] == '\0') {
	if (++i == argc)
	  usage(argv[0]);
	start_dir = argv[i];
      }
      else
        start_dir = argv[i] + j + 1;
      i++;
    }
    else if (argv[i][j] == 'w') {
      if (argv[i][j + 1] == '\0') {
	if (++i == argc)
	  usage(argv[0]);
	inter_file = argv[i];
      }
      else
        inter_file = argv[i] + j + 1;
      i++;
    }
    else if (argv[i][j] == '\0' && j > 1)
      i++;
    else
      usage(argv[0]);
  }
  if (i != argc)
    filename = argv[i];

  /* init expat and install callbacks */
  
  parser = XML_ParserCreate(encoding);
  XML_SetUserData(parser, fp);
  XML_SetElementHandler(parser, startElement, endElement);
  XML_SetCharacterDataHandler(parser, characterData);
  XML_SetProcessingInstructionHandler(parser, processingInstruction);

  if (verbose) {
    fprintf(stderr, i18n("output-dir: %s\n"), start_dir);
    if (inter_file)
      fprintf(stderr, i18n("intermediate stuff to: %s\n"), inter_file);
    if (filename)
      fprintf(stderr, i18n("bookmark file: %s\n"), filename);
  }

  if (filename) {
    fd = open(filename, O_BINARY|O_RDONLY);
    if (fd < 0) {
      perror(filename);
      return 1;
    }
  }
  else {
    filename = "(stdin)";
    fd  = 0;                 /* stdin */
  }

  if (inter_file) {
    int res =  make_wellformed(fd, 2);
    if (fd)
      close (fd);
    exit (res);
  }
  
#ifdef SIM_SED

  /* simulate what sed has done in V0.1*/

  if (pipe(sed_pipe)) {
    fprintf (stderr, i18n("could not create pipe: %s\n"), strerror (errno) );
    exit (1);
  }

  child_pid = fork();
  if (child_pid < (pid_t) 0) {
    fprintf (stderr, i18n("could not fork(): %s\n"), strerror (errno) );
    exit (1);
  }
  else if (child_pid == (pid_t) 0) {
    /* child process is making a wellformed xml file */
    make_wellformed(fd, sed_pipe[1]);
    if (fd)
      close (fd);
    close (sed_pipe[1]);
    exit (0); /* parent will do the rest if possible */
  }
  else {
    /* parent process is doing the hard work */
    if (!inter_file)
      result = processStream(sed_pipe[0], filename, parser);
    close (sed_pipe[0]);
  }

#else

  result = processStream(fd, filename, parser);
  if (fd)
    close (fd);

#endif

  if (create_alias && !inter_file && result) {
    if (verbose)
      fprintf (stderr, i18n("generating alias links\n"));
    ps = aliasof;
    while (ps) {
      char *src,
           *dst;
      dst = ps->str;
      src = getList (aliasid, ps->id);
  
      if (!src) {
        fprintf (stderr, i18n("no ALIASID=%d found, can\'t generate link\n"), ps->id);
      }
      else {
        if (verbose)
          fprintf (stderr, i18n("generating alias %s -> %s \n"), src, dst);
        if (link (src, dst)) {                   
          fprintf (stderr, i18n("could\'t generate hard link %s\n"), dst);
        }
      }
      ps = ps->next;
    }
  }

  freeList (aliasid);
  freeList (aliasof);
  
  XML_ParserFree(parser);
  if (verbose && result)
    fprintf (stderr, i18n("successfully ready\n"));
  
#ifdef SIM_SED

  if (!result)  /* kill well-forming child if still running */
    kill (child_pid, SIGKILL);

#endif

  return 0;
}
