/* vim: set sw=8 ts=8 si: */
/* Author: Guido Socher, Copyright: GPL */

#include <stdio.h>
#include <stdlib.h>
#include "hash.h"
#include <string.h>
#include "config.h"
/* 
 * This implements a simple fifo-buffer class where any number of
 * strings can be stored in fifo order and where one can quckliy
 * check if a given string is already stored 
 */
struct buffer{
        char *dat;
        char *val;
	int hashnumber;
        struct buffer *nextbuff;
};
/* b_tail is the latest element and we add always to the b_tail. 
 * The pointers go from head to tail. When you search the list
 * then you must start at the head. Which is the oldest element.
 *  b_head->[     ]
 *          [nextbuff->[      ]
 *                     [nextbuff->[      ]
 *                                [nextbuff->NULL
 *                                ^
 *                      b_tail ---|
 */
static struct buffer *b_head[3],*b_tail[3];

/* init the class */
void init_fifo_class(int which){
	if (which >=0 && which <3){
		b_head[which]=NULL;
		b_tail[which]=NULL;
	}else{
		printf("ERROR: max 3 fifo buffers available\n");
		exit(1);
	}
}

/* add a new string to the fifo hash table, data is the key in this
 * table and val is just any additional string. */
void add_to_fifo(int which,char *dat, char *val){
        struct buffer *newelem;
        char *i_dat,*i_val;
        i_dat=(char *)malloc(strlen(dat)+1);
        if (i_dat==NULL) exit(8);
        i_val=(char *)malloc(strlen(val)+1);
        if (i_val==NULL) exit(8);
        newelem=(struct buffer *)malloc(sizeof(struct buffer));
        if (newelem==NULL) exit(9);
        strcpy(i_dat,dat);
        strcpy(i_val,val);

        newelem->dat=i_dat;
        newelem->val=i_val;
        newelem->hashnumber=genhashkey(i_dat);
        if (b_tail[which]==NULL){
                newelem->nextbuff=NULL;
                b_head[which]=newelem;
                b_tail[which]=newelem;
        }else{
                /*add at the end*/
                b_tail[which]->nextbuff=newelem;
                newelem->nextbuff=NULL;
                b_tail[which]=newelem;
        }
}

/* 
 * remove the oldest element. Return 1 if data was read out of the
 * fifo and zero if there was nothing in the fifo.
 * dat and val are out variables. Use like this:
 * char *d,*v;
 * read_out_of_fifo(1,&d,&v);
 * &d,&v change their values the next time you call this function.
 */
int read_out_of_fifo(int which, char **o_dat, char **o_val){
        static char dat[MAXFIFO_STRLEN+1]; /*avoid that the user of the class does the free*/
        static char val[MAXFIFO_STRLEN+1]; /*avoid that the user of the class does the free*/
        struct buffer *tmp;
        if (b_head[which]==NULL){
                return(0);
        }
        strncpy(dat,b_head[which]->dat,MAXFIFO_STRLEN);
        strncpy(val,b_head[which]->val,MAXFIFO_STRLEN);
        dat[MAXFIFO_STRLEN]='\0';
        val[MAXFIFO_STRLEN]='\0';
        tmp=b_head[which];
        b_head[which]=b_head[which]->nextbuff;
        if (b_head[which]==NULL) b_tail[which]=NULL;
        free(tmp->dat);
        free(tmp);
	*o_dat=dat;
	*o_val=val;
        return(1);
}

/* check if a given string is already stored in the fifo 
 * 1 as return value means already stored*/
int is_in_fifo(int which,char *dat){
        struct buffer *bufelem;
	int hkey;
	hkey=genhashkey(dat);
        if (b_head[which]==NULL){
                return(0);
        }
	bufelem=b_head[which];
	while (bufelem != NULL){
		if(bufelem->hashnumber == hkey && strcmp(bufelem->dat,dat)==0){
			/* hash key equal and string equal */
			return(1);
		}
		bufelem=bufelem->nextbuff;
	}
        return(0);
}

/* add a  string to the fifo unless it is already in there 
 * return 1 if string was added and zero if it was already there */
int add_to_fifo_unless_there(int which,char *dat,char *val){
	if (is_in_fifo(which,dat)) return(0);
	add_to_fifo(which,dat,val);
	return(1);
}

