#!/usr/bin/env perl
#
# FILE:
# workload.pl
#
# FUNCTION:
# Perl script to run and control multiuser runs of "webclient"
# Performs coordinated rampup and launch of individual copies,
# prints summary statistics, and detects stalled/stopped clients.
#
# In normal operation, should be invoked only after setup from 
# the 'run.workload' perl script, as shown in the examples directory. 
#
# =======================================================
#
# to round to thousandths, call as &round(value,1000)
#
sub round 
{
	return (int($_[0]*$_[1]+0.5)/$_[1]);
}

# =======================================================
#
# interrupt handler
#
sub int_handler
{
	print "Warning: $myname: user hit ^C or sent -kill.........\n";
	print "Info: $myname: killing $prog child processes.....\n";
	&exit_routine;
}

# =======================================================
#
# routine to read the shared memory communication area
#
sub read_mem
{
	local $i;
	local $j;
	local $string;
	local $tmp;

	# There are 16 bytes of global comm area at the start
	# of the shared memory area. 
	$sline = __LINE__;
	shmread($shmid,$string,16,$bytes_per_child*$number_of_users) 
		|| die "$myname: shmread failed at line $sline!\n";
	$len = length($string);
	$total_sessions = 0;
	$total_requests = 0;
	$total_connects = 0;
	$total_gifs = 0;

	$total_kbytes = 0;
	$total_html_kbytes = 0;
	$total_gif_kbytes = 0;

	$sum_first_response = 0;
	$sum_end_to_end = 0;

	$stopped = 0;
	$running = 0;
	$stopped_this_interval = 0;
	$no_progress = 0;
	$no_progress_this_interval = 0;
	$made_progress_this_interval = 0;
	#
	# scan through the list of runs, recording statistics as we go
	#
	for($j=0;$j<$number_of_users;$j++) {
		$tmp = substr($string,$j*$bytes_per_child,$bytes_of_integers);
		@ivalues = unpack("i*",$tmp);

		# tmpchar contains client status string
		$tmpchar = substr($string,$j*$bytes_per_child+$bytes_of_integers,4);
		$status[$j] = $tmpchar;

		# decode the rest of the shared mem area
		$sessions[$j]     = $ivalues[0] - $starting_sessions[$j];

		$request[$j]      = $ivalues[2];
		$requests[$j]     = $ivalues[3] - $starting_requests[$j];
		$connects[$j]     = $ivalues[4] - $starting_connects[$j];
		$gifs_fetched[$j] = $ivalues[5] - $starting_gifs[$j];

		$response[$j]     = $ivalues[6] - $starting_response[$j];
		$end_to_end[$j]   = $ivalues[7] - $starting_end_to_end[$j];

		$total_bytes[$j]  = $ivalues[8] - $starting_total_bytes[$j];
		$html_bytes[$j]   = $ivalues[9] - $starting_html_bytes[$j];
		$gif_bytes[$j]    = $ivalues[10] - $starting_gif_bytes[$j];

		# accumulate totals of various sorts
		$total_sessions += $sessions[$j];
		$total_requests += $requests[$j];
		$total_connects += $connects[$j];
		$total_gifs     += $gifs_fetched[$j];

		$total_kbytes      += $total_bytes[$j];
		$total_html_kbytes += $html_bytes[$j];
		$total_gif_kbytes  += $gif_bytes[$j];

		$sum_first_response += $response[$j];
		$sum_end_to_end     += $end_to_end[$j];

		#
		# did this run stop?
		#
		if ($tmpchar eq "STOP") {
			$stopped++;
			#
			# did this run stop this interval?
			#
			if ($stopped_run[$j] == 0) {
				$stopped_this_interval++;
				$stopped_run[$j] = 1;
			}
		}
		#
		# Has this client ever run?
		if (($tmpchar eq "RUN ") && ($requests[$j] > 0)) {
			$running ++;
		}
		#
		# has this run stopped making progress?
		#
		if (($html_bytes[$j]  == $saved_html_bytes2[$j])
		  && ($gifs_fetched[$j] == $saved_gifs_fetched2[$j])
		  && ($tmpchar eq "RUN ")) {
			$no_progress++;
			#
			# did this run stop making progress this interval?
			#
			if ($no_progress_run[$j] == 0) {
				$no_progress_this_interval++;
				$no_progress_run[$j] = 1;
			}
		} 
		#
		# did this run start making progress this interval?
		#
		elsif (($no_progress_run[$j] == 2) && ($tmpchar ne "STOP")) {
			$made_progress_this_interval++;
			$no_progress_run[$j] = 3;
		}
	}

	# report stats in kbytes
	$total_kbytes      /= 1024.0;
	$total_html_kbytes /= 1024.0;
	$total_gif_kbytes  /= 1024.0;

	#
	# update the arrays used to keep track of stalled runs
	#
	for($j=0;$j<$number_of_users;$j++) {
		$saved_html_bytes2[$j] = $saved_html_bytes1[$j];
		$saved_gifs_fetched2[$j] = $saved_gifs_fetched1[$j];
		$saved_html_bytes1[$j] = $saved_html_bytes[$j];
		$saved_gifs_fetched1[$j] = $saved_gifs_fetched[$j];
		$saved_html_bytes[$j] = $html_bytes[$j];
		$saved_gifs_fetched[$j]= $gifs_fetched[$j];
	}

	#
	# compute the number of requests and kbytes for this interval
	#
	$requests_this_interval = $total_requests - $last_requests;
	$last_requests = $total_requests;

	$total_kb_this_interval = $total_kbytes - $last_kbytes;
	$last_kbytes = $total_kbytes;

	$currtime = time(); 
	$interval = $currtime - $last_time;
	if ($interval > 0) {
		$req_rate_this_interval = &round($requests_this_interval/$interval,100);
		$kb_rate_this_interval  = &round($total_kb_this_interval/$interval,100);
	} else {
		$req_rate_this_interval = "UNDEF";
		$kb_rate_this_interval  = "UNDEF";
	}
	$last_time = $currtime;
	$run_time = $currtime - $start_time;
	if ($run_time > 0) {
		$req_rate = &round($total_requests/($run_time),100);
		$kb_rate  = &round($total_kbytes/($run_time),100);
	} else {
		$req_rate = "UNDEF";
		$kb_rate = "UNDEF";
	}

	#
	# compute average response times overall, and for this interval
	#
	$resp_this_interval = $sum_first_response - $last_resp;
	$last_resp = $sum_first_response;

	$end_this_interval = $sum_end_to_end - $last_end_to_end;
	$last_end_to_end = $sum_end_to_end;

	if ($total_requests > 0) { 
		$avg_first_response = $sum_first_response / $total_requests; 
		$avg_end_to_end     = $sum_end_to_end / $total_requests; 
	} else { 
		$avg_first_response = "UNDEF";
		$avg_end_to_end     = "UNDEF";
	}
	if ($requests_this_interval > 0) { 
		$resp_this_interval /= $requests_this_interval; 
		$end_this_interval  /= $requests_this_interval; 
	} else {
		$resp_this_interval = "UNDEF";
		$end_this_interval  = "UNDEF";
	}

}

# =======================================================
#
# routine to print the status about forward progress or lack thereof.
#
format STALL =
S @### @######      @### @########  @#########  @#########    @<<<<
$j,$sessions[$j],$request[$j],$requests[$j],$html_bytes[$j],$gifs_fetched[$j],$status[$j]
.

#00000000111111111122222222223333333333444444444455555555556666666666777777777788888888889
#23456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
format OVERALL = 
T @#### @# @# @###### @###### @###### @##### @##### @##### @##.## @#.### @#.###
$run_time, $stopped, $no_progress, $total_sessions, $total_requests, $total_connects, $total_gifs, $total_kbytes, $total_gif_kbytes, $req_rate, $avg_first_response, $avg_end_to_end
.

format INTERVAL = 
I                                           @####.##       @##.## @#.### @#.###
$kb_rate_this_interval, $req_rate_this_interval, $resp_this_interval, $end_this_interval
.

$purty = 0;

sub print_status
{
	if ($purty > 10) { $purty = 0; }
	if ($purty == 0) {
		#        @#### @# @# @###### @###### @###### @##### @##### @##### @##.## @#.### @#.###
		print "T  ELAP STP STL  SESS    REQS   CONNS   GIFS KBYTES GIF-KB   REQ   FIRST END to\n";
		#                                                  @####.##       @##.## @#.### @#.###
		print "I   for this interval:                       KB-RATE        RATE   DATA   END\n";
		print "------------------------------------------------------------------------------- \n";
	}
	$purty ++;
	$~ = OVERALL;
	write;
	$~ = INTERVAL;
	write;

	$printed_header = 0;
	if ($stopped_this_interval > 0) {
		print "------------------------------------------------------------------------------- \n";
		print "Warning: The following runs stopped this interval:\n";
		$~ = STALL;
		if ($printed_header == 0) {
			$printed_header = 1;
			print "S child    sess  request  requests  html_Kbytes   gif_files  status\n";
		}
		for($j=0;$j<$number_of_users;$j++) {
			if ($stopped_run[$j] == 1) {
				$stopped_run[$j] = 2;
				write;
			}
		}
	}
	if ($no_progress_this_interval > 0) {
		print "------------------------------------------------------------------------------- \n";
		print "Warning: The following runs appear to be stalled:\n";
		print "    (They have made no progress for 2 intervals.)\n";
		$~ = STALL;
		if ($printed_header == 0) {
			$printed_header = 1;
			print "S child    sess  request  requests  html_Kbytes   gif_files  status\n";
		}
		for($j=0;$j<$number_of_users;$j++) {
			if ($no_progress_run[$j] == 1) {
				$no_progress_run[$j] = 2;
				write;
			}
		}
	}
	if ($made_progress_this_interval > 0) {
		print "------------------------------------------------------------------------------- \n";
		print "Info: The following runs became un-stalled this interval:\n";
		$~ = STALL;
		if ($printed_header == 0) {
			$printed_header = 1;
			print "S child    sess  request  requests  html_Kbytes   gif_files  status\n";
		}
		for($j=0;$j<$number_of_users;$j++) {
			if ($no_progress_run[$j] == 3) {
				$no_progress_run[$j] = 0;
				write;
			}
		}
	}

	if ($stopped == $number_of_users) {
		print "------------------------------------------------------------------------------- \n";
		printf "$myname: All children have stopped.... run being aborted.....\n";
		$sline = __LINE__;
		shmctl($shmid,&IPC_RMID,0) || die "$myname: shmctl failed at line $sline.\n";
		print "Info: $myname: exits\n";
		exit;
	}
}

#
# we always exit via way of this routine
#
# (whether we exit due to ^C or by normal completion)
sub exit_routine 
{
	local $s;
	local $n;

	print "------------------------------------------------------------------------------- \n";
	#
	# look for instances of prog being run by this user
	#
	# $logname = $ENV{"LOGNAME"};
	# Do a Berkley style ps command, this is far more common & 
	# standardized across a large variety of operating systems.  
	# Make sure that only a breif listing is used, otherwise we 
	# risk scrolling the executable name off the screen.
	open(PS,"ps x |");
	$found = 0;
	while(<PS>) 
	{
		if(index($_,$prog)>0) 
		{
			# found one -- get the process id
			@split = split(/ +/,$_);
			$pid = $split[1];
			# send it a SIGTERM
			# print "kill -TERM $pid\n";
			system("kill -TERM $pid");
			$found++;
		}
	}
	close PS;
	print "Info: $myname: found and will stop $found copies of $prog\n";

	# loop and wait for all of them to report they've stopped ...
	sleep (1);
	$s = 0;
	$n = 0;
	while (($s != $number_of_users) && ($n <$number_of_users*10))
	{
		&read_mem();
		if ($s != $stopped) {
			print "$stopped of $number_of_users have stopped.\n";
		}
		$s = $stopped;
		if ($number_of_users - $s > 10) { sleep (5); $n +=5; }
		sleep (1);
		$n ++;
	}

	# We've given them 10 seconds each to shut down. 
	# If they've still not shut down, whack em dead.
	if ($s != $number_of_users) {
	open(PS,"ps x |");
	$found = 0;
	while(<PS>) 
	{
		if(index($_,$prog)>0) 
		{
			# found one -- get the process id
			@split = split(/ +/,$_);
			$pid = $split[1];
			system("kill -9 $pid");
			$found++;
		}
	}
	close PS;
	print "Warning: $myname: found and killed dead $found hung copies of $prog\n";
	}
	# release the shared memory area
	shmctl($shmid,&IPC_RMID,0);
	print "Info: $myname: exits\n";
	exit;
}

#
# main program ===========================================================
#

# get the program name
@COMPONENTS = split(/\//,$0);
$myname = $COMPONENTS[$#COMPONENTS];
# $short_version is used by webclient to make sure that this script
# and webclient use compatible shared memory layouts
# following MUST be 4 chars long
$short_version = "4.1 ";
# updates to this program that do not change short_version
# are flagged by letters at the end of the version name
$version = $short_version."e";
print "Info: $myname($version):starts.................\n";

# set up some default search paths
$INC[$#INC+1] = $bindir;
$INC[$#INC+1] = "bin";
$INC[$#INC+1] = "../bin";
$INC[$#INC+1] = "../../bin";
($inky = $0) =~ s/workload.pl//;
$INC[$#INC+1] = $inky;
$INC[$#INC+1] = "/usr/lib/perl5";
$INC[$#INC+1] = "/usr/lib/perl5/site_perl/aix";
$INC[$#INC+1] = "/usr/lib/perl5/site_perl/aix/sys";
$INC[$#INC+1] = "/usr/lib/perl5/site_perl/i386-linux";
$INC[$#INC+1] = "/usr/lib/perl5/site_perl/i386-linux/sys";
$INC[$#INC+1] = "/usr/lib/perl5/site_perl/i386-linux/linux";


#
# We need the shared-memory header to get access to the shared mem functions
require "shm.ph";

# verify that the user has done some minimal configuration
die "The number of web servers must be specified" if ($nservers == 0);
die "A port number for each server must be sepcified" if ($#port < ($nservers-1));

# make sure that there are some rational defaults
# Set $prog to the name of the program to run. Normally this is webclient.
if ($prog eq "") {$prog  =  "webclient"; }

if ($max_sessions == 0) {
die "Missing parameter: call as: $myname url.file number_of_users think_time run_duration dir_prefix max_sessions \n" if ($#ARGV<5);
} 

if ($run_duration == 0) {
die "Missing parameter: call as: $myname url.file number_of_users think_time run_duration\n" if ($#ARGV<3);
} 

if ($number_of_users == 0) {
die "Missing parameter: call as: $myname url.file number_of_users \n" if ($#ARGV<1);
} 

if ($url_file eq "") {
die "Missing parameter: call as: $myname url.file\n" if ($#ARGV<0);
} 

if ($#ARGV >= 0) { $url_file        = $ARGV[0]; }
if ($#ARGV >= 1) { $number_of_users = $ARGV[1]; }
if ($#ARGV >= 2) { $think_time      = $ARGV[2]; }
if ($#ARGV >= 3) { $run_duration    = $ARGV[3]; }
if ($#ARGV >= 4) { $dir_prefix      = $ARGV[4]; }
if ($#ARGV >= 5) { $max_sessions    = $ARGV[5]; }


if ($dir_prefix eq "") { $dir_prefix = "run_"; }
if ($output_prefix eq "") { $output_prefix = "webclient_output_"; }
if ($report_prefix eq "") { $report_prefix = "webclient_report_"; }
if ($trace_prefix eq "") { $trace_prefix = "webclient_trace_"; }

$dest = $dir_prefix.$run_duration."_".$number_of_users;
print "Data will be stored in directory named: $dest.\n";

#
#
# read usernames, passwords and pins
#
open (PASSWD, $passwdfile) || die "Can't open file $passwdfile";
$i=0;
while (<PASSWD>) {
	chop;
	if ($_ =~ /\#/) { next; }
	($custid[$i], $pin[$i], $passwd[$i]) = split(/:/, $_);
	$i++;
}
close PASSWD;

#
# if we don't have enough parameter array entries, then we give up
#
$ncust = $#custid+1;
if ($number_of_users > $ncust) {
	print "Fatal Error: $myname: Asked to run with $number_of_users users, but only $ncust logon id's supplied in the $passwdfile file.\n";
	exit 99;
}

# 
# xxxxxxxxx config stuff xxxxxxxxxxxx
# set up control variables
# 
$| = 1;
$input = $url_file;
$output = "$dest/output/$output_prefix";
$report = "$dest/reports/$report_prefix";
$trace = "$dest/traces/$trace_prefix";
#
# unique shmkey's allow multiple independent runs to run simultaneously
# $shmkey = 1234567;
$shmkey = time();
$quiet = 1;
$error_log = "$dest/$prog.error_log";

#
# build the output directories
#
if (! -e $dest) {
	system("mkdir $dest");
}
if (! -e "$dest/reports") {
	system("mkdir $dest/reports");
}
if (! -e "$dest/output") {
	system("mkdir $dest/output");
}
if (! -e "$dest/traces") {
	system("mkdir $dest/traces");
}

#
# initialize
#
$last_requests = 0;
$last_kbytes   = 0;
$last_resp     = 0;
$last_end_to_end = 0;
for($j=0;$j<$number_of_users;$j++) {
	$starting_sessions[$j]    = 0;
	$starting_requests[$j]    = 0;
	$starting_connects[$j]    = 0;
	$starting_gifs[$j]        = 0;

	$starting_response[$j]    = 0;
	$starting_end_to_end[$j]  = 0;

	$starting_total_bytes[$j] = 0;
	$starting_html_bytes[$j]  = 0;
	$starting_gif_bytes[$j]   = 0;

	$stopped_run[$j]          = 0;
	$no_progress_run[$j]      = 0;
	$saved_html_bytes2[$j]    = -1;
	$saved_gifs_fetched2[$j]  = -1;
	$saved_html_bytes1[$j]    = -1;
	$saved_gifs_fetched1[$j]  = -1;
	$saved_html_bytes[$j]     = -1;
	$saved_gifs_fetched[$j]   = -1;
}

# 
# create the shared memory area
# 
# these values must match the one in webbot/webclient
$SIZE = 65536;
$PERM = 0600;
$shmid = shmget($shmkey,$SIZE,$PERM | &IPC_CREAT | &IPC_EXCL );
if ($shmid == "") {
	print "Warning: First shmget failed... will delete segment and try again.\n";
	print "\$!=$!\n";
	$shmid = shmget($shmkey,$SIZE,$PERM);
	shmctl($shmid,&IPC_RMID,0);
	$shmid = shmget($shmkey,$SIZE,$PERM | &IPC_CREAT | &IPC_EXCL );
	if ($shmid == "") {
		print "Error: Second shmget failed... giving up!\n";
		print "\$!=$!\n";
		exit;
	}
}
print "Info: shmget returned $shmid\n";

# 
# now initialize the shared memory area
# 
# there are 48 bytes per child in the shared memory array
$bytes_per_child = 48;
# of which 44 bytes are integers and 4 are status characters
$integers = 11;
$bytes_of_integers=$integers*4;
# there are 4 integers (16 bytes) of shared comm area at
# the start of the shared memory area--init 1st three of
# them to 0, -1, and 0 -- put version name in 4th one
# ($short_version MUST be 4 chars exactly)
# word 0 -- # of children who have initialized and are ready
#           to start submitting requests
# word 1 -- # of children allowed to start submitting requests
# word 2 -- rampup complete flag
$string = pack("i*",0,-1,0);
$string = $string.$short_version;
# concatenate on the rest of the structures
for($i=0;$i<$number_of_users;$i++) {
	$init_integer = pack("i",-1*($i+1));
	for($j=0;$j<$integers;$j++) {
		$string = $string.$init_integer;
	}
	$string = $string."INIT";
}
$len = length($string);
print "Info: Writing $len bytes to shared memory area.\n";
$sline = __LINE__;
$rc=shmwrite($shmid,$string,0,length($string)) || die "$myname: shmwrite failed at line $sline.\n";

#
# read it back to make sure we got it correct
#
# &read_mem();  &print_staus();

# 
# start off the child runs
# 
# $n = $number_of_users."_";
$nice = "";
if ($number_of_users > 15) {
	$nice = "nice -+5";
}

# we want repeatable "random" seeds
srand (10);

for($i=0;$i<$number_of_users;$i++) {
	$server_sub = $i % $nservers;

	$seed = int(rand (123456));

	print "$nice $bindir/$prog $options 
               --think-time=$think_time 
               --num-sessions=$max_sessions 
               --input-file=$input 
               --webserver=$webserver:$port[$server_sub] 
               --report-file=$report$i 
               --user-pin-pw=$custid[$i]:$pin[$i]:$passwd[$i] 
               --random-seed=$seed 
               --shmem=$i:$shmkey 
               --log-file=$error_log 
               --fork --quiet > $output$i 2>&1 \n\n";

	$rc = system("$nice $bindir/$prog $options --think-time=$think_time --num-sessions=$max_sessions --input-file=$input --webserver=$webserver:$port[$server_sub] --report-file=$report$i --user-pin-pw=$custid[$i]:$pin[$i]:$passwd[$i] --random-seed=$seed --shmem=$i:$shmkey --log-file=$error_log --fork --quiet > $output$i 2>&1 ");

	# if the return code is non-zero, then there was some sort of obvious
	# failure, such as a bad argument, missing input file, etc.  Abort the
	# run immediately, instead of plowing on.
	if ($rc != 0) { 
		print "\n";
		print "************************************************************\n";
		print "************************************************************\n";
		print "Error: $prog failed to start! \n";
		print "The output file $output$i shows that the error is: \n\n";
		system ("cat $output$i");
		print "\n";
		print "************************************************************\n";
		print "************************************************************\n";
		print "\n";
		goto time_to_exit; 
	}
}

#
# establish an interrupt handler 
#
$SIG{'INT'} = 'int_handler';
$SIG{'QUIT'} = 'int_handler';
$SIG{'TERM'} = 'int_handler';

# ignore death-of-child signals, otherwise zombies remain unharvested.
$SIG{'CLD'} = 'IGNORE';
$SIG{'CHLD'} = 'IGNORE';

# 
# wait for the children to indicate they have started up
# 

# the following is not the real start time.  this is done here
# so that it has a value in case we exit during the startup loop
$start_time = time();

$tries = 0;
while(1) {
	&read_mem();
	$sline = __LINE__;
	shmread($shmid,$string,0,4) || die "$myname: shmread failed at line $sline.\n";
	$nstarted = unpack("i",$string);

	if ($nstarted < $number_of_users) {
		print "Info: Number of children who have started:$nstarted\n";
		if ($tries < 60) {
			$tries++;
			print "Sleeping..... (try=$tries)\n";
			sleep(10);
		} else {
			print "Error: Children did not start after 600 seconds.\n";
			print "Aborting the run. .......................\n";
			goto time_to_exit;
		}
	} else {
		goto start_children;
	}
}

#
# In order to keep everyone from running in lockstep when running
# a throughput run (which has think times set to zero), we start
# the children off in groups of $nstart children at a time.
# We then wait for the indicated children to run at least one
# request before we start the next group of children.
#
# This can be disabled by setting $nstart = $number_of_users
#

start_children:
$nstart = 10;
print "Info: Releasing $number_of_users children now.... ";
if ($number_of_users > $nstart) 
{
	print "in groups of $nstart\n";
}
print "\n";
for($i=0;$i<$number_of_users;$i+=$nstart) 
{
	$low = $i;
	$high = $i + $nstart - 1;
	if ($high > $number_of_users-1) {$high = $number_of_users-1;}
	print "Releasing children $low through $high...\n";
#
######################################################
# the following code does not work because shmwrite
# appears to have a bug in it that causes the write
# to fail the second time it is done (with a segment 
# violation).  So, we have the little program updshm
# that updates shared memory for us and we use it
# instead. ...........................................
######################################################
#	$flag = pack("i",$high);
#	$strlen = length($flag);
#	printf "strlen flag=$strlen\n";
#	$sline = __LINE__;
#	print "just before shmwrite..........\$shmid=$shmid\n";
#	$mstart = 4;
#	$mlen   = 4;
#	shmwrite($shmid,$flag,$mstart,$mlen*16) || die "$myname: shmwrite failed at line $sline.\n";
#	print "just after  shmwrite..........\n";
	# store the value $high at offset 4 in the shared memory segment
	system("$bindir/updshm $shmid 4 $high");
	$running = 0;
	$ntry = 0;
	if (! defined $ramp_interval) { $ramp_interval = 20; }
	$maxtry = 600 / $ramp_interval;

	while ($running < $high) 
	{
		sleep($ramp_interval);
		&read_mem();

		#
		# get the status information for child $j
		#
		for($j=0;$j<=$high;$j++) {
			if ($status[$j] eq "STOP") {
				print "Error: Child $j stopped during ramp up....\n";
				print "Run is being aborted...............\n";
				goto time_to_exit;
			}
		}
		$ntry++;
		if ($ntry > $maxtry) {
			print "Error: Children did not start after $ntry * $ramp_interval seconds....\n";
			print "Run is being aborted........................\n";
			goto time_to_exit;
		}
	}
}

print "Info: All children started.....\n";

#
# indicate to the children that rampup is complete
# this will cause statistics to start to be collected.   
#
# store a "1" at offset 8 in the shared memory area
system("$bindir/updshm $shmid 8 1");
$date = `date`;
print "$myname: rampup complete at time: $date\n";

# 
# record the start time of the test
# 
$start_time = time();
$last_time = $start_time;

#
# Here we record the initial values for progress made during ramp up.
# We will subtract these values off for reporting statistics.
#
$last_requests = 0;
$last_kbytes   = 0;
$last_resp     = 0;
$last_end_to_end = 0;

for($j=0;$j<$number_of_users;$j++) {
	$starting_sessions[$j]    = $sessions[$j];
	$starting_requests[$j]    = $requests[$j];
	$starting_connects[$j]    = $connects[$j];
	$starting_gifs[$j]        = $gifs_fetched[$j];

	$starting_response[$j]    = $response[$j];
	$starting_end_to_end[$j]  = $end_to_end[$j];

	$starting_total_bytes[$j] = $total_bytes[$j];
	$starting_html_bytes[$j]  = $html_bytes[$j];
	$starting_gif_bytes[$j]   = $gif_bytes[$j];
}

# 
# Wake up periodically, print statistics and go back to sleep
# repeat until the run end conditions are met. ..............
# 
$run_time = 0;
if (! defined $sleep_interval) { $sleep_interval = 30; }
$sleep_time = $sleep_interval;
while(1) {
	sleep($sleep_time);
	$awake = time();
	&read_mem ();
	&print_status ();
	$now   = time();
	$sleep_time = $sleep_interval - ($now - $awake);
	$run_time = $now - $start_time;
	if (($run_time >= $run_duration) || ($total_sessions >= $max_sessions)) {
		goto time_to_exit;
	}
}

# 
# exit
# 
time_to_exit:
print "Info: $myname: awake, killing $prog child processes.....\n";
$date = `date`;
print "$myname: run complete at time: $date\n";

# int_handler calls this subroutine as well
&exit_routine;
