//
// class wlCookie
// class wlHeader
// class wlURLEncoding
//
// FUNCTION:
// This header file describes three classes that deal with key-value
// pairs encoded in strings.  Because of this, they are closesly related.
// Yet, because the actual string encoding is slightly different in each 
// case, there is no easy, convenient way to generalize/abstract thier
// actual function.  Thus, there are three classes.
//
// Note that the implementation of all three classes is based on that
// of wlString.  This allows for rather efficient storage and manipulation
// with a minimum of string copying and memory fragmentation.  Note 
// however, that the programming interface is a little more complex/subtle
// than the 'programming 101' design might have been.
//
// ----------------------------------------------------------------------
// class wlHeader implements an HTTP header management class, useful for 
// creating and manipulating headers.  HTTP headers consist of lists of
// key-value pairs, where each pair appears on a new line, and the key is 
// separated from the value by a colon. An example of an HTTP header is 
// shown below:
//    
//    Connection: Keep-Alive
//    User-Agent: Mozilla/4.04 [en] (X11; I; AIX 4.2)
//    Host: accountlink.edwardjones.com
//    Accept: image/gif, image/x-xbitmap, image/jpeg, image/png, */*
//    Accept-Language: en
//    Accept-Charset: iso-8859-1,*,utf-8
//
// Both HTTP requests and responses have headers.
//
// ----------------------------------------------------------------------
// class wlURLEncoding implements a url-encoded string management class,
// useful for creating and manipulating url-encoded strings. URL-encoded
// strings consist of lists of key-value pairs, where each pair is separated
// by an ampersand, and the key is separated from the value by an equals sign.
// An example url-encoded string is shown below:
//
//     xtr=Frameset&SID15=uIjUpW&Token=75789&NLS=EN&UserID=TESTACCT108
//
// ----------------------------------------------------------------------
// class wlCookie implements a Cookie management class, useful for 
// creating and manipulating cookies.  HTTP Cookies consist of lists of
// key-value pairs, where each pair is separated by a semicolon, and the 
// key is separated from the value by an equals sign. An example of an 
// HTTP cookie string is shown below:
//    
//    IBONK=0EE40968; path=/; expires=Sunday, 30-Apr-2000 01:00:00 GMT; domain=.asdf.com;
//
// Cookies are transmitted in HTTP headers in both requests and responses.
// Servers send cookies to browsers with the following types of header lines:
//    Set-cookie: IBONK=0EE40968; path=/; secure;
//    Set-Cookie: sesessionid=4E13Q2YAAAAAACLVYMGAAAA;Path=/
// Note that the second example is missing a semicolon, is missing whitespace,
// and has different capitalization.  Notice the first example includes a keyword
// without a value ('secure', which denotes taht cookie should be used only on 
// an SSL connection).
//
// Note that since the keyword 'secure' is an exception, and is not followed by
// an '=' sign, the behaviour of the GetValue() routine is modified slightly:
// GetValue ("secure") will return null if that keyword is present, else non-null.
//
// Browsers return cookies to servers with the following kinds of header lines:
//    Cookie: IBONK=0EE40968;
//    Cookie: IBONK=C355DABC; sesessionid=4E13Q2YAAAAAACLVYMGAAAA
//
// Note that the path, domain, and 'secure' parts are not returned by the browser.
// Note that the second example sends two cookies and is not terminated by
// a semicolon.
//
// ----------------------------------------------------------------------
// Because of the general similarities between these three classes, they 
// derive from a common base class and thus share some method documentation:
//
// METHODS:
// The RemoveField() method will remove the (first occurance of) the 
//    field and its value from the string.  If the field was found and 
//    removed, then '1' is returned, else 0 is returned.
//
// The AddIfAbsent() method searches for the field "field" in the string.
//    If the field is not found, then it is added, with the value "value",
//    and '1' is returned. If the field is found, then nothing is done, 
//    and a value of '0' is returned.
//
//    class wlHeader: A colon is automatically placed after the field, 
//    and the value is terminated with a CRLF.
//
//    class wlURLEncoding: the ampersand and equals sign are added as 
//    appropriate.
//
//    class wlCookie: the semicolon and equals sign are added as 
//    appropriate.
//
// The ReplaceField() method searches the header for the string  "field".
//    If it is found, then the new value is used to replace the old value.
//    A '1' is returned if  replacement occured, otherwise, a '0' is returned.
//    Note that a null-string value causes the filed to be removed.
//
//    class wlHeader: everything following the first colon after "field" 
//    until CRLF is replaced by value. This may remove old whitespace.
//    New whitespace is not stripped from the new value.  
// 
//
// The ReplaceOrAddField() method is similar, except that if the field
//    is not found, then it is added. More precisely, the field, a colon, 
//    a space, the value, and a CRNL is appended.  A '1' is returned if
//    a replacement or addition was done, otherwise, a '0' is returned.
//
//
// Thus, for example, if the value of the header is "Host: foo.com\r\n", 
// then 
// ReplaceField ("Ho", "bar.com") will change it to "Host: bar.com\r\n"
// while
// ReplaceOrAddField ("Fnord", "bar.com") will change it to
// "Host: foo.com\r\nFnord: bar.com\r\n"
// and
// ReplaceField ("Fnord", "bar.com") will have no effect.
//
// The ReplaceFields() and ReplaceOrAddFields() routines will do the same, 
//    but for a null-terminated list of field-value pairs.  They return a
//    count of the number of substitutions made.
//
// The GetValue() method returns the value associated with "field", else NULL.
//    A case-insensitive search for the field name is made.  The returned
//    value is *not* NULL terminated, its terminated the way it would
//    normally appear in its context (ie. by a semicolon, ampersand, newline)
//
// The Protected methods:
// The replace() method does the actual heavy lifting.
// The fld_start() method takes a pointer to a value, null-terminates
//    the value, and returns a pointer to the next field start after the 
//    value.
// The val_start() method takes a pointer to a field, null-terminates
//    the field, and returns a pointer to the next value start after the 
//    field.
// Both of these methods 'destroy' thier input by writing NULL's into 
// the string.


#ifndef __WL_HEADER_H__
#define __WL_HEADER_H__

#include "super.h"

class wlKeyPairs :
   public wlString
{
   public:
      virtual ~wlKeyPairs();
      int Remove (const char *field);
      int AddIfAbsent (const char *field, const char *value);
      int ReplaceField (const char *field, const char *value);
      int ReplaceOrAddField (const char *field, const char *value);
      int ReplaceFields (char **fields, char **values);
      int ReplaceFields (wlKeyPairs &pairs);
      int ReplaceOrAddFields (char **fields, char **values);
      int ReplaceOrAddFields (wlKeyPairs &pairs);

      virtual char * GetValue (const char  *field) = 0;
      
   protected:
      virtual char *fld_start (char  *start) = 0;
      virtual char *val_start (char  *start) = 0;
      virtual int replace (const char * field, 
                           const char * value, 
                           int do_add, int do_replace) = 0;
};

class wlHeader :
   public wlKeyPairs
{
   public:
      virtual ~wlHeader();
      wlHeader& operator= (const wlString &);
      virtual char * GetValue (const char  *field);

   protected:
      virtual char *fld_start (char  *start);
      virtual char *val_start (char  *start);
      virtual int replace (const char * field, 
                           const char * value, 
                           int do_add, int do_replace);
};


char * is_content_binary (wlHeader &);

class wlURLEncoding :
   public wlKeyPairs
{
   public:
      virtual ~wlURLEncoding();
      wlURLEncoding& operator= (const wlString &);
      virtual char * GetValue (const char  *field);
      
   protected:
      virtual char *fld_start (char  *start);
      virtual char *val_start (char  *start);
      virtual int replace (const char * field, 
                           const char * value, 
                           int do_add, int do_replace);
};


class wlCookie :
   public wlKeyPairs
{
   public:
      wlCookie& operator= (const wlString &);
      virtual char * GetValue (const char  *field);
      
   protected:
      virtual char *fld_start (char  *start);
      virtual char *val_start (char  *start);
      virtual int replace (const char * field, 
                           const char * value, 
                           int do_add, int do_replace);
};


#endif /* __WL_HEADER_H__ */

/* ============================================================= */
