// 
// FILE:
// pool.C
//
// FUNCTION:
// Implements a thread pool. 
//
// HISTORY:
// Created by Linas Vepstas October 1999
//

#include <signal.h>
#include <string.h>

#include "generic.h"
#include "pool.h"

// =====================================================================

int 
wlThread :: GetId (void)
{
   return id;
}

// =====================================================================

void 
wlThread :: Go (void)
{
   pthread_mutex_lock (&lock);
   PDBG ("about to cond_signal to cut loose id=%d \n", id);
   pthread_cond_signal (&got_work);
   pthread_mutex_unlock (&lock);
}

// =====================================================================

void 
wlThread :: NeverMind (void)
{

   // release a thread which was going to be used, but wasn't.
   pthread_mutex_lock (&home->pool_lock);
   if (busy)
   {
      busy = 0;
      home->free_threads ++;
      PDBG ("freeing thread id=%d total free=%d\n", 
          id, home->free_threads);
      // signal anyone who might have been waiting for a free thread.
      pthread_cond_signal (&home->pool_empty);
   
      // signal anyone who was waiting for everything to finish.
      if (home->free_threads == home->total_threads) 
      {
         pthread_cond_signal (&home->pool_full);
      }
   }
   pthread_mutex_unlock (&home->pool_lock);
}


// =====================================================================

void 
wlThread :: Dispatch (void)
{
   PDBG ("enter thread dispatch loop for id=%d \n", id);
   pthread_mutex_lock (&lock);

   // Block the signals that we expect the main thread to catch.
   // If we don't block them, then they'll get lost.
   // Note that the list of blocked signals should coincide with 
   // those set up in webopts.C
   sigset_t new_mask, old_mask;
   sigemptyset (&new_mask);
   sigemptyset (&old_mask);
   sigaddset (&new_mask, SIGHUP);
   sigaddset (&new_mask, SIGINT);
   sigaddset (&new_mask, SIGQUIT);
   sigaddset (&new_mask, SIGPIPE);
   sigaddset (&new_mask, SIGTERM);
   sigaddset (&new_mask, SIGUSR1);
   sigaddset (&new_mask, SIGUSR2);
   sigaddset (&new_mask, SIGALRM);

#ifdef AIX
#define pthread_sigmask sigthreadmask
#endif /* AIX */
   pthread_sigmask (SIG_BLOCK, &new_mask, &old_mask);

   // Let parent know that this thread has initialized
   pthread_mutex_lock (&home->pool_lock);
   home->free_threads ++;
   id = -id;
   pthread_setspecific (home->tid_key, (void *) id);
   pthread_setspecific (home->self_key, (void *) this);
   pthread_cond_signal (&home->pool_full);
   pthread_mutex_unlock (&home->pool_lock);

   PDBG ("done initializing thread dispatch loop for id=%d \n", id);
   while (1) {
      // Wait for work to be given to us.
      pthread_cond_wait (&got_work, &lock);
      busy = 1;

      PDBG ("about to dispatch thread on id=%d\n", id);
      // Negative id means we should shut down and exit
      if (0 > id) break;

      RunInThread();

      // signal anyone who might have been waiting for a free thread.
      pthread_mutex_lock (&home->pool_lock);
      home->free_threads ++;
      busy = 0;
      PDBG ("freeing thread id=%d total free=%d\n", 
          id, home->free_threads);
      pthread_cond_signal (&home->pool_empty);

      // signal anyone who was waiting for everything to finish.
      if (home->free_threads == home->total_threads) 
      {
         pthread_cond_signal (&home->pool_full);
      }
      pthread_mutex_unlock (&home->pool_lock);
   }

   // If we exit the loop, we should self-destruct.  We do this
   // here instead of the destructor because that way no one can 
   // accidentally start us up again.

   // First, take ourselves out of the pool.
   pthread_mutex_lock (&home->pool_lock);
   PDBG ("self destructing on id=%d\n", id);
   id = 0;
   home->total_threads --;
   if (home->free_threads == home->total_threads) 
   {
      pthread_cond_signal (&home->pool_full);
   }
   pthread_mutex_unlock (&home->pool_lock);
   pthread_mutex_unlock (&lock);

   home = 0x0;
   pthread_cond_destroy (&got_work);
   pthread_mutex_destroy (&lock);
}

// =====================================================================

void 
wlThread :: ChkStk (void)
{
   size_t sz;
   sz = stack_base - (unsigned long) &sz;  // assume stack grows down
   if (sz > stack_used) 
   {
      stack_used = sz;
      if (64000 < sz) PERR("stack size %d=0x%x is too large\n", sz, sz);
   }
}

// =====================================================================

void 
wlThreadPool :: ChkStk (void)
{
   wlThread *th = (wlThread*) pthread_getspecific (wlThreadPool::self_key);
   if (th) th->ChkStk();
}

// =====================================================================

void *
wlThreadPool :: dispatch (void *arg)
{
   wlThread *th = (wlThread*) arg;
   th->stack_base = (unsigned long) &arg; // arg is in the frame nearby
   th->Dispatch();
   return 0;
}

// =====================================================================

wlThreadPool :: wlThreadPool (int nthreads, wlThread **array)
{
   int i;

   total_threads = nthreads;
   pool_entries = nthreads;
   free_threads = 0;
   pool = new wlThread* [nthreads];
   for (i=0; i<nthreads; i++) {
      pool[i] = array[i];
   }

   // Create a key to store the per-thread id.
   pthread_key_create (&tid_key, NULL);
   pthread_setspecific (tid_key, (void *) 0);
   pthread_key_create (&self_key, NULL);
   pthread_setspecific (self_key, (void *) 0);
   tid_key_is_init = 1;

   // First, create a bunch of locks that we will use to control
   // the dispatching of work units into each thread.  All we 
   // want are 'fast' locks.

   pthread_mutexattr_t matter;
   pthread_mutexattr_init (&matter);
   pthread_condattr_t catter;
   pthread_condattr_init (&catter);
   for (i=0; i<nthreads; i++) 
   {
      pool[i]->id = -(i+1);
      pool[i]->home = this;
      pool[i]->busy = 0;
      pool[i]->stack_base = 0;
      pool[i]->stack_used = 0;
      pthread_mutex_init (&pool[i]->lock, &matter);
      pthread_cond_init (&pool[i]->got_work, &catter);
   }

   // initialize the pool lock as well
   pthread_mutex_init (&pool_lock, &matter);

   // initialize pool-empty condition
   pthread_cond_init (&pool_empty, &catter);
   pthread_cond_init (&pool_full, &catter);

   pthread_mutexattr_destroy (&matter);
   pthread_condattr_destroy (&catter);

   // Create a bunch of threads.  The only place where 
   // we cancel/join them is in the destructor.
   pthread_attr_t  attr;
   pthread_attr_init(&attr);

   // Under AIX, the default stacksize is 96KB.  
   // ... XXX mystery hang is gone; it was *not* a stack problem.
   // pthread_attr_setstacksize (&attr, 256*1024); 

   /* create a bunch of threads */
   pthread_mutex_lock (&pool_lock);
   for (i=0; i<nthreads; i++) 
   {
       int rc = pthread_create(&pool[i]->thread, &attr, dispatch, 
                              (void *)pool[i]);
       if (rc) 
       {
          int norr = errno;
          PERR ("pthread_create failed for id=%d with errno %d\n", 
             i+1, norr, strerror (norr));
       }
   }

   // Close out the thread attr structure.
   pthread_attr_destroy(&attr);

   // wait for all of the threads to initialize themselves
   while (total_threads > free_threads) {
      PDBG ("waiting for thread initialization, %d of %d complete\n", 
         free_threads, total_threads);
      pthread_cond_wait (&pool_full, &pool_lock);
   }
   PDBG ("done with thread initialization, %d of %d complete\n", 
      free_threads, total_threads);
   pthread_mutex_unlock (&pool_lock);
}

// =====================================================================

wlThreadPool :: ~wlThreadPool ()
{
   int i;
   pthread_mutex_lock (&pool_lock);

   for (i=0; i<pool_entries; i++) 
   {
      // If the thread has already self-destructed, no-op.
      // Otherwise tell it to self-destruct.
      if (0 != pool[i]->id)
      { 
         // if the thread is idle, tell it to self-destruct.
         if (0 == pool[i]->busy) 
         {
            pthread_mutex_lock (&pool[i]->lock);
            pool[i]->busy = 1;

            // The lock can free up right before its destroyed ...
            if (0 == pool[i]->id) 
            {
               pthread_mutex_unlock (&pool[i]->lock);
               continue;
            }
            free_threads --;
            pool[i]->id = - pool[i]->id;
            pthread_cond_signal (&pool[i]->got_work);
            pthread_mutex_unlock (&pool[i]->lock);
         }
         else
         {
            // try to kill it.  XXX this won't really work, see below.
            free_threads --;
            pthread_cancel(pool[i]->thread);
         }
      }
   }

   // Wait for all of the threads to self-destruct
   // We have to do this in addition to the pthread_join below, 
   // as it seems that AIX 4.2 has a broken pthread_join.  Note that
   // the XXX hack alert message about hangs applies here as well.
   while (total_threads > 0) {
      PDBG ("waiting for termination, %d left of %d\n", 
          total_threads, pool_entries);
      pthread_cond_wait (&pool_full, &pool_lock);
   }
   PDBG ("done with thread termination of %d threads\n", pool_entries);
   pthread_mutex_unlock (&pool_lock);

   // Wait for each thread to complete.
   // XXX hack alert -- we can hang here in some not-uncommon 
   // situations.  Problem is, if any of the threads were busy,
   // then the pthread_cancel() won't do a thing because none 
   // of them will be at a cancellation point.  As a result, 
   // we won't be able to join them until they finish, and
   // if they never finish, then we hang.   One thing we can 
   // is to skip doing the join.  But then, doing the delete[]
   // risks corrupting memory.  Which risk would you rather have?
   for (i=0; i<pool_entries; i++) 
   {
      pthread_join (pool[i]->thread, NULL);
   }

   if (0 != total_threads) 
   {
      PERR ("unable to destroy all threads; %d left of %d\n", 
         total_threads, pool_entries);
      return;
   }
   pool_entries = 0;
   delete [] pool;
   pool = 0x0;

   pthread_key_delete (tid_key);
   pthread_key_delete (self_key);

   pthread_cond_destroy (&pool_empty);
   pthread_cond_destroy (&pool_full);
   pthread_mutex_destroy (&pool_lock);
}

// =====================================================================

wlThread *
wlThreadPool::GetThread (void)
{
   int i;

   // Serialize pool dispatches
   pthread_mutex_lock (&pool_lock);

   PDBG ("pool has %d free threads of total=%d\n", free_threads, total_threads);
   // If there aren't any free threads, we'll just have to wait
   // until one frees up.
tryagain:
   while (0 == free_threads) {
      PDBG  ("waiting for a free thread\n");
      pthread_cond_wait (&pool_empty, &pool_lock);
   }

   // The locks and condition guarentee that there will be
   // a free thread in the pool by the time we get to here.
   for (i=0; i<pool_entries; i++) 
   {
      if (0 != pool[i]->id) 
      {
         if (0 == pool[i]->busy)
         {
            pthread_mutex_lock (&pool[i]->lock);
            
            // The thread might get busy after we test, 
            // but before we lock, so test again ...
            // The lock can free up right before its destroyed ...
            if ((1 == pool[i]->busy) ||
                (0 == pool[i]->id)) 
            {
               pthread_mutex_unlock (&pool[i]->lock);
               goto tryagain;
            }
            PDBG ("dispatch thread id=%d\n", pool[i]->id);
            free_threads --;
            pool[i]->busy = 1;

            pthread_mutex_unlock (&pool[i]->lock);
            pthread_mutex_unlock (&pool_lock);
            return pool[i];
         }
      }
   }

   // Should never get to here; should have found a thread.
   PERR ("free thread count is wrong. "
         "entries=%d total_threads=%d free_threads=%d \n", 
         pool_entries, total_threads, free_threads);
   for (i=0; i<pool_entries; i++) 
   {
      PERR ("slot=%d id=%d busy=%d\n", i, pool[i]->id, pool[i]->busy); 
   }
   pthread_mutex_unlock (&pool_lock);

   return 0x0;
}

// =====================================================================

void
wlThreadPool::JoinAll (void)
{
   // Keep tabs on the use count, spin till they're all free
   pthread_mutex_lock (&pool_lock);
   while (total_threads != free_threads) {
      pthread_cond_wait (&pool_full, &pool_lock);
   }
   pthread_mutex_unlock (&pool_lock);
}

// =====================================================================

pthread_key_t wlThreadPool :: tid_key;
pthread_key_t wlThreadPool :: self_key;
int wlThreadPool :: tid_key_is_init = 0;

int get_tid (void) 
{
   if (wlThreadPool::tid_key_is_init) {
      return (int) pthread_getspecific (wlThreadPool::tid_key);
   }
   return 0;
}

// ===================== END OF FILE ====================================
