//
// class wlRequest
//
// FUNCTION:
// Implement utilities that assemble and manipulate HTTP requests and 
//    responses.  The structure of these classes should loosely resemble 
//    the IETF HTTP/1.1 spec; i.e. these classes are meant to mirror
//    the true HTTP protocol & how it works.
//
// The HeaderLength() method searches for the canonical end-of-header 
//    tokens \n\n or \n\r\n in the message buffer.  It returns the 
//    number of characters until that point, including the \n\n or \n\r\n.
//    If neither of these tokens are found, it returns 0, even if the length 
//    of the message is non-zero.
//
// The Split() method splits appart the message into its start-line, 
//    header and body components. The "message" is used as input, and 
//    the old value of the members "startline", "header" and "body" are
//    over-written with the new values.
//
// The Join() method joins together the members "startline", "header" and
//    "body" into a single string.  The old contents of the member 
//    "message" are replaced with this new join.
//
// A Split() followed by a Join() or a Join() followed by a Split() is 
//    idempotent only for canonical HTTP messages.
//
// The bug_compat flag, if set, enables bug-compatible (non-spec-compliant)
//    behaviour.  Currently, only one bug is supported:
//    -- 'Content-Length-off-by-two': in this bug, Netscape Navigator
//       appends a spurious \r\n to a POST body, but fails to count these
//       as part of the Content-Length sent in the POST header. As a 
//       result, some servers (notably, the Sun Java URL-decoding servlets)
//       are unable to correctly parse the POST body, resulting in a
//       server core dump and/or a Java NullPointerException. Sigh.
//       Of course, both the browser and the server are at fault ...
//       Set bug_compat to 1 to enable this bug.


#ifndef __WL_REQUEST_H__
#define __WL_REQUEST_H__

#include "header.h"
#include "super.h"

class wlMessage
{
   public:
      wlMessage (void);
      void Split (void);
      void Join (void);
      size_t HeaderLength (void);
   public:
      wlString message;

      wlString startline;
      wlHeader header;
      wlString body;
};


class wlRequest :
   public wlMessage
{
   public:
      wlRequest (void);
      void Assemble (void);
      void Analyze (void);
   public:
      wlString method;      // e.g. GET POST
      wlString url;         // e.g. /some/url.html or http://fqdn.com/yamon.html
      wlString version;     // e.g. HTTP/1.0
      int bug_compat;		// support netscape, domino go bugs
};

class wlResponse :
   public wlMessage
{
   public:
      wlResponse (void);
      void AnalyzeHeader (void);
   public:
      wlString version;     // e.g. HTTP/1.0
      wlString status;      // e.g. 200 OK
      int status_code;      // e.g. 200
};

// A *very small* subset of the HTTP/1.0 return codes.
enum {
   HTTP_OK         = 200, // OK
   HTTP_MOVED_PERM = 301, //  
   HTTP_MOVED_TEMP = 302, // Moved Temporarily (redirect)
   HTTP_USE_LOCAL  = 304, // use local copy, often for gif's 
   HTTP_AUTH_REQ   = 401  // Authorization Required 

};

#endif /* __WL_REQUEST_H__ */

