/*
 * FILE: webopts.c
 *
 * FUNCTION:
 * Parse webbot/webclient command-line arguments.
 *
 * DESIGN NOTES:
 * Should be updated with a unified command-line/config-file utility.
 *
 */

#include <errno.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "cookie.h"
#include "generic.h"
#include "shhopt.h"
#include "socket.h"
#include "sysdep.h"
#include "webopts.h"

#define DEFAULT_LISTEN_PORT 5080

wlOpts :: wlOpts (void)
{
    progname = NULL;

    version = VERSION;
#ifdef USE_SOCKS
    version += " (SOCKS)";
#endif /* USE_SOCKS */

    argc = 0;
    argv = NULL;
    web_portnum = DEFAULTWWWPORT;
    proxy_portnum = DEFAULTWWWPORT;
    local_fqdn_host_name = NULL;
    local_host_name = NULL;
    local_portnum = DEFAULT_LISTEN_PORT;

    protocol_minor_version = 1;
    http_version = "HTTP/1.1";
    keep_alive = 1;
    alarm_time = -1;
    fetch_gifs = 0;
    do_cache_emulation = 0;
    nthreads = 4;
    print_each = 0;
    ignore_checksum_errors = 0;
    warn_checksum_errors = 0;
    skip_stats = 0;
    debug = 0;
    tdebug = 0;
    quiet_stdout = 0;
    write_completion_times = 0;
    write_url_progress = 0;
    run_in_background = 0;
    prt_version = 0;
    wait_interval = 0;
    do_host_translation = 1;
    act_as_proxy = 0;
    trace_server = 0;
    trace_client = 0;

    seed = 7309165;  /* random number generator seed .............. */
    num_sessions = -1;
    bug_compat = 1;     // enable bug compatability mode by default

    think_distribution = THINK_EXPONENTIAL;
    mean_think_time = -1.0e20;

    access_log_file_name = NULL;
    clean_exit_file_name = NULL;
    error_log_file_name = NULL;
    header_file_name = NULL;
    input_file_name = NULL;
    report_file_name = NULL;
    trace_file_name = NULL;
    url_out_file_name = NULL;

    access_log_file = NULL;
    error_log_file = NULL;
    report_file = NULL;
    trace_file = NULL;
    url_out_file = NULL;
    error_log = -1;


    /* Database of key-value pairs that will be used in making 
     * replacements (substitiutions) in the input URL's
     */
    substitution_key= NULL;
    substitution_value= NULL;

    header_subst_field= NULL;
    header_subst_value= NULL;
    header_add_field= NULL;
    header_add_value= NULL;

    handle_field= NULL;
    handle_value= NULL;

    child_number = -1;
    shm_key = 0;
    shm_base = NULL; 

}


/* ================================================================ */
/* ================================================================ */
/* ================================================================ */
/* ================================================================ */

static void 
add_substitution_kv (char ***key_anchor, char ***value_anchor,
                     char *key, char *value)
{
    if ((!key_anchor) || (!value_anchor))
    {
        /* should probably be an assert */
        perr ("Internal Fatal Error: add_substitution(): "
             "failed to specify anchor chain \n");
        return;
    }
    if (!key)
    {
        perr ("Error: add_substitution(): "
             "failed to specify a substitution key \n");
        return;
    }

    /* strip out any quotation marks */
    if (('\"' == key[0]) || ('\'' == key[0]))
    {
        char q = *key, *closeq;
        key ++;
        key = strdup (key);
        closeq = strchr (key, q);
        /* find matching quote mark and kill it */
        if (closeq) *closeq = 0x0;
    } 
    else
    {
        key = strdup (key);
    }
    
    if (!value) value = "XXX";
    /* strip out any quotation marks */
    if (('\"' == value[0]) || ('\'' == value[0]))
    {
        char q = *value, *closeq;
        value ++;
        value = strdup (value);
        closeq = strchr (value, q);
        /* find matching quote mark and kill it */
        if (closeq) *closeq = 0x0;
    } 
    else
    {
        value = strdup (value);
    }
    

    /* simply append "key" to the stored array; malloc the array 
     * if needed.  */
    if (!(*key_anchor)) {
        (*key_anchor) =  (char **) malloc (2 * sizeof (char *));
        (*key_anchor)[0] = key;
        (*key_anchor)[1] = NULL;
        (*value_anchor) =  (char **) malloc (2 * sizeof (char *));
        (*value_anchor)[0] = value;
        (*value_anchor)[1] = NULL;
    } else {
        char **old;

        /* count the number of elements in the array */
        int nkey = 0;
        char *ptr = (*key_anchor)[0];
        while (ptr) { 
            nkey ++;
            ptr = (*key_anchor)[nkey];
        }

        /* malloc a new, longer array */
        old = (*key_anchor);
        (*key_anchor) =  (char **) malloc ((nkey+2) * sizeof (char *));

        /* copy old array to new longer array */
        nkey = 0;
        ptr = old[0];
        while (ptr) { 
            (*key_anchor)[nkey] = ptr;
            nkey ++;
            ptr = old[nkey];
        }

        /* append to new longer array */
        (*key_anchor)[nkey] = key;
        (*key_anchor)[nkey+1] = NULL;
        free (old);

        /* malloc a new, longer array */
        old = (*value_anchor);
        (*value_anchor) =  (char **) malloc ((nkey+2) * sizeof (char *));

        /* copy old array to new longer array */
        nkey = 0;
        ptr = old[0];
        while (ptr) { 
            (*value_anchor)[nkey] = ptr;
            nkey ++;
            ptr = old[nkey];
        }

        /* append to new longer array */
        (*value_anchor)[nkey] = value;
        (*value_anchor)[nkey+1] = NULL;
        free (old);

    }
}

void 
add_substitution (char ***keyanc, char ***valanc, char *keyval)
{
    char *key=0x0, *value=0x0;

    if (!keyval)
    {
        perr ("Error: add_substition(): "
            " failed to specify a substituion key-value pair\n");
        return;
    }


    /* strip out any quotation marks */
    if (('\"' == keyval[0]) || ('\'' == keyval[0]))
    {
        char q = *keyval, *closeq;
        keyval ++;
        keyval = strdup (keyval);
        closeq = strchr (keyval, q);
        /* find matching quote mark and kill it */
        if (closeq) *closeq = 0x0;
    } 
    else
    {
        keyval = strdup (keyval);
    }
    

    key = keyval;
    value = strchr (key, ':');
    if (!value) value = strchr (key, '=');
    if (value) {
        *value = 0x0;
        value ++;
    } 
    add_substitution_kv (keyanc, valanc, key, value);
    free (keyval);
}
    
/* Yuck hack alert */
/* we really should fix option parsing so that it doesn't 
 * require globals to operate correctly ... 
 */
static char ***ska = NULL;  /* this */
static char ***sva = NULL;  /* this */
static char ***hsf = NULL;  /* this */
static char ***hsv = NULL;  /* this */
static char ***haf = NULL;  /* this */
static char ***hav = NULL;  /* this */
static char ***trf = NULL;  /* this */
static char ***trv = NULL;  /* this */
static void add_hsub (char *keyval) { add_substitution (hsf, hsv, keyval); }
static void add_hadd (char *keyval) { add_substitution (haf, hav, keyval); }
#ifndef WEBMON
static void add_sub (char *keyval) { add_substitution (ska, sva, keyval); }
#endif /* WEBMON */
static void add_handle (char *keyval) { add_substitution (trf, trv, keyval); }

static wlCache *cp = NULL;
static void add_cookie_path (char *path) {cp->AddToCache (path); }

/* ================================================================ */
/* ================================================================ */
/* ================================================================ */
/* ================================================================ */

static char *uprogname = NULL;

static void
usage(void)
{
    printf("Usage: %s [options] \n\n", uprogname);

/* ----------------------------------------------------------- */

#ifdef WEBMON
    printf(
"-a, --act-as-proxy        behave as a proxy server\n" 
"-A, --alarm=<time>        turns on timeout alarms (delay 'time')\n" 
"-d, --debug               print basic debugging messages \n"
"-D, --Debug               turns on verbose debugging\n"
"-E, --Debug-debug         turns on extra verbose debugging\n"
"-e, --print-each          print individual response time observations\n"
"-h, --help                print this message\n"
"-n, --no-xlation          disable host/url translation/rewriting \n"
"-p, --listen-port=<port>  specifies the port to listen to\n"
"-P, --proxy=<proxy:port>  specifies the proxyserver and port\n"
"-q, --debug-time          turns on debugging output for timing statistics\n"
"-Q, --Debug-time          turns on verbose timing debug output\n"
"-r, --report-file=<file>  specifies the report file name\n"
"-t, --trace-file=<file>   write HTTP traces to file\n"

"-U, --user-agent=<string> specify value of 'User-Agent:' in HTTP header\n"
#ifdef WIN32
"                          Blanks should be encoded with the hash-mark #\n"
#else 
"                          string must be enclosed in single quotes\n"
#endif /* WIN32 */

"-v, --request-file=<file> record browser requests in this file\n"
"-w, --webserver=<server:port> specifies the webserver and port\n"
"\n"
"Flags without short-form equivalents:\n"
"--access-log=<file>       write webserver-style access log\n"
"--no-bug-compat           enable strict conformance to HTTP standard\n"
"--quiet                   minimize messages written to stdout\n"
"--trace-client            write exchanges with client to trace file\n"
"--trace-server            write exchanges with server to trace file\n"
"--version                 print version info and exit\n"
"\n"
    );
#endif /* WEBMON */

/* ----------------------------------------------------------- */

#ifdef WEBCLIENT  
    printf(
"-A, --alarm=<time>       turns on timeout alarms (delay 'time')\n" 
"-c, --cache-emulation    turns on the gif cache emulation (requires -g flag)\n"
"-d, --debug              print basic debugging messages \n"
"-D, --Debug              turns on verbose debugging\n"
"-E, --Debug-debug        turns on extra verbose debugging\n"
"-e, --print-each         print individual response time observations\n"
"-f, --input-file=<file>  specifies the url list file to be run\n"
"-g, --fetch-gifs         turns on fetching of gif files\n"
"-h, --help               print this message\n"
"-i, --ignore-checksums   don't validate web pages with checksums\n"
"-L, --log-file=<file>    specify name of error log file\n"

#ifndef WIN32
"-m, --shmem=<child:shmkey> defines common shared memory segment\n"
#endif /* WIN32 */

"-P, --proxy=<proxy:port> specifies the proxyserver and port\n"
"-q, --debug-time         turns on debugging output for timing statistics\n"
"-Q, --Debug-time         turns on verbose timing debug output\n"

"-R, --random-seed=<seed> specify seed used to generate random think times\n"
"-r, --report-file=<file> specifies the report file name\n"
"-t, --trace-file=<file> write HTTP traces to file\n"
"-u, --user-pin-pw=<username:userpin:userpasswd> same as specifying \n"
    "\t--substitute=<<USER>>:username \n"
    "\t--substitute=<<PIN>>:userpin \n"
    "\t--substitute=<<PASS>>:userpasswd \n\n"

"-U, --user-agent=<string> specify value of 'User-Agent:' in HTTP header\n"
#ifdef WIN32
    "\tBlanks should be encoded with the hash-mark #\n"
#else /* WIN32 */
    "\tstring must be enclosed in single quotes if it contains [{( etc...\n"
#endif /* WIN32 */

"-v, --new-url-file=<file>     recalculate checksums, write new input file\n"
"-W, --wait-interval=<seconds> pause after each trial, before starting next\n"
"-w, --webserver=<server:port> specifies the webserver and port\n"
"-x, --timestamps              write request start and end timestamps\n"
"\n"
"Flags without short-form equivalents:\n"
"--access-log=<file>           write webserver-style access log\n"
"--clean-exit=<file>           specifies urls to run on error or interrupt\n"
"--cookie-path=<URL>           make sure that cookie was set for path\n"
#ifndef WIN32
"--fork                        run in the background after validating args\n"
#endif /* WIN32 */
"--handle=<field>              replace handle values in header from returned page data\n"
"--header-add=<field:val>      substitute or add field-value to HTTP header\n"
"--header-file=<file>          use the HTTP header found in this file\n"
"--header-subst=<field:val>    substitute field-value to HTTP header\n"
"--http-version=<float>        use HTTP/1.0 or 1.1 protocol\n"
"--no-bug-compat               enable strict conformance to HTTP standard\n"
"--no-keep-alive               do not use keep-alive sockets to feth gifs\n"
"--num-threads=<int>           number of threads to use for gif fetching\n"
"--num-sessions=<int>          override number of times to replay session\n"
"--quiet                       minimize messages written to stdout\n"
"--skip-stats                  don't collect or print performance statistics\n"
"--show-progress               write out each URL as its fetched\n"
"--substitute=<key:value>      replace \"key\" with \"value\" in POST or URL\n"
"--think-time=<float>          override the think time in the input file\n"
"--think-fixed                 think time specifies a fixed time interval\n"
"--think-exponential           think time is random exponential distribution\n"
"--think-gaussian              think time is random gaussian distribution\n"
"--version                     print version info and exit\n"
"--warn-checksums              print warnings when web page checksums are bad\n"
);
#endif /* WEBCLIENT */

#if (defined (USE_SKIT) || defined (USE_SSLEAY))
    print_ssl_usage ();
#endif /* USE_SKIT ||  USE_SSLEAY */

    printf ("\n");
    exit (1);
} /* END usage() */



/* ********************************************************************** */
/* print options summary on file outfile. ............................... */
/* ********************************************************************** */
void 
wlOpts :: PrintOptionsSummary (void)
{
    int i;

    prt ("\n");

    prt("%s: %s (built on %s) \n", progname, (char *)version, __DATE__);

    time_t today = time (0);
    prt ("The current time is %s", ctime (&today));

    wlHost resolv;
    resolv.ResolveAddr (local_fqdn_host_name, "tcp");
    prt ("The local host is %s (%s) \n", local_fqdn_host_name, resolv.ipnum);
    prt ("The pid is %d\n", getpid());

    prt ("\n============================================================\n");
    /* print the command line */
    prt ("Command line:");
    for (i = 0; i < argc; i++) prt ( "%s ", argv[i] ); 
    prt ( "\n\n" );
    prt ("Summary of command line settings:\n");

    if (webserver.Memlen()) {
        resolv.ResolveAddr ((char *) webserver, "tcp");
        prt ("The webserver will be %s:%hu (%s:%hu)\n", 
            (char *)webserver, web_portnum,
            resolv.ipnum, web_portnum);
    }
#ifdef WEBMON
    if (0 < act_as_proxy) {
        prt ("Will act as a proxy server\n");
        if (0 < do_host_translation) {
            prt ("\tNote: you may want to disable host translation when"
                "acting as a proxy\n");
        }
    }
#endif /* WEBMON */
    if (proxyserver.Memlen()) {
        resolv.ResolveAddr ((char *) proxyserver, "tcp");
        prt ("The proxyserver will be %s:%hu (%s:%hu)\n", 
            (char *)proxyserver, proxy_portnum,
            resolv.ipnum, proxy_portnum);
    } else {
        prt ("No proxyserver specified.\n");
    }
    if (user_agent_name.Memlen()) {
        prt ("User-Agent name for this run is: %s\n", (char *)user_agent_name);
    }
#ifdef WEBMON
    else {
        prt ("User-Agent name will not be altered\n");
    }
#endif /* WEBMON */

    if (0 < alarm_time) {
        prt ("Timeout alarms are enabled "
                        "(delay=%d seconds)\n", alarm_time);
    } else {
        prt ("No timeout alarm set. \n");
    }

    if (clean_exit_file_name) {
        prt ("On error, cleanup URLS will be run from file: %s\n", 
             clean_exit_file_name);
    } 
    if (trace_client) {
        prt ("Trace all exchenges with the client\n");
    }
    if (trace_server) {
        prt ("Trace all exchanges with the server\n");
    }
    if (trace_file_name) {
        prt ("Trace file is: %s\n", trace_file_name);
    } else {
        prt ("Trace is off.\n");
    }
    if (report_file_name) {
        prt ("Report file is: %s\n", report_file_name);
    } else {
        prt ("No report file will be written.\n");
    }
    if (access_log_file_name) {
        prt ("Access log file is: %s\n", access_log_file_name);
    } 
    if (error_log_file) {
        prt ("Error log file is: %s\n", error_log_file_name);
    }
    if (url_out_file_name) {
        prt ("Writing an url list to %s\n",
            url_out_file_name);
    }
    if (print_each) {
        prt ("Each response time observation will be printed in the report file.\n");
    }
    if (ignore_checksum_errors) {
        prt ("Checksum errors will be ignored.\n");
    } 
    if (warn_checksum_errors) {
        prt ("Checksum errors will generate warnings.\n");
    }
    if ((!ignore_checksum_errors) && (!warn_checksum_errors)) {
        prt ("Checksum errors will be fatal.\n");
    }
    if (skip_stats) {
        prt ("Will not collect or print timing statistics.\n");
    }
    if (substitution_key) {
        int nkey = 0;
        char *key, *value;

        prt ("Will make the following key-value "
             "substitutions in URL's & POST data:\n");
        key = substitution_key[nkey];
        while (key) {
            value = substitution_value[nkey];
            prt ("\tsubstitution key=%s \tvalue=%s\n", key, value);
            nkey ++;
            key = substitution_key[nkey];
        }
    }
#ifdef WEBCLIENT
    if (header.Memlen()) {
        prt ("Using the following HTTP header:\n%s\n", (char *) header);
    }
    if (header_file_name) {
        prt ("Header was obtained from file %s\n", header_file_name);
    }

    if (header_subst_field) {
        int nkey = 0;
        char *key, *value;

        prt ("Will make the following field "
             "substitutions in HTTP header:\n");
        key = header_subst_field[nkey];
        while (key) {
            value = header_subst_value[nkey];
            prt ("\tfield=%s \tvalue=%s\n", key, value);
            nkey ++;
            key = header_subst_field[nkey];
        }
    }
    if (header_add_field) {
        int nkey = 0;
        char *key, *value;

        prt ("Will make the following field "
             "substitutions or additions in the HTTP header:\n");
        key = header_add_field[nkey];
        while (key) {
            value = header_add_value[nkey];
            prt ("\tfield=%s \tvalue=%s\n", key, value);
            nkey ++;
            key = header_add_field[nkey];
        }
    }
    if (handle_field) {
        int nkey = 0;
        char *key;

        prt ("Will track the following handles "
             "in the HTTP header:\n");
        key = handle_field[nkey];
        while (key) {
            prt ("\tfield=%s\n", key);
            nkey ++;
            key = handle_field[nkey];
        }
    }
    if (do_cache_emulation && !(fetch_gifs)) 
    {
        prt ("*********************************\n");
        prt ("   -c flag requires -g flag.\n");
        prt ("   setting -g flag NOW!!!\n");
        prt ("*********************************\n");
        fetch_gifs = 1;
    }
    if (fetch_gifs) {
        prt ("gifs will be downloaded and timed.\n");
    } else {
        prt ("gifs will NOT be downloaded and timed.\n");
    }
    if (do_cache_emulation) {
        prt ("The browser's gif cache will be emulated.\n");
    }
    prt ("Will use %d threads to fetch gifs in parallel.\n", nthreads);

    prt ("Will use HTTP protocol version %s\n", (char *) http_version);

    if (keep_alive) {
       prt ("Will use keep-alive sockets to fetch gifs more quickly.\n");
    } else {
       prt ("Will not use keep-alive.\n");
    }

    if (input_file_name) {
        prt ("Input file is: %s\n", input_file_name);
    } else {
        prt ("No input file specified.  URL's to be read from stdin.\n");
        input_file_name = strdup ("/dev/tty");
    }

    prt ("Will check that these URL's return cookies:\n");
    i = 0;
    char * ptr = path_requires_cookie.GetEntry(0);
    while (ptr) {
        prt ("\t%s\n", ptr);
        i++;
        ptr = path_requires_cookie.GetEntry(i);
    }
#ifndef WIN32
    if (shm_key > 0) {
        prt ("This is child process %d; shared memory id=%d\n",
            child_number, shm_key);
    }
#endif /* WIN32 */ 
    prt ("random number seed: %d\n", seed);

    if (1.0e6 > (FABS(mean_think_time))) {
        switch (think_distribution) {
            case THINK_FIXED:
                prt ("Will use fixed think time of %f seconds\n",
                     FABS(mean_think_time));
                break;
            case THINK_EXPONENTIAL:
                prt ("Will use exponentially distributed random "
                     "think time with mean of %f seconds\n", mean_think_time);
                break;
            case THINK_GAUSSIAN:
                prt ("Will use gaussian distributed random "
                     "think time with mean of %f seconds\n", mean_think_time);
                break;
        }
    } 

    if (0 < wait_interval) {
        prt ("Will wait %d seconds between sessions\n", wait_interval);
    }
#endif /* WEBCLIENT */

#ifdef WEBMON
    if (0 < do_host_translation) {
        prt ("Host translation enabled\n");
    } else {
        prt ("Host translation disabled\n");
    }
#endif /* WEBMON */

    
#if (defined (USE_SKIT) || defined (USE_SSLEAY))
    if (!quiet_stdout) print_ssl_options_summary (stdout, client_ssl_opts, server_ssl_opts);
    if (report_file) {
        print_ssl_options_summary (report_file, client_ssl_opts, server_ssl_opts);
    }
#endif /* USE_SKIT  USE_SSLEAY*/

    if (bug_compat) prt ("Netscape bug compatibility mode enabled.\n");

    if (0 < debug) prt ("Running in debug mode\n");
    if (1 < debug) prt ("Running in detailed debug mode\n" );
    if (2 < debug) prt ("Running in extra detailed debug mode\n");
    if (0 < tdebug) prt ("Timing debug code is enabled.\n");
    if (1 < tdebug) prt ("Verbose timing debug code is enabled.\n");
    prt ("============================================================\n\n");

    /* if only summary was asked for, exit */
    if (prt_version) exit (0);
}



/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */

// standard option
#define SOPT(a,b,c,d) {             \
    opt[iopt].shortName = (a);      \
    opt[iopt].longName = (b);       \
    opt[iopt].type = (c);           \
    opt[iopt].arg = (void *) (d);   \
    opt[iopt].flags = 0;            \
    opt[iopt].flagDefault = 0x3fff3f1f; \
    iopt ++;                        \
};

// callback option
#define CALL(a,b,c,d) {             \
    opt[iopt].shortName = (a);      \
    opt[iopt].longName = (b);       \
    opt[iopt].type = (c);           \
    opt[iopt].arg = (void *) (d);   \
    opt[iopt].flags = OPT_CALLFUNC; \
    opt[iopt].flagDefault = 0x3fff3f1f; \
    iopt ++;                        \
};

// flag option
#define FLAG(a,b,c,d) {             \
    opt[iopt].shortName = (a);      \
    opt[iopt].longName = (b);       \
    opt[iopt].type = OPT_FLAG;      \
    opt[iopt].arg = (void *) (c);   \
    opt[iopt].flags = 0;            \
    opt[iopt].flagDefault = (d);    \
    iopt ++;                        \
};


void bad_option (const char *fmt, ...)
{
    va_list ap;

    perr("Fatal Error: bad option \n");
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fflush(stdout);
    fflush(stderr);
    exit(99);
}

void
wlOpts :: ParseArgs (int *argc_in, char *argv_in[])
{
    optStruct opt[50];
    int i, iopt=0;

    double http_proto = 1.1;
    int dbug=0, Dbug=0, Ebug=0;
    int qbug=0, Qbug=0;
    int no_bug_compat = 0;
    int no_keep_alive = 0;
    char *upinpw = 0x0;
    char *uagent = 0x0;
    char *wserv  = 0x0;
    char *pserv  = 0x0;
#ifdef WEBMON
    int disable_host_translation=0;
    unsigned int pno = DEFAULT_LISTEN_PORT;
#else
    char *shmarg =0x0;
#endif /* WEBMON */

    cp = &path_requires_cookie;
    ska = &(substitution_key);
    sva = &(substitution_value);
    hsf = &(header_subst_field);
    hsv = &(header_subst_value);
    haf = &(header_add_field);
    hav = &(header_add_value);
    trf = &(handle_field);
    trv = &(handle_value);

    /* make a copy of the command line,
     * since parsing it will destroy it.
     */
    argc = *argc_in;
    argv = (char **) malloc ((argc+1)* sizeof (char *));
    for (i = 0; i < argc; i++) {
        argv[i] = strdup (argv_in[i]);
    }
    argv[argc] = NULL;

    progname = strdup (argv[0]);
    uprogname = progname;

    /* parse the SSL command line options first ... */
#if (defined (USE_SKIT) || defined (USE_SSLEAY))
    parse_ssl_args (client_ssl_opts, server_ssl_opts, argc_in, argv_in);
#endif /* USE_SKIT USE_SSLEAY */


    optSetFatalFunc (bad_option);

    /* short   longname         type        var/func           flagval  */
    SOPT('A', "alarm",        OPT_INT,    &(alarm_time)             );

    SOPT('d', "debug",        OPT_FLAG,   &(dbug)                   );
    SOPT('D', "Debug",        OPT_FLAG,   &(Dbug)                   );
    SOPT('E', "Debug-debug",  OPT_FLAG,   &(Ebug)                   );
    SOPT('e', "print-each",   OPT_FLAG,   &(print_each)             );
    CALL('h', "help",         OPT_FLAG,   usage                     );

    SOPT('P', "proxy",        OPT_STRING, &(pserv)                  );
    SOPT('q', "debug-time",   OPT_FLAG,   &(qbug)                   );
    SOPT('Q', "Debug-time",   OPT_FLAG,   &(Qbug)                   );
    SOPT('r', "report-file",  OPT_STRING, &(report_file_name)       );

    SOPT('t', "trace-file",   OPT_STRING, &(trace_file_name)        );
    SOPT('U', "user-agent",   OPT_STRING, &(uagent)                 );

    SOPT('u', "user-pin-pw",  OPT_STRING, &(upinpw)                 );
    SOPT('v', "request-file", OPT_STRING, &(url_out_file_name)      );
    SOPT('w', "webserver",    OPT_STRING, &(wserv)                  );

    SOPT('x', "timestamps",   OPT_FLAG,   &(write_completion_times) );
    SOPT( 0 , "access-log",   OPT_STRING, &(access_log_file_name)   );
    SOPT( 0 , "clean-exit",   OPT_STRING, &(clean_exit_file_name)   );
    CALL( 0 , "header-add",   OPT_STRING, add_hadd                  );
    CALL( 0 , "header-subst", OPT_STRING, add_hsub                  );
    SOPT( 0 , "header-file",  OPT_STRING, &(header_file_name)       );
    CALL( 0 , "handle",       OPT_STRING, add_handle                );
    SOPT( 0 , "no-bug-compat",OPT_FLAG,   &(no_bug_compat)          );
    SOPT( 0 , "no-keep-alive",OPT_FLAG,   &(no_keep_alive)          );
    SOPT( 0 , "quiet",        OPT_FLAG,   &(quiet_stdout)           );
    SOPT( 0 , "show-progress",OPT_FLAG,   &(write_url_progress)     );
    SOPT( 0 , "version",      OPT_FLAG,   &(prt_version)            );

#ifdef WEBMON
    SOPT('a', "act-as-proxy", OPT_FLAG,   &(act_as_proxy)           );
    SOPT('n', "no-xlation",   OPT_FLAG,   &(disable_host_translation));
    SOPT('p', "listen-port",  OPT_UINT,   &(pno)                    );
    SOPT( 0 , "trace-client", OPT_FLAG,   &(trace_client)           );
    SOPT( 0 , "trace-server", OPT_FLAG,   &(trace_server)           );
#endif /* WEBMON */

#ifdef WEBCLIENT
    SOPT('c', "cache-emulation", OPT_FLAG,&(do_cache_emulation)     );
    SOPT('f', "input-file",   OPT_STRING, &(input_file_name)        );
    SOPT('g', "fetch-gifs",   OPT_FLAG,   &(fetch_gifs)             );
    SOPT('i', "ignore-checksums", OPT_FLAG, &(ignore_checksum_errors) );
    SOPT('L', "log-file",     OPT_STRING, &(error_log_file_name)    );
#ifndef WIN32
    SOPT('m', "shmem",        OPT_STRING, &(shmarg)                 );
#endif /* WIN32 */
    SOPT('R', "random-seed",  OPT_INT,    &(seed)                   );
    SOPT('W', "wait-interval", OPT_INT,   &(wait_interval)          );

    CALL( 0 , "cookie-path",  OPT_STRING, add_cookie_path           );
#ifndef WIN32
    SOPT( 0 , "fork",         OPT_FLAG,   &(run_in_background)      );
#endif /* WIN32 */
    SOPT( 0 , "http-version", OPT_DOUBLE, &(http_proto)             );
    SOPT( 0 , "num-sessions", OPT_INT,    &(num_sessions)           );
    SOPT( 0 , "num-threads",  OPT_INT,    &(nthreads)               );
    SOPT( 0 , "skip-stats",   OPT_FLAG,   &(skip_stats)             );
    CALL( 0 , "substitute",   OPT_STRING, add_sub                   );
    SOPT( 0 , "think-time",   OPT_DOUBLE, &(mean_think_time)        );
    FLAG( 0 , "think-fixed",       &(think_distribution),  THINK_FIXED      );
    FLAG( 0 , "think-exponential", &(think_distribution),  THINK_EXPONENTIAL);
    FLAG( 0 , "think-gaussian",    &(think_distribution),  THINK_GAUSSIAN   );
    SOPT( 0 , "warn-checksums", OPT_FLAG, &(warn_checksum_errors) );
#endif /* WEBCLIENT */

    SOPT(0, 0, OPT_END, 0);  /* no more options */

    /* parse all options */
    optParseOptions(argc_in, argv_in, opt, 0);

    if (dbug) debug = 1;
    if (Dbug) debug = 2;
    if (Ebug) debug = 3;

    if (qbug) tdebug = 1;
    if (Qbug) tdebug = 2;

    if (no_bug_compat) bug_compat = 0;
    if (no_keep_alive) keep_alive = 0;

    if (uagent) user_agent_name = uagent;
    if (pserv) proxyserver = pserv;
    if (wserv) webserver = wserv;

#ifdef WEBCLIENT
    if (1.0 < http_proto) {
        protocol_minor_version = 1;
        http_version = "HTTP/1.1";
    } else {
        protocol_minor_version = 0;
        http_version = "HTTP/1.0";
    }

    if (1 > nthreads) nthreads = 4;

    /* fix up the think flags into a self-consistent state:
     * if the think time is negative, tis fixed,
     * if its fixed, then its negative */
    if (-1.0e-6 > mean_think_time) think_distribution = THINK_FIXED;

    if (THINK_FIXED == think_distribution) {
        mean_think_time = - FABS(mean_think_time);
    }

#ifndef WIN32
    if (shmarg) {
        sscanf (shmarg,"%d:%d", &(child_number), &(shm_key));
    }
#endif /* WIN32 */

    if (upinpw) 
    {
        char * ptr;
        char *username = NULL;
        char *userpin = NULL;
        char *userpw = NULL;


        /* should be arranged in the format "username:pin:password" */
        username = upinpw;
        ptr = strchr (username, ':');
        if (!ptr) {
            perr("Fatal Error: badly formed argument "
                 "to -u option: %s\n", upinpw);
            exit (33);
        } else {
            *ptr = 0x0;
            userpin = ++ptr;
            ptr = strchr (userpin, ':');
            if (!ptr) {
                perr("Fatal Error: badly formed argument "
                     "to -u option: %s\n", upinpw);
                exit (33);
            } else {
                *ptr = 0x0;
                userpw = ++ptr;
            }
        }
        add_substitution_kv (ska, sva, "<<USERNAME>>", username);
        add_substitution_kv (ska, sva, "<<USER>>", username);
        add_substitution_kv (ska, sva, "<<PIN>>", userpin);
        add_substitution_kv (ska, sva, "<<PASSWORD>>", userpw);
        add_substitution_kv (ska, sva, "<<PASSWD>>", userpw);
        add_substitution_kv (ska, sva, "<<PASS>>", userpw);
    }
#endif /* WEBCLIENT */
    cp=NULL;
    ska=NULL;
    sva=NULL;
    hsf=NULL;
    hsv=NULL;
    haf=NULL;
    hav=NULL;
    trf=NULL;
    trv=NULL;

#ifdef WEBMON
    /* cast to short */
    local_portnum = (unsigned short) pno;

    if (disable_host_translation) do_host_translation = 0;
#endif /* WEBMON */

    if (trace_file_name && !(trace_client || trace_server)) trace_server = 1;
    if (!trace_file_name && (trace_client || trace_server)) 
    {
        perr ("Fatal Error: tracing requires that a trace file be specified!\n");
        exit (44);
    }

    if (1 < *argc_in)  /* skip argv[0] which is name of process */
    {
        int i;
        perr ("Fatal Error: Unknown options remaining:\n");
        for (i=1; i<*argc_in; i++) perr ("\t%d %s\n", i, argv_in[i]);
        exit (66);
    }

}

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */

#define OPEN_FILE(name,fh,access,errmsg)			\
    if (name) 							\
    {								\
        fh = fopen (name,access);				\
        if (NULL == fh)						\
        {							\
            int norr = errno;  					\
            perr("****************************************\n"	\
                 "Fatal Error: can't open " #errmsg 		\
                 "file: %s (%d:%s)\n", 				\
                 name, norr, strerror(norr)); 			\
            return 1;						\
        }									


int
wlOpts :: ProcessOpts (void)
{
    int rc;
#ifndef WIN32
    struct sigaction *action, *old_action;
#endif

#ifdef WIN32
#ifdef MICROSOFT_VISUAL_C
    LARGE_INTEGER li_win32performance_counter_frequency;
#endif /* MICROSOFT_VISUAL_C */
    WORD wVersionRequested;
    WSADATA wsaData;
#endif /* WIN32 */

#ifdef WEBCLIENT
    if (!input_file_name) 
    {
        perr("****************************************\n"
             "Fatal Error: must specify input file of URL's\n"
             "\tType \"%s -h\" to get a command line summary\n", progname); 
        return 1;
    }
#endif /* WEBCLIENT */

    OPEN_FILE (report_file_name, report_file, "w", "Report File")
        setlinebuf (report_file);
    }

    OPEN_FILE (trace_file_name, trace_file, "w", "Trace File")
        setvbuf (trace_file, NULL, _IONBF, 0);
    }

    OPEN_FILE (access_log_file_name, access_log_file, "a", "Access Log") 
        setlinebuf (access_log_file);
    }

    OPEN_FILE (url_out_file_name, url_out_file, "w", "URL Output") 
        setlinebuf (url_out_file);

        /* override value of the value of the -i flag.  We need to */
        /* be computing the checksums so that we can write them out. */
        ignore_checksum_errors = 0;
        warn_checksum_errors = 0;
        skip_stats = 1;
    }

    /* ************************************************************* */
    /* open the error log_file in append mode. ..................... */
    /* ************************************************************* */
    if (!error_log_file_name) {
        error_log_file_name = strdup ("error.log");
    }
    OPEN_FILE (error_log_file_name, error_log_file, "a", "Error Log")
        error_log = fileno (error_log_file);
    }


    /* validate the SSL command line options now ... */
#if (defined (USE_SKIT) || defined (USE_SSLEAY))
    validate_ssl_opts (client_ssl_opts);
    validate_ssl_opts (server_ssl_opts);
#endif /* USE_SKIT USE_SSLEAY */

    /* if there is a port number on the web server name, get it........... */
    if (webserver.Memlen()) {
        char * start = (char *) webserver;
        char * end = strchr (start,':');
        if (end) {
            web_portnum = (unsigned short) atoi (end+1);
            wlString tmp;
            tmp.Memcpy (start, end-start);  // truncate at port number
            webserver = tmp;
        } else {
            if (client_ssl_opts.use_ssl) {
                web_portnum = DEFAULTHTTPSPORT;
            } else {
                web_portnum = DEFAULTWWWPORT;
            }
        }
    }

    if (proxyserver.Memlen()) {
        char * start = (char *) proxyserver;
        char * end = strchr (start, ':');
        if (end) {
            proxy_portnum = (unsigned short) atoi (end+1);
            wlString tmp;
            tmp.Memcpy (start, end-start);  // truncate at port number
            proxyserver = tmp;
        }
    }

    if (write_completion_times && (NULL == report_file)) {

        perr("****************************************\n"
             "Fatal Error: -x option requires -r reportfile option.\n"
             "****************************************\n");
        return 1;
    }

    // initialize the user_agent_name array to default values
    // but do this only for webclient, since the default for
    // webmon is to not change the user-agent name
    if (0 == user_agent_name.Memlen())
    {
#ifdef WEBCLIENT
        user_agent_name = progname;
        user_agent_name += "/";
        user_agent_name += version;
#endif /* WEBCLIENT */
    } else {
        char * ptr = (char *)user_agent_name;
        while (*ptr) {
            if ('#' == *ptr) *ptr = ' ';
            ptr++;
        }
    }

    if (0 == path_requires_cookie.NumEntries()) 
    {
        /* hard-wire /proclogin.ns to remain backwards compatible */
        path_requires_cookie.AddToCache ("/proclogin.ns");
    }

    if (header_file_name) 
    {
        FILE * hf = fopen (header_file_name, "r");
        if(!hf)
        {
            int norr = errno;  /* avoid having printfs mangle errno value */
            perr("****************************************\n"
                 "Fatal Error: Can't open header_file: %s %d %s\n", 
                header_file_name, norr, strerror (norr)); 
            return 1;
        }

#define HSIZE 4000
        /* read in the header file */
        header = "";
        while (!feof (hf)) {
            char *p, buff[HSIZE];
            p = fgets (buff, HSIZE, hf);
            if (p) header.Strcat (p);
        }
        fclose (hf);
    }

    /* hack alert-- we want to build a slightly different header
     * when we are acting in proxy mode ... */
    if (0 == header.Memlen()) 
    {
        char port[30];
        
        header.Strcat ("User-Agent: ");
        header.Strcat (user_agent_name);
        header.Strcat ("\r\nHost: ");
        header.Strcat (webserver);
        sprintf (port, ":%hu", web_portnum);
        header.Strcat (port);
        header.Strcat ("\r\n"
                       "Accept: */*\r\n" 
                       "Accept-Language: en\r\n" 
                       "Accept-Charset: iso-8859-1,*,utf-8\r\n");

        if (keep_alive && 0 == protocol_minor_version)
        {
            header.Strcat ("Connection: Keep-Alive\r\n");
        }
    }

    /* *************************************************************** */
    /* establish a signal handler for Sigusr1 ........................ */
    /* *************************************************************** */
#ifndef WIN32
    action = (struct sigaction *) malloc(sizeof(struct sigaction));
    old_action = (struct sigaction *) malloc(sizeof(struct sigaction));
    action->sa_handler = signal_handler;
#ifdef AIX
    action->sa_mask.losigs = 0;
    action->sa_mask.hisigs = 0;
#endif /* AIX */
    action->sa_flags= 0;


#define SIGACTION(SIGNAME) {					\
    rc = sigaction(SIGNAME,action,old_action);			\
    if (rc != 0) {						\
        perr("Error: %s: sigaction failed \n",progname);	\
        perr("\t%s processing disabled.\n", #SIGNAME);		\
    } else {							\
        prt("Info: %s: signal handler established for %s\n",	\
             progname, #SIGNAME);				\
    }								\
}

    // Note: the signals enabled here should be blocked in pool.C
    // We need to block these siggy's in the child threads so that
    // they get delivered in the main thread (POSIX semantics).
    SIGACTION (SIGHUP);
    SIGACTION (SIGINT);
    SIGACTION (SIGQUIT);
    SIGACTION (SIGPIPE);
    SIGACTION (SIGTERM);
    SIGACTION (SIGUSR1);
    SIGACTION (SIGUSR2);

    if (0 < alarm_time) {
        SIGACTION (SIGALRM);
    }

#else  /* WIN32 */
    prt ("%s: Testing for Pentium Processor..\n",progname);
    if (!is_good_cpu()) {
        prt ("%s: Pentium Processor or higher required....\n",progname);
        return;
    }
    prt ("%s: Processor test succeeded..\n",progname);


#ifdef MICROSOFT_VISUAL_C
    /* get the frequency of the peformance counters (needed for converstion  */
    /* from time_struc (typdef'd to LARGE_INTEGER) to seconds............... */
    rc = QueryPerformanceFrequency(&li_win32performance_counter_frequency);
    if (! rc) {
        perr("Fatal Error: "
            "QueryPerformanceFrequency returned %d at line %d\n",rc,__LINE__);
        perr("%s: unable to continue.\n",progname);
        exit(999);
        }
    /* we assume here that the counter frequency is less than 2**32-1....... */
    win32performance_counter_frequency = li_win32performance_counter_frequency.LowPart;
#endif /* MICROSOFT_VISUAL_C */


    /* ************************************************ */
    /* initialize winsock interface ................... */
    /* ************************************************ */
 
    wVersionRequested = MAKEWORD( 1, 1 );
 
    rc = WSAStartup( wVersionRequested, &wsaData );
    if ( rc != 0 ) {
        /* Tell the user that we couldn't find a usable */
        /* WinSock DLL................................. */
        prt("Info: wlPorcessOpts(): WSAStartup returned rc=%d\n",rc);
        return 1;
    }
    prt("Info: wlProcessOpts(): WSAStartup successful.\n");

    /* Confirm that the WinSock DLL supports 2.0.........*/
    /* Note that if the DLL supports versions greater    */
    /* than 2.0 in addition to 2.0, it will still return */
    /* 2.0 in wVersion since that is the version we      */
    /* requested.                                        */

    if ( LOBYTE( wsaData.wVersion ) != 1 ||
            HIBYTE( wsaData.wVersion ) != 1 ) {
        /* Tell the user that we couldn't find a usable */
        /* WinSock DLL.                                 */
        perr("Fatal Error: wlProcessOpts(): Version mismatch\n");
        perr("\twsaData.wVersion=%d.%d\n",
            LOBYTE(wsaData.wVersion), HIBYTE(wsaData.wVersion));
        WSACleanup( );
        return 1; 
    }
    /* The WinSock DLL is acceptable. Proceed. */

#endif /* WIN32 */

#ifdef WEBCLIENT
    /* set the random number seed. ............................ */
    SRANDOM(seed);
#endif

    
    /* ******************************************************** */
    /* get the local host name ................................ */
    /* ******************************************************** */
#define HOSTNAMELEN 256
    local_fqdn_host_name = (char *) malloc (HOSTNAMELEN);
    rc = gethostname(local_fqdn_host_name, HOSTNAMELEN);
    if (0 > rc) {
        int norr = errno;
        perr ("Warning: wlProcessOpts: unable to obtain the local hostname\n");
        perr ("\terrno=%d (%s)\n", norr, strerror(norr));
        perr ("\tdefaulting to 127.0.0.1 \n");

        /* of course, we could try gethostid() before using localhost,
         * but of course, gethostname only fails on win95, and win95
         * doesn't support gethostid() or uname() or unamex(), so
         * its pointless to get fancy here.
         */
        free (local_fqdn_host_name);
        local_fqdn_host_name = strdup ("127.0.0.1");
        local_host_name = strdup("127.0.0.1");
    } else {
        local_host_name = strdup (local_fqdn_host_name);
        char * start = strchr (local_host_name, '.');
        if (start) *start = 0x0;
    }

    return 0;
}

/* ================================================================= */
// destructor

wlOpts :: ~wlOpts (void)
{
    CloseFiles();
}

/* ================================================================= */

#define CLOSE_FILE(fh) 	\
    if (fh) {		\
        fclose (fh);	\
        fh = NULL;	\
    }

void
wlOpts ::CloseFiles (void)
{
    CLOSE_FILE (access_log_file);
    CLOSE_FILE (error_log_file);
    CLOSE_FILE (report_file);
    CLOSE_FILE (trace_file);
    CLOSE_FILE (url_out_file);
    error_log = -1;
}

/* ========================== END OF FILE ====================== */
