// Thread_Manager.cpp
#define ACE_BUILD_DLL
#include "ace/Thread_Manager.h"

#if !defined (__ACE_INLINE__)
#include "ace/Thread_Manager.i"
#endif /* __ACE_INLINE__ */

ACE_ALLOC_HOOK_DEFINE(ACE_Thread_Control)
ACE_ALLOC_HOOK_DEFINE(ACE_Thread_Manager)

#if defined (ACE_HAS_THREADS)

void
ACE_Thread_Manager::dump (void) const
{
  ACE_TRACE ("ACE_Thread_Manager::dump");
}

ACE_Thread_Descriptor::ACE_Thread_Descriptor (void)
  : thr_id_ (ACE_OS::NULL_thread),
    grp_id_ (0),
    thr_state_ (ACE_THR_IDLE)
{
  ACE_TRACE ("ACE_Thread_Descriptor::ACE_Thread_Descriptor");
}

// Return the thread descriptor (indexed by ACE_thread_t).

int 
ACE_Thread_Manager::thread_descriptor_i (ACE_thread_t thr_id,
					 ACE_Thread_Descriptor &descriptor)
{
  ACE_TRACE ("ACE_Thread_Descriptor::thread_descriptor_i");

  for (size_t i = 0; i < this->current_count_; i++)
    if (ACE_OS::thr_equal (thr_id, this->thr_table_[i].thr_id_))
      {
	descriptor = this->thr_table_[i];
	return 0;
      }

  return -1;
}

int 
ACE_Thread_Manager::thread_descriptor (ACE_thread_t thr_id,
				       ACE_Thread_Descriptor &descriptor)
{
  ACE_TRACE ("ACE_Thread_Descriptor::thread_descriptor");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  return this->thread_descriptor_i (thr_id, descriptor);
}

// Return the thread descriptor (indexed by ACE_hthread_t).

int 
ACE_Thread_Manager::hthread_descriptor_i (ACE_hthread_t thr_handle, 
					  ACE_Thread_Descriptor &descriptor)
{
  ACE_TRACE ("ACE_Thread_Descriptor::hthread_descriptor_i");  

  for (size_t i = 0; i < this->current_count_; i++)
    if (ACE_OS::thr_cmp (thr_handle, this->thr_table_[i].thr_handle_))
      {
	descriptor = this->thr_table_[i];
	return 0;
      }

  return -1;
}

int 
ACE_Thread_Manager::hthread_descriptor (ACE_hthread_t thr_handle, 
					ACE_Thread_Descriptor &descriptor)
{
  ACE_TRACE ("ACE_Thread_Descriptor::hthread_descriptor");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  return this->hthread_descriptor_i (thr_handle, descriptor);
}

// Return the thread descriptor (indexed by ACE_hthread_t).

int 
ACE_Thread_Manager::thr_self (ACE_hthread_t &self)
{
  ACE_TRACE ("ACE_Thread_Descriptor::thr_self");  
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  // Try to get the cached HANDLE out of TSS to avoid lookup.
  ACE_hthread_t *handle = ACE_LOG_MSG->thr_handle ();

  // Wasn't in the cache, so we'll have to look it up and cache it.
  if (handle == 0) 
    { 
      ACE_Thread_Descriptor td;
      ACE_thread_t id = ACE_OS::thr_self ();
      
      int i = this->thread_descriptor_i (id, td);

      if (i == -1) 
	return -1; 
      handle = &this->thr_table_[i].thr_handle_;

      // Update the TSS cache. 
      ACE_LOG_MSG->thr_handle (handle);
    } 
  self = *handle;
  return 0;
}

int
ACE_Thread_Manager::resize (size_t size)
{
  ACE_TRACE ("ACE_Thread_Manager::resize");
  ACE_Thread_Descriptor *temp;
  
  ACE_NEW_RETURN (temp, ACE_Thread_Descriptor[size], -1);

  for (size_t i = 0; i < this->max_table_size_; i++)
    temp[i] = this->thr_table_[i]; // Structure assignment.

  this->max_table_size_ = size;

  delete [] this->thr_table_;

  this->thr_table_ = temp;
  return 0;
}

// Create and initialize the table to keep track of the thread pool.

int
ACE_Thread_Manager::open (size_t size)
{
  ACE_TRACE ("ACE_Thread_Manager::open");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  if (this->max_table_size_ < size)
    this->resize (size);
  return 0;
}

// Initialize the synchronization variables.

ACE_Thread_Manager::ACE_Thread_Manager (size_t size)
  : zero_cond_ (lock_),
    max_table_size_ (0), 
    thr_table_ (0),
    current_count_ (0),
    grp_id_ (1)
{
  ACE_TRACE ("ACE_Thread_Manager::ACE_Thread_Manager");
  if (this->open (size) == -1)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_Thread_Manager"));
}

// Close up and release all resources.

int
ACE_Thread_Manager::close (void)
{
  ACE_TRACE ("ACE_Thread_Manager::close");
  if (this->thr_table_ != 0)
    {
      delete [] this->thr_table_;
      this->thr_table_ = 0;
      this->max_table_size_ = 0;
      this->current_count_ = 0;
    }
  return 0;
}

ACE_Thread_Manager::~ACE_Thread_Manager (void)
{
  ACE_TRACE ("ACE_Thread_Manager::~ACE_Thread_Manager");
  this->close ();
}

// Call the appropriate OS routine to spawn a thread.  Should *not* be
// called with the lock_ held...

int 
ACE_Thread_Manager::spawn_i (ACE_THR_FUNC func, 
			     void *args, 
			     long flags, 
			     ACE_thread_t *t_id, 
			     ACE_hthread_t *t_handle,
			     u_int priority,
			     int grp_id,
			     void *stack, 
			     size_t stack_size)
{
  ACE_TRACE ("ACE_Thread_Manager::spawn_i");
  ACE_thread_t thr_id;
  ACE_hthread_t thr_handle;

  if (t_id == 0)
    t_id = &thr_id;

  if (t_handle == 0)
    t_handle = &thr_handle;
  
  int result = ACE_Thread::spawn (func, args, flags, 
				  t_id, t_handle, priority,
				  stack, stack_size);
  if (result != 0)
    {
      errno = result;
      return -1;
    }
  else
    return this->append_thr (*t_id, *t_handle, 
			     ACE_THR_SPAWNED, 
			     grp_id);
}

// Create a new thread running FUNC.  *Must* be called with the lock_
// held...

int 
ACE_Thread_Manager::spawn (ACE_THR_FUNC func, 
			   void *args, 
			   long flags, 
			   ACE_thread_t *t_id, 
			   ACE_hthread_t *t_handle,
			   u_int priority,
			   int grp_id,
			   void *stack, 
			   size_t stack_size)
{
  ACE_TRACE ("ACE_Thread_Manager::spawn");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  if (grp_id == -1)
    grp_id = this->grp_id_++; // Increment the group id.

  if (this->spawn_i (func, args, flags, t_id, t_handle, 
		     priority, grp_id, stack, stack_size) == -1)
    return -1;
  else
    return grp_id;
}

// Create N new threads running FUNC.

int 
ACE_Thread_Manager::spawn_n (int n, 
			     ACE_THR_FUNC func, 
			     void *args, 
			     long flags,
			     u_int priority,
			     int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::spawn_n");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  if (grp_id == -1)
    grp_id = this->grp_id_++; // Increment the group id.

  for (int i = 0; i < n; i++)
    {
      // @@ What should happen if this fails?! e.g., should we try to
      // cancel the other threads that we've already spawned or what?
      if (this->spawn_i (func, args, flags, 0, 0, priority, grp_id) == -1)
	return -1;
    }

  return grp_id;
}

// Append a thread into the pool (does not check for duplicates).
// Must be called with locks held.

int
ACE_Thread_Manager::append_thr (ACE_thread_t t_id, 
				ACE_hthread_t t_handle,
				ACE_Thread_State thr_state,
				int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::append_thr");
  // Try to resize the array to twice its existing size if we run out
  // of space...
  if (this->current_count_ >= this->max_table_size_ 
      && this->resize (this->max_table_size_ * 2) == -1)
    return -1;
  else
    {
      ACE_Thread_Descriptor &thr_desc = 
	this->thr_table_[this->current_count_];

      thr_desc.thr_id_ = t_id;
      thr_desc.thr_handle_ = t_handle;
      thr_desc.grp_id_ = grp_id;
      thr_desc.thr_state_ = thr_state;

      this->current_count_++;
      return 0;
    }
}

// Insert a thread into the pool (checks for duplicates and doesn't
// allow them to be inserted twice).

int
ACE_Thread_Manager::insert_thr (ACE_thread_t t_id, 
				ACE_hthread_t t_handle,
				int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::insert_thr");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  // Check for duplicates and bail out if they're already
  // registered...
  if (this->find (t_id) != -1)
    return -1;

  if (grp_id == -1)
    grp_id = this->grp_id_++;
    
  if (this->append_thr (t_id, t_handle, 
			ACE_THR_SPAWNED,
			grp_id) == -1)
    return -1;
  else
    return grp_id;
}

// Remove a thread from the pool.  Must be called with locks held.

void
ACE_Thread_Manager::remove_thr (int i)
{
  ACE_TRACE ("ACE_Thread_Manager::remove_thr");
  if (this->current_count_ > 1)
    // Structure assignment.
    this->thr_table_[i] = this->thr_table_[this->current_count_ - 1];

  this->current_count_--;

  // Tell all waiters when there are no more threads left in the pool.
  if (this->current_count_ == 0)
    this->zero_cond_.broadcast ();
}

// Factory out some common behavior to simplify the following methods.
#define ACE_THR_OP(OP,STATE) \
  int result = OP (this->thr_table_[i].thr_handle_); \
  if (result != 0) { \
    this->remove_thr (i); \
    errno = result; \
    return -1; \
  } \
  else { \
    this->thr_table_[i].thr_state_ = STATE; \
    return 0; \
  }

int
ACE_Thread_Manager::suspend_thr (int i)
{
  ACE_TRACE ("ACE_Thread_Manager::suspend_thr");

  ACE_THR_OP (ACE_Thread::suspend, ACE_THR_SUSPENDED);
}

int
ACE_Thread_Manager::resume_thr (int i)
{
  ACE_TRACE ("ACE_Thread_Manager::resume_thr");

  ACE_THR_OP (ACE_Thread::resume, ACE_THR_RUNNING);
}

int
ACE_Thread_Manager::cancel_thr (int i)
{
  ACE_TRACE ("ACE_Thread_Manager::cancel_thr");
  this->thr_table_[i].thr_state_ = ACE_THR_CANCELLED;
  return 0;
}

int
ACE_Thread_Manager::kill_thr (int i, int signum)
{
  ACE_TRACE ("ACE_Thread_Manager::kill_thr");

  int result = ACE_Thread::kill ((ACE_thread_t) this->thr_table_[i].thr_handle_,
				 signum);

  if (result != 0) 
    { 
      this->remove_thr (i); 
      errno = result; 
      return -1; 
    } 
  else 
    return 0; 
}

// Locate the index in the table associated with <t_id>.  Must be
// called with the lock held.

int 
ACE_Thread_Manager::find (ACE_thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::find");

  for (size_t i = 0; i < this->current_count_; i++)
    if (ACE_OS::thr_equal (t_id, this->thr_table_[i].thr_id_))
      return i;

  return -1;    
}

#define ACE_EXECUTE_OP(OP) \
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1); \
  int i = this->find (t_id); \
  if (i == -1) return -1; \
  return OP (i);

// Suspend a single thread.

int 
ACE_Thread_Manager::suspend (ACE_thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::suspend");
  ACE_EXECUTE_OP (this->suspend_thr);
}

// Resume a single thread.

int 
ACE_Thread_Manager::resume (ACE_thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::resume");
  ACE_EXECUTE_OP (this->resume_thr);
}

// Cancel a single thread.

int 
ACE_Thread_Manager::cancel (ACE_thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::cancel");
  ACE_EXECUTE_OP (this->cancel_thr);
}

// Send a signal to a single thread.

int 
ACE_Thread_Manager::kill (ACE_thread_t t_id, int signum)
{
  ACE_TRACE ("ACE_Thread_Manager::kill");
  return ACE_Thread::kill (t_id, signum);
}

int
ACE_Thread_Manager::check_state (ACE_Thread_State state, 
				 ACE_thread_t id) 
{
  ACE_TRACE ("ACE_Thread_Manager::check_state");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  ACE_Thread_State *thr_state = 0;
#if 0
  int self_check = ACE_OS::thr_equal (id, ACE_OS::thr_self ());

  // If we're checking the state of our thread, try to get the cached
  // value out of TSS to avoid lookup.
  if (self_check) 
    thr_state = ACE_LOG_MSG->thr_state ();

  // Wasn't in the cache, so we'll have to look it up.
  if (thr_state == 0) 
    { 
      int i = this->find (id);

      if (i == -1) 
	return -1; 
      thr_state = &this->thr_table_[i].thr_state_;

      if (self_check) // Update the TSS cache. 
	ACE_LOG_MSG->thr_state (thr_state); 
    } 
#else
   // Turn off caching for the time being until we figure out 
   // how to do it correctly in the face of deletions...
   int i = this->find (id);

   if (i == -1) 
     return -1; 
   thr_state = &this->thr_table_[i].thr_state_;
#endif /* 0 */
  return *thr_state == state;
}

// Test if a single thread is suspended.

int 
ACE_Thread_Manager::testsuspend (ACE_thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::testsuspend");
  return this->check_state (ACE_THR_SUSPENDED, t_id);
}

// Test if a single thread is active (i.e., resumed).

int 
ACE_Thread_Manager::testresume (ACE_thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::testresume");
  return this->check_state (ACE_THR_RUNNING, t_id);
}

// Test if a single thread is cancelled.

int 
ACE_Thread_Manager::testcancel (ACE_thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::testcancel");
  return this->check_state (ACE_THR_CANCELLED, t_id);
}

// Get group ids for a particular thread id.

int 
ACE_Thread_Manager::get_grp (ACE_thread_t t_id, int &grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::get_grp");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  int i = this->find (t_id);
  if (i == -1)
    return -1;
  grp_id = this->thr_table_[i].grp_id_;
  return 0;
}

// Set group ids for a particular thread id.

int 
ACE_Thread_Manager::set_grp (ACE_thread_t t_id, int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::set_grp");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  int i = this->find (t_id);
  if (i == -1)
    return -1;
  this->thr_table_[i].grp_id_ = grp_id;
  return 0;
}

// Suspend a group of threads.

int
ACE_Thread_Manager::apply_grp (int grp_id, 
			       THR_FUNC func,
			       int arg)
{
  ACE_TRACE ("ACE_Thread_Manager::apply_grp");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  int result = 0;

  for (size_t i = 0; i < this->current_count_; i++)
    if (this->thr_table_[i].grp_id_ == grp_id
	&& (this->*func)(i, arg) == -1)
      result = -1;

  return result;
}

int 
ACE_Thread_Manager::suspend_grp (int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::suspend_grp");
  return this->apply_grp (grp_id, 
			  THR_FUNC (&ACE_Thread_Manager::suspend_thr));
}

// Resume a group of threads.

int 
ACE_Thread_Manager::resume_grp (int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::resume_grp");
  return this->apply_grp (grp_id, 
			  THR_FUNC (&ACE_Thread_Manager::resume_thr));
}

// Kill a group of threads.

int 
ACE_Thread_Manager::kill_grp (int grp_id, int signum)
{
  ACE_TRACE ("ACE_Thread_Manager::kill_grp");
  return this->apply_grp (grp_id, 
			  THR_FUNC (&ACE_Thread_Manager::kill_thr), signum);
}

// Cancel a group of threads.

int 
ACE_Thread_Manager::cancel_grp (int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::resume_grp");
  return this->apply_grp (grp_id, 
			  THR_FUNC (&ACE_Thread_Manager::cancel_thr));
}

int
ACE_Thread_Manager::apply_all (THR_FUNC func, int arg)
{
  ACE_TRACE ("ACE_Thread_Manager::apply_all");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  int result = 0;

  for (size_t i = 0; i < this->current_count_; i++)
    if ((this->*func)(i, arg) == -1)
      result = -1;

  return result;
}

// Resume all threads that are suspended.

int 
ACE_Thread_Manager::resume_all (void)
{
  ACE_TRACE ("ACE_Thread_Manager::resume_all");
  return this->apply_all (THR_FUNC (&ACE_Thread_Manager::resume_thr));
}

int 
ACE_Thread_Manager::suspend_all (void)
{
  ACE_TRACE ("ACE_Thread_Manager::suspend_all");
  return this->apply_all (THR_FUNC (&ACE_Thread_Manager::suspend_thr));
}

int 
ACE_Thread_Manager::kill_all (int sig)
{
  ACE_TRACE ("ACE_Thread_Manager::kill_all");
  return this->apply_all (&ACE_Thread_Manager::kill_thr, sig);
}

int 
ACE_Thread_Manager::cancel_all (void)
{
  ACE_TRACE ("ACE_Thread_Manager::cancel_all");
  return this->apply_all (THR_FUNC (&ACE_Thread_Manager::cancel_thr));
}

// Must be called when thread goes out of scope to clean up its table
// slot.

void *
ACE_Thread_Manager::exit (void *status, int do_thr_exit)
{
  ACE_TRACE ("ACE_Thread_Manager::exit");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, 0);

  int i = this->find (ACE_Thread::self ());

  // Locate thread id.
  if (i != -1)
    this->remove_thr (i);

  if (do_thr_exit)
    {
      // Note, destructor is never called, so we must manually release
      // the lock...

      ACE_MT (ace_mon.release ()); 

      ACE_Thread::exit (status);
      // On reasonable systems ACE_Thread::exit() should not return.
      // However, due to horrible semantics with Win32 thread-specific
      // storage this call can return (don't ask...).  Therefore, we
      // need to reacquire the mutex so that we don't get burned when
      // the Guard automatically releases it when this method returns.
      ACE_MT (ace_mon.acquire ()); 
    }
  return 0;
}

// Wait for all the threads to exit.

int
ACE_Thread_Manager::wait (const ACE_Time_Value *timeout)
{
  ACE_TRACE ("ACE_Thread_Manager::wait");
  ACE_GUARD_RETURN (ACE_Thread_Mutex, ace_mon, this->lock_, -1);

  while (this->current_count_ > 0)
    if (this->zero_cond_.wait (timeout) == -1)
      return -1;

  return 0;
}

void
ACE_Thread_Control::dump (void) const
{
  ACE_TRACE ("ACE_Thread_Control::dump");
}

int
ACE_Thread_Control::insert (ACE_Thread_Manager *tm)
{
  ACE_TRACE ("ACE_Thread_Control::insert");
 
  ACE_hthread_t t_id;
  ACE_Thread::self (t_id);
  this->tm_ = tm;
 
  return this->tm_->insert_thr (ACE_Thread::self (), t_id);
}

// Initialize the thread controller.

ACE_Thread_Control::ACE_Thread_Control (ACE_Thread_Manager *t, 
					int insert)
  : tm_ (t),
    status_ (0)
{
  ACE_TRACE ("ACE_Thread_Control::ACE_Thread_Control");
  if (this->tm_ != 0 && insert)
    {
      ACE_hthread_t t_id;
      ACE_Thread::self (t_id);
      this->tm_->insert_thr (ACE_Thread::self (), t_id);
    }
}

// Automatically kill thread on exit.

ACE_Thread_Control::~ACE_Thread_Control (void)
{
  ACE_TRACE ("ACE_Thread_Control::~ACE_Thread_Control");
  this->exit (this->status_);
}

// Exit from thread (but clean up first).

void *
ACE_Thread_Control::exit (void *exit_status)
{
  ACE_TRACE ("ACE_Thread_Control::exit");
  if (this->tm_ != 0)
    return this->tm_->exit (exit_status);
  else
    {
      ACE_Thread::exit (exit_status);
      return 0;
    }
}

#endif /* !defined (ACE_HAS_THREADS) */
