// This program illustrates how the ACE Proactor can be used to
// implement an application that reads from a file and sends/recvs
// message to a peer via a TCP socket.

#include "ace/ReactorEx.h"
#include "ace/Proactor.h"
#include "ace/SOCK_Connector.h"
#include "ace/SOCK_Acceptor.h"
#include "ace/Get_Opt.h"
#include "ace/Time_Value.h"
#include "ace/Service_Config.h"

class Peer_Handler : public ACE_Event_Handler
// = TITLE
//     Connect to a server.  Receive messages from FILE_Handler 
//     and forward them to the server using proactive I/O.
{
public:
  Peer_Handler (void);
  
  int open (const char *host, u_short port);
  // Initialize the <Peer_Handler> with the <host> and <port> of its
  // peer.

  virtual int handle_output_complete (ACE_Message_Block *msg,
				      long bytes_transfered);
  // One of our asynchronous writes to the remote peer has completed.
  // Make sure it succeeded and then delete the message.

  virtual int handle_input_complete (ACE_Message_Block *msg,
				     long bytes_transfered);
  // The remote peer has sent us something.  If it succeeded, print
  // out the message and reinitiate a read.  Otherwise, fail.  In both
  // cases, delete the message sent.

  virtual ACE_Message_Block *get_message (void);
  // This is so the Proactor can get a message to read into.

  virtual ACE_HANDLE get_handle (void) const;
  // This is so the Proactor can get our handle.

  virtual int handle_close (ACE_HANDLE, ACE_Reactor_Mask);
  // We've been removed from the ReactorEx.

private:
  ACE_SOCK_Stream stream_;
  // Socket that we have connected to the server.
};

class FILE_Handler : public ACE_Event_Handler
// = TITLE
//   Using proactive I/O, receive message from FILE and forward 
//   them to the Peer_Handler.
{
public:
  FILE_Handler (void);

  int open (Peer_Handler *ph);
  // This method opens the file and starts to read it using overlapped
  // I/O.

  virtual int handle_timeout (const ACE_Time_Value &tv,
			      const void *arg = 0);
  // Prints out a message periodically.

  virtual int handle_input_complete (ACE_Message_Block *msg, 
				     long bytes_transfered);
  // One of our asynchronous reads from the file has completed.  Make
  // sure it succeeded and then delete the message and reinitiate
  // another read.

  virtual int handle_close (ACE_HANDLE, ACE_Reactor_Mask);
  // We've been removed from the ReactorEx.

  virtual ACE_Message_Block *get_message (void);
  // This is so the Proactor can get a message to read into.

  virtual ACE_HANDLE get_handle (void) const;
  // This is so the Proactor can get our handle.

private:
  Peer_Handler *ph_;
  // Event_Handler responsible for handling asynchronous I/O
  // completions.

  ACE_Overlapped_File file_;
  // The file we're reading using overlapped I/O.
};

Peer_Handler::Peer_Handler (void)
{
}

int 
Peer_Handler::open (const char *host, u_short port)
{
  if (host == 0) // Acceptor
    {
      ACE_SOCK_Acceptor acceptor;
      ACE_INET_Addr local_addr (port);

      if (acceptor.open (local_addr) == -1 ||
	  acceptor.accept (this->stream_) == -1)
	ACE_ERROR_RETURN ((LM_ERROR, "%p\n", 
			   "accept failed"), -1);

      ACE_DEBUG ((LM_DEBUG, "accepted.\n"));
    }
  else // Connector
    {
      ACE_INET_Addr addr (port, host);
      ACE_SOCK_Connector connector;

      // Establish connection with server.
      if (connector.connect (stream_, addr) == -1)
	ACE_ERROR_RETURN ((LM_ERROR, "%p\n", 
			   "connect"), -1);

      ACE_DEBUG ((LM_DEBUG, "connected.\n"));
    }

  // Start off the overlapped I/O operation.
  return ACE_Service_Config::proactor ()->initiate 
    (this, ACE_Event_Handler::READ_MASK);
}

int 
Peer_Handler::handle_output_complete (ACE_Message_Block *msg, 
				      long bytes_transfered)
{
  if (bytes_transfered <= 0)
    ACE_DEBUG ((LM_DEBUG, "%p\n", "Message failed"));

  // This was allocated by the FILE_Handler.
  delete msg;
  return 0;
}

int 
Peer_Handler::handle_input_complete (ACE_Message_Block *msg, 
				     long bytes_transfered)
{
  if ((bytes_transfered > 0) && (msg->length () > 0))
    {
      // NUL terminate the buffer.
      msg->rd_ptr ()[bytes_transfered] = '\0';

      // Print out the message received from the server.
      ACE_DEBUG ((LM_DEBUG, "%s\n", msg->rd_ptr ()));

      delete msg;

      return 1; // Reinvokes the recv() operation!
    }

  delete msg;
  return -1; // Close down.
}

ACE_Message_Block *
Peer_Handler::get_message (void)
{
  // An extra byte for NUL termination.
  ACE_Message_Block *message = 
    new ACE_Message_Block (BUFSIZ + 1);

  message->size (BUFSIZ);

  return message;
}

ACE_HANDLE 
Peer_Handler::get_handle (void) const
{
  return this->stream_.get_handle ();
}

int 
Peer_Handler::handle_close (ACE_HANDLE, ACE_Reactor_Mask)
{
  ACE_DEBUG ((LM_DEBUG, "Peer_Handler closing down\n"));
  return 0;
}

FILE_Handler::FILE_Handler (void) 
{
}

int 
FILE_Handler::open (Peer_Handler *ph)
{
  this->ph_ = ph;

  if (this->file_.open ("test_proactor.cpp", 
			GENERIC_READ,
			FILE_SHARE_READ, 0,
			OPEN_EXISTING, 
			FILE_FLAG_OVERLAPPED) == -1)
    ACE_ERROR_RETURN ((LM_ERROR, "%p open failed.\n", 
		       "FILE_Handler"), -1);

  ACE_Time_Value timeout (5);

  ACE_Service_Config::proactor ()->schedule_timer 
    (this, 0, timeout, timeout);

  return ACE_Service_Config::proactor ()->initiate 
    (this, ACE_Event_Handler::READ_MASK, 0, &this->file_);
}

int 
FILE_Handler::handle_timeout (const ACE_Time_Value &tv, 
			       const void *arg)
{
  ACE_DEBUG ((LM_DEBUG, "timeout.\n"));
  return 0;
}

int 
FILE_Handler::handle_input_complete (ACE_Message_Block *msg, 
				     long bytes_transfered)
{
  if (bytes_transfered > 0)
    {
      // NUL terminate the message.
      msg->base ()[msg->length ()] = '\0';

      // Print out the message received from the server.
      //	  ACE_DEBUG ((LM_DEBUG, "%s", msg->rd_ptr ()));

      // Initiate an asynchronous send() operation.
      ACE_Service_Config::proactor ()->initiate 
	(this->ph_, ACE_Event_Handler::WRITE_MASK, msg);

      return 1; // Reinvoke a recv() operation.
    }

  return -1; // Close down.
}

int 
FILE_Handler::handle_close (ACE_HANDLE, ACE_Reactor_Mask)
{
  ACE_DEBUG ((LM_DEBUG, "closing down\n"));
  return 0;
}

ACE_Message_Block *
FILE_Handler::get_message (void) 
{
  ACE_Message_Block *message = 
    new ACE_Message_Block (BUFSIZ + 1);

  message->size (BUFSIZ);

  return message;
}

ACE_HANDLE 
FILE_Handler::get_handle (void) const
{
  return this->file_.get_handle ();
}

static char *host = 0;
static int port = ACE_DEFAULT_SERVER_PORT;

static void
parse_args (int argc, char *argv[])
{
  ACE_Get_Opt get_opt (argc, argv, "h:p:");
  int c;

  while ((c = get_opt ()) != EOF)
    switch (c)
      {
      case 'h':
        host = get_opt.optarg;
	break;
      case 'p':
        port = ACE_OS::atoi (get_opt.optarg);
	break;
      }
}

int
main (int argc, char *argv[])
{
  parse_args (argc, argv);

  Peer_Handler peer_handler;

  if (peer_handler.open (host, port) == -1)
    ACE_ERROR_RETURN ((LM_ERROR, 
		       "%p open failed, errno = %d.\n",
		       "peer_handler", errno), 0);

  ACE_DEBUG ((LM_DEBUG, "Opening FILE_Handler.\n"));

  FILE_Handler file_handler;

  if (file_handler.open (&peer_handler) == -1)
    ACE_ERROR_RETURN ((LM_ERROR, 
		       "%p open failed, errno = %d.\n",
		       "file_handler", errno), 0);

  ACE_DEBUG ((LM_DEBUG, "Entering main event loop.\n"));

  ACE_Time_Value timeout (3);

  // Block in the event loop handling all the I/O events.  If the
  // proactor times out after 1 second without handling any events,
  // then we'll assume that everything is done and exit.  If we were
  // not using the timeout, we would use the Service_Config proactor
  // event loop.  Note that this is NOT using the ReactorEx.  See
  // examples/Reactor/ReactorEx/ for an example that does.

  for (;;)
    {
      // Could be ACE_Service_Config::run_proactor_event_loop () if we
      // were not using the timeout.
      if (ACE_Service_Config::proactor ()->handle_events (timeout) == -1)
	ACE_ERROR_RETURN ((LM_ERROR, "handle_events failed errno = %d.\n",
			   ACE_OS::last_error ()), -1);

      // If a timeout has occurred, exit.
      if (timeout.sec () == 0 && timeout.usec () == 0)
	break;
      else
	timeout.set (3);
    }

  return 42;
}
