// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of APE.
// 
// The exception is that, if you link the APE library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the APE library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name APE.  If you copy code from other releases into a copy of
// APE, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for APE, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include <stdlib.h>
#include "config.h"
#include "macros.h"
#include "thread.h"
#include "misc.h"

int Keydata::count = 0;
int Keydata::sequence = 1;
ifstream Keydata::cfgFile;
char Keydata::lastpath[KEYDATA_PATH_SIZE + 1]; 

Keydata::Keydata() : 
MemPager(KEYDATA_PAGER_SIZE)
{
	link = 0;
	memset(&keys, 0, sizeof(keys)); 
}

Keydata::Keydata(const char *path) :
MemPager(KEYDATA_PAGER_SIZE)
{
	link = 0;
	memset(&keys, 0, sizeof(keys));
	Load(path);		
}

Keydata::~Keydata()
{
	Unlink();
	if(count < 1)
		endKeydata();
}

keysym_t *Keydata::getSymbol(const char *sym, bool create)
{
	unsigned path = getIndex(sym);
	keysym_t *key = keys[path];

	while(key)
	{
		if(!stricmp(sym, key->sym))
			return key;
		key = key->next;
	}
	if(!create)
		return NULL;

	key = (keysym_t *)alloc(sizeof(keysym_t) + strlen(sym));
	strcpy(key->sym, sym);
	key->count = 0;
	key->next = keys[path];
	key->data = NULL;
	key->list = NULL;
	keys[path] = key;
	return key;
}

unsigned Keydata::getIndex(const char *str)
{
	unsigned key = 0;
	
	while(*str)
		key = (key << 1) ^ *(str++);

	return key % KEYDATA_INDEX_SIZE;
}

int Keydata::getCount(const char *sym)
{
	keysym_t *key = getSymbol(sym, false);
	if(!key)
		return 0;

	return key->count;
}

char *Keydata::getFirst(const char *sym)
{
	keysym_t *key = getSymbol(sym, false);
	keyval_t *val;

	if(!key)
		return NULL;

	val = key->data;
	if(!val)
		return NULL;

	while(val->next)
		val = val->next;

	return val->val;
}

char *Keydata::getLast(const char *sym)
{
	keysym_t *key = getSymbol(sym, false);
	if(!key)
		return NULL;

	if(!key->data)
		return NULL;

	return key->data->val;
}

int Keydata::getIndex(char **data, int max)
{
	int idx;
	keysym_t *key;
	int icount = 0;

	for(idx = 0; idx < KEYDATA_INDEX_SIZE; ++idx)
	{
		if(icount >= max)
			break;
		key = keys[idx];
		while(key && icount < max)
		{
			*(data++) = key->sym;
			++icount;
		}
	}
	*data = NULL;
	return icount;
}

void Keydata::setValue(const char *sym, const char *data)
{
	keysym_t *key = getSymbol(sym, true);
	keyval_t *val;

	if(!data)
		data = "";

	val = (keyval_t *)alloc(sizeof(keyval_t) + strlen(data));
	++key->count;
	key->list = NULL;
	val->next = key->data;
	key->data = val;
	strcpy(val->val, data);
}	

char **Keydata::getList(const char *sym)
{
	int icount;
	keysym_t *key = getSymbol(sym, false);
	keyval_t *data;

	if(!key)
		return NULL;

	icount = key->count;
	if(!icount)
		return NULL;

	++icount;
	if(!key->list)
	{
		key->list = (char **)first(sizeof(char **) * icount);
		key->list[--icount] = NULL;
		data = key->data;
		while(icount && data)
		{
			key->list[--icount] = data->val;
			data = data->next;
		}
		while(icount)
			key->list[--icount] = "";
	}
	return key->list;
}

void Keydata::clrValue(const char *sym)
{
	keysym_t *key = getSymbol(sym, false);
	if(!key)
		return;

	key->count = 0;
	key->list = NULL;
	key->data = NULL;
}

void Keydata::Load(KEYDEF *defs)
{
	keysym_t *key;

	while(defs->keyword)
	{
		key = getSymbol(defs->keyword, true);
		if(!key->data)
			setValue(defs->keyword, defs->value);
		++defs;
	}
}	

void Keydata::Load(const char *keypath)
{
	char path[512];
	char seek[17];
	char find[17];
	char *prefix = NULL;
	char *cp, *ep;
	int fpos;
	keysym_t *sym;
	keysym_t *val;

	if(*keypath == '~')
	{
		prefix = getenv("HOME");
		strcpy(path, prefix);
		strcat(path, "/.");
		++keypath;
	}

	if(!prefix)
	{
		prefix = getenv("CONFIG_KEYDATA");
		if(!prefix)
			prefix = "/etc";
		strcpy(path, prefix);
		prefix = NULL;
	}

	if(*keypath == '/')
		++keypath;

	strcat(path, keypath);
	cp = strrchr(path, '/');
	strncpy(seek, cp + 1, 16);
	seek[16] = 0;
	*cp = 0;

	if(!prefix)
		strcat(path, ".conf");

	if(strcmp(path, lastpath))
	{
		endKeydata();
		cfgFile.open(path, ios::in);
		if(!cfgFile && !prefix)
			cfgFile.open(path + 5, ios::in);
		if(!cfgFile)
			return;
		strcpy(lastpath, path);
	}

	if(link != sequence)
	{
		link = sequence;
		++count;
	}

	find[0] = 0;
	cfgFile.seekg(0);
	while(stricmp(seek, find))
	{
		cfgFile.getline(path, sizeof(path) - 1);
		if(cfgFile.eof())
			return;

		cp = path;
		while(*cp == ' ' || *cp == '\n' || *cp == '\t')
			++cp;

		if(*cp != '[')
			continue;

		ep = strchr(cp, ']');
		if(ep)
			*ep = 0;
		else
			continue;

		strncpy(find, ++cp, 16);
		find[16] = 0;
	}

	for(;;)
	{
		cfgFile.getline(path, sizeof(path) - 1);
		if(cfgFile.eof())
			return;

		cp = path;
		while(*cp == ' ' || *cp == '\t' || *cp == '\n')
			++cp;

		if(!*cp || *cp == '#' || *cp == ';' || *cp == '!')
			continue;

		if(*cp == '[')
			return;

		fpos = 0;
		while(*cp && *cp != '=')
		{
			if(*cp == ' ' || *cp == '\t')
			{
				++cp;
				continue;
			}
			find[fpos] = *(cp++);
			if(fpos < 16)
				++fpos;
		}
		find[fpos] = 0;
		if(*cp != '=')
			continue;

		++cp;
		while(*cp == ' ' || *cp == '\t' || *cp == '\n')
			++cp;

		ep = cp + strlen(cp);
		while((--ep) > cp)
		{
			if(*ep == ' ' || *ep == '\t' || *ep == '\n')
				*ep = 0;
			else
				break;
		}

		if(*cp == *ep && (*cp == '\'' || *cp == '\"'))
		{
			++cp;
			*ep = 0;
		}

		setValue(find, cp);
	}
}

void Keydata::Unlink(void)
{
	if(link != sequence)
	{
		link = 0;
		return;
	}
	
	link = 0;
	--count;
}
			
void endKeydata(void)
{
	Keydata::count = 0;
	++Keydata::sequence;
	if(!Keydata::sequence)
		++Keydata::sequence;

	Keydata::lastpath[0] = 0;
	if(Keydata::cfgFile)
		Keydata::cfgFile.close();
}

