// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of APE.
// 
// The exception is that, if you link the APE library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the APE library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name APE.  If you copy code from other releases into a copy of
// APE, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for APE, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "fileio.h"
#include "thread.h"
#include "file.h"

File::File(const char *fname, int mode)
{
	DWORD access = 0, dispose = OPEN_EXISTING;
	DWORD attr = FILE_ATTRIBUTE_NORMAL | FILE_FLAG_RANDOM_ACCESS | FILE_FLAG_POSIX_SEMANTICS;

	_append = false;

	switch(mode)	{
	case FILE_OPEN_READONLY:
		access = GENERIC_READ;
		break;
	case FILE_OPEN_WRITEONLY:
		access = GENERIC_WRITE;
		break;
	case FILE_OPEN_READWRITE:
		access = GENERIC_READ | GENERIC_WRITE;
		break;
	case FILE_OPEN_APPEND:
		access = GENERIC_READ | GENERIC_WRITE;
		_append = true;
		break;
	case FILE_OPEN_SYNC:
		access = GENERIC_READ | GENERIC_WRITE;
		attr |= FILE_FLAG_OVERLAPPED;
		break;
	case FILE_OPEN_TRUNCATE:
		access = GENERIC_READ | GENERIC_WRITE;
		dispose = TRUNCATE_EXISTING;
		break;
	}
	
	hFile = CreateFile(fname, access, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, dispose, attr, NULL);
	if(hFile == INVALID_HANDLE_VALUE)
		throw(this);
};

File::File(const char *fname, int mode, int perm)
{
	DWORD access = 0, dispose = OPEN_ALWAYS;
	DWORD attr = FILE_ATTRIBUTE_NORMAL | FILE_FLAG_RANDOM_ACCESS | FILE_FLAG_POSIX_SEMANTICS;
	_append = false;

	switch(mode)	{
	case FILE_OPEN_READONLY:
		access = GENERIC_READ;
		break;
	case FILE_OPEN_WRITEONLY:
		access = GENERIC_WRITE;
		break;
	case FILE_OPEN_READWRITE:
		access = GENERIC_READ | GENERIC_WRITE;
		break;
	case FILE_OPEN_APPEND:
		access = GENERIC_READ | GENERIC_WRITE;
		_append = true;
		break;
	case FILE_OPEN_SYNC:
		access = GENERIC_READ | GENERIC_WRITE;
		attr |= FILE_FLAG_OVERLAPPED;
		break;
	case FILE_OPEN_TRUNCATE:
		access = GENERIC_READ | GENERIC_WRITE;
		dispose = TRUNCATE_EXISTING;
		break;
	}

//	switch(perm)
//	{
//	}
	
	hFile = CreateFile(fname, access, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, dispose, attr, NULL);
	if(hFile == INVALID_HANDLE_VALUE)
		throw(this);
};

File::File(const File &f)
{
	hFile = hDup(f.hFile);
	if(INVALID_HANDLE_VALUE == hFile)
		throw(this);
};

File &File::operator=(const File &f)
{
	hClose(hFile);
	hFile = hDup(f.hFile);
	if(INVALID_HANDLE_VALUE == hFile)
		throw(this);

	return *this;
}

LockedFile::LockedFile(const char *fname, int access) :
File(fname, access) {};

LockedFile::LockedFile(const LockedFile &f) :
File((const File &)f) {};

int LockedFile::Append(void *buf, size_t len)
{
	int rc;
	OVERLAPPED over;

	over.hEvent = 0;
	over.OffsetHigh = 0;
	over.Offset = hSeek(hFile, 0, FILE_END);

	LockFileEx(hFile, LOCKFILE_EXCLUSIVE_LOCK, 0, 0x7fffffff, 0, &over);
	hSeek(hFile, 0, FILE_END);
	rc = hWrite(hFile, buf, len);
	hSeek(hFile, over.Offset, FILE_BEGIN);
	UnlockFileEx(hFile, 0, 0x7fffffff, 0, &over);
	return rc;
}

int LockedFile::Request(pos_t pos, void *buf, size_t len)
{
	OVERLAPPED over;

	over.hEvent = 0;
	pos_t npos = hSeek(hFile, pos, FILE_BEGIN);
	if(npos != pos)
		return -1;

	over.Offset = pos;
	over.OffsetHigh = 0;
	LockFileEx(hFile, LOCKFILE_EXCLUSIVE_LOCK, 0, len, 0, &over);
	return hRead(hFile, buf, len);
}

int LockedFile::Update(pos_t pos, void *buf, size_t len)
{
	OVERLAPPED over;
	int rc;

	over.hEvent = 0;
	pos_t npos = hSeek(hFile, pos, FILE_BEGIN);
	if(npos != pos)
		return -1;

	rc = hWrite(hFile, buf, len);
	over.Offset = pos;
	over.OffsetHigh = 0;
	UnlockFileEx(hFile, 0, len, 0, &over);
	return rc;
}

int LockedFile::Clear(pos_t pos, size_t len)
{
	OVERLAPPED over;

	over.hEvent = 0;
	pos_t npos = hSeek(hFile, pos, FILE_BEGIN);
	if(npos != pos)
		return -1;

	over.Offset = pos;
	over.OffsetHigh = 0;
	UnlockFileEx(hFile, 0, len, 0, &over);
	return 0;
}

SharedFile::SharedFile(const char *fname, int access, int mode) :
	File(fname, access, mode), Mutex() {};

SharedFile::SharedFile(const char *fname, int access) :
	File(fname, access), Mutex() {};

SharedFile::SharedFile(const SharedFile &f) :
	File((const File &)f), Mutex() {};

int SharedFile::Append(void *buf, size_t len)
{
	int	rc;

	ENTER_CRITICAL
	hSeek(hFile, 0, FILE_END);
	rc = hWrite(hFile, buf, len);
	LEAVE_CRITICAL
	return rc;
}

int SharedFile::Read(pos_t pos, void *buf, size_t len)
{
	int rc;
	pos_t npos;

	ENTER_CRITICAL
	npos = (pos_t)hSeek(hFile, pos, FILE_BEGIN);
	if(npos == pos)
		rc = hRead(hFile, buf, len);
	else
		rc = -1;

	LEAVE_CRITICAL
	return rc;
}

int SharedFile::Write(pos_t pos, void *buf, size_t len)
{
	int rc;
	pos_t npos;

	ENTER_CRITICAL
	npos = (pos_t)hSeek(hFile, pos, FILE_BEGIN);
	if(npos == pos)
		rc = hWrite(hFile, buf, len);
	else
		rc = -1;

	LEAVE_CRITICAL
	return rc;
}

SharedFile &SharedFile::operator=(SharedFile &f)
{
	ENTER_CRITICAL
	hClose(hFile);
	hFile = hDup(f.hFile);
	LEAVE_CRITICAL
	if(!hFile)
		throw(this);

	return *this;
}

