// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of APE.
// 
// The exception is that, if you link the APE library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the APE library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name APE.  If you copy code from other releases into a copy of
// APE, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for APE, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#ifndef	__APE_FILE_H__
#define	__APE_FILE_H__

#ifndef	__APE_THREAD_H__
#include "thread.h"
#include "fileio.h"
#endif

typedef	unsigned long pos_t;

enum{
	FILE_OPEN_READONLY = 0,
	FILE_OPEN_WRITEONLY,
	FILE_OPEN_READWRITE,
	FILE_OPEN_APPEND,
	FILE_OPEN_SYNC,
	FILE_OPEN_TRUNCATE
};

enum{
	FILE_PERM_PRIVATE,
	FILE_PERM_GROUP,
	FILE_PERM_PUBLIC
};

class File
{
protected:
	HANDLE hFile;
	bool _append;

public:
	File(const char *fname, int access);
	File(const char *fname, int access, int perm);
	File(const File &f);
	
	virtual ~File()
		{hClose(hFile);};

	int Read(void *buf, size_t len)
		{return hRead(hFile, (char *)buf, len);};

	int Write(void *buf, size_t len)
		{return hWrite(hFile, (char *)buf, len);};

	int Write(char *buf)
		{return hWrite(hFile, buf, strlen(buf));};

	pos_t Position(pos_t pos)
		{return (pos_t)hSeek(hFile, pos, FILE_BEGIN);}; 

	pos_t Append(void)
		{return (pos_t)hSeek(hFile, 0l, FILE_END);}; 

	File &operator=(const File &f);

	friend read(File &f, void *buf, size_t len)
		{return hRead(f.hFile, (char *)buf, len);};

	friend write(File &f, void *buf, size_t len)
		{return hWrite(f.hFile, (char *)buf, len);};

	friend write(File &f, char *buf)
		{return hWrite(f.hFile, buf, strlen(buf));};

	friend pos_t append(File &f)
		{return (pos_t)hSeek(f.hFile, 0l, FILE_END);};

	friend pos_t position(File &f, pos_t pos)
		{return (pos_t)hSeek(f.hFile, pos, SEEK_SET);};
};

class LockedFile : protected File 
{
public:
	LockedFile(const char *fname, int access);
	LockedFile(const LockedFile &f);

	~LockedFile()
		{hClose(hFile);};

	int Append(void *buf, size_t len);
	int Request(pos_t pos, void *buf, size_t len);
	int Update(pos_t pos, void *buf, size_t len);
	int Clear(pos_t pos, size_t len);
};

class SharedFile : public File, public Mutex 
{
public:
	SharedFile(const char *fname, int access, int perm);
	SharedFile(const char *fname, int access);
	SharedFile(const SharedFile &f);

	~SharedFile()
		{hClose(hFile);};

	int Append(void *buf, size_t len);
	int Read(pos_t pos, void *buf, size_t len);
	int Write(pos_t pos, void *buf, size_t len);

	SharedFile &operator=(SharedFile &f);
	
	friend inline read(SharedFile &f, pos_t pos, void *buf, size_t len)
		{f.Read(pos, buf, len);};

 	friend inline write(SharedFile &f, pos_t pos, void *buf, size_t len)
 		{f.Write(pos, buf, len);};

	friend inline append(SharedFile &f, void *buf, size_t len)
		{f.Append(buf, len);};
};

class __EXPORT Pipe 
{
private:
	HANDLE reader;
	HANDLE writer;
	int objsize, objcount;

public:
	Pipe(int size, int count);
	Pipe(const Pipe &p);
	~Pipe();

	inline Receive(void *addr)
		{return hRead(reader, addr, objsize);};

	inline Send(void *addr, size_t len)
		{return hWrite(writer, addr, objsize);};

	Pipe &operator=(const Pipe &p);
};

inline clear(LockedFile &f, pos_t pos, size_t len)
	{return f.Clear(pos, len);};

inline append(LockedFile &f, void *buf, size_t len)
	{return f.Append(buf, len);};

inline request(LockedFile &f, pos_t pos, void *buf, size_t len)
	{return f.Request(pos, buf, len);};

inline update(LockedFile &f, pos_t pos, void *buf, size_t len)
	{return f.Update(pos, buf, len);};

#endif

