// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your
// choice whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "config.h"
#include "thread.h"

Mutex::Mutex()
{
	_mutex = 0;
	_waiting = 0;
	_owner = 0;
	_reent = 0;
	_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	if(!_event)
		throw(this);
}

bool Mutex::TryEnterMutex(void)
{
	// we got it
	DWORD tid = GetCurrentThreadId();
	if(!InterlockedExchange(&_mutex, 1))
	{
		_owner = tid;
		_reent = 0;
		return true;
	}

	// we already owned it
	if(_owner == tid)
	{
		++_reent;
		return true;
	}

	// already claimed
	return false;
}

void Mutex::EnterMutex(void)
{
	DWORD tid = GetCurrentThreadId();
	if(!InterlockedExchange(&_mutex, 1))
	{
		_owner = tid;
		_reent = 0;
		return;
	}

	if(_owner == tid)
	{
		++_reent;
		return;
	}

	while(true)
	{
		InterlockedIncrement(&_waiting);
		if(!InterlockedExchange(&_mutex, 1))
		{
			InterlockedDecrement(&_waiting);
			_owner = tid;
			_reent = 0;
			return;
		}
		WaitForSingleObject(_event, INFINITE);
		if(!InterlockedExchange(&_mutex, 1))
		{
			_owner = tid;
			_reent = 0;
			return;
		}
	}
}

void Mutex::LeaveMutex(void)
{
	if(_owner != GetCurrentThreadId())
		throw this;

	if(_reent)
	{
		--_reent;
		return;
	}

	_owner = 0;
	InterlockedExchange(&_mutex, 0);

	if(_waiting > 0)
	{
		InterlockedDecrement(&_waiting);
		SetEvent(_event);
		return;
	}
}

MutexCounter::MutexCounter() : Mutex()
{
	counter = 0;
};

int AtomicCounter::operator+=(int change)
{
	while(change)
		InterlockedIncrement(&atomic);
	
	return atomic;
}

int AtomicCounter::operator-=(int change)
{
	while(change)
		InterlockedDecrement(&atomic);

	return atomic;
}

int operator++(MutexCounter &mc) 
{
	int rc = 0;

	mc.EnterMutex();
	rc = mc.counter++;
	mc.LeaveMutex();
	return rc;
};

int operator--(MutexCounter &mc)
{
	int rc = 0;

	mc.EnterMutex();
	rc = --mc.counter;
	if(!rc)
		throw(mc);
	mc.LeaveMutex();
	return rc;
};

/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
