// Copyright (C) 1999 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "audio.h"
#include "thread.h"

RTSThread::RTSThread(Semaphore *start, int pri) :
Thread(start, pri, 0)
{
	_startup = 0;
	Reset();
}

RTSThread::~RTSThread()
{
	Pause(0);
	Terminate();
}

void RTSThread::Reset(void)
{
	struct timezone tz;
	
	gettimeofday(&_tv, &tz);
}

timeout_t RTSThread::getTimeUsed(void)
{
	struct timeval tv;
	struct timezone tz;

	gettimeofday(&tv, &tz);
	return (tv.tv_sec - _tv.tv_sec) * 1000l + (tv.tv_usec - _tv.tv_usec) / 1000l;
}

bool RTSThread::isServiceDelay(void)
{
	if(_startup)
	{
		--_startup;
		return true;
	}
	return false;
}

void RTSThread::addTimeUsed(timeout_t timer)
{
	_tv.tv_sec += (timer / 1000l);
	_tv.tv_usec += (timer % 1000l) * 1000l;
	if(_tv.tv_usec > 1000000l)
	{
		++_tv.tv_sec;
		_tv.tv_usec %= 1000000l;
	}
}

bool RTSThread::Pause(timeout_t timer)
{
	struct timeval tv;
	struct timezone tz;

	addTimeUsed(timer);
	gettimeofday(&tv, &tz);
	
	timer = (_tv.tv_sec - tv.tv_sec) * 1000l + (_tv.tv_usec - tv.tv_usec) / 1000l;
	if(timer < 1)
		return false;

	Sleep(timer);
	return true;
}

RTSSendThread::RTSSendThread(Semaphore *start, int pri, timeout_t timer) :
RTSThread(start, pri)
{
	setTimeout(timer);
}

RTSSendThread::~RTSSendThread()
{
	Pause(getTimeout());
	Terminate();
}


void RTSSendThread::RunSender(void)
{
	size_t len;
	
	Reset();        

	// slight offset for frame timer adjustment.
	addTimeUsed(getTimeout() / 3);

	for(;;)
	{
		while(isServiceDelay())
		{
			Pause(getTimeout());
			RTSUpdate();
		}

		len = RTSFetch(rtpdata, getTimeout());
		if(len > 0)
			RTSSend(rtpdata, len);
		else
			RTSUpdate();

		// if true, we're probably not feeding from a realtime
		// data source or have fallen out of sync, so we must use
		// synchronized delays

		if(getTimeUsed() < getTimeout() / 3 * 2)
		{
			Pause(getTimeout());
			continue;
		}

		// see if we are overruning the time of the
		// current frame and if so skip now empty time marks
		// in the session.

		while(getTimeUsed() > getTimeout() + getTimeout() / 3)
		{
			RTSUpdate();
			addTimeUsed(getTimeout());
		}
	}
}

RTSRecvThread::RTSRecvThread(Semaphore *start, int pri, timeout_t timer) :
RTSThread(start, pri) 
{
	setTimeout(timer);
	_starting = 0;
	setServiceDelay(4);
}

void RTSRecvThread::RunReceiver(void)
{
	size_t len;
	unsigned short encoding;
	unsigned long timestamp;
	
	Reset();
	for(;;)
	{
		if(getTimeUsed() >= getTimeout())
		{
			if(!isServiceDelay())
				RTSOutput(getTimeout());
			addTimeUsed(getTimeout());
			continue;
		}
		
		if(RTSWait(getTimeout() - getTimeUsed()))
		{
			if(!isServiceDelay())
				RTSOutput(getTimeout());
			addTimeUsed(getTimeout());
		}
		
		len = RTSRecv(rtpdata, &encoding, &timestamp);
		if(len < 1)
		{
			RTSRecvError();
			continue;
		}
			
		timestamp = getStarting(timestamp);
		len = RTSPost(rtpdata, encoding, timestamp, len);
	}
}

unsigned long RTSRecvThread::addStarting(unsigned long samples)
{
	_starting = (_starting + samples) & 0xffffffff;
	return _starting;
}

unsigned long RTSRecvThread::getStarting(unsigned long stamp)
{
	return (stamp - _starting) & 0xffffffff;
}

void RTSRecvThread::setStarting(unsigned long stamp)
{
	_starting = stamp & 0xffffffff;
}
