/* arrman-t.h

   Array-Manager with Templates.

   Author: Marko Meyer <derived from listman-t.h on 95/10/22>
   Time-stamp: <95/12/02 11:58:32 mme>

   This file belongs to the library BACKNET.
   It is Copyright (C) 1995 Marko Meyer.
   Please read the files README, doc/LICENSE and doc/DISCLAIMER.
   If you miss one of these files, please contact me:

   Email: mme@pub.th-zwickau.de
   Ordinary Mail: Marko Meyer
                  Teichstrasse 27
				  D-08289 Schneeberg
				  Germany
   History:
   95/09/02 -- Marko Meyer: Due to some new preferences I changed the 
               ListManager slightly.

   95/10/22 -- Marko Meyer: This Array Manager was derived from the 
               ListManager to improve speed

   95/11/11&12 -- Marko Meyer: included new functions:
                  int Get_List(int listid, X *Dest);
                  int Set_List(int listid, X *Src [,int len]);
				  for use in ArrayServer 

   95/11/12 -- Marko Meyer: included condition 'if(elemnr == 0) ...' into
               several functions, for ensuring the actual numbering scheme
               of elements, which starts at 1.
			   Made MAXARRS be setable at compile time without changing the
			   code.

*/

#include <dosbug.h>

#include __LIST_ER__

#include <string.h>

#ifdef _DEBUG_ON_
#define DBG(x) x
#else
#define DBG(x)  
#endif

#ifndef MAXARRS
#define MAXARRS 10000
#endif /* !MAXARRS */

template <class X> class C_Array
{
	int i_ArrCount;                                 /*number of arrays*/
	X   *AS_FirstElem[MAXARRS];              /*pointers to arrayheads*/
	int ai_ElemCount[MAXARRS];	 /*number of elements stored in array.*/
	
	public:


C_Array()
{
	i_ArrCount=0;
	for(int i=0;i<MAXARRS;ai_ElemCount[i]=0,AS_FirstElem[i++]=NULL);
}

/* We need a new create. The old one did create a list by creating one
   Element with a given content and appending it to a free listnode.
   The new one will only mark a listnode as taken by writing -1 to
   ai_ElemCount[listnode] or create the amount of Elements specified.
   mme - 09/02 1995. */

int
ncreate(int amount, int *i_Error)
{
	int listnr,i;
	
	*i_Error=NO_ERROR;
	if((i_ArrCount < (MAXARRS - 1)))
	{
		for(listnr=1 ; (listnr < (MAXARRS - 1) && 
						(lexist(listnr) == LIST_EXISTS)); listnr++);
		i_ArrCount++;
		ai_ElemCount[listnr] = -1; /* Mark as taken. */

		if(amount > 0)
		{
			/* We use append to append the amount of Elements. */
			for(i = 1;(i <= amount)&&(*i_Error == NO_ERROR); i++)
				*i_Error = append(listnr,0);
		}
		if(*i_Error == NO_ERROR) return listnr;
		else 
		{
			destroy(listnr);
			return 0;
		}
	}
	else *i_Error=ERR_OVERFL_LIST;
	return 0;
}

						

int 
create(X continp,int *i_Error)
{
	int listnr;

	*i_Error=NO_ERROR;
	if((i_ArrCount<(MAXARRS-1)))
	{
	 	for(listnr=1;(listnr<MAXARRS-1)&&lexist(listnr)==LIST_EXISTS;
			listnr++);	
		if((AS_FirstElem[listnr] = (X *) calloc(1, sizeof(X)))) 
		{
			i_ArrCount++;
			ai_ElemCount[listnr]++;
			AS_FirstElem[listnr][0] = continp;
			return listnr;	
		}
		else *i_Error=ERR_MEMORY_LIST;
	}
	else *i_Error=ERR_OVERFL_LIST;
	return 0;
	
}

int 
lexist(int listnr)
{
	if(((ai_ElemCount[listnr]!=0)/*&&(AS_FirstElem[listnr])*/))
		return LIST_EXISTS;
	/* I commented this, because it seems to be okay, when we only check
	   ai_ElemCount[], because this is set properly when things change.
	   Did this because -1 as ai_ElemCount[listnr] shows that the list is
	   taken, though no Element is appended yet. mme - 09/02 1995 */
	else return NO_LIST;
}

int 
eexist(int listnr,int elemnr)
{
	if(((lexist(listnr)==LIST_EXISTS)
	   &&(ai_ElemCount[listnr]>=elemnr))) return ELEM_EXISTS;
	else return NO_ELEM;
}

int 
read(int listnr,int elemnr,X* content)
{
	if(elemnr == 0) return ERR_NOELEM_LIST;
	if((eexist(listnr,elemnr)==ELEM_EXISTS))
	{
		*content=AS_FirstElem[listnr][elemnr-1];
		return NO_ERROR;
	}
	else return ERR_NOELEM_LIST;
}

int 
write(int listnr,int elemnr,X continp)
{
	if(elemnr == 0) return ERR_NOELEM_LIST;
	if((eexist(listnr,elemnr)==ELEM_EXISTS))
	{
		AS_FirstElem[listnr][elemnr - 1] = continp;
		return NO_ERROR;
	}
	else return ERR_NOELEM_LIST;
}

int 
append(int listnr,X continp)
{
	if((lexist(listnr)==LIST_EXISTS))
	{
		/* We have to check if we only set the list taken and no Element
		   is present in this list.  mme - 09/02 1995*/

		if(ai_ElemCount[listnr] == -1)
		{
			/* Okay, that's the case!! */
			if((AS_FirstElem[listnr] = (X *) calloc(1, sizeof(X))))
			{
				AS_FirstElem[listnr][0] = continp;
				ai_ElemCount[listnr] = 1;
			}
			else return ERR_MEMORY_LIST;
		}
		else
		{
			/* Otherwise there's still an Element to append after. */

			if((AS_FirstElem[listnr] = 
				(X *) realloc(AS_FirstElem[listnr],
							  (ai_ElemCount[listnr]+1) * sizeof(X))))
			{
				AS_FirstElem[listnr][ai_ElemCount[listnr]] = continp;
				ai_ElemCount[listnr]++;
				return NO_ERROR;
			}
			else return ERR_MEMORY_LIST;
		}
	}
	else return ERR_LNOTEX_LIST;
	return NO_ERROR;
}

int 
insert(int listnr,int before_elem,X continp)
{
	int len = 0;
	
	if(before_elem == 0) return ERR_NOELEM_LIST;
	if((lexist(listnr)==LIST_EXISTS))
	{
		if(((before_elem<=ai_ElemCount[listnr])&&(ai_ElemCount[listnr]>0)))
		{
			if((AS_FirstElem[listnr] = 
				(X *) realloc(AS_FirstElem[listnr],
							  (ai_ElemCount[listnr] + 1) * sizeof(X))))
			{
				len = ((ai_ElemCount[listnr] - before_elem + 1) *
					sizeof(X)) - 1;
				memmove(&AS_FirstElem[listnr][before_elem + 1],
						&AS_FirstElem[listnr][before_elem], len);
				AS_FirstElem[listnr][before_elem] = continp;
				ai_ElemCount[listnr]++;
				return NO_ERROR;
			}
			else return ERR_MEMORY_LIST;
		}
		else return ERR_NOELEM_LIST;
	}
	else return ERR_LNOTEX_LIST;
}

int 
del(int listnr,int elemnr)
{
	int len = 0;

	if(elemnr == 0) return ERR_NOELEM_LIST;
	if((eexist(listnr,elemnr)==ELEM_EXISTS))
	{
		len = ((ai_ElemCount[listnr] - elemnr) * sizeof(X)) - 1;
		memmove(&AS_FirstElem[listnr][elemnr],
				&AS_FirstElem[listnr][elemnr + 1], len);
		AS_FirstElem[listnr] = 
			(X *) realloc(AS_FirstElem[listnr], 
						  (ai_ElemCount[listnr] - 1) * sizeof(X));
		ai_ElemCount[listnr]--;
		if(!ai_ElemCount[listnr]) destroy(listnr);
		return NO_ERROR;
	}
	else return ERR_NOELEM_LIST;
}

int 
destroy(int listnr)
{
  int i_Error = NO_ERROR;
  
  if(ai_ElemCount[listnr] == -1)
  {
	  /* The list was taken, no Element present. We just set back
		 the values. mme - 09/02 1995. */
	  
	  ai_ElemCount[listnr] = 0;
	  i_ArrCount--;
	  return NO_ERROR;
  }

  if(((ai_ElemCount[listnr] > 0) && AS_FirstElem[listnr] ))
  {
	  free(AS_FirstElem[listnr]);
	  AS_FirstElem[listnr]=NULL;
	  ai_ElemCount[listnr]=0;
	  i_ArrCount--;
  }
  else i_Error=ERR_LNOTEX_LIST;
  
  return i_Error;
}

int
Get_List(int listid, X **Dest)
{
	/* Returns the list listid to Dest, which mustn't be NULL when calling.
	   For security aims the list is copied into an herein allocated buffer,
	   whiches starting address is returned then. */

	if(lexist(listid))
	{
		if(Dest == NULL) return ERR_OVERFL_LIST;
		if( (*Dest = (X *) calloc(ai_ElemCount[listid], sizeof(X))) != NULL)
		{
			memcpy(*Dest,AS_FirstElem[listid],ai_ElemCount[listid]*sizeof(X));
			return NO_ERROR;
		}
		else return ERR_MEMORY_LIST;
	}
	else return ERR_LNOTEX_LIST;
}

int
Set_List(int listid, X *Src)
{
	/* Set_List sets the list listid to the values specified in Dest.
	   There are only that much elements copied, as are actually present
	   (that means: no further allocation is done).
	   So when having a bigger list to save as listid, specify the new size
	   and another function with another signature is called. */

	if(lexist(listid))
	{
		if(Src != NULL)
		{
			memcpy(AS_FirstElem[listid], Src, ai_ElemCount[listid]*sizeof(X));
			return NO_ERROR;
		}
		else return ERR_OVERFL_LIST;
	}
	else return ERR_LNOTEX_LIST;
}

int
Set_List(int listid, X *Src, int len)
{
	/* This version of Set_List is used when having a bigger list to
	   attach to listid. The missing memory is realloced and then the
	   Src copied to it. The value of len _must_ be in _bytes_!!*/

	if(lexist(listid))
	{
		if(Src != NULL)
		{
			if(len == (signed)(ai_ElemCount[listid] * sizeof(X)))
				return Set_List(listid, Src);
			else
			{
				if((AS_FirstElem[listid] = (X *) realloc(AS_FirstElem[listid],
														 len)) != NULL)
				{
					memcpy(AS_FirstElem[listid], Src, len);
					return NO_ERROR;
				}
				else return ERR_MEMORY_LIST;
			}
		}
		else return ERR_OVERFL_LIST;
	}
	else return ERR_LNOTEX_LIST;
}

~C_Array()
{
  int i=0;
	if((i_ArrCount>0)) for(i=1;i<=i_ArrCount;destroy(i++));
}

};


