/* array_serv_udp_3.cc
   Author: Marko Meyer
   Date: 95/12/21
   Time-stamp: <96/01/27 19:19:52 mme>
  
   This file belongs to BACKNET. It realizes a server for the array 
   management. [This time with classes and a pipe.]

	BACKNET - A library for simulating neural BACKPROPAGATION-Networks
    Copyright (C) 1995, 1996	Marko Meyer

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


	If you have any suggestions, ideas or comments regarding this library
	feel free to contact me:

	Email:	mme@pub.th-zwickau.de
	Ordinary Mail:	Marko Meyer, Teichstrasse 27, D-08289 Schneeberg, Germany

	History:
    95/12/21 -- Marko Meyer: derived this from array_serv_udp_2.cc
	                         replaced the SHM-Request-Table with a pipe.
							 Started implementing Process_Request(...).
	
    95/12/24 -- Marko Meyer: As a Christmas present, Process_Request(...)
	                         seems to work now. Noticed  a strange
							 endless-loop under strange circumstances.
							 Was not replicable when trying. (Heisenbug!?)
							 ** FIXED THE HEISENBUG **
							 
    95/12/26 -- Marko Meyer: Fixed some minor things: RDat free'ing ...
*/

#ifdef _WITH_ARRAY_
#undef _WITH_ARRAY_
#endif
#include <arrserv_udp_2.h>

C_ArrayServ::C_ArrayServ(char *serv_name, short port)
{
	/* This one is the Constructor of the Array Server. It is called when
	   starting to work. The expected two control flows are still united
	   when this is executed. */

	name = new char[strlen("UDP Array Server 0.1") + 1];
	copyright = new char[strlen("Copyright (C) 1995 Marko Meyer") + 1];

	strcpy(name, "UDP Array Server 0.1");
	strcpy(copyright, "Copyright (C) 1995 Marko Meyer");

	DBG(cerr<<NAME_ID<<" "<<VERSION_NR<<" "<<name<<endl<<copyright<<endl);

	/* Get the communication socket. */

	Req_Sock = set_up_udp(serv_name, port);
	if(Req_Sock < 0)
	{
		perror("C_ArrServ(): Error at set_up_udp");
		exit(RET_NOSOCK_ARS);
	}

	/* FRESH!!!! ;-) Making a pipe as replacement of Req_T-SHM. */

	if(pipe(Req_T_Pipe) < 0)
	{
		perror("Error at pipe");
		exit(RET_MKPIPE_ARS);
	}
	

	DBG(cerr<<"C_ArrServ(): Req_T = "<<Req_T_Pipe[0]<<" (rd), "
		<<Req_T_Pipe[1]<<" (wr)"<<endl);

	/* Setting the signal handler for SIGCLD. */

	signal(SIGCLD, (void (*) (int)) handle_SIGCLD);
	signal(SIGINT, (void (*) (int)) handle_SIGINT);

	Par_ID = getppid();
	Own_ID = getpid();
	Cld_ID = 0;
	
}

void 
C_ArrayServ::handle_SIGCLD()
{
	/* A handler for SIGCLD. Shifts a message, time and PID (of dying
	   process) into cerr. */

	int Cld_ID;
	int Own_ID = getpid();
	int status;
	int estat;

	Cld_ID = wait(&status);
#ifdef __LINUX__
	estat = WIFEXITED(&status) ? WEXITSTATUS(&status) : WTERMSIG(&status);
#else
	estat = WIFEXITED(status) ? WEXITSTATUS(status) : WTERMSIG(status);
#endif	/* __LINUX__ */

	cerr<<"Array_Server: Announcing death of child process."<<endl
		<<"Child-PID: "<<Cld_ID<<"\tOwn PID: "<<Own_ID<<endl
#ifdef __LINUX__
			<<(WIFEXITED(&status)?"exited with exit code: "
#else
			<<(WIFEXITED(status)?"exited with exit code: "
#endif /* __LINUX __ */
			   :"died with signal: ")<<estat<<endl; 
	exit(RET_SIGCLD_ARS);
	
}

void 
C_ArrayServ::handle_SIGINT()
{
	/* This is a handler for SIGINT (usually ^C). */

	if(getpgrp() == getpid())
	{
		cerr<<"Array_Server: Received SIGINT, going down."<<endl;
		cerr<<"\t Killing child process ..."<<endl;
		kill(-getpgrp(),SIGINT);
		wait(NULL);
		cerr<<"\t Stopping now."<<endl;
	}
	else
	{
		cerr<<"Array_Server_Child: Received SIGINT, going down."<<endl;
		cerr<<"Array_Server_Child: Killed."<<endl;
	}
	exit(RET_SIGINT_ARS);
}

	
int
C_ArrayServ::Run()
{
	/* Run forks the child and calls the appropriate methods. */

	Cld_ID = fork();
	
	if(Cld_ID < 0) 
	{
		perror("Run(): Error at fork");
		exit(RET_NOFORK_ARS);
	}
	
	if(Cld_ID == 0)
		return Run_Server();
	else
		return Run_T_Fill();
}

int
C_ArrayServ::Run_T_Fill()
{
	/* This is the table filler. */

	Buffer_Info *Req_Buf = 0;
	struct sockaddr_in cladr;
	int cllen = sizeof(sockaddr_in);
	

	/* Setting the PID's correctly. */

	Own_ID = getpid();
	Par_ID = getppid();

	/* Closing the Req_T_Pipe's read-port. */

	DBG(cerr<<"Run_T_Fill(): Closing Req_T_Pipe's read port."<<endl);
	
	close(Req_T_Pipe[0]);

	/* This is the table-filler. */
	DBG(cerr<<"\tLooping now."<<endl);

	for( ;; )
	{
		memset(&cladr, 0, sizeof(cladr));
		DBG(cerr<<"Run_T_Fill(): Address now: "<<cladr.sin_addr.s_addr<<endl);
		Req_Buf = recv_req(Req_Sock, &cladr, &cllen);
		
		if(Req_Buf == NULL)
			DBG(cerr<<"Run_T_Fill(): Request failed."<<endl);
		else
		{
			DBG(cerr<<"Run_T_Fill(): cladr: "<<cladr.sin_addr.s_addr<<endl);
			Note_and_Wakeup(&cladr,Req_Buf);
		}
	}
	return NO_ERROR;
}

int 
C_ArrayServ::Note_and_Wakeup(sockaddr_in *Claddr, Buffer_Info *Request)
{
	/* Note and wakeup notifies the request in the request table.
	   Since the request table is a pipe, the server is awoken 
	   automagically when new data resides in the pipe. We are stopping, 
	   when the pipe is full.

	   The Buf value describes a Buffer_Info struct as returned from
	   recv_req(int). A shared memory segment is allocated and the
	   data pointed to by Buf->address is copied into it. Then the
	   Buf structures size value is copied into the Req_Table and the
	   shmkey is bummed into the appropriate field, so that the 
	   server process on the other side can attach the shm. */

	struct timeval tv;
	key_t shmkey;
	int shmid;
	char *New_Buf;
	Req_Table Req_Line;

	DBG(cerr<<"Note_and_Wakeup"<<endl);

	shmkey=get_key();
	
	if((shmid = shmget(shmkey, Request->size, 0666|IPC_CREAT)) < 0)
	{
		perror("N_a_W: Error at shmget");
		exit(RET_SHMGET_ARS);
	}
	
	if( (New_Buf = (char *) shmat(shmid, NULL, 0)) == (char *) -1)
	{
		perror("N_a_W: Error at shmat");
		exit(RET_SHMATT_ARS);
	}

	memcpy(New_Buf, Request->address, Request->size);
	Req_Line.sock = Req_Sock;
	Req_Line.address = *Claddr;	
	Req_Line.Req_Buf.size = Request->size;
	Req_Line.Req_Buf.key = shmkey;
	Req_Line.Req_Buf.bstruct = Request->bstruct;
	Req_Line.Req_Buf.address = New_Buf;
	gettimeofday(&tv,NULL);
	Req_Line.ins_time = tv;
	Req_Line.valid = IS_VALID;

	free(Request->address);             /* Request->address was calloc'ed. */
	delete(Request);                    /* Request was new'ed. */

	if(shmdt(New_Buf) < 0)
	{
		perror("N_a_W: Error at shmdt");
		exit(RET_SHMDTH_ARS);
	}

	/* Write the Req_Line to the Req_T_Pipe. */

	if( write(Req_T_Pipe[1], &Req_Line, sizeof(Req_Table)) 
	   != sizeof(Req_Table))
	{
		perror("N_a_W: Error at write");
		exit(RET_WRPIPE_ARS);
	}
	
	return NO_ERROR;
}

int
C_ArrayServ::Run_Server()
{
	/* This is the Server. */

	Req_Table Req_Line;

	/* Setting the PID's correctly. */

	Own_ID = getpid();
	Par_ID = getppid();

	/* Closing Req_T_Pipe's write port. */

	DBG(cerr<<"Run_Server(): Closing Req_T_Pipe's write port."<<endl);

	close(Req_T_Pipe[1]);
	
	while(1)
	{
		/* Read from the Req_T_Pipe. [Blocking when nothing to do.] */

		DBG(cerr<<"Run_Server(): Reading from Req_T_Pipe."<<endl);

		if(read(Req_T_Pipe[0], &Req_Line, sizeof(Req_Table)) 
		   < (signed) sizeof(Req_Table))
		{
			perror("Run_Server(): Error at read");
			exit(RET_RDPIPE_ARS);
		}
		
		/* Read returned; we got something to do. */

		/* First attaching the shm segment, where the data resides. */

		DBG(cerr<<"Run_Server(): getting shm"<<endl); 
		if( Req_Line.valid == IS_VALID)
		{
			if( (Shm_ID = shmget(Req_Line.Req_Buf.key,
								Req_Line.Req_Buf.size,0)) < 0)
			{
				perror("Run_Server(): Error at shmget");
				exit(RET_SHMGET_ARS);
			}
			
			DBG(cerr<<"Run_Server(): attaching shm"<<endl); 
			if( (Req_Line.Req_Buf.address = (char *) shmat(Shm_ID,NULL,0)) < 0)
			{
				perror("Run_Server(): Error at shmat");
				exit(RET_SHMATT_ARS);
			}
			
			if(Process_Request(&(Req_Line.address),&(Req_Line.Req_Buf)) < 0)
			{
				cerr<<"Run_Server(): Process_Request returned with -1."
					<<endl<<"\tSeems to be an internal Error."<<endl;
			}

			DBG(cerr<<"Run_Server(): removing shm"<<endl);
			if(shmdt(Req_Line.Req_Buf.address) < 0)
			{
				perror("Run_Server(): Error at shmdt");
				exit(RET_SHMDTH_ARS);
			}
			
			if(shmctl(Shm_ID, IPC_RMID, NULL)<0)
			{
				perror("Run_Server(): Error at shmctl");
				exit(RET_SHMCTL_ARS);
			}
			
			Req_Line.valid = IS_INVALID;

		}
	}
	return NO_ERROR;
}


int	
C_ArrayServ::Process_Request(sockaddr_in *Claddr, Buffer_Info *Req_Buf)
{
	/* Now we will start to do real processing on the data presented by
	   the client. There are different things to do here. We have to
	   evaluate the kind of request and according to this we must do
	   the appropriate actions. */

	/* Strategy is the following:
	   
	   * if there is more than one RS, only the first is taken, the 
	     others ignored, no error generated.

	   * if there is no RS, NULL is returned, DBG(cerr) done.

	   * the structures may have any arbitrary order.
	   
	   * all RDSt's are taken and no limit for this.
	   
	   * NISt or AS within the buffer are ignored without error.

	   * if the parser comes to an unmagic number at a magic position,
	     a NULL is returned and an error written with DBG.
		 This may occure when some strange sizes are inside the buffer
		 which don't match the current structure sizes. 
		 Nevertheless, this should _never_ occur.

	*/

	RS *Req = NULL;
	RDSt **RDat = NULL;
	int RDat_Cnt = 0;
	int size = 0;
	char *address = NULL;
	int Error= NO_ERROR;
	AS *Ans = (AS *) new AS;
	RDSt *ADat = NULL;
	char *setl_buf = 0;
	char *getl_buf;
	int getl_size;

	/* First filling in the AS-template. */

	DBG(cerr<<"Process_Request(): Started ..."<<endl);
	
	if(!Ans) return -1;

	Ans->Magic = MAG_AS;
	Ans->ReqID = 0;
	Ans->RSID = 0;
	Ans->Error = NO_ERROR;
	Ans->RSFollow = FLW_NO;
	Ans->NIStFollow = FLW_NO;

	/* ... then we check if the request_buffer is a buffer at all. */

	DBG(cerr<<"Process_Request(): filled in AS-template."<<endl);
	
	if(Req_Buf)
	{
		/* ... then if we've got a fine bstruct. */

		DBG(cerr<<"Process_Request(): Got Req_Buf non zero."<<endl);

		if(Req_Buf->bstruct != NO_BUF)
		{
			/* ... then let's search for some structs. */
			
			DBG(cerr<<"Process_Request(): Got fine bstruct: "
				<<Req_Buf->bstruct<<endl);
			
			if((size = Req_Buf->size) <= 0) return -1;
			address = Req_Buf->address;

			do
			{
				switch(*((int *)address))
				{
				  case MAG_RS:
					DBG(cerr<<"Process_Request(): Got RS!"<<endl);
					if(Req == NULL)
						Req = (RS *) address;
					else DBG(cerr<<"Process_Request(): Got _multiple_ RS!"
							 <<" Ignoring!"<<endl);
					address += sizeof(RS);
					size -= sizeof(RS);
					break;
				  case MAG_RDSt:
					DBG(cerr<<"Process_Request(): Got RDSt!"<<endl);
					RDat = (RDSt **) realloc(RDat, sizeof(RDSt*)*(++RDat_Cnt));
					if(RDat == NULL) 
					{
						DBG(cerr<<"Process_Request(): Error at realloc"<<endl);
						return -1;
					}
					RDat[RDat_Cnt-1] = (RDSt *) address;
					address += sizeof(RDSt);
					size -= sizeof(RDSt);
					break;
				  case MAG_NISt:
					DBG(cerr<<"Process_Request(): Got NISt! Ignoring."<<endl);
					address += sizeof(NISt);
					size -= sizeof(NISt);
					break;
				  case MAG_AS:
					DBG(cerr<<"Process_Request(): Got AS! Ignoring."<<endl);
					address += sizeof(AS);
					size -= sizeof(AS);
					break;
				  default:
					DBG(cerr<<"Process_Request(): Unaligned Structure!"<<endl);
					return -1;
					break;
				}
			}
			while(size > 0);

			/* Now we have the structures where they should be. */

			if(Req == NULL) 
			{
				DBG(cerr<<"Process_Request(): Buffer came without RS!"<<endl);
				return -1;
			}
			
			/* We have an RS, fine! */

			Ans->ReqID = Req->ReqID;
			Ans->RSID = Req->RSID;


			switch(Req->ReqID)
			{
			  case REQ_INITA:
				/* This is an INIT-Request. We take it here as an create()
				   or ncreate() to the appropriate Array Manager. 
				   The ObjectID-field of the RS gives us informations about
				   the number of elements, we brought with us. 
				   There's much intelligence needed to determine what we 
				   have and what we need.*/

				DBG(cerr<<"Process_Request(): Got REQ_INITA!"<<endl);

				Ans->Error = Store.ncreate(Req->ObjectID, &Error);
				
				if(Error != NO_ERROR)
				{
					DBG(cerr<<"Process_Request(): Error "<<Error
						<<" at ncreate"<<endl);
					Ans->Error = -Error;
					break;
				}
				
				if(Req->ObjectID > 0)
				{
					/* Now we read the attached RDSt's. */
					DBG(cerr<<"Process_Request(): Reading RDSt's ..."<<endl);

					if(RDat != NULL)
					{
						int i,j,k;

						DBG(cerr<<"Process_Request(): Reading (R) and "
							<<" writing (W) ..."<<endl<<"\t");

						for(i=0,k=1; 
							i<((Req->ObjectID + MAXREQEL - 1)/MAXREQEL);
							i++)
						{
							if(RDat[i] != NULL)
							{
								DBG(cerr<<"R ");
								for(j=0;(j<RDat[i]->DVSize)&&(Error==NO_ERROR);
									j++,k++)
								{
									DBG(cerr<<"W ");
									Error=Store.write(Ans->Error,k,
													  RDat[i]->DataVector[j]);
								}
								if(Error!=NO_ERROR)
								{
									Store.destroy(Ans->Error);
									Ans->Error = -Error;
									break;
								}
								
							}
							else break;
						}
						DBG(cerr<<endl);
					}
					else break;
				}
				break;
			  case REQ_QUERY:

				/* A REQ_QUERY is here interpreted as a 'Get_List()' to
				   the Array Manager. 
				   All possibly connected RDSt's are ignored, since there
				   is no use for them. There's an AS together with several
				   RDSt's returned. */
				
				DBG(cerr<<"Process_Request(): Got REQ_QUERY!"<<endl);

				getl_size=Store.Get_List(Req->AttList, &getl_buf);
				
				if(getl_size < 0)
				{
					DBG(cerr<<"Process_Request(): Error "<<getl_size
						<<" at Store.Get_List()"<<endl);
					Ans->Error = -getl_size;
					if(getl_buf) free(getl_buf);
					break;
				}
				else
				{
					/* getl_size contains the number of Elements read
					   in getl_buf. Allocating ADat-Structs for return. */
				
					int i,j,k = (getl_size + MAXREQEL - 1) / MAXREQEL;
					
					if((ADat=(RDSt *) calloc(k, sizeof(RDSt))) == NULL)
					{
						free(getl_buf);
						Ans->Error = -ERR_MEMORY_NWK;
						break;
					}

					DBG(cerr<<"Process_Request(): Preparing (P) and copying "
						<<"(C)"<<endl<<"\t");
					
					for(i = 0; i < k; i++)
					{
						DBG(cerr<<"P ");
						ADat[i].Magic = MAG_RDSt;
						ADat[i].RSID = Req->RSID;
						ADat[i].RSSID = i;
						ADat[i].RSFollow = k-i-1;
						ADat[i].DVSize = ((i<(k-1))?MAXREQEL:
										   (getl_size - (i*MAXREQEL)));

						for(j = 0; j < ADat[i].DVSize; j++)
						{
							DBG(cerr<<"C ");
							memcpy(ADat[i].DataVector[j], 
								   getl_buf + ((i*MAXREQEL)+(j*DVLEN)), DVLEN);
						}
						
					}
					DBG(cerr<<endl);
					Ans->RSFollow = k;
					free(getl_buf);
				}
				break;

			  case REQ_STORE:
				/* A REQ_STORE is here interpreted as a 'Set_List()' to the
				   Array Manager. We expect the leading RS to contain a lot
				   of useful information, especially:
				         * AttList to contain the Array the data belongs
						   to. If unset we return an Error. [In future 
						   possible call the methods for REQ_INIT instead.]

						   */

				DBG(cerr<<"Process_Request(): Got REQ_STORE!"<<endl);
				
				if(RDat != NULL)
				{
					int i,j,k;
				
					DBG(cerr<<"Process_Request(): Preparing (P) and copying "
						<<"(C)"<<endl<<"\t");

					for(i = 0, k = 0; RDat[i] != NULL; i++)
					{
						DBG(cerr<<"P ");
						if((setl_buf=(char *) realloc(setl_buf, 
													  RDat[i]->DVSize * DVLEN))
						   == NULL)
						{
							Ans->Error = -ERR_MEMORY_NWK;
							break;
						}
						
						for(j = 0; j < RDat[i]->DVSize; j++,k++)
						{
							DBG(cerr<<"C ");
							memcpy(setl_buf + ((i*MAXREQEL)+(j*DVLEN)),
								   RDat[i]->DataVector[j], DVLEN);
						}
						
					}
					DBG(cerr<<endl);
					Ans->Error = -Store.Set_List(Req->AttList, setl_buf, k);

					free(setl_buf);
				}
				break;

			  case REQ_DESTR:
				/* A REQ_DESTR is here treated as a destroy() to the 
				   Array Manager. It expects the AttList-field of the
				   RS set to the number of the array subject to destroy. */
				
				DBG(cerr<<"Process_Request(): Got REQ_DESTR!"<<endl);
				
				Ans->Error = -Store.destroy(Req->AttList);
				break;
			  default:
				DBG(cerr<<"Process_Request(): Got unknown request "
					<<Req->ReqID<<endl);
					Ans->Error = -ERR_UNKREQ_NWK;
				break;
			}
		}
	}

	/* Now we must send the return-buffer. */

	DBG(cerr<<"Process_Request(): Sending Ret_Buf."<<endl);
	DBG(cerr<<"Process_Request(): Sending AS."<<endl);
	if(send_(Req_Sock, Ans, 0, Claddr, sizeof(sockaddr_in)) < 0)
		perror("Error at AS-send");
	if(ADat)
	{
		DBG(cerr<<"Process_Request(): Have ADat."<<endl);
		for(int i = 0; i < Ans->RSFollow; i++)
			if(send_(Req_Sock, &(ADat[i]), 0, Claddr, sizeof(sockaddr_in)) < 0)
				perror("Error at RDSt-Send.");
	}
	delete Ans;
	if(ADat) free(ADat);
	if(RDat) free(RDat);
	DBG(cerr<<"Process_Request(): Returning ..."<<endl);
	return NO_ERROR;
}

C_ArrayServ::~C_ArrayServ()
{
}

void
usage(char *name)
{
	cerr<<"usage: "<<name<<" [-p port_nr [-s [serv_name]]]"<<endl;
	exit(-1);
}

int
main(int argc, char **argv)
{
	char *p_name=NULL;
	char *ch;
	char *port_chars = NULL;
	char *serv_name = NULL;
	short port = -1;
	C_ArrayServ *C_AS;

	/* Arguments:
	     -p <Port_Nr>
		 -s <Serv_Name>
    */

	p_name=argv[0];
	if(argc <= 1) usage(p_name);
	while((--argc > 0) && ((*++argv)[0] == '-'))
	{
		for(ch = argv[0]+1; *ch != '\0'; ch++)
		{
			switch(*ch)
			{
			  case 'p':
				if((--argc <= 0) || ((*++argv)[0] == '-'))
				{
					cerr<<"-p requires a parameter."<<endl;
					usage(p_name);
				}
				else port_chars = *argv;
				break;
			  case 's':
				if((--argc <= 0) || ((*++argv)[0] == '-'))
					serv_name = p_name;
				else
					serv_name = *argv;
				break;
				
			  default:
				cerr<<p_name<<": Unknown command line option "<<*ch<<endl;
				usage(p_name);
				break;
			}
		}
	}

	if(port_chars) port = (short) atoi(port_chars);

	if((C_AS = (C_ArrayServ *) new C_ArrayServ(serv_name, port)) == NULL)
	{
		perror("Main: Error at new");
		exit(-1);
	}

	C_AS->Run();

	return 0;
}
