/* test_cli_udp_1.cc
   Author: Marko Meyer
   Date:  95/12/03
   Time-stamp: <96/01/18 10:37:29 mme>

   Test - client for the Array Server.

   	BACKNET - A library for simulating neural BACKPROPAGATION-Networks
    Copyright (C) 1995	Marko Meyer

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


	If you have any suggestions, ideas or comments regarding this library
	feel free to contact me:

	Email:	mme@pub.th-zwickau.de
	Ordinary Mail:	Marko Meyer
					Teichstrasse 27
					D-08289 Schneeberg
					Germany

*/

/* Version 1 cooperates with Version 3 of the server. 
   It should have a little more testing facility.
   If you received this file with a distribution of BACKNET, please ignore.
   This is just for me testing the Array Server.
*/

#include <stdio.h>
#include <netbacknet.h>
#include <unistd.h>
#include <sys/time.h>

#define MAXERAND 100

char name[] = "Test Client v0.1 UDP";
char copyright[] = "Copyright (C) 1995,1996	Marko Meyer";

void
usage(char *name)
{
	cout<<name<<" [-h [host_addr]] [-p port_nr] [-s [serv_name]]"<<endl;
	exit(-1);
}

int *
test_create(int sock, sockaddr_in *ServAddr, int count)
{
	/* The create-test is not too difficult to understand: There's a 
	   random number of data for each create. A medium access time and
	   a medium access time per created element is written to the 
	   terminal. The test takes 'count' steps before it terminates. */

	float f_MAT = 0.0;
	float f_MATpE = 0.0;
	float tmadd = 0.0;
	int i_RandCnt = 0;
	int i;
	int i_Try = 1;
	int len = sizeof(sockaddr_in);
	timeval tv,tv2, timeout = {2,0};
	RDSt RDat;
	RS RStruct;
	AS AStruct;
	int *ret_list;

	cout<<"Creating "<<count<<" arrays ..."<<endl;

	if((ret_list = (int *) calloc(count, sizeof(int))) == NULL)
	{
		perror("Error at calloc");
		exit(-1);
	}
	
	gettimeofday(&tv,NULL);
	
	srand(tv.tv_usec);

	do
	{
		/* First draw a random number between 1 and MAXERAND/MAXREQEL. */
		
		i_RandCnt = (int) ((float)(rand() * 1.0 / RAND_MAX ) * MAXERAND) + 1;
		
		RStruct.Magic = MAG_RS;
		RStruct.ReqID = REQ_INITA;
		RStruct.RSID = i_Try;
		RStruct.ObjectID = i_RandCnt * MAXREQEL;
		RStruct.AttList = 0;
		RStruct.RSFollow = i_RandCnt;
		RStruct.StFollow = FLW_NO;

		if(send_(sock, &RStruct, 0, ServAddr, len) < 0)
		{
			perror("Error at send_(RS)");
			exit(-1);
		}
		
		
		for(i = 0; i < i_RandCnt; i++)
		{
			/* filling the structures ... */
			
			RDat.Magic = MAG_RDSt;
			RDat.RSID = i_Try;
			RDat.RSSID = i;
			RDat.RSFollow = i_RandCnt - i - 1;
			RDat.DVSize = MAXREQEL;
			
			for(int j = 0; j < RDat.DVSize; j++)
				sprintf(RDat.DataVector[j],"%19d",i);
			
			if(send_(sock, &RDat, 0, ServAddr, len) < 0)
			{
				perror("Error at send_(RDSt)");
				exit(-1);
			}
		}

		/* TIME measure 1 */
		gettimeofday(&tv,NULL);

		/* Receive the AS. */

		SETTIMEOUT(timeout,5,0);
		if(recv_(sock, &AStruct, 0, ServAddr, &len, &timeout) < 0)
		{
			perror("Error at recv_(AS)");
			exit(-1);
		}
		/* TIME measure 2 */

		gettimeofday(&tv2,NULL);
		
		if(AStruct.Error > 0)
		{
			tmadd = ((float)tv2.tv_sec - (float)tv.tv_sec - 1.) * 1.E+06 
				+ (float)tv2.tv_usec - (float)tv.tv_usec + 1.E+06;
			f_MAT += tmadd;
			f_MATpE += tmadd / ((float)i_RandCnt * (float)MAXREQEL);
			ret_list[i_Try-1] = AStruct.Error;
			i_Try++;
			cout<<".";
			cout.flush();
		}		
	}
	while((AStruct.Error > 0) && (i_Try <= count));

	if(i_Try > 1) i_Try--;
	
	f_MAT/=i_Try;
	f_MATpE/=i_Try;

	cout<<endl<<endl<<"Results:"<<endl;
	cout<<"\tCreated "<<i_Try<<" Arrays with:"<<endl
		<<"\tMedium Access Time: "<<f_MAT<<" s\t ... per Element: "<<f_MATpE
			<<" s."<<endl;

	return ret_list;
}

void
test_destroy(int sock, sockaddr_in *ServAddr, int count, int *deslist = NULL)
{
	/* This routine tests the destroy() call to the Array Server. 
	   Therefore it uses either a deslist of already present lists with
	   count specifying the amount of lists to destroy or it first
	   creates a set of count lists (with test_create) which are then 
	   destroyed. */

	float f_MAT = 0.0;
	RS RStruct;
	AS AStruct;
	timeval tv1, tv2, timeout = {2,0};
	int len = sizeof(sockaddr_in);

	cout<<"Destroying "<<count<<" lists ..."<<endl;
	if(!deslist)
		deslist = test_create(sock, ServAddr, count);

	RStruct.RSFollow = FLW_NO;
	RStruct.StFollow = FLW_NO;
	
	for(int i = 0; i < count; i++)
	{
		if(deslist[i])
		{
			RStruct.Magic = MAG_RS;
			RStruct.ReqID = REQ_DESTR;
			RStruct.RSID = i + 1;
			RStruct.AttList = deslist[i];
			
			if(send_(sock, &RStruct, 0, ServAddr, len) < 0)
			{
				perror("Error at send_(RS)");
				exit(-1);
			}
			
			/* TIME measure 1 */
			
			gettimeofday(&tv1,NULL);

			SETTIMEOUT(timeout,5,0);
			if(recv_(sock, &AStruct, 0, ServAddr, &len, &timeout) < 0)
			{
				perror("Error at recv_(AS)");
				exit(-1);
			}
			/*TIME measure 2*/
			gettimeofday(&tv2,NULL);
			
			if(AStruct.Error == NO_ERROR)
			{
				f_MAT += ((float)tv2.tv_sec - (float)tv1.tv_sec - 1.) * 1.E+06 
					+ (float)tv2.tv_usec - (float)tv1.tv_usec + 1.E+06; 
				cout<<".";
				cout.flush();
			}
			else
				cout<<"Error at destroying array nr. "<<deslist[i]<<": "
					<<AStruct.Error<<endl;
		}
	}

	if(count > 0) f_MAT/=count;
	cout<<endl<<endl<<"Results:"<<endl;
	cout<<"\tDestroyed "<<count<<" Arrays with:"<<endl
		<<"\tMedium Access Time: "<<f_MAT<<" s."<<endl;
}

void
test_invalid_amount(int sock, sockaddr_in *ServAddr)
{
	/* This is for testing timeout capabilities. The server gets one too less
	   RDSt's. It should have the timeout and then return to its correct
	   work. So we do three things:

	   1. Send the invalid data.
	   2. Send a normal data set to create a list.
	   3. Send a normal destroy to destroy the by (2) created list.
	*/

	int len = sizeof(sockaddr_in), i;
	struct timeval timeout = {2,0};
							  
	cout<<"Testing timeout handling."<<endl;

	/* Preparing the incorrect data: */

	struct RS ThisRS =

	{
		MAG_RS,                 /* Magic Number for identification */
		1,                      /* An arbitrary RSID. */
		REQ_INITA,              /* A Request INITArray. */
		MAXREQEL+2,             /* The amount of data. */
		0,                      /* No list attached yet. */
		2,                      /* Number of RDSt's to send. */
		FLW_NO                  /* No St's to attach. */
		};

	struct RDSt OneRDSt;

	OneRDSt.Magic = MAG_RDSt;  /* Magic number of RDSt's ... */
	OneRDSt.RSID =	1;         /* The same RSID like in the RS above ... */
	OneRDSt.RSSID =	1;         /* This is the first RDSt in RSID 1 ... */
	OneRDSt.RSFollow = 1;      /* One RDSt should follow (but will never do)*/
	OneRDSt.DVSize = MAXREQEL; /* This thing is full of nice data ... */

	for(i = 0; i < OneRDSt.DVSize ; i++)
		sprintf(OneRDSt.DataVector[i], " 1 !");

	cout<<"Sending incorrect data ..."<<endl;

	if(send_(sock, &ThisRS, 0, ServAddr,len) < 0)
	{
		perror("Error at send_(RS)");
		exit(-1);
	}
	
	if(send_(sock, &OneRDSt, 0, ServAddr, len) < 0)
	{
		perror("Error at send_(RDSt)");
		exit(-1);
	}

	/* Now the server should wait for the specified timeout-time and then
	   return to it's usual service. */
	/* We should receive a timeout-AS! */

	struct AS ToutAS;

	SETTIMEOUT(timeout,5,0);
	if(recv_(sock, &ToutAS, 0, ServAddr, &len, &timeout) < 0)
	{
		perror("Error at recv_(AS)");
		exit(-1);
	}
	
	if(ToutAS.Error == -ERR_TIMOUT_NWK)
		cout<<endl<<"Received timeout-message, theoretically resending."<<endl;
	else
		cout<<endl<<"Received unexpected value "<<ToutAS.Error
			<<" in returned AS."<<endl;
	
	/* Now sending correct data, to verify that the Server did the 
	   Right Thing: to interrupt the old transmission and wait for a
	   new connection. */

	ThisRS.Magic = MAG_RS;     /* The magic number */
	ThisRS.RSID = 2;          /* An arbitrary RSID */
	ThisRS.ReqID = REQ_INITA;  /* The Request InitArray */
	ThisRS.ObjectID = MAXREQEL; /* For safety: only one RDSt needed. */
	ThisRS.AttList = 0;          /* No list attached yet. */
	ThisRS.RSFollow = 1;          /* One RDSt to send. */
	ThisRS.StFollow = FLW_NO;      /* No St to send. */
	
	OneRDSt.Magic = MAG_RDSt;  /* Magic number of RDSt's ... */
	OneRDSt.RSID =	2;         /* The same RSID like in the RS above ... */
	OneRDSt.RSSID =	1;         /* This is the first RDSt in RSID 1 ... */
	OneRDSt.RSFollow = 0;      /* One RDSt should follow (but will never do)*/
	OneRDSt.DVSize = MAXREQEL; /* This thing is full of nice data ... */

	for(i = 0; i < OneRDSt.DVSize ; i++)
		sprintf(OneRDSt.DataVector[i], " 1 !");

	cout<<"Sending correct data to verify server-timeout."<<endl;

	if(send_(sock, &ThisRS, 0, ServAddr, len) < 0)
	{
		perror("Error at send_(RS)");
		exit(-1);
	}
	
	if(send_(sock, &OneRDSt, 0, ServAddr, len) < 0)
	{
		perror("Error at send_(RDSt)");
		exit(-1);
	}
	
	/* Now the server should do his work and send us a correct answer. */

	SETTIMEOUT(timeout,5,0);
	if(recv_(sock, &ToutAS, 0, ServAddr, &len, &timeout) < 0)
	{
		perror("Error at recv_(AS)");
		exit(-1);
	}
	
	if(ToutAS.Error > 0)	
	{
		cout<<endl<<"Server created list correctly. Therefore timeout is ok."
			<<endl;
		
		/* In this case we can destroy it. */

		ThisRS.Magic = MAG_RS;
		ThisRS.RSID = 3;
		ThisRS.ReqID = REQ_DESTR;
		ThisRS.ObjectID = 0;
		ThisRS.AttList = ToutAS.Error;
		ThisRS.RSFollow = FLW_NO;
		ThisRS.StFollow = FLW_NO;

		if(send_(sock, &ThisRS, 0, ServAddr, len) < 0)
		{
			perror("Error at send_(RS)");
			exit(-1);
		}
		
		/* Now we look for the answer: */

		SETTIMEOUT(timeout,5,0);
		if(recv_(sock, &ToutAS, 0, ServAddr, &len, &timeout) < 0)
		{
			perror("Error at recv_(AS)");
			exit(-1);
		}
		
		if(ToutAS.Error == NO_ERROR)
			cout<<endl<<"Ok. List destroyed with no error."<<endl;
		else
			cout<<endl<<"Answer contained unexpected value "<<ToutAS.Error
				<<" in field Error."<<endl;
	}
	else
		cout<<endl<<"Answer contained unexpected value "<<ToutAS.Error
			<<" as Error."<<endl;
	
}		
	

int
main(int argc, char **argv)
{
	int sock;
	char *p_name;
	char *ch;
	char *host_addr = NULL;
	short port = 0;
	char *serv_name = NULL;
	char *port_chars = NULL;
	int *deslist;
	
	sockaddr_in serv_address;

	cout<<NAME_ID<<" "<<VERSION_NR<<" "<<name<<endl<<copyright<<endl;

	p_name = argv[0];
	if(argc <= 1) usage(p_name);

	while((--argc > 0) && ((*++argv)[0] == '-'))
	{
		for(ch = argv[0]+1; *ch != '\0'; ch++)
		{
			switch(*ch)
			{
			  case 'h':
				if((--argc <=0) || ((*++argv)[0] == '-'))
					host_addr = "127.0.0.1";
				else
					host_addr = *argv;
				break;
			  case 'p':
				if((--argc <= 0) || ((*++argv)[0] == '-'))
				{
					cerr<<"-p requires a parameter."<<endl;
					usage(p_name);
				}
				else
					port_chars = *argv;
				break;
			  case 's':
				if((--argc <= 0) || ((*++argv)[0] == '-'))
					serv_name = p_name;
				else
					serv_name = *argv;
				break;
				
			  default:
				cerr<<p_name<<": Unknown command line option "<<*ch<<endl;
				usage(p_name);
				break;
			}
		}
	}

	if(port_chars) port = (short) atoi(port_chars);
	sock = set_up_udp(NULL, 0);
	if(sock > 0)
	{
		
		serv_address.sin_family = AF_INET;
		serv_address.sin_addr.s_addr = inet_addr(host_addr);
		serv_address.sin_port = htons(port);
		
		deslist = test_create(sock, &serv_address, 100);
		
		test_destroy(sock, &serv_address, 100, deslist);

		test_invalid_amount(sock, &serv_address);
		
		close(sock);
	}
}
