/* layer.cc

   Realizes functionality of a neural-network-layer.

   Author: Marko Meyer
   Date:   29.03.1995 (creation)
           22.07.1995 (last modification)

	BACKNET - A library for simulating neural BACKPROPAGATION-Networks
    Copyright (C) 1995	Marko Meyer

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


	If you have any suggestions, ideas or comments regarding this library
	feel free to contact me:

	Email:	mme@pub.th-zwickau.de
	Ordinary Mail:	Marko Meyer
					Teichstrasse 27
					D-08289 Schneeberg
					F.R.G.

*/

#include <layer.h>

C_Layer::C_Layer(C_List<ELEMTYPE> *C_NetLists,
				                          /*pointer to ListManager<ELEMTYPE>*/
				 int i_NeuCnt,            /*number of neurons in curr. layer*/
				 C_Layer *C_LastLay,          /*pointer to last layer*/
				 int i_InpList,             /*number of first inputlist*/
				 float f_LearnRate,           /*learning rate*/
				 int i_BiasQuest)
{
    S_Layer_Param *S_LastLayData=NULL;
	int i_Error=NO_ERROR;
	
	DBG(cerr<<"Layer: Entering constructor."<<endl
		/*<<"\tLayer-Net-ID: "<<i_LayNetID -- rsc 95/11/29 */
		<<"\tNeuron-Count: "<<i_NeuCnt<<endl
		<<"\tLearning Rate: "<<f_LearnRate<<endl);
	
    if(C_LastLay)   /*if layer is not first layer*/
	{
		/*transfer data of previous layer*/
		S_LastLayData=C_LastLay->hand_lay_data();
		
		/*the input list for the current layer is the output list of the
		  previous layer*/
		S_LayerData.i_Input_List=S_LastLayData->i_Output_List;
		
		/*the list to transfer backpropagated errors to is the error list
		  of the previous layer*/
		S_LayerData.i_Last_Error_List=S_LastLayData->i_Error_List;

		/*increment the layer nr. and store it*/
		S_LayerData.i_Layer_Nr=S_LastLayData->i_Layer_Nr+1;
	}
	else      /*the layer is the first layer*/
	{
		/*the input list is given in the function-parameters*/
		S_LayerData.i_Input_List=i_InpList;
		
		/*there's no error list where errors should be transferred to*/
		S_LayerData.i_Last_Error_List=0;

		/*the layer nr is always 1 if the layer is the first layer!*/
		S_LayerData.i_Layer_Nr=1;
	}
	if((C_ListMan=C_NetLists)!=NULL) 
	{
		DBG(cerr<<"Layer: ListMan is present."<<endl);
		/*the NeuroStore must be invoked here*/
		C_NeuStor=(C_List<C_Neuron*>*)new C_List<C_Neuron*>;
		if(C_NeuStor)
		{
			DBG(cerr<<"Layer: NeuroStore created."<<endl);
			/*The NeuroStore is from now on available. First create the 
			  appropriate list...*/
			i_NeuList=C_NeuStor->create(NULL,&i_Error);
			if(i_Error==NO_ERROR)
			{
				/*Everything's okay, so let the show go on ...*/
				if((S_LayerData.i_Output_List=Create_List(i_NeuCnt,0))==-1) 
					i_Error=ERR_CRLIST_LAY;
				if((S_LayerData.i_Error_List=Create_List(i_NeuCnt,0))==-1)
					i_Error=ERR_CRLIST_LAY;
				if(i_Error==NO_ERROR)
				{
					S_LayerData.i_Neuro_Cnt=i_NeuCnt;
					S_LayerData.f_Learn_Rate=f_LearnRate;
					if(S_LastLayData)
					{
						inCount=S_LastLayData->i_Neuro_Cnt;
						i_Bias=i_BiasQuest;
					}
					else
					{
						inCount=i_NeuCnt;
						i_Bias=0;
					}
				}
			}
		}
		else i_Error=ERR_NSINIT_LAY;
	}
	else i_Error=ERR_NOLMAN_LAY;
	/*that's all*/
	DBG(cerr<<"Layer: Leaving constructor with Error: "<<i_Error<<endl);
}

int 
C_Layer::Init(S_Fermi_Param  *S_Fermi_Data, /*pointer to Fermi-data*/
			  int i_InitMode,              /*init-mode for neurons*/
			  ifstream *I_GWI_File)            /*file with stored weights*/
{
	S_FermiData=S_Fermi_Data;
	/*init for all the neurons*/
	return Init_Neurons(i_InitMode, i_Bias, inCount, I_GWI_File);
}

int 
C_Layer::Action(int i_action_nr)
{
	int i_Error,i;
	C_Neuron *C_Tmp_Neuron; /*This is needed for the function calls.*/
	
	i_Error=NO_ERROR;
	
	DBG(cerr<<"Layer: Entering Action ..."<<endl);
    /*is there a Neuro-Store available?*/
	if(C_NeuStor!=NULL)
	{
		for(i=1;i<=S_LayerData.i_Neuro_Cnt;i++)
		{
			/*First determine the current Neuron ...*/
			i_Error=C_NeuStor->read(i_NeuList,i,&C_Tmp_Neuron);
			if(i_Error==NO_ERROR)
			{
				/*...either call neuron's function Recall() or Learn()...*/
				switch(i_action_nr)
				{
				  case ACTN_RECAL:
					i_Error=C_Tmp_Neuron->Recall();
					break;
				  case ACTN_LEARN:
					i_Error=C_Tmp_Neuron->Learn();
					break;
				  default:
					i_Error=ERR_WRACTN_LAY;
				}
			}
			else return i_Error;
		}
	}
	else return ERR_NONSTR_LAY;
	DBG(cerr<<"Layer: Leaving Action with Error: "<<i_Error<<endl);
	return i_Error;
}

int C_Layer::Save(ofstream *O_GWI_Save)
{
    int	i_Error;
	C_Neuron *C_ActNeuron;

	i_Error=NO_ERROR;
	DBG(cerr<<"Layer: Entering Save ..."<<endl);

	for(int i=1;i<=S_LayerData.i_Neuro_Cnt;i++)
	{
		i_Error=C_NeuStor->read(i_NeuList,i,&C_ActNeuron);
		if(i_Error==NO_ERROR)
			i_Error=C_ActNeuron->Save(O_GWI_Save);
		else return i_Error;
	}
	
	return i_Error;
}

 C_Layer::~C_Layer()
{
	Destroy_Neurons();
	Destroy_List(S_LayerData.i_Output_List);
	Destroy_List(S_LayerData.i_Error_List);
   	delete C_NeuStor;
}



S_Layer_Param *C_Layer::hand_lay_data()
{
	/*This function is only for sending any customer the layerdata.
	  (You know: Data is owned by the object and has to be released by
	  objects methods.)*/

	return &S_LayerData;
}

int C_Layer::Create_List(int i_Count,ELEMTYPE Cont)
{
	int i_ListIDX;
	int i;
	int i_Error;
	
	i_Error=NO_ERROR;

	DBG(cerr<<"Layer: Entering Create_List ... "<<endl);
	/*first creating*/
	i_ListIDX=C_ListMan->create(Cont,&i_Error);

	/*then appending i_Count - 1 elements*/
	for(i=1;(i<i_Count)&&(i_Error==NO_ERROR);\
		i_Error=C_ListMan->append(i_ListIDX,Cont),i++);
	
	DBG(cerr<<"Layer: Leaving Create_List with Error: "<<i_Error<<endl);
	/*in case of an Error -1 is returned, otherwise the ListIDX*/
	return (i_Error==NO_ERROR) ? i_ListIDX : -1;
}

int C_Layer::Destroy_List(int i_Numb)
{
	return C_ListMan->destroy(i_Numb);
}


int C_Layer::Init_Neurons(int i_InitMode,int i_BiasQuest,
						  int inCount,ifstream *I_GWI_File)
{
	int i;
	int i_Error;
	C_Neuron *C_Tmp_Neuron;
	
	DBG(cerr<<"Layer: Entering Init_Neurons ..."<<endl);
	/*Initialize all the neuron's being wanted by the net*/
	i_Error=NO_ERROR;
	for(i=1;(i<=S_LayerData.i_Neuro_Cnt)&&(i_Error==NO_ERROR);
		i++)
	{
		C_Tmp_Neuron=(C_Neuron*)new C_Neuron(&S_LayerData,\
											 C_ListMan,\
											 0,\
											 i,\
											 inCount,
											 i_BiasQuest);

		if(C_Tmp_Neuron==NULL) return ERR_NONEUR_LAY;
		
		i_Error=C_Tmp_Neuron->Init(S_FermiData, i_InitMode, I_GWI_File);

		if(i_Error==NO_ERROR)
		{
			if(i==1) C_NeuStor->write(i_NeuList,i,C_Tmp_Neuron);
			else C_NeuStor->append(i_NeuList,C_Tmp_Neuron);
		}
	}
	DBG(cerr<<"Layer: Leaving Init_Neurons with Error: "<<i_Error<<endl);
	return i_Error;
}

int C_Layer::Destroy_Neurons()
{
	int i,i_Error,i_Error2;
	C_Neuron *C_ActNeuron;
	
	
	DBG(cerr<<"Layer: Entering Destroy_Neurons ..."<<endl);
	/* Call destructor of every neuron.*/
	i_Error2=NO_ERROR;
	i_Error=NO_ERROR;
	/* ---rsc: 95/12/17 --- */
	if(C_NeuStor)
	{
		
		/*While a Neuron exists in the Neuron-list ...*/
		for(i=1;i<=S_LayerData.i_Neuro_Cnt;i++)
		{
			i_Error2=C_NeuStor->read(i_NeuList,i,&C_ActNeuron);
			if(i_Error2==NO_ERROR)
				delete C_ActNeuron;
			
			if(i_Error!=NO_ERROR) 
			{ 
				if(i_Error2!=NO_ERROR) i_Error2=i_Error;
				else (i_Error2=ERR_MULTIP_NET);
			}
		}
	} /* --- rsc: 95/12/17 --- */
	
	DBG(cerr<<"Layer: Leaving Destroy_Neuron with Error: "<<i_Error2<<endl);
	return i_Error2;
}
	
