/* neuron.cc
   Realizes one single neuron of a neural network.
   
   Author: Marko Meyer
   Date: 10.02.1995 (creation)
   Time-stamp: <95/10/22 13:20:15 mme>
   
	BACKNET - A library for simulating neural BACKPROPAGATION-Networks
    Copyright (C) 1995	Marko Meyer

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


	If you have any suggestions, ideas or comments regarding this library
	feel free to contact me:

	Email:	mme@pub.th-zwickau.de
	Ordinary Mail:	Marko Meyer
					Teichstrasse 27
					D-08289 Schneeberg
					F.R.G.

    Particular copyrights / code comments:

	   - unsigned int getseed() was suggested by Heiko Schellhorn,
         hsc@pub.th-zwickau.de, for compatibility between Borland-C++-Compilers
		 and UNIX-C++-Compilers. Thanks.

*/

#include <neuron.h>

#ifdef __BORLANDC__ /* Check this please! */

unsigned int
getseed()
{
	struct time tm;
	gettime(&tm);

	return (unsigned int) tm.ti_hund;
}

#else

unsigned int 
getseed()
{
	struct timeval tm;
	struct timezone tz;

	gettimeofday(&tm,&tz);
	
	return (unsigned int) tm.tv_usec;
}

#endif /* __BORLANDC__ */


C_Neuron::C_Neuron(S_Layer_Param *S_LayerData, 
				   /*pointer to struct with layer-data*/
			 C_List<ELEMTYPE> *C_NetLists, /*pointer to ListManager*/
			 int i_NeuNetID,             /*position of neuron within net*/
			 int i_NeuLayID,             /*position of neuton within layer*/
			 int i_InputCount,           /*number of inputs*/
			 int i_BiasQuest)            /*using BIAS or not*/
{
	/*This is the constructor of a single neuron.*/
	DBG(cerr<<"Neuron: Entering Neuron's constructor.\n");
	DBG(cerr<<"\tNeuron-Nr.: "<<i_NeuNetID
		<<"\tNeuron-LID: "<<i_NeuLayID
		<<"\tInput-Cnt: "<<i_InputCount<<endl);

    int i_LocError;

	S_Layer=S_LayerData;                
	C_ListMan=C_NetLists;
	i_InpCount=i_InputCount;

	DBG(cerr<<"Neuron: Before allocating memory ..."<<endl);

	i_LocError=NO_ERROR;

	/*alloc mem for neuron-information*/
	if((S_Neuron=(S_Neuro_Param*)new S_Neuro_Param))
	{   
		/*if done create weight list for neuron*/
		DBG(cerr<<"Neuron: Allocated Information-struct.\n");
		i_NetID=i_NeuNetID;
		S_Neuron->i_UseBias=i_BiasQuest;
		S_Neuron->i_Neuro_Nr=i_NeuLayID;
		S_Neuron->i_WeightCnt=i_InputCount+i_BiasQuest;
	}
	else S_Neuron=NULL;
	DBG(cerr<<"Neuron: Finished construction. Error is: "<<i_LocError<<endl);
}

int
C_Neuron::Init(S_Fermi_Param *S_FermiData,/*pointer to struct with fermi-data*/
			   int i_InitMode,             /*number of initmode*/
			   ifstream *I_GWI_File)
{
	/* Init - Initializes everything. */
	int i_LocError=NO_ERROR;

	S_Fermi=S_FermiData;
	if(S_Neuron==NULL) return ERR_MEMORY_NEU;
	i_LocError=Create_Wgt_List(S_Neuron->i_WeightCnt,0);
	if(i_LocError==NO_ERROR)
	{
		/*if done initialize weights in weightlist*/
		DBG(cerr<<"Neuron: Switching Init-Mode: "<<i_InitMode<<endl);
		
		switch(i_InitMode)
		{
		  case INIT_FROM_GWI:
			i_LocError=Init_GWI(I_GWI_File);
			break;
		  case INIT_GV_0_1:
			i_LocError=Init_GV(0,1);
			break;
		  case INIT_GV_N_P:
			i_LocError=Init_GV(-1,1);
			break;
		  case INIT_0:
			i_LocError=Init_(0);
			break;
		  case INIT_1:
			i_LocError=Init_(1);
			break;
		  default:
			i_LocError=ERR_INITMD_NEU;
		}
	}
	else return i_LocError;
	return i_LocError;
}

int C_Neuron::Create_Wgt_List(int i_Count,ELEMTYPE Cont)
{
	int i_Error;
	int i_ListNr;
	int i;
	
	i_Error=NO_ERROR;
	DBG(cerr<<"Neuron: Creating Weightlist ...\n");
	i_ListNr=C_ListMan->create(Cont,&i_Error);
	if(i_Error==NO_ERROR)
	{
		/*creating list for weights*/
		DBG(cerr<<"Neuron: List created: "<<i_ListNr<<endl);
		S_Neuron->i_Weights=i_ListNr;
		/*appending i_Count-1 weights*/
		DBG(cerr<<"Neuron: Appending "<<i_Count-1<<" Elements ...\n");
		for(i=1;(i<i_Count)&&(i_Error==NO_ERROR);i++)
			i_Error=C_ListMan->append(i_ListNr,Cont);
		if(i_Error==NO_ERROR)
		{
			DBG(cerr<<"Neuron: Appending successful !\n");
			S_Neuron->i_WeightCnt=i_Count;
		}
		else 
		{
			DBG(cerr<<"Neuron: Appended with Error "<<i_Error<<" !\n");
			S_Neuron->i_WeightCnt=0;
		}
   	}
	else 
	{
	    S_Neuron->i_Weights=0;
		DBG(cerr<<"Neuron: Error "<<i_Error<<" occured at creating list!\n");
	}
	DBG(cerr<<"Neuron: Leaving with Error "<<i_Error<<endl);
	return i_Error;	
}

int C_Neuron::Destroy_Wgt_List()
{
	int i_WgtLst,i_Error;

	i_Error=NO_ERROR;
	DBG(cerr<<"Neuron: Entering Destroy_Wgt_list.\n");
	i_WgtLst=S_Neuron->i_Weights;
	DBG(cerr<<"Neuron: Destroying List Nr. "<<i_WgtLst<<endl);
	i_Error=(*C_ListMan).destroy(i_WgtLst);
	return i_Error;
}


int C_Neuron::Init_GV(ELEMTYPE Min,ELEMTYPE Max)
{
	ELEMTYPE RandWeight;
	int i,i_Error;

	i_Error=NO_ERROR;
	DBG(cerr<<"Neuron: Entering Init_GV.\n");
	srand(getseed());	
	for(i=1;(i<=S_Neuron->i_WeightCnt)&&(i_Error==NO_ERROR);i++)
	{
		RandWeight=Min+((ELEMTYPE)rand()*(fabs(Max)+fabs(Min))/RAND_MAX);
		DBG(cerr<<"Neuron: Rand() says: "<<RandWeight<<" !\n");
		i_Error=C_ListMan->write(S_Neuron->i_Weights,i,RandWeight);
	}
	DBG(if(i_Error) cerr<<"Neuron: Error "<<i_Error\
		<<" occured initing GV !\n";
		else cerr<<"Neuron: Initialization GV finished !\n");
	return i_Error;
}

int C_Neuron::Init_(ELEMTYPE value)
{
	int i,i_Error;
	
	i_Error=NO_ERROR;
	DBG(cerr<<"Neuron: Entering Init_ ...\n");
	for(i=1;(i<=S_Neuron->i_WeightCnt)&&(i_Error==NO_ERROR);i++)
		C_ListMan->write(S_Neuron->i_Weights,i,value);
	DBG(if(i_Error!=NO_ERROR) cerr<<"Neuron: Error "<<i_Error
		<<" occured initing with constval !\n";
		else cerr<<"Neuron: Initing finished!\n");
	return i_Error;
}

int C_Neuron::Init_GWI(ifstream *I_GWI_File)
{
	DBG(cerr<<"Neuron: Entering Init_GWI ...\n");
	return Load_GWI(I_GWI_File);
}

int C_Neuron::Load_GWI(ifstream *I_GWI_File)
{
	/* This one uses C++ iostreams-lib and reads the weights from the 
	   GWI-file-stream. For learning more about the format, the GWI-file
	   is saved in, please refer to the file doc/FILEFORMATS ! */

	char c_LineBuf[LINELENGTH]; /* LineBuf for reading GWI-lines.*/
	char **cc_Endpoint;
	int i_IntError = NO_ERROR;
	
	ELEMTYPE weight;
	
	DBG(cerr<<"Neuron: Entering Load_GWI!\n");
	cc_Endpoint=(char**)new char[1][LINELENGTH];
	
	if((I_GWI_File->good())&&(I_GWI_File->rdbuf()->is_open!=0)&&cc_Endpoint)
	{
		/* The GWI-file is available. */
		DBG(cerr<<"Neuron: Reading weights from file.\n");
		DBG(cerr<<"Neuron: Actual position: "<<I_GWI_File->tellg()<<endl);
		
		/* Getting first line. */
		I_GWI_File->getline(c_LineBuf,LINELENGTH);
		
		DBG(cerr<<"Neuron: Got "<<I_GWI_File->gcount()<<" characters.\n");
		
		if(c_LineBuf[0]=='#')
		{
			/* The first character, that a neuron in the file sees, must be
			   a '#'. Refer to doc/FILEFORMATS for informations about the
			   correct GWI-format.*/
			DBG(cerr<<"Neuron: Got preceeding '#' !\n");
			
			/* It's possible to have more than one '#'-lines, so that we
			   have to look this up. */
			
			while((c_LineBuf[0]=='#')&&(!I_GWI_File->eof()))
			{
				I_GWI_File->getline(c_LineBuf,LINELENGTH);
				DBG(if(c_LineBuf[0]=='#') 
					cerr<<"Neuron: Got more than one '#' !\n");
			}
			
			/*Now we have to read all the weights. We do it in a loop that
			  tests, whether eof or lastweight is reached. */
			for(int i=1;(i<=S_Neuron->i_WeightCnt)&&(i_IntError==NO_ERROR);
				i++)
			{
				/* The next line is subject to change :-) */
				if(i>1) I_GWI_File->getline(c_LineBuf,LINELENGTH); 
				/* If reached eof return the error !*/
				if(I_GWI_File->eof()) return ERR_UXPEOF_NEU;

				/* Looking up first character on line. */
				if(c_LineBuf[0]=='#') return ERR_READWG_NEU;
				DBG(cerr<<"Neuron: Found no '#' as first character. Okay.\n");
				*cc_Endpoint="\0";
				weight=strtod(c_LineBuf,cc_Endpoint);
				if(*cc_Endpoint==c_LineBuf) return ERR_READWG_NEU;
				if(errno==ERANGE) return ERR_WGTOVF_NEU;
				
				DBG(cerr<<"Neuron: Got weight "<<weight<<" !\n");
				
				/* Storing weight. */
				i_IntError=C_ListMan->write(S_Neuron->i_Weights,i,weight);
			}
			/* All weights are read ! And this correctly ... */
			DBG(cerr<<"Neuron: Reading weights completely done!\n");
			return i_IntError;
		}
		else return ERR_READWG_NEU;
	}
	else return ERR_GWINOP_NEU;
}

int C_Neuron::Save_GWI(ofstream *O_GWI_Save)
{
	int i_ActWeight;
	int i_ReadErr;
	ELEMTYPE Weight;
	
	
	DBG(cerr<<"Neuron: Entering Save_GWI!\n");
	if((O_GWI_Save->good())&&(O_GWI_Save->rdbuf()->is_open!=0))
	{
		/* The GWI - File is okay, first we write a '#' on a single line,
		   added by the name and the number of the Neuron and the Layer. */

		DBG(if(O_GWI_Save->good()) cerr<<"Neuron: GWI_Save is good"<<endl);
		
		O_GWI_Save->clear();

		(*O_GWI_Save)<<"# Neuron: "<<S_Neuron->i_Neuro_Nr<<" ; Layer: "
			<<S_Layer->i_Layer_Nr<<" ; Weights: "<<S_Neuron->i_WeightCnt
				<<"; BIAS: "<<S_Neuron->i_UseBias<<endl;
		DBG(cerr<<"Neuron: Preceeding Neuron-information written! \n");

		O_GWI_Save->clear();

		DBG(if(O_GWI_Save->good()) cerr<<"Neuron: GWI-Save is good"<<endl);
		
		/* Then the weights! */

		DBG(cerr<<"Neuron: Reading Weights from List!\n");
		for(i_ActWeight=1;
			i_ActWeight<=S_Neuron->i_WeightCnt;
			i_ActWeight++)
		{
			i_ReadErr=C_ListMan->read(S_Neuron->i_Weights,i_ActWeight,&Weight);
			if(i_ReadErr==NO_ERROR)
			{
				DBG(cerr<<"Neuron: Weight read!\n");
				/* Now saving the weight. */
				
				(*O_GWI_Save)<<Weight<<endl;
				
				DBG(cerr<<"Neuron: Weight write attempt!\n");
				
				if(!O_GWI_Save->good())
				{
					DBG(cerr<<"Neuron: Error at saving weight!\n");
					return ERR_SAVEWG_NEU;
				}
				else DBG(cerr<<"Neuron: Weight written!\n");
			}
			else
			{
				DBG(cerr<<"Neuron: Error "<<i_ReadErr
					<<" at reading from list!\n");
				return i_ReadErr;
			}
		}
		DBG(cerr<<"Neuron: Weights saved successfully!\n");
	}
	else		
			

	{
		DBG(cerr<<"Neuron: GWI-Save not open!\n");
		return ERR_GWINOP_NEU;
	}
	DBG(cerr<<"Neuron: Leaving GWI-Save!\n");
	return NO_ERROR;
}

int C_Neuron::Save(ofstream *O_GWI_Save)
{

	/* This is just a method to save the weights of a neuron to the
	   file pointed to by O_GWI_Save.  It does not destroy something, 
	   it really only saves weights. */

	int i_Error;

	DBG(cerr<<"Neuron: Entered Neuron's method Save.\n");
	i_Error=NO_ERROR;
	i_Error=Save_GWI(O_GWI_Save);
    if(i_Error==NO_ERROR)
	{

	}
	else
		/*An error occured at saving.*/
		cerr<<"Neuron: Error "<<i_Error<<" at saving weights.\n";
	
	return i_Error;
}

 C_Neuron::~C_Neuron()
{
	/*This is the destructor of a single Neuron.  */
	/*Deleting Neuron-information-struct if existent.  */
	/*Deleting the weightlist.  */
	/*Although it is possible that errors occure during 'Destroy_Wgt_List()'
	  we don't care.  This is the destructor of the Neuron and everything
	  must be destroyed.  What can't be destroyed by the program itself is
	  either not present or can be destroyed by the operating system.  */
    Destroy_Wgt_List();
	if(S_Neuron) delete(S_Neuron);
	DBG(cerr<<"Neuron: Leaving Neuron."<<endl);
}


 int C_Neuron::Recall()
{
	/* This one does a recall for all the inputs. It uses the given InputList,
	   the own WeightList and writes its output to the OutputList. */

	int i_IntError;

	DBG(cerr<<"Neuron: Entering Recall ...\n");
	
	i_IntError=calc_effective_input();
	if(i_IntError==NO_ERROR) 
	{
		i_IntError=calc_activity_value();
		if(i_IntError==NO_ERROR)
		{
			i_IntError=calc_output_value(CALC_NO_DERIV);
			if(i_IntError==NO_ERROR)
			{
				/* The 'Exit'-value is stored as a neurons parameter. It is  
				   now written to the layer's OutputList. */
				DBG(cerr<<"Neuron: OutputValue is calculated to: "<<Exit
					<<endl);
				
				i_IntError=C_ListMan->write(S_Layer->i_Output_List,\
											S_Neuron->i_Neuro_Nr,Exit);
				DBG(cerr<<"Neuron: Error "<<i_IntError<<" at saving output."
					<<endl);
				DBG(cerr<<"Neuron: Leaving Recall."<<endl);
				return i_IntError;
			}
			else return i_IntError;	   
		}
		else return i_IntError;
	}
	else return i_IntError;
}

 int C_Neuron::Learn()
{
	/* This one does all calculations useful for learning the Neuron. 
	   It uses the ErrList, changes the weights and distributes the calculated
	   error to the previous ErrList. Therefore it calls modify_weight()
	   and distribute_errors().*/

	int i_IntError;

	DBG(cerr<<"Neuron: Entering Learn ..."<<endl);
	
	i_IntError=distribute_error();
	if(i_IntError==NO_ERROR) return modify_weights();
	else return i_IntError;
}


 int C_Neuron::calc_effective_input()
{
	/* This is the virtual function for calculating the effective Input of a
	   Neuron. It takes the values from the InputList, multiplicates them
	   with the appropriate weights and adds the results. So one value is
	   created: 'Entrance'. It is saved as a parameter of the Neuron. */

	int i_IntError=NO_ERROR;
	ELEMTYPE Input=0,Weight=0,PreEntrance=0;
	
	DBG(cerr<<"Neuron: Entering calc_effective_input ..."<<endl);
	i_IntError=NO_ERROR;
	DBG(cerr<<"Neuron: 1"<<endl);
	
	Entrance=0.;
	DBG(cerr<<"Neuron: 2"<<endl);
	
	PreEntrance=0.;
	DBG(cerr<<"Neuron: 3"<<endl);
	
	for(int i=1;i<=(S_Neuron->i_WeightCnt-S_Neuron->i_UseBias);i++)
	{
		/* Reading all Inputs ...*/
		DBG(cerr<<"Neuron: Reading all Inputs ..."<<endl);
		
		i_IntError=C_ListMan->read(S_Layer->i_Input_List,i,&Input);
		if(i_IntError==NO_ERROR)
		{
			/* ... and all Weights ...*/
			DBG(cerr<<"Neuron: ... and all Weights ..."<<endl);
			
			i_IntError=C_ListMan->read(S_Neuron->i_Weights,i,&Weight);
			/* ... multiplicating them and adding to a Pre-Entrance-variable.*/
			DBG(cerr<<"Neuron: ... multiplicating and adding ..."<<endl);
			
			if(i_IntError==NO_ERROR) PreEntrance+=(Input*Weight);
			else return i_IntError;
		}
		else return i_IntError;
		
	}

	if(S_Neuron->i_UseBias==USE_BIAS)
	{
		/* In case of USE_BIAS, there's one Weight (the BIAS-Weight) that is
		   not interacting with a real Input-value, but with '1', so we add
		   it to the Pre-Entrance-variable.*/
		DBG(cerr<<"Neuron: ... reading BIAS-Weight ..."<<endl);
		
		i_IntError=C_ListMan->read(S_Neuron->i_Weights,S_Neuron->i_WeightCnt,
								   &Weight);
		DBG(cerr<<"Neuron: ... adding BIAS-Weight ..."<<endl);
		
		if(i_IntError==NO_ERROR) PreEntrance+=Weight;
		else return i_IntError;
	}
	/* That's the end: Entrance is calculated, NO_ERROR returned.*/
	DBG(cerr<<"Neuron: Finished calculating without Error."<<endl);
	DBG(cerr<<"Neuron: Entrance is: "<<Entrance<<endl);
	Entrance=PreEntrance;
	return NO_ERROR;
}

 int C_Neuron::calc_activity_value()
{
	/* This one calculates the value of the activity function. Because in
	   a standard Backpropagation-Neuron the activity function is the 
	   equality, it is really simple! */
	
	DBG(cerr<<"Neuron: Entering calc_activity_value ..."<<endl);
	Activity=Entrance;
	DBG(cerr<<"Neuron: Finished calculation."<<endl);
	DBG(cerr<<"Neuron: Activity is: "<<Activity<<endl);
	return NO_ERROR;
}

 int C_Neuron::calc_output_value(int i_Derivation)
{
	/* This one calculates the output value. It really only calculates them,
	   it is saved as a parameter for the Neuron.
	   In this special Neuron the Fermi-function is used to calculate the 
	   output. For learning aims we need the derivation too, so that we do
	   this as an if-condition.*/

	ELEMTYPE Glob_Numerator;
	ELEMTYPE Glob_Denominator;
	ELEMTYPE Denominator_Exp;
	ELEMTYPE Denominator_Exp_Mult1;
	ELEMTYPE Denominator_Exp_Mult2;
	ELEMTYPE Denominator_Exp_Mult2_Numerator;
	ELEMTYPE Denominator_Exp_Mult2_Denominator;

	DBG(cerr<<"Neuron: Entering calc_output_value ..."<<endl);
	DBG(cerr<<"Neuron: Calculating derivation ? "
		<<((i_Derivation==CALC_NO_DERIV)?"NO":"YES")<<endl);
	
	DBG(cerr<<"Neuron: Actual Fermi-Values:"<<endl);
	DBG(cerr<<"\tMinimum: "<<S_Fermi->f_Minimum<<"\tMaximum: "<<S_Fermi->\
		f_Maximum<<endl<<"\tTheta: "<<S_Fermi->f_Theta<<"\tSigma: "<<\
		S_Fermi->f_Sigma<<endl);

	Glob_Numerator=0.;
	Glob_Denominator=0.;
	Denominator_Exp=0.;
	Denominator_Exp_Mult1=0.;
	Denominator_Exp_Mult2=0.;
	Denominator_Exp_Mult2_Numerator=0.;
	Denominator_Exp_Mult2_Denominator=0.;
	
	Denominator_Exp_Mult1         = ((-4.)*(S_Fermi->f_Sigma));
	Denominator_Exp_Mult2_Numerator = ((Activity) - (S_Fermi->f_Theta));
	Denominator_Exp_Mult2_Denominator  = ((S_Fermi->f_Maximum) - 
										  (S_Fermi->f_Minimum));
	Denominator_Exp_Mult2         = (Denominator_Exp_Mult2_Numerator / 
								Denominator_Exp_Mult2_Denominator);
	Denominator_Exp               = (ELEMTYPE)exp(Denominator_Exp_Mult1 * 
											 Denominator_Exp_Mult2);
	Glob_Numerator             = Denominator_Exp_Mult2_Denominator;
	Glob_Denominator              = (1. + Denominator_Exp);
	
	if(i_Derivation==CALC_NO_DERIV)
		Exit= S_Fermi->f_Minimum + (Glob_Numerator / Glob_Denominator);
	else 
		ExitDerv= (ELEMTYPE) \
			(((-1. *Denominator_Exp_Mult1)/(Glob_Numerator * Glob_Numerator)) \
				   * (Exit - (S_Fermi->f_Minimum)) * \
				   (S_Fermi->f_Maximum - Exit));
	DBG(cerr<<"Neuron: Finished calculation."<<endl);
	DBG(cerr<<"Neuron: Output is: "<<Exit<<"\tDerv is: "<<ExitDerv<<endl);
	return NO_ERROR;
}


 int C_Neuron::distribute_error()
{
	/* This one distributes errors back to a previous layer, if there is one.*/
	
	ELEMTYPE new_own_error;
	ELEMTYPE own_weight;
	ELEMTYPE calc_error;
	ELEMTYPE present_error;
	
	int i_IntError=NO_ERROR;

	DBG(cerr<<"Neuron: Entering distribute_error ..."<<endl);
	/*First of all, we test the position in the net */
	new_own_error=0.;
	own_weight=0.;
	calc_error=0.;
	present_error=0.;
	if(S_Layer->i_Layer_Nr!=1)
	{
		/* We find out the derivative Exit ...*/
		i_IntError=calc_output_value(CALC_DERIV);
		if(i_IntError==NO_ERROR)
		{
			/* ... multiply it to the own_error read from the ErrorList ... */
			i_IntError=C_ListMan->read(S_Layer->i_Error_List,\
									   S_Neuron->i_Neuro_Nr,&new_own_error);
			if(i_IntError==NO_ERROR)
			{
				new_own_error*=ExitDerv;
				for(int i=1;(i<=(S_Neuron->i_WeightCnt-S_Neuron->i_UseBias))
					&&(i_IntError==NO_ERROR);i++)
				{
					/* Read weight of actual Neuron to Neuron i of previous
					   Layer. */
					i_IntError=C_ListMan->read(S_Neuron->i_Weights,i,\
											   &own_weight);
					if(i_IntError==NO_ERROR)
					{
						calc_error=(new_own_error * own_weight);
						DBG(cerr<<"Neuron: calc_error is: "<<calc_error<<endl);
						
						/*In LErrList: read present entry for cumulative cal-
						  culation.*/
						i_IntError=C_ListMan->read(S_Layer->i_Last_Error_List,
												   i,&present_error);
						if(i_IntError==NO_ERROR)
						{
							/* Add own calculated error to present entry.*/
							present_error+=calc_error;
							
							/* Push present_error back to the LErrList.*/
							i_IntError=C_ListMan->write(S_Layer->\
														i_Last_Error_List,i,
														present_error);
						}
						else return i_IntError;
					}
					else return i_IntError;
				}
			}
			else return i_IntError;
		}
		else return i_IntError;
	}
	DBG(cerr<<"Neuron: Errors distributed correctly."<<endl);
	return i_IntError;
}


 int C_Neuron::modify_weights()
{
	/* This one modifies the weight using the ErrorList of the Layer.*/
	
	ELEMTYPE new_own_error;
	ELEMTYPE own_input;
	ELEMTYPE weight_delta;
	ELEMTYPE own_weight;
	
	int i_IntError;

	DBG(cerr<<"Neuron: Entering modify_weight ..."<<endl);
	new_own_error=0.;
	own_input=0.;
	weight_delta=0.;
	own_weight=0.;
	i_IntError=calc_output_value(CALC_DERIV);
	if(i_IntError==NO_ERROR)
	{
		/* ... multiply it to the own_error read from the ErrorList ... */
		i_IntError=C_ListMan->read(S_Layer->i_Error_List,\
								   S_Neuron->i_Neuro_Nr,&new_own_error);
		if(i_IntError==NO_ERROR)
		{
			new_own_error*=ExitDerv;
			for(int i=1;(i<=(S_Neuron->i_WeightCnt))&&(i_IntError==NO_ERROR);i++)
			{
				/* Read the Input. */
				if((i==S_Neuron->i_WeightCnt)&&(S_Neuron->i_UseBias==USE_BIAS))
					own_input=1; /* In case of BIAS-Weight, the input is '1'.*/
				else
					i_IntError=C_ListMan->read(S_Layer->i_Input_List,i,
											   &own_input);
				if(i_IntError==NO_ERROR)
				{
					/* Multiply everything ... (nearly, of course! :-))*/
					weight_delta=S_Layer->f_Learn_Rate*new_own_error*own_input;
					
					/* Read old weight.*/
					i_IntError=C_ListMan->read(S_Neuron->i_Weights,i,
											   &own_weight);
					if(i_IntError==NO_ERROR)
					{
						/* Calculate new weight. */
						own_weight+=weight_delta;
						
						/* Save new weight. */
						i_IntError=C_ListMan->write(S_Neuron->i_Weights,i,
													own_weight);
					}
					else return i_IntError;
				}
				else return i_IntError;
			}
			
		}
		else return i_IntError;
	}
	else return i_IntError;
	DBG(cerr<<"Finished modification."<<endl);
	
	/* Everything is corrected, we have to write a '0' to the ErrorList,
	   because the following LearningStep will use the value in the ErrorList
	   as the value a previous Neuron has written to it. */
	if(i_IntError==NO_ERROR)
		i_IntError=C_ListMan->write(S_Layer->i_Error_List,S_Neuron->i_Neuro_Nr,0);
	return i_IntError;
}

