//  libeasy, C++ library to encapsulate things and make life easy.
//  Copyright (C) 2000 Hans Dijkema 
//
//  This program/library is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License as published 
//  by the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program/library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program/library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// This software is maintained by Hans Dijkema.
// See the Makefile.lsm for your primary and secundary site. 
// email me at hnmdijkema@softhome.net
//
#ifndef __LOG__HXX
#define __LOG__HXX

#ifdef WIN32
#include <vc6.hxx>
#endif

#include <stdarg.h>

//sec
//h1 Algemene gegevens
// library : libeasy.a
// module  : hlog
// include : hlog.hxx
// link    : -leasy
//
// Versie Wie              Datum       Wijziging/Opdracht/Omschrijving
// --------------------------------------------------------------------------
// 1.00   HNM Dijkema      27-12-1999  Algemeen bruikbare library ten behoeve
//                                     van C++ projecten.
//
//h2 Revisies
// Versie Revisieinformatie                                           
// --------------------------------------------------------------------------
// 1.00   Initiele versie
//
//
//h1 Doel van de library
// Doel van de library hlog is om een framework te geven voor logging
// van informatie. Deze logging kan algemene logging of debug logging
// betreffen. Debug logging is hierbij een optie die aan en uit kan worden
// gezet.
//
// De voorziening wordt gemaakt met behulp van een base class 'log_base',
// welke in de basis functionaliteit voorziet en afgeleide classes die
// voorzien in specifieke functionaliteit.
//
// Met behulp van een programmabrede statische module variabele LOG, wordt
// vervolgens de huidige logging geimplementeerd. Afgeleide classes 
// implementeren de manier van loggen. Bijvoorbeeld, logging naar stderr,
// of logging naar syslog, of zelfs logging naar een listbox o.i.d.
// 
// Default wordt een afgeleide class voor logging naar de stderr meegegeven 
// en een afgeleide class voor logging naar syslog meegegeven. 
//
//h2 class structuur
// Framework is dus als volgt:
//
//   log_base ----> log_cerr
//            |
//            |---> log_syslog
//            |
//            |---> etc.
//
//
//h1 Classes en functies
//h2 class log_base
// Deze class vormt een base class voor het loggen en het debuggen.
// Debuggen is per default uitgeschakeld (_debugLevel==0). Het loggen
// staat altijd aan.
//
// Voor het loggen van 'log' informatie en 'debug' informatie worden
// twee virtuele functies geimplementeerd welke de informatie afdrukken.
// Deze virtuele functies worden in een afgeleide class geimplementeerd.
//
// De hlog module implementeert default de class die output verzorgt
// op de stderr, class log_cerr. Deze class implementeert de virtuele
// functies in log_base en zorgt dat ze de output op de stderr stream 
// zetten.
//
// Een programma dat een willekeurig object dat afgeleid is van log_base
// wordt instantieert, kan met behulp van twee functies 'log' en 'debug'
// zorgen dat informatie wordt gelogged, zonder dat het over de log variabele
// hoeft te beschikken. Een globale statische variabele log_base *LOG, wordt
// bij iedere instantiatie van een log_base afgeleide class opnieuw ingesteld
// naar dat object. De oude verwijzing wordt tijdelijk opgeslagen in het
// geinstantieerde object zelf.
//
// De log functies worden met een log_type aangeroepen. De gebruiker 
// bepaalt met behulp van de afgeleide class wat er met het log_type 
// wordt gedaan. Het kan zijn, alleen afdrukken; maar het kan ook zijn
// dat het samenhangt met het debug level. Of er is een mogelijkheid
// om bepaalde types aan te zetten en andere uit. De afgeleide class
// bepaalt hoe en wat.
//
//  Verder is de aanroep van de algemene log functies hetzelfde als
// die van printf() (man printf). De algemene log functies 
// roepen LOG->log()/__debug() aan, om de uiteindelijke implementatie 
// te verkrijgen. De parameters worden hierbij in het vprintf() formaat 
// gegoten (man vprintf).
//
//end

//h2 class log_base

class log_base 
{
  private:
//def
    log_base   *prev;
//
// variabele 'prev' krijgt bij iedere constructie de vorige
// waarde van static variabele LOG.
//end
  protected:
//def
    int _debugLevel;
//
// variabele _debugLevel wordt bij constructie op 0 geinitialiseerd.
// Indien _debugLevel<>0, dan wordt er debug informatie gelogged, anders
// niet. Dit is de semantiek waar afgeleide classes zich aan moeten houden.
//end
  public:
//def
    virtual void log(int log_type,char *format,va_list arg) = 0;
//
// pre : -
// post: de log functie van de afgeleide class wordt uitgevoerd.
//end
//def
    virtual void __debug(int log_type,char *format,va_list arg) = 0;
//
// pre : -
// post: de __debug functie van de afgeleide class wordt uitgevoerd.
//end
  public:
//def
    bool debugOn(void)  
    { return _debugLevel!=0; }
//
// pre : -
// post: =true, _debugLevel!=0
//       =false, otherwise
//end
//def
    void debugLevel(int level=1);
//
// pre : -
// post: Het debug level is ingesteld op level.
//       De afgeleide class bepaald voor level!=0 de semantiek.
//       level==0 betekent: Geen debug informatie loggen.
//end
//def
    void debugOff(void)	
    { debugLevel(0); }
//
// pre : -
// post: Het debuggen is uitgezet (debug level==0).
//end
  public:
//def
    log_base();
//
// pre : -
// post: static log_base *LOG=this
//       prev=LOG
//end
//def
    virtual ~log_base();
//
// pre : -
// post: static log_base *LOG=prev.
// opm.: Deze functie is virtual, zodat ook de destructor van de
//       afgeleide class wordt aangeroepen, indien een delete op
//       log_base * wordt gedaan.
//end
};

//*************************************************************************

//h2 Algemene hlog functies.

//def
void log(int log_type,char *format,...);
//
// pre : -
// post: Indien LOG==NULL, instantieert een locale variabele van type log_cerr.
//       Roept LOG->log() aan.
//end

//def
void __debug(int debug_type,char *format,...);
//
// pre : -
// post: Indien LOG==NULL, instantieert een locale variabele van type log_cerr.
//       Roept LOG->__debug() aan.
//end
//def
bool debugOn(void);
//
// pre : -
// post: =false, LOG==NULL 
//       =log->debugOn(), otherwise
//end


//def
#define debug if (debugOn()) __debug
//
// Gebruik in plaats van __debug, altijd deze constructie om 
// debug regels in de programmatuur te plaatsen.
// In dit geval, wordt __debug alleen aangeroepen, indien debugOn()
// waar is.
//end

//*************************************************************************

//h2 class log_cerr : public log_base
//Deze class implementeert de virtuele functies van log_base.
//
//De virtuele destructor wordt niet gebruikt, het default gedrag
//is voldoende.
//end

class log_cerr : public log_base
{
  public:
//def
    virtual void log(int log_type,char *format,va_list arg);
//
// pre : -
// post: vfprintf(stderr,format,arg);fprintf(stderr,"\n");
//end
//def
    virtual void __debug(int debug_type,char *format,va_list arg);
//
// pre : -
// post: als debugOn(), 
//         dan fprintf(stderr,"debugging (level=%d)", format, arg, "\n")
//end
};

//h2 class log_syslog : public log_base

class log_syslog : public log_base
{
  public:
//def
    log_syslog::log_syslog(char *prgname);
//
// pre  : prgname!=NULL
// post : syslog object geinstantieerd, openlog() uitgevoerd.
//
//def
    log_syslog::~log_syslog();
//
// pre  : -
// post : closelog() uitgevoerd.
//end
  public:
//def
    virtual void log(int log_type,char *format,va_list arg);
//
// pre : -
// post: vsprintf(msg,format,arg);syslog(LOG_INFO,msg)
//end
//def
    virtual void __debug(int debug_type,char *format,va_list arg);
//
// pre : -
// post: als debugOn(),
//          dan syslog(LOG_DEBUG,"debugging (level=%d",format,arg)
//end
};

//sec
//h1 Verwijzingen
//    n.v.t.


//h1 Gegevensbenadering
//    n.v.t.

//h1 Voorbeeldprogramma(s)
//h2 Voorbeeld 1
// Dit programma logt z'n output naar stderr. Maar, kan,
// zoals te zien is, eenvoudig omgeconfigureerd worden naar syslog.
// Merk op dat LG.debugLevel(2) uitgecommentarieerd is, d.w.z.
// er wordt geen debuginformatie gelogd.
//
// int main(int argc,char *argv[])
// {//log_syslog LG(argv[0]);
//  log_cerr LG;
////  LG.debugLevel(2);
//    log(SUB_PRG_1,"Dit is een log %d",10);
//    debug(SUB_PRG_1,"Dit is een debug");
// } 
//end

#endif

