//  libeasy, C++ library to encapsulate things and make life easy.
//  Copyright (C) 2000 Hans Dijkema 
//
//  This program/library is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License as published 
//  by the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program/library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program/library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// This software is maintained by Hans Dijkema.
// See the Makefile.lsm for your primary and secundary site. 
// email me at hnmdijkema@softhome.net
//
#include <harray.hxx>
#include <hregex.hxx>
#include <stdio.h>
#ifndef WIN32
  #include <sys/mman.h>
  #include <unistd.h>
#else
  #include <winbase.h>
  #include <io.h>
  #include <fcntl.h>
#endif

//imp

//doc Module hregex - implementatie

//h2 Caching van regex's

//Voor de cache van reguliere expressies worden vier variabelen
//gebruikt van type h(s)array. In 'regex_ids' worden de gevraagde
//reguliere expressies als string opgeslagen. In de 'regex_flags'
//de gebruikte flags bij het compileren. In 'regex_buf' worden de
//gecompileerde reguliere expressies opgeslagen en in 'regex_used'
//wordt het aantal keer dat een variabele van een regex_id is
//geinstantieerd bijgehouden. Dit is een semafoor variabele. Bij
//constructie van een regex variabele, wordt deze met 1 opgehoogd.
//Bij destructie met 1 verlaagd.

static hsarray<re_pattern_buffer *> regex_buf;
static hsarray<int>                 regex_flags;
static harray<string>               regex_ids;
static hsarray<int>                 regex_used;

//end

//doc Constructie en Destructie

//def
hregex::hregex(char *reg,int flags)
//
{
  cache_index=-1;
//        Variabele cache_index moet initieel op -1 staan, om goed gebruik bij
//        newrx te waarborgen.
  _matched=false;
//         Er is nog niets gematched.
  newrx(reg,flags);
//         De reguliere expressie is geinitialiseerd.
  regs.num_regs=0;
//         Initialisatie van re_registers (zie regex.info).
}

//endimp
//def
hregex::hregex(string reg,int flags)
// 
// Functionaliteit zie hregex(char *reg,int flags)
//
{
  cache_index=-1;
  _matched=false;
  newrx((char *) reg.c_str(),flags);
  regs.num_regs=0;
}

//imp
//def
hregex::~hregex()
//
{
  if (_matched) {
    free(regs.start);
    free(regs.end);
  }
//        Indien er gematched is, dan is er ruimte gealloceerd in
//        re_registers (zie regex.info). Deze ruimte moet bij destructie
//        van een hregex object verwijderd worden.
//
  regex_used[cache_index]-=1;
//        De semafoor in de cache van reguliere expressies wordt nu
//        verlaagd, omdat deze reguliere expressie variabele wordt
//        gedeinstantieerd.
}

//def
void hregex::newrx(char *reg,int flags)
//
// Deze functie (die ook als member functie kan worden aangeroepen)
// initialiseerd de reguliere expressie die meegegeven is.
// Voor de semantiek van 'flags', zie regex.info.
//
{
char *ret;
int  fmret;
int i,N;

  if (cache_index>=0) { regex_used[cache_index]-=1; }
//         indien cache_index<0, (-1), wordt newrx() voor de eerste keer
//         aangeroepen (zie de constructors). Deze regex staat dus nog
//         niet in de cache.

  for(i=0,N=regex_ids.len();
           i<N && (regex_ids[i]!=reg || regex_flags[i]!=flags);
      i++
  );
//         Zoek de reguliere expressie op in de cache. Hiervoor moet
//         de reguliere expressie reg en de flags (de interpretatie van 
//         reg) gelijk zijn aan die van de cache.
//

  if (i==N) {
//         Indien niet gevonden
    R=new re_pattern_buffer;
    if (R==NULL) {
      THROW_FATAL("kan pattern buffer niet alloceren");
    }
    R->fastmap=new char[256];
    if (R->fastmap==NULL) {
      THROW_FATAL("kan pattern buffer niet alloceren");
    }
//         Alloceer de regex variabelen (zie regex.info) voor het compileren
//         van de reguliere expressie.
    regex_ids[i]=reg;
    regex_flags[i]=flags;
    regex_used[i]=1;
    regex_buf[i]=R;
//         Zet deze variabelen in de cache.
    R->allocated=0;
    R->buffer=NULL;
    R->translate=NULL;
    R->regs_allocated=REGS_FIXED;
    re_set_syntax(flags);
    _flags=flags;

    ret=(char *) re_compile_pattern(reg,strlen(reg),R);

    if (ret!=NULL) {
      THROW_FATAL2("Can't compile regex '%s'",reg);
    }
//         Initialiseer R en compileer de reguliere expressie 
//         (zie regex.info).

    fmret=re_compile_fastmap(R);
    if (fmret!=0) {
      THROW_FATAL2("Can't compile fastmap for regex '%s'",reg);
    }
//         Compileer de fastmap (met de fastmap worden de begin
//         characters van de reguliere expressie bepaald, zie regex.info).
  }
  else {
//         Indien gevonden in de cache
    regex_used[i]+=1;
    R=regex_buf[i];
//         Verhoog de semafoor en laat R naar de gecompileerde reguliere
//         expressie in de cache wijzen.
  }

  cache_index=i;
//         Aan het eind van het liedje moet worden vastgehouden, welke
//         reguliere expressie in de cache deze instantiatie (dit object)
//         gebruikt.
}

//endimp
//def
void hregex::newrx(string reg,int flags)
//
// Roept newrx(char *reg,int flags) aan.
//
{
  newrx((char *) reg.c_str(),flags);
}
//end


//imp


//doc Het zoeken met hregex objecten (matching) 

//def
bool hregex::match(char *s,long start_at,int maxmatch,long maxlen)
//
{
int i,r;

  if (((int) regs.num_regs)!=maxmatch) {
    if (regs.num_regs!=0) {
      delete regs.start;
      delete regs.end;
    }
    regs.start=new regoff_t[maxmatch+1];
    regs.end=new regoff_t[maxmatch+1];
    if (regs.start==NULL || regs.end==NULL) {
      THROW_FATAL("Kan geen geheugen voor registers alloceren");
    }
    regs.num_regs=maxmatch;
  }
//        Indien het maximaal aantal gevraagde matches (interpretatie,
//        zie regex.info). Afwijkt van regs.num_regs (bij constructie
//        geinitialiseerd op 0), dan worden de indexen regs.start en 
//        regs.end gealloceerd.

  _matched=true;
//        Nu hebben we een keer een match gedaan.

  if (maxlen==-1) { maxlen=strlen(s); }
//        Match maximaal de lengte van de gegeven string

  re_set_syntax(_flags);
  r=re_search(R,s+start_at,maxlen,0,maxlen,&regs);
//        Zet de gevraagde interpretatie van de reguliere expressie op
//        _flags en ga zoeken.

  if (r<0) {
//        Indien het resultaat <0 is, dan is er niets gevonden.
    regs.start[0]=0;
    regs.end[0]=0;
    _matches=0;
//        Het aantal matches is nu 0
  }
  else {
//        Anders, indien het resultaat >=0 is, dan is er wel iets gevonden.
    for(i=0;i<maxmatch;i++) {
      regs.end[i]=regs.end[i]-regs.start[i];
      regs.start[i]=regs.start[i]+start_at;
    }
    _matches=regs.num_regs;
//        regs.end[] bevat nu de lengtes van de matches, regs.start[] de
//        offsets in de string. _matches bevat het aantal gematchte strings
  }

  regs.num_regs=maxmatch;
//        Omdat we zelf een statische buffer hebben gealloceerd, maken
//        we nu regs.num_regs gelijk aan de maximale buffergrootte, zodat
//        (re)allocatie de volgende keer goed gaat (zie begin van deze
//        functie).

return r>=0;
//        =true, er is iets gevonden
//        =false, otherwise
}
//endimp

//def
bool hregex::match(string & s,long start_at,int maxmatch,long maxlen)
//
// roept match(char *,... aan).
//
{
return match((char *) s.c_str(),start_at,maxmatch,maxlen);
}

//def
bool hregex::match(FILE *f,long start_at,int maxmatch,long maxlen)
//
// roept match(fileno(f),...) aan.
//
{
return match(fileno(f),start_at,maxmatch,maxlen);
}

//imp
//def
bool hregex::match(int fh,long start_at,int maxmatch,long maxlen)
//
// Deze funtie roept match(char *,...) voor een file aan. 
// Om dit te kunnen doen, wordt de file als memory map behandeld.
// NB. Na aanroep van deze functie, zijn de indexen reg_start en
// reg_end, offsets in een file, niet meer offsets in een string.
//
{
long  filelen;
bool  r;
hmmap map(fh);
char *mem=map.memory();

  if (maxlen==-1) { maxlen=filelen; }
//      Bepaal de lengte van de file en memory map de file op char *mem.
//      Roep match(char *,...) aan.
  r=match(mem,start_at,maxmatch,maxlen);
return r;
//      Retourneer het resultaat van match(char *,...)
}
//endimp
//end


//imp
//doc Besturing van de cache

//def
void hregexCleanCache(void)
//
{
int i,N;
  for(i=0,N=regex_ids.len();i<N && regex_used[i]==0;i++);
//        Bepaal of er nog variabelen worden gebruikt van de cache.
//        Semafoor(s) regex_used[] zijn dan ongelijk 0.
  if (i==N) {
//        Indien dit niet zo is
    regex_ids.reinitialize();
    for (i=0;i<N;i++) {
      regfree(regex_buf[i]);
    }
    regex_buf.reinitialize();
    regex_flags.reinitialize();
    regex_used.reinitialize();
//        Dealloceer alle gecompileerde reguliere expressies (zie regex.info).
//        En maak alle arrays van de cache leeg (zie boven voor de variabelen).
  }
}
//endimp

//def
int hregexCacheLen(void)
//
// retourneert de lengte van 1 van de arrays van de cache.
//
{
return regex_ids.len();
}

//end

