//
// turbomac.h : a class for simple turbomachines modeling
// (c) 1995--1996 Alessandro Cella <cella@marvin.conecta.it>
//
// distributed under GNU GPL, see license doc
//
// this offers NO WARRANTY... 
//

#ifndef _TURBOMAC_H
#define _TURBOMAC_H

#define TBMNAME "turbomac"
#define TBMVERS "v.0.8.1"
#define TBMDATE "24 jun 96"
#define TBMAUTH "a.cella (cella@marvin.conecta.it)"

#include <String.h>

//
// each turbomachine deserves an operating
// fluid to work upon :)
//
#ifndef _FLUID_H
#include "fluid.h"
#endif

//
// used to display info or errors
// note --however-- that for that purpose only stderr will be used
//
#include <iostream.h>

#ifndef _PLANTELEMENT_H
#include "plantElement.h"
#endif

class turbomac : public plantElement
{
  friend ostream& operator << (ostream&, turbomac&);

 public:
  //
  // we all need a default constructor...
  //
  turbomac(String name="default turbomachine",int id=0);

  //
  // ...and a destructor
  //
  virtual ~turbomac() {};
  
  //
  // sets working fluid, lookups on a table somewhere
  //
  int setFluid(String fluidID);

  //
  // set mass flow, efficiency, pressure ratio and inlet temperature
  //
  int setMassFlow(double tmMflow);
  int setEfficiency(double tmEff);
  int setPRatio(double tmPratio);
  int setInletT(double tmInT);

  //
  // this is the main thing i am currently interested in:
  // getting the oulet temperature (in SI units, K) once that
  // efficiency of the whole turbomachine, as well as massflow,
  // pressure ratio and other stuff has been stated (alas, has been set).
  //
  virtual double getOutletT()=0;
 
  //
  // info methods
  //
  String getVersion();

 protected:
  //
  // not just private, since data must be accessed from descendants.
  // data hiding, only we have to know how to manage this stuff
  // let the user be left only with the choice of fluid ID 
  // eg. "CO2", "N2", "He"...
  //
    
  //
  // note that fluid is a complete class, which enables
  // its user to cope with complex --and tedious-- stuff without fear...
  //
  fluid workingFluid;
 
  //
  // so, how much are you sucking, babe?
  //
  double massFlow;
  
  //
  // pressure ratio, adiabathic efficiency, inlet temperature
  // SI units:    [1],       [1],       [K]   
  // ranges  :  >1 && <...,  >0 && <1 , more than 0 
  //
  double pRatio,adEff,inletT;
  double outletT;
  
  //
  // let us compute k and k-1/k hierarchically, leaving the descendants
  // facing the problem of how to calc m-1/m
  //
  double estimK(double T1, double T2);  
  virtual double estimKm1OverK(double T1, double T2);

  //
  // since it's likely that the fluid's properties change
  // more than slightly during the compression/expansion, it's
  // better to use these methods to get some close estimates
  // before returning outletT.
  //
  double estimOutletT(double T1, double Km1OverK);

};

#endif
