#include <ctype.h>

#include <qfile.h>
#include <qfileinf.h>
#include <qregexp.h>
#include <qdstream.h>
#include <qstrlist.h>
#include <qfiledlg.h>

#include <kmsgbox.h>

#include "TFileIO.h"
#include "TSignalSlot.h"
#include "TMainWindow.h"

#define MAX_LINE_LEN 1024

TFileIO::TFileIO(TStore &st, TLayoutWindow &bw) {
  fStore = &st;
  fLayoutWindow = &bw;

  fCodePrefix = "";
  fLayoutFilename = "";
}



void TFileIO::layoutFilename(QFileInfo fi) {
  fLayoutFilename = fi.filePath();
}


QString TFileIO::layoutFilename(void) {
  return fLayoutFilename;
}



QString TFileIO::CFilename(void) {
  return (layoutWindow()->cCodeFile().fileName());
}
QString TFileIO::HeaderFilename(void) {
  return (layoutWindow()->hCodeFile().fileName());
}
QString TFileIO::DataFilename(void) {
  return (layoutWindow()->dataFile().fileName());
}


int TFileIO::writeLayout(void) {
  if (layoutFilename()=="") {
    KMsgBox::message(NULL, "Error", "There has been no layout file selected!");
    return 1;
  }


  // Make sure that we have a backup
  QString fPath = layoutFilename() + "~";
  if (0!=unlink(fPath) && errno!=ENOENT) {
	  QString errmsg;
	  errmsg.sprintf("Unable to delete old backup dialog file:\n\t%s\n\t%s\nContinue ?", 
					 (const char *)fPath, strerror(errno));
	  if (KMsgBox::DB_SECOND==KMsgBox::yesNo(NULL, "Error", errmsg, KMsgBox::EXCLAMATION))
		  return FALSE;
  }
  
  QFileInfo fi(layoutFilename());
  if (fi.exists() && 0!=rename (layoutFilename(), fPath)) {
	  QString errmsg;
	  errmsg.sprintf("Unable to rename dialog file to backup:\n\t%s\n\t%s\nContinue ?", 
					 (const char *)layoutFilename(), strerror(errno));
	  if (KMsgBox::DB_SECOND==KMsgBox::yesNo(NULL, "Error", errmsg, KMsgBox::EXCLAMATION))
		  return FALSE;
  }

  QFile layoutFile(layoutFilename());
  QTextStream layoutStream(&layoutFile);

  layoutFile.open(IO_WriteOnly);

  writeLayoutString(layoutStream, "", "CFilename", layoutWindow()->cCodeFile().relFilePath(fi.dir()));
  writeLayoutString(layoutStream, "", "HFilename", layoutWindow()->hCodeFile().relFilePath(fi.dir()));
  writeLayoutString(layoutStream, "", "DataFilename", layoutWindow()->dataFile().relFilePath(fi.dir()));
  writeLayoutRecursive(&layoutWindow()->objectTree(), layoutStream, 0);
  return 0;
}



int TFileIO::writeLayoutRecursive(TStoreType *tree, QTextStream &ts, int tabs) {
  if (!tree)
    return 0;


  QString tabstr0 = setTabs(tabs);
  ts << tabstr0 << "Object {" << endl;
  
  QString tabstr = setTabs(++tabs);

  QString cname = tree->storeTypeName();
  if (cname == "TLayoutWindow")
    cname = "QWidget";
  
  writeLayoutString(ts, tabstr, "Classname", cname);
  tree->writeLayout(ts, tabs);

  for (uint i=0 ; i<tree->numNodes() ; i++)
    writeLayoutRecursive((TStoreType*)tree->child(i), ts, tabs);

  ts << tabstr0 << "}" << endl;
  return 1;
}





int TFileIO::readLayout(void) {
  if (layoutFilename()=="") {
    KMsgBox::message(NULL, "Error", "There has been no layout file selected!");
    return 1;
  }

  QFileInfo fi(layoutFilename());
  //cout << "About to set directory to: " << fi.dirPath(TRUE) << endl;
  QDir::setCurrent(fi.dirPath(TRUE));

  QFile layoutFile(fi.fileName());
  QTextStream layoutStream(&layoutFile);

  if (!layoutFile.open(IO_ReadOnly)) {
	  QString errmsg = layoutFilename();
	  errmsg.prepend("Unable to open the file: \n\t");
		 
	  KMsgBox::message(NULL, "Error", errmsg, KMsgBox::EXCLAMATION, "OK");
	  return -1;
  }

  readLayoutRecursive(NULL, layoutStream);

  //layoutWindow()->dumpObjectTree();
  TStoreType *newTree = &layoutWindow()->objectTree();

  for ( ; newTree ; newTree = newTree->next() )
    newTree->postLayoutLoad();
  return 0;
}



TStoreType *TFileIO::readLayoutRecursive(TStoreType *tree, QTextStream &is) {
	QString      tmp, propname;
	TStoreType *curTree = NULL;

  
	is >> propname;
	if (propname == "}") {
		return 0;
	}
	is >> tmp;
	while (tmp != "{") {
		propname = propname + " " + tmp;
		is >> tmp;
	}

	while (propname != "}" && !is.eof()) {
		if (propname == "CFilename") {
			layoutWindow()->cCodeFile(TFileInfo(readLayoutString(is)));
			if (!loadFile(layoutWindow()->cCodeFile())) {
				QString errmsg = layoutWindow()->cCodeFile().filePath();
				errmsg.prepend("Unable to open the C++ file: \n\t");
				
				KMsgBox::message(NULL, "Error", errmsg, KMsgBox::EXCLAMATION, "OK");
				return NULL;
			}
		}
		else if (propname == "HFilename") {
			layoutWindow()->hCodeFile(TFileInfo(readLayoutString(is)));
			if (!loadFile(layoutWindow()->hCodeFile())) {
				QString errmsg = layoutWindow()->hCodeFile().filePath();
				errmsg.prepend("Unable to open the header file: \n\t");
				
				KMsgBox::message(NULL, "Error", errmsg, KMsgBox::EXCLAMATION, "OK");
				return NULL;
			}
		}
		else if (propname == "DataFilename") {
			layoutWindow()->dataFile(TFileInfo(readLayoutString(is)));
		}
		else if (propname == "Classname") {
			// Create an instance of this class here
			QString typeString = readLayoutString(is);
      
			TStoreTypeCreateFn type = store()->type(typeString);
      
			//if (tree == NULL) 
			//	curTree = &layoutWindow()->objectTree();
			//else
			curTree = layoutWindow()->create(tree, type);
		}
		else if (propname == "Object") {
			readLayoutRecursive(curTree, is);
		}
		else {
			if (curTree)
				curTree->readLayoutItem(is, propname);
		}
    
		is >> propname;
		if (propname == "}"  || is.eof()) {
			return curTree;
		}
		is >> tmp;
		while (tmp != "{") {
			propname = propname + " " + tmp;
			is >> tmp;
		}
	}
	return curTree;
}



bool TFileIO::loadFile(QFileInfo fi) {
  QFile f(fi.filePath());
  if (f.open(IO_ReadOnly)) {
	  QTextStream ts(&f);
	  QString foo;
	  foo = ts.readLine();
	  while (!ts.eof())
		  foo = foo + "\n" + ts.readLine();
	  f.close();
	  layoutWindow()->setText(fi.fileName(), foo);

	  return TRUE;
  }
  else
	  return FALSE;
}



QString TFileIO::delimitString(QString str) {
  QString rVal = str;
  
  rVal.replace(QRegExp("\""), "\\\"");
  return rVal;
}

QString TFileIO::undelimitString(QString str) {
  QString rVal = str;
  
  rVal.replace(QRegExp("\\\""), "\"");
  return rVal;
}


void TFileIO::writeLayoutInt(QTextStream &os, QString tabstr, QString name, int value) {
  os << tabstr << name << " { " << value << " }" << endl;
}

void TFileIO::writeLayoutBool(QTextStream &os, QString tabstr, QString name, bool value) {
  QString foo = value ? "True" : "False";
  os << tabstr << name << " { " << foo << " }" << endl;
}

void TFileIO::writeLayoutString(QTextStream &os, QString tabstr, QString name, QString value) {
  os << tabstr << name << " { \"" << delimitString(value) << "\" }" << endl;
}
void TFileIO::writeLayoutData(QTextStream &os, QString tabstr, QString name, QByteArray array) {
  char buffer[100];
  char *ptr = buffer;

  static int num=1;
  QString s;
  s.sprintf ("out%02d", num++);
  QFile file(s);
  file.open(IO_WriteOnly);
  QDataStream ds(&file);
  ds << array;
  file.close();

  os << tabstr << name << " { " << endl;

  memset(buffer, 0, 100);
  ulong l = array.size();
  sprintf (ptr, "%02X", (uchar)(l >> 24)) ; ptr += 2;
  sprintf (ptr, "%02X", (uchar)(l >> 16)) ; ptr += 2;
  sprintf (ptr, "%02X", (uchar)(l >>  8)) ; ptr += 2;
  sprintf (ptr, "%02X", (uchar)l) ; ptr += 2;

  for (uint i=0 ; i<array.size() ; i++) {
    sprintf(ptr, "%02X", (uint)((uchar)array[i]) );
    ptr += 2;

    if (((i+4)%32)==31) {
      os << tabstr << TABSTRING << buffer << endl;
      memset(buffer, 0, 100);
      ptr = buffer;
    }
  }
  if (ptr != buffer)
    os << tabstr << TABSTRING << buffer << endl;

  os << tabstr << "}" << endl;
}



void TFileIO::readLayoutUntil(QTextStream &is, char c) {
  char ch[4];
  do {
    is.readRawBytes(ch, 1);
  } while (ch[0] != c && !is.eof());
}

int TFileIO::readLayoutInt(QTextStream &is) {
  int value;

  //readLayoutUntil(is, '{');
  is >> value;
  readLayoutUntil(is, '}');
  return value;
}

bool TFileIO::readLayoutBool(QTextStream &is) {
  QString str;

  //readLayoutUntil(is, '{');
  is >> str;
  readLayoutUntil(is, '}');

  bool value = toupper(str[0])=='T' ? TRUE : FALSE;
  return value; 
}

QString TFileIO::readLayoutString(QTextStream &is) {
  QString rVal;
  
  rVal = readDelimitedString(is);
  readLayoutUntil(is, '}');

  return rVal;
}

QString TFileIO::readDelimitedString(QTextStream &is) {
  QString rVal;
  char ch[3];

  do {
    is.readRawBytes(ch, 1);
  }
  while((ch[0] != '\"') && (ch[0]!='}'));
  
  if (ch[0]=='}')
    return "";

  while (1) {
    is.readRawBytes(ch, 1);

    if (ch[0]=='\"') // delimited strings are always ended by a space
      return rVal;

    if (ch[0]=='\\') // read the next char, no matter what it is
      is.readRawBytes(ch, 1);
    rVal += ch[0];
  }
  return rVal;
}






inline uchar TFileIO::readByte(QTextStream &is) {
  char ch1, ch2;
  int rVal;

  do {
    is >> ch1;
  } while (!isalnum(ch1));
  do {
    is >> ch2;
  } while (!isalnum(ch2));

  ch1 = toupper(ch1);
  ch1 = (ch1>='0' && ch1<='9') ? ch1-'0' : 10+ch1-'A';

  ch2 = toupper(ch2);
  ch2 = (ch2>='0' && ch2<='9') ? ch2-'0' : 10+ch2-'A';

  rVal = (ch1<<4)+ch2;
  return rVal;
}



QByteArray TFileIO::readLayoutData(QTextStream &is) {
  QByteArray array;
  ulong a,b,c,d;
  static int num=1;
  //readLayoutUntil(is, '{');

  a = readByte(is);
  b = readByte(is);
  c = readByte(is);
  d = readByte(is);
  ulong size = (a<<24) + (b<<16) + (c<<8) + d;

  array.resize(size);

  for (uint i=0 ; i<array.size() ; i++)
    array[i] = readByte(is);

  QString s;
  s.sprintf ("in%02d", num++);
  QFile file(s);
  file.open(IO_WriteOnly);
  QDataStream ds(&file);
  ds << array;
  file.close();

  readLayoutUntil(is, '}');
  return array;
}





int TFileIO::writeCode() {
  bool rVal = dumpH();
  if (rVal) 
    rVal = dumpC();
  if (rVal)
    rVal = dumpData();
  return rVal;
}


bool TFileIO::dumpH() {
  QString hCodeFileName;
  
  if (!layoutWindow()->hFileEverSaved()) {
    QFileDialog *fd = new QFileDialog(layoutWindow()->mainWindow()->saveCodePath(), 
				      "*.h", NULL, "Save header file as", TRUE);
    fd->setCaption("Save header file as");
    if (fd->exec()) {
      layoutWindow()->mainWindow()->setSaveCodePath(fd->dirPath());
      hCodeFileName = fd->selectedFile();
      layoutWindow()->hCodeFile(TFileInfo(hCodeFileName));
    }
    else 
      return FALSE;
  }  
  else {
	 hCodeFileName = layoutWindow()->hCodeFile().filePath();
	 QString fPath = hCodeFileName + "~";

	 if (0!=unlink(fPath) && errno!=ENOENT) {
		 QString errmsg;
		 errmsg.sprintf("Unable to delete old backup header file:\n\t%s\n\t%s\nContinue ?", 
						(const char *)fPath, strerror(errno));
		 if (KMsgBox::DB_SECOND==KMsgBox::yesNo(NULL, "Error", errmsg, KMsgBox::EXCLAMATION))
			 return FALSE;
	 }
	 if (0!=rename (hCodeFileName, fPath)) {
		 QString errmsg;
		 errmsg.sprintf("Unable to rename header file to backup:\n\t%s\n\t%s\nContinue ?", 
						(const char *)hCodeFileName, strerror(errno));
		 if (KMsgBox::DB_SECOND==KMsgBox::yesNo(NULL, "Error", errmsg, KMsgBox::EXCLAMATION))
			 return FALSE;
	 }
  }

  QFile file(hCodeFileName);
  QString hCode = layoutWindow()->text(layoutWindow()->hCodeFile().fileName());

  file.open (IO_WriteOnly);
  file.writeBlock(hCode, hCode.length());
  file.close();

  layoutWindow()->hFileEverSaved(TRUE);
  return TRUE;
}



bool TFileIO::dumpC() {
  QString cCodeFileName;

  if (!layoutWindow()->cFileEverSaved()) {
    QFileDialog *fd = new QFileDialog(layoutWindow()->mainWindow()->saveCodePath(), 
				      "*.cpp", NULL, "Save CPP file as", TRUE);
    fd->setCaption("Save cpp file as");
    if (fd->exec()) {
      layoutWindow()->mainWindow()->setSaveCodePath(fd->dirPath());
      cCodeFileName = fd->selectedFile();
      layoutWindow()->cCodeFile(TFileInfo(cCodeFileName));
    }
    else 
      return FALSE;
  }  
  else {
	  cCodeFileName = layoutWindow()->cCodeFile().filePath();
	  QString fPath = cCodeFileName + "~";

	 if (0!=unlink(fPath) && errno!=ENOENT) {
		 QString errmsg;
		 errmsg.sprintf("Unable to delete old backup CPP file:\n\t%s\n\t%s\nContinue ?", 
						(const char *)fPath, strerror(errno));
		 if (KMsgBox::DB_SECOND==KMsgBox::yesNo(NULL, "Error", errmsg, KMsgBox::EXCLAMATION))
			 return FALSE;
	 }
	 if (0!=rename (cCodeFileName, fPath)) {
		 QString errmsg;
		 errmsg.sprintf("Unable to rename CPP file to backup:\n\t%s\n\t%s\nContinue ?", 
						(const char *)cCodeFileName, strerror(errno));
		 if (KMsgBox::DB_SECOND==KMsgBox::yesNo(NULL, "Error", errmsg, KMsgBox::EXCLAMATION))
			 return FALSE;
	 }
  }

  QFile file(cCodeFileName);
  QString cCode = layoutWindow()->text(layoutWindow()->cCodeFile().fileName());

  file.open (IO_WriteOnly);
  file.writeBlock(cCode, cCode.length());
  file.close();

  layoutWindow()->cFileEverSaved(TRUE);

  return TRUE;
}



void TFileIO::clearTempNames(void) {
  TStoreType *ptr = &layoutWindow()->objectTree();
  while(ptr) {
    ptr->tempName("");
    ptr = (TStoreType*)ptr->next();
  }
}

void TFileIO::setTempNames(void) {
  TStoreType *ptr = &layoutWindow()->objectTree();
  int count=0;
  QString foo;
  
  while(ptr) {
    foo.sprintf("tmp%d", count++);
    while (!ptr->tempName(foo))
      foo.sprintf("tmp%d", count++);
    
    ptr = (TStoreType*)ptr->next();
  }
}












bool TFileIO::dumpData() {
  QString dataFileName;

  if (!layoutWindow()->dataEverSaved()) {
    QFileDialog *fd = new QFileDialog(layoutWindow()->mainWindow()->saveCodePath(), 
				      "*.cpp", NULL, "Save data file as", TRUE);
    fd->setCaption("Save data file as");
    bool foolish = TRUE;

    while (foolish) {
      if (fd->exec()) {
	layoutWindow()->mainWindow()->setSaveCodePath(fd->dirPath());
	dataFileName = fd->selectedFile();
	
	QFileInfo dataFileInfo(dataFileName);
	if (dataFileInfo.fileName() == layoutWindow()->cCodeFile().fileName()) {
	  KMsgBox::message(NULL, "Error", "You can not save the data file with the same name as the C file!.", KMsgBox::EXCLAMATION);
	}
	else {
	  foolish = FALSE;
	  layoutWindow()->dataFile(TFileInfo(dataFileName));
	}
      }
      else 
	return FALSE;
    }
  }
  else {
    dataFileName = layoutWindow()->dataFile().filePath();
  }

  QFile lDataFile(dataFileName);
  QTextStream dataStream(&lDataFile);

  lDataFile.open(IO_WriteOnly);


  if (0==strcmp(layoutWindow()->hCodeFile().dirPath(TRUE), 
		layoutWindow()->dataFile().dirPath(TRUE)))
    dataStream << "#include \"" << layoutWindow()->hCodeFile().fileName() << "\"\n" << endl;
  else
    dataStream << "#include <" << layoutWindow()->hCodeFile().fileName() << ">\n" << endl;

  setTempNames();
  TStoreType *tree = &layoutWindow()->objectTree();
  for ( ; tree ; tree = tree->next() ) {
    tree->writeCFileData(dataStream, tree->validName());
    dataStream << endl;
  }

  dataStream << "void " << layoutWindow()->name() << "::initLayout(void) {" << endl;
  dumpCConstructor(dataStream);
  dataStream << "}" << endl;

  layoutWindow()->dataEverSaved(TRUE);

  return TRUE;
}



void TFileIO::dumpCConstructor(QTextStream &ts) {
	clearTempNames();
	setTempNames();
  
	TStoreType *ptr = &layoutWindow()->objectTree();
	for ( ; ptr ; ptr = ptr->next() ) {
		dumpSingleObject(ts, ptr);
	}

	ptr = &layoutWindow()->objectTree();
	for ( ; ptr ; ptr=ptr->next() ) {
		if (ptr->object() == layoutWindow())
			ptr->writeCFileProperties(ts, "this");
		else 
			ptr->writeCFileProperties(ts, ptr->validName());
		ts << endl;
	}

	for ( ptr=&layoutWindow()->objectTree() ; ptr ; ptr=ptr->next()) {
		TLinkedConnectionsListIt it(*ptr->connections());
		for ( ; it.current() ; ++it) {
			// for every set of signals and slots connected to 
			// this signal...
			TLinkedConnections *link = it.current();
			TConnectionListIt cit(link->connections);

			for ( ; cit.current() ; ++cit ) {
				TConnection *conn = cit.current();

				ts << "\tconnect(" << ptr->validName() 
				   << ", SIGNAL("  // we are always comming from a SIGNAL
				   << link->source->name << "("
				   << link->source->args << ")), ";

				// What receives the SIGNAL
				if (conn->dest == &layoutWindow()->objectTree())
					ts << "this";
				else
					ts << conn->dest->validName();

				// Is the receiver a SLOT or a SLOT
				if (conn->connection->type==ctSlot)
					ts << ", SLOT(" ;
				else
					ts << ", SIGNAL(";

				// Which of the source SIGNALS args do we want the dest to have
				ts << (const char *)conn->connection->name 
				   << "(" << (const char *)conn->connection->args
				   << ")));" << endl;
			}
		}
	}
}



void TFileIO::dumpSingleObject(QTextStream &ts, TStoreType *tree) {
  // If the object does not have a valid name, then declare variable with the 
  // temp name in it.
  if (tree->object() != layoutWindow()) {
    if (tree->validName()==tree->tempName())
      ts << "\t" << tree->storeTypeName() << " *"
	 << tree->validName() << ";" << endl;
    
    if (tree->object()->parent()==layoutWindow())
      // Object is created as a child of the class
      tree->writeCFileConstructor(ts, tree->validName(), "this");
    else
	// Object is created as a child of a member in the class
      tree->writeCFileConstructor(ts, tree->validName(),
				  tree->parent()->validName());
    ts << endl;
  }
}


QString TFileIO::defineWrapperName(const char *str) {
  QFileInfo info(str);
  
  QString tmp;
  tmp.sprintf("_%s_", info.fileName().data());
  
  tmp.replace(QRegExp("[^a-zA-Z0-9]"), "_");
  return tmp.upper();
}
