#include <stdio.h>
#include <iostream.h>
#include <stdlib.h>
#include <assert.h>

#include <qapp.h>
#include <qpainter.h>
#include <qpushbt.h>
#include <qdialog.h>
#include <qlined.h>
#include <qfiledlg.h>
#include <qmsgbox.h>
#include <qmlined.h>

#include <kmsgbox.h>

#include "TLayoutWindow.h"

#include "TPropertyWidget.h"
#include "THandleGroup.h"
#include "TStore.h"
#include "TMainWindow.h"
#include "TQuestionDialog.h"
#include "TStoreTypeQWidget.h"
#include "TFileIO.h"
#include "TConnectionDlg.h"
#include "TCodeEdit.h"
#include "TEditWindow.h"

#define min(a,b) (((a)<(b)) ? (a) : (b))
#define Max(a,b) ((a)>(b) ? (a) : (b))

/*!
  Constructor for TLayoutWindow
  */
TLayoutWindow::TLayoutWindow(TMainWindow *mw, QWidget *parent, const char *name) : QWindow::QWindow(parent, name) {
  initMetaObject();
  
  popupMenu = new QPopupMenu;
  popupMenu->setCheckable(TRUE);
  popupMenu->insertItem("Properties", this, SLOT(showProperties()) );
  popupMenu->insertItem("Alignment", this, SLOT(showAlignment()) );
  popupMenu->insertItem("Connections", this, SLOT(showConnections()) );
  popupMenu->insertItem("Grid Size", this, SLOT(setGrid()) );
  fGridEnabledItem = popupMenu->insertItem("Show Grid", this, SLOT(toggleGridEnabled()));
  fSnapToGridItem = popupMenu->insertItem("Snap to Grid", this, SLOT(toggleSnapToGrid()) );
  
  QPopupMenu *editMenu = new QPopupMenu;
  editMenu->insertItem("copy", this, SLOT(copy()) );
  editMenu->insertItem("cut", this, SLOT(cut()) );
  editMenu->insertItem("paste", this, SLOT(paste()) );
  editMenu->insertItem("delete", this, SLOT(remove()) );
  popupMenu->insertItem("Edit", editMenu);

  
  fHandleGroupList.setAutoDelete(TRUE);
  fMovePosList.setAutoDelete(TRUE);

  fObjectTree = NULL;
  fFileName = "";
  fGridSize = 0;
  fInGeometryLoop = FALSE;
  fSnapToGrid = TRUE;
  fGridVisible = TRUE;
  fModified = TRUE;
  
  //installEventFilter(this);

  setMainWindow(mw);

  connect(this, SIGNAL(textChanged(const char *)), SLOT(makeModified()));
  fBusy = FALSE;  

  emit(sigSetActive(fObjectTree));
}



TLayoutWindow::~TLayoutWindow(void) {
  delete(popupMenu);
}








/*!
  \fn TLayoutWindow::eventFilterMousePress(QObject*, QMouseEvent*)
  Event Filter for mouse Presses
  */
bool TLayoutWindow::eventFilterMousePress(QObject *sender, QMouseEvent *ev) {
	QObject *working = NULL;
	int container;

	mainWindow()->setActiveLayoutWindow(this);

	if (ev->button() & RightButton) {
		popupMenu->popup(((QWidget*)sender)->mapToGlobal(ev->pos()));
	}
	else {
		if (sender->isWidgetType()) {
			working = sender;
      
			TStoreType *tree = objectTree().find(sender);
			if (tree)
				container = tree->canItHaveChildren() ? AltButton : 0;
			else 
				container = 0;
      
			// toggle Container bit
			int state = ev->state() ^ container;
      
      
			if (state == NoButton) {
				// Start a new List
				deselectAll();
				selectObject(sender);
			}
			else if (state & ControlButton) {
				// Toggle sender with the selected list
				if (fSelectedList.find(sender)==-1)
					selectObject(sender);
				else
					deselectObject(sender);
			}
			else if (state & AltButton) {
				TStoreType *leaf = NULL;

				// Make a new object as a child of this one
				if (fSnapToGrid && gridSize()>1) {
					int x = gridSize() * (int)(ev->pos().x() / gridSize());
					int y = gridSize() * (int)(ev->pos().y() / gridSize());
					leaf = createHere(objectTree().find(sender), QPoint(x,y));
				}
				else
					leaf = createHere(objectTree().find(sender), ev->pos());
	
				if (leaf)
					working = leaf->object();
				else
					working=sender;
				deselectAll();
				selectObject(working);
			}
      
			emit(sigSetActive( objectTree().find(working) ));
      
			if (fSelectedList.find(working) != -1) {
				if (working != fObjectTree->object()) {
					fMovingWidget = TRUE;
	
					fOffset = QCursor::pos();
					fMovePosList.clear();
					QObjectListIt it(fSelectedList);
					for ( ; it.current() ; ++it) {
						if (it.current()->isWidgetType())
							fMovePosList.append(new QPoint(((QWidget*)it.current())->pos()));
						else
							fMovePosList.append(new QPoint(0,0));
					}
				}
			}
		}
	}
	return TRUE;
}




bool TLayoutWindow::eventFilterMouseRelease(QObject *, QMouseEvent *) {
  fMovingWidget = FALSE;
  return FALSE;
}




bool TLayoutWindow::eventFilterMouseMove(QObject *, QMouseEvent *) {
  if (fMovingWidget) {
    QPoint p = QCursor::pos();
    
    p -= fOffset; // p now holds the new position of the widget 
    if (fSnapToGrid && gridSize()>1) {
      p /= (int) gridSize();
      p *= (int) gridSize();
    }
    QObjectListIt it(fSelectedList);
    QListIterator<QPoint> pit(fMovePosList);
    
    for ( ; it.current() ; ++it, ++pit) {
      if (it.current()->isWidgetType()) {
	QWidget *w = (QWidget *)it.current();
      
	w->move(*pit.current() + p);
      }
    }
  }
  return FALSE;
}





bool TLayoutWindow::eventFilter(QObject *sender, QEvent *ev) {
  /* Event filter which is applied to all children created from
     this widget.
     
     This is used to handle current selection, multiple selection,
     resize signaling and other interaction between the widgets and
     their handlers.
     */
  
  switch (ev->type()) {
  case Event_MouseButtonPress:
    return eventFilterMousePress(sender, Q_MOUSE_EVENT(ev));
    break;
  case Event_MouseButtonRelease:
    return eventFilterMouseRelease(sender, Q_MOUSE_EVENT(ev));
    break;
  case Event_MouseMove:
    return eventFilterMouseMove(sender, Q_MOUSE_EVENT(ev));
    break;
  case Event_Move:
  case Event_Resize:
  {
    QWidget *w = (QWidget *)sender;
    if (fSnapToGrid && gridSize()>1) {
      QRect r;
      r.setTop( gridSize() * ((int)(w->pos().y()/gridSize())));
      r.setLeft( gridSize() * ((int)(w->pos().x()/gridSize())));
      r.setWidth( gridSize() * ((int)(w->size().width()/gridSize())));
      r.setHeight( gridSize() * ((int)(w->size().height()/gridSize())));
      
      emit(sigChildMoved(w, r));

    }
    else
      emit(sigChildMoved(w, QRect(w->pos(), w->size())));
    break;
  }
  case Event_Paint:
  {
	  QPainter p;
	  QBrush brush(yellow);
	  QPaintEvent *pev = (QPaintEvent*)ev;

	  if (fGridVisible && gridSize()>1) {
		  int left = (pev->rect().left() / gridSize());
		  left *= gridSize();
    
		  int top = (pev->rect().top() / gridSize());
		  top *= gridSize();
    
		  int right = (pev->rect().right() / gridSize()) + 1;
		  right *= gridSize();
    
		  int bottom = (pev->rect().bottom() / gridSize()) + 1;
		  bottom *= gridSize();
    
		  p.begin((QWidget*)sender);
		  p.setPen(QPen(black, 1, SolidLine));
		  for (int y=top ; y<bottom ; y+=gridSize())
			  for (int x=left ; x<right ; x+=gridSize())
				  p.drawPoint(x,y);
		  p.end();
	  }
	  break;
  }
  }
  
  return FALSE;
}




void TLayoutWindow::childDestroyed() {
  const QObject *obj = sender();

  if (obj) {
    if (!fBusy) {
      TStoreType *tree = objectTree().find(obj);
      if (tree)
	// This has the side effect of removing the declaration from the Header
	codeEdit.removeVariable(tree);
    }
  }
}




void TLayoutWindow::selectObject(const QObject *w) {
  THandleGroup *handler;
  
  // Make sure that we remove it first, so that it becomes the last element
  deselectObject(w);
  
  // Turn what was the last entry from red to yellow
  if (fHandleGroupList.count())
    fHandleGroupList.getLast()->color(yellow);
  
  // We may have changed the first widget, do make sure that the
  // first one is tagged green
  if (fHandleGroupList.count())
    fHandleGroupList.getFirst()->color(green);
  
  // Set up a handler group for this object
  if (w->isWidgetType() && w!=this) {
    QWidget *foo = (QWidget *)w;

    handler = new THandleGroup( foo->parentWidget() ? foo->parentWidget() : this);
    connect (this,    SIGNAL(sigChildMoved(QObject*, const QRect&)),
	     handler, SLOT(updateWatched(QObject*, const QRect&)));
    
    connect (handler, SIGNAL(sigHandleGroupRectChanged(QObject*, const QRect&)),
	     this,    SLOT(slotChildMoved(QObject*, const QRect&)));
    
    handler->gridSize(snapToGrid() ? gridSize() : 0);
    handler->watched(foo);
    handler->color(red);

	handler->showHandles();
	handler->raiseHandles();
	if (foo == fObjectTree->object())
		handler->hideHandles();
  }
  else
    handler = new THandleGroup(this);

  fHandleGroupList.append(handler);
  fSelectedList.append(w);
}


void TLayoutWindow::deselectObject(const QObject *w) {
  int pos = fSelectedList.find(w);
  if (pos != -1) {
    fSelectedList.remove(pos);
    fHandleGroupList.remove(pos);
    
    if (fHandleGroupList.count()>1)
      fHandleGroupList.getFirst()->color(green);
  }
}

void TLayoutWindow::deselectAll(void) {
  fSelectedList.clear();
  fHandleGroupList.clear();
}


void TLayoutWindow::show(void) {
  QWindow::show();
}

void TLayoutWindow::hide(void) {
  QWindow::hide();
}


TStoreType *TLayoutWindow::create(TStoreType *parent, TStoreTypeCreateFn createtype) {
	TStoreType *tree=NULL;

	if (createtype) {
		tree = createtype(parent, this);
		if (!parent) {
			fObjectTree = tree;

			QString a,b,c;
			a.sprintf ("%s.h", tree->object()->name());
			b = tree->storeTypeName();
			c.sprintf ("%s.data", tree->object()->name());
			codeEdit.initialise(tree->object()->name(), a,b,c);
			
			TEditWindow *ew = fMainWindow->editWindow();
			ew->editFile(this, codeEdit.cCodeFile().fileName(), codeEdit.cCode(), 
						 codeEdit.cCodeLine());
			ew->editFile(this, codeEdit.hCodeFile().fileName(), codeEdit.hCode(), 
						 codeEdit.hCodeLine());


			QHBoxLayout *layout = new QHBoxLayout(this);
			layout->addWidget((QWidget*)tree->object(), 1);
			layout->activate();
		}

		tree->object()->installEventFilter(this);
		connectToChildDestroyed(tree);

		if (tree->object()->isWidgetType()) {
			((QWidget*)tree->object())->show();
			((QWidget*)tree->object())->setCursor(sizeAllCursor);
		}

		propertyWidget()->addNewObject(tree);
		codeEdit.addNewObject(tree);
		emit(textChanged(codeEdit.hCodeFile().fileName()));

		emit(sigCreatedObject(QRect(QPoint(0,0), ((QWidget*)tree->object())->size() )));
		emit(sigCreatedObject(tree));
		makeModified();
	}
	return tree;
}

TStoreType *TLayoutWindow::createHere(TStoreType *parent, QPoint pos) {
  TStoreType *tree=NULL;
  TStoreTypeCreateFn createtype = fStore->type(mainWindow()->typeToCreate());
  
  if (createtype) {
    tree = create(parent, createtype);

    if (tree->object()->isWidgetType()) {
      ((QWidget*)tree->object())->move(pos);
    }
    
    // Make sure that this is the only selected object;
    deselectAll();
    
    emit(sigSetActive(tree));
  }
  return tree;
}




void TLayoutWindow::setActive(QObject *obj) {
  deselectAll();
  selectObject(obj);
  emit(sigSetActive(objectTree().find(obj)));
}


void TLayoutWindow::setActive(void) {
  if (sender()->isWidgetType()) {
    selectObject(sender());
    emit(sigSetActive( objectTree().find(sender()) ));
  }
}

void TLayoutWindow::closeEvent(QCloseEvent *ev) {
  if (modified()) {
    int i = QMessageBox::warning(this, "Window not saved!", 
				 "Save this window?", "Yes",
				 "No", "Cancel", 0, 2);
    switch (i) {
    case 0:
      save();
      break;
    case 1:
      ev->accept();
      break;
    case 2:
      ev->ignore();
      break;
    }
  }
}



void TLayoutWindow::slotChildMoved(QObject *wid, const QRect &r) {
  if (fInGeometryLoop)
    return;
  fInGeometryLoop=TRUE;
  
  if (wid) {
    QRect bar;
    if (fSnapToGrid && gridSize()>1) {
      int foo = (int)gridSize() * ((int)(r.top()/gridSize()));
      bar.setTop( foo );
      
      foo = (int)gridSize() * ((int)(r.left()/gridSize()));
      bar.setLeft( foo );
      
      foo =  (int)gridSize() * ((int)(r.width()/gridSize()));
      bar.setWidth( foo );
      
      foo = (int)gridSize() * ((int)(r.height()/gridSize()));
      bar.setHeight( foo );
    }
    else
      bar = r;
    
    emit(sigChildMoved(wid, bar));
	if (wid==fObjectTree->object()) {
		resize(((QWidget*)wid)->size());
	}
  }
  fInGeometryLoop=FALSE;
}


void TLayoutWindow::gridSize(uint s) {
  if (s!=fGridSize) {
    QListIterator<THandleGroup> it(fHandleGroupList);
    for ( ; it.current() ; ++it )
      it.current()->gridSize(s);
    fGridSize = s;


	TStoreType *tree = &objectTree();
	while (tree) {
		if (tree->canItHaveChildren())
			if (tree->object()->isWidgetType())
				((QWidget*)tree->object())->repaint(TRUE);
		tree = tree->next();
	}
  }
}


void TLayoutWindow::setGrid() {
  QString tmp;
  tmp.setNum(gridSize());
  TQuestionDialog *qd = new TQuestionDialog("Grid Size:", tmp);
  qd->adjustSize();
  if (qd->exec()) {
    gridSize(atoi(qd->text()));
  }
  delete qd;
}


void TLayoutWindow::toggleGridEnabled() {
  bool newState = popupMenu->isItemChecked(fGridEnabledItem);

  setGridVisible(!newState);

}

void TLayoutWindow::setGridVisible(bool vis) {
  fGridVisible = vis;
  popupMenu->setItemChecked(fGridEnabledItem, vis);  

  TStoreType *tree = &objectTree();
  while (tree) {
	  if (tree->canItHaveChildren())
		  if (tree->object()->isWidgetType())
			  ((QWidget*)tree->object())->repaint(TRUE);
	  tree = tree->next();
  }
  repaint(TRUE);
}




void TLayoutWindow::toggleSnapToGrid() {
  bool newState = popupMenu->isItemChecked(fSnapToGridItem);

  setSnapToGrid(!newState);
  popupMenu->setItemChecked(fSnapToGridItem, !newState);
}



void TLayoutWindow::setSnapToGrid(bool snap) {
  fSnapToGrid = snap;

  QListIterator<THandleGroup> it(fHandleGroupList);
  for ( ; it.current() ; ++it )
    it.current()->gridSize(snap ? gridSize() : 0);
  popupMenu->setItemChecked(fSnapToGridItem, snap);
}

void TLayoutWindow::setMainWindow(TMainWindow *m) {
  if (fMainWindow!=m) {
    fMainWindow = m;

    store(m->store());
    if (propertyWidget()) {
      TPropertyWidget *p = propertyWidget();
      connect (this, SIGNAL(sigSetActive(TStoreType*)),
	       p, SLOT(activeObject(TStoreType *)));
      
      connect (this, SIGNAL(sigChildMoved(QObject*, const QRect&)),
	       p, SLOT(widgetGeometryChanged(QObject*, const QRect &)));
      
      connect (p, SIGNAL(sigObjectChanged(QObject *)),
	       this, SLOT(setActive(QObject *)));

      if (fObjectTree)
		  propertyWidget()->addNewObject(fObjectTree);
    }

    TConnectionWindow *c = fMainWindow->connectionWindow();
    connect(this, SIGNAL(sigSetActive(TStoreType*)), 
	    c, SLOT(activeObject(TStoreType*)));
    connect(this, SIGNAL(sigCreatedObject(TStoreType*)), 
	    c, SLOT(addNewObject(TStoreType*)));
    connect (c, SIGNAL(sigObjectChanged(QObject *)),
	     this, SLOT(setActive(QObject *)));

  }
}

TPropertyWidget *TLayoutWindow::propertyWidget() { 
  return mainWindow() ? mainWindow()->propertyWidget() : 0; 
};





void TLayoutWindow::remove() {
  QObjectListIt it(fSelectedList);
  emit(sigSetActive( &objectTree() ));
  for ( it.toFirst() ; it.current() ; it.toFirst()) {
    QObject *rootObject = it.current();

    TStoreType *origTree = fObjectTree->find(rootObject);

    if (rootObject==this) {
      continue;
    }
    
    removeItem(origTree);
  }
}





void TLayoutWindow::removeItem(TStoreType *origTree) {
  if (!origTree) {
    return;
  }

  for (int i=origTree->numNodes() ; i>0 ; i--)
    removeItem( origTree->child(i) );

  TStoreType *par = origTree->parent();
  if (!par) {
    return;
  }

  deselectObject(origTree->object());
  delete origTree->object();

  par->removeChild(origTree);
  delete origTree;

}

void TLayoutWindow::cut() {
  copy();
  remove();
}



void TLayoutWindow::copy() {
  mainWindow()->clearClipBoard();

  QObjectListIt it(fSelectedList);
  for ( it.toFirst() ; it.current() ; ++it) {
    QObject *rootObject = it.current();

    TStoreType *origTree = fObjectTree->find(rootObject);
    if (!origTree)
      continue;
    
    TStoreType *tree = copyTree(NULL, origTree, FALSE);
    mainWindow()->addToClipBoard(tree);
  }
}


TStoreType *TLayoutWindow::copyTree(TStoreType *parentTree, TStoreType *sourceTree, bool onLayout) {
  TStoreType *newTree = copyObject(parentTree, sourceTree, onLayout);

  for (uint i=0 ; i<sourceTree->numNodes() ; i++) {
    copyTree(newTree, (TStoreType*) sourceTree->child(i), onLayout);
  }
  return newTree;
}

TStoreType* TLayoutWindow::copyObject(TStoreType *parentTree, TStoreType *source, bool onLayout) {
  QString str = source->storeTypeName();
  TStoreTypeCreateFn createtype = fStore->type( str );
  TStoreType *tree = NULL;

  if (!createtype)
    return NULL;
  
  if (onLayout) {
    tree = create(parentTree, createtype);
    if (tree->object()->isWidgetType())
      ((QWidget*)tree->object())->show();
  }
  else
    tree = createtype(parentTree, this);
  tree->copyDetails(source );
  
  tree->variableName(source->variableName());
  tree->tempName(source->tempName());

  return tree;
}






void TLayoutWindow::paste() {
  QObject *rootObject = fSelectedList.getLast();

  TStoreType *origTree = fObjectTree->find(rootObject);
  if (!origTree)
    return;

  if (rootObject != this) {
    if (!origTree->canItHaveChildren()) {
      QString msg;
      msg.sprintf("%s::%s does not normally have children,\nare you sure you wish to paste into it?", (const char *)origTree->storeTypeName(), rootObject->name());
      if (!KMsgBox::yesNo(NULL, "Paste", msg, KMsgBox::QUESTION))
	return;
    }
  }
  
  QListIterator<TStoreType> it(mainWindow()->clipBoard());
  for ( ; it.current() ; ++it)
    copyTree(origTree, it.current(), TRUE);
}


void TLayoutWindow::showProperties() {
  mainWindow()->showProperties();
}

void TLayoutWindow::showConnections() {
  mainWindow()->showConnections();
}

void TLayoutWindow::showAlignment() {
  mainWindow()->showAlignment();
}

void TLayoutWindow::makeModified() {
  fModified = TRUE;
}
void TLayoutWindow::clearModified() {
  fModified = FALSE;
}




void TLayoutWindow::save() {
  if (objectTree().object()->name()=="") {
    KMsgBox::message(NULL, "Save Error", "The window does not have a class name.");
    return;
  }

  if (filename()=="")
    saveAs();
  else {
    TFileIO fileio(*store(), *this);

    fileio.layoutFilename(QFileInfo(filename()));
    fileio.writeCode();
    fileio.writeLayout();
    clearModified();
  }
}


void TLayoutWindow::saveAs() {
  QFileDialog *fd = new QFileDialog(mainWindow()->saveDialogPath(), "*.dlg", NULL, 
				    "Save Dialog As", TRUE);

  if (fd->exec()) {
    mainWindow()->setSaveDialogPath(fd->dirPath());
    QString fname = fd->selectedFile();

    QFileInfo info(fname);
    
    TFileIO fileio(*store(), *this);

    fileio.layoutFilename(info);
    fileio.writeCode();
    fileio.writeLayout();
    clearModified();

    filename(fname);
  }
  delete fd;
}

bool TLayoutWindow::open(QFileInfo info) {
  fBusy = TRUE;

  TFileIO fileio(*store(), *this);

  fileio.layoutFilename(info);
  if (0==fileio.readLayout()) {
	  filename(info.filePath());
	  cFileEverSaved(TRUE);
	  hFileEverSaved(TRUE);
	  dataEverSaved(TRUE);
	  clearModified();

	  fBusy = FALSE;
	  //dumpObjectTree();

	  // set the geometry of this to be the geometry of the 
	  // first item that we have
	  setGeometry(QRect(((QWidget*)fObjectTree->object())->pos(),
						((QWidget*)fObjectTree->object())->size()));
	  return TRUE;
  }
  else
	  return FALSE;
}


void TLayoutWindow::generateCode() {
  if (objectTree().variableName()=="") {
    KMsgBox::message(NULL, "Save Error", "The window does not have a class name.");
    return;
  }

  if (CFilename().isEmpty() || ""==CFilename())
    generateCodeAs();
  else {
    TFileIO fileio(*store(), *this);
    fileio.writeCode();
  }
}

void TLayoutWindow::generateCodeAs() {
  QFileDialog *fd = new QFileDialog(mainWindow()->saveCodePath(), "*.cpp", NULL,
				    "Generate Code As", TRUE);
  if (fd->exec()) {
    mainWindow()->setSaveCodePath(fd->dirPath());
    QString filename = fd->selectedFile();

    QFileInfo info(filename);
    
    TFileIO fileio(*store(), *this);

    CFilename(filename);
    makeModified();
    
    fileio.writeCode();
  }
  delete fd;
}

void TLayoutWindow::editCode() {
  TEditWindow *ew = mainWindow()->editWindow();

  ew->show();
  ew->raise();
}

QString &TLayoutWindow::text(const char *name) {
  QString tmp=name;
  int i = tmp.findRev('/');

  if (i)
    tmp = tmp.mid(i+1, tmp.length());
  
  if (tmp==cCodeFile().fileName())
    return codeEdit.cCode();
  else
    return codeEdit.hCode();
}


void TLayoutWindow::setText(const char *filename, const char *text) {
  makeModified();
  if (0==strcmp(filename, codeEdit.cCodeFile().fileName() ))
    codeEdit.cCode(text);
  else
    codeEdit.hCode(text);
}



int TLayoutWindow::getTextRow(const char *file) {
  if (file==cCodeFile().fileName())
    return codeEdit.cCodeLine();
  else
    return codeEdit.hCodeLine();
}

void TLayoutWindow::setTextRow(const char *file, int row) {
  if (file==cCodeFile().fileName())
    codeEdit.cCodeLine(row);
  else
    codeEdit.hCodeLine(row);
}




void TLayoutWindow::renameClass(const char *oldName) {
  QString newName = fObjectTree->object()->name();

  codeEdit.renameClass(oldName, newName);

  fObjectTree->object()->setName(newName);
  propertyWidget()->updateObject(fObjectTree, oldName);

  emit(textChanged(codeEdit.cCodeFile().fileName()));
  emit(textChanged(codeEdit.hCodeFile().fileName()));
}


void TLayoutWindow::addSignalCode(const char *name, const char *args) {
  TSignalList *list = (TSignalList*)fObjectTree->findData("CreatedSignals");
  if (list) {
    TSignalListIt sit(*list);
    for ( ; sit.current() ; ++sit) {
      if ((0==strcmp(name, sit.current()->name)) &&
	  (0==strcmp(args, sit.current()->args))) {
	codeEdit.addSignalCode(sit.current());
	emit(textChanged(codeEdit.hCodeFile().fileName()));
      }
    }
  }
}
void TLayoutWindow::addSlotCode(const char *name, const char *args) {
  TSlotList *list = (TSlotList*)fObjectTree->findData("CreatedSlots");
  if (list) {
    TSlotListIt sit(*list);
    for ( ; sit.current() ; ++sit) {
      if ((0==strcmp(name, sit.current()->name)) &&
	  (0==strcmp(args, sit.current()->args))) {
	codeEdit.addSlotCode(sit.current());
	emit(textChanged(codeEdit.cCodeFile().fileName()));
	emit(textChanged(codeEdit.hCodeFile().fileName()));
      }
    }
  }
}

void TLayoutWindow::cCodeFile(TFileInfo f) { 
  TEditWindow *ew = mainWindow()->editWindow();
  
  ew->renameTab(this, codeEdit.cCodeFile().fileName(), f.fileName());
  codeEdit.cCodeFile(f); 
};

void TLayoutWindow::hCodeFile(TFileInfo f) { 
  TEditWindow *ew = mainWindow()->editWindow();
  
  ew->renameTab(this, codeEdit.hCodeFile().fileName(), f.fileName());
  codeEdit.hCodeFile(f); 
  emit(textChanged(codeEdit.cCodeFile().fileName()));
};

void TLayoutWindow::dataFile(TFileInfo f) { 
  codeEdit.dataFile(f); 
};


void TLayoutWindow::changeVariableName(TStoreType *tree, QString oldName) {
  codeEdit.changeVariableName(tree, oldName);
  emit(textChanged(codeEdit.hCodeFile().fileName()));
}


void TLayoutWindow::connectToChildDestroyed(TStoreType *tree) {
  connect(tree->object(), SIGNAL(destroyed()), this, SLOT(childDestroyed()));
}




// Sponge

void TLayoutWindow::setAlignTop(void) {
  int top=0;
  bool set=FALSE;
  
  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	set = TRUE;
	top = w->y();
      }
      else
	w->move(w->x(), top);
    }
  }
}




void TLayoutWindow::setAlignLeft(void) {
  int left=0;
  bool set=FALSE;
  
  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	set = TRUE;
	left = w->x();
      }
      else
	w->move(left, w->y());
    }
  }
}

void TLayoutWindow::setAlignRight(void) {
  int right=0;
  bool set=FALSE;
  
  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	set = TRUE;
	right = w->x() + w->width();
      }
      else
	w->move((right - w->width())+1, w->y());
    }
  }
};



void TLayoutWindow::setAlignBottom(void) {
  int bottom=0;
  bool set=FALSE;
  
  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	set = TRUE;
	bottom = w->y() + w->height();
      }
      else
	w->move(w->x(), ( bottom - w->height() ) + 1 );
    }
  }
};



void TLayoutWindow::setAlignWidths(void) {
  int width=0;
  bool set=FALSE;
  
  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	set = TRUE;
	width = w->width();
      }
      else
	w->resize(width, w->height());
    }
  }
};



void TLayoutWindow::setAlignHeights(void) {
  int height=0;
  bool set=FALSE;

  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	set = TRUE;
	height = w->height();
      }
      else
	w->resize(w->width(), height);
    }
  }
};



void TLayoutWindow::setAlignHCentres(void) {
  int centre=0;
  bool set=FALSE;

  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	set = TRUE;
	centre = w->x() + (w->width()/2);
      }
      else
	w->move(centre - w->width()/2, w->y());
    }
  }
};





void TLayoutWindow::setAlignVCentres(void) {
  int centre=0;
  bool set=FALSE;
  

  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	set = TRUE;
	centre = w->y() + (w->height()/2);
      }
      else
	w->move( w->x(), centre - w->height()/2 );
    }
  }
};



typedef QListIterator<QWidget> TWidgetTopListIterator;
class TWidgetTopList: public QList<QWidget> {
public:
private:
  int compareItems(GCI s1, GCI s2) {
    QWidget *w1 = (QWidget*)s1;
    QWidget *w2 = (QWidget*)s2;

    return (w1->x() - w2->x());
  }
};

void TLayoutWindow::setAlignVSpread(void) {
  int top=0, bottom=0, totalheight=0, totalspace=0;
  int cumulativegap=0, cumulativeheight=0, i=0;
  TWidgetTopList sortedlist;
  
  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  
  for ( ; it.current() ; ++it)
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      bottom = Max(bottom, w->y() + w->height());
      totalheight += w->height();
      sortedlist.inSort(w);
    }

  
  TWidgetTopListIterator it2(sortedlist);
  top = it2.current()->y();
  totalspace = (bottom - top) - totalheight;
  for ( i=0 ; it2.current() ; ++it2, i++) {
    cumulativegap = (i * totalspace) / (it2.count()-1);

    int ny = top + cumulativeheight + cumulativegap;
    it2.current()->move(it2.current()->x(), ny);
    cumulativeheight += it2.current()->height();
  }
};



typedef QListIterator<QWidget> TWidgetLeftListIterator;
class TWidgetLeftList: public QList<QWidget> {
public:
private:
  int compareItems(GCI s1, GCI s2) {
    QWidget *w1 = (QWidget*)s1;
    QWidget *w2 = (QWidget*)s2;

    return (w1->y() - w2->y());
  }
};

void TLayoutWindow::setAlignHSpread(void) {
  int left=0, right=0, totalwidth=0, totalspace=0;
  int cumulativegap=0, cumulativewidth=0, i=0;
  TWidgetTopList sortedlist;


  QObjectListIt it(selectedList());
  if (it.count()<2)
    return;
  
  for ( ; it.current() ; ++it)
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      right = Max(right, w->x() + w->width());
      totalwidth += w->width();
      sortedlist.inSort(w);
    }

  
  TWidgetTopListIterator it2(sortedlist);
  left = it2.current()->x();
  totalspace = (right - left) - totalwidth;
  for ( i=0 ; it2.current() ; ++it2, i++) {
    cumulativegap = (i * totalspace) / (it2.count()-1);

    int nx = left + cumulativewidth + cumulativegap;
    it2.current()->move(nx, it2.current()->y());
    cumulativewidth += it2.current()->width();
  }
};




QRect TLayoutWindow::calcBounds(void) {
  QRect bounds = QRect(0,0,0,0);
  bool set = FALSE;
  QObjectListIt it(selectedList());
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      if (!set) {
	bounds = QRect(w->pos(), w->size());
	set = TRUE;
      }
      else {
	if (w->x() < bounds.left())
	  bounds.setLeft(w->x());
	if ((w->x()+w->width()) > bounds.right())
	  bounds.setRight(w->x() + w->width());
	if (w->y() < bounds.top())
	  bounds.setTop(w->y());
	if ((w->y()+w->height()) > bounds.bottom())
	  bounds.setBottom(w->y() + w->height());
      }
    }
  }
  return bounds;
}



void TLayoutWindow::setPosHCentre(void) {
  QRect bounds = calcBounds();
  int offsetx = (width() - bounds.width())/2 - bounds.left();

  QObjectListIt it(selectedList());
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      w->move(w->x()+offsetx, w->y());
    }
  }
};



void TLayoutWindow::setPosVCentre(void) {
  QRect bounds = calcBounds();
  int offsety = (height() - bounds.height())/2 - bounds.top();

  QObjectListIt it(selectedList());
  for ( ; it.current() ; ++it) {
    if (it.current()->isWidgetType()) {
      QWidget *w = (QWidget*)it.current();
      w->move(w->x(), w->y()+offsety);
    }
  }
};



