#include "pics/QObject.xpm"
#include "TStoreProperty.h"
#include "TStoreType.h"
#include "TFileIO.h"
#include "TLayoutWindow.h"

#include <qbutton.h>
#include <qcombo.h>
#include <qdir.h>
#include <qfile.h>
#include <qfiledlg.h>

#include <dlfcn.h>
#include <assert.h>



class TLoadConnection {
public:
	QString sourceName;
	QString sourceArgs;
	QString destObjectName;
	TConnectionType destType;
	QString destName;
	QString destArgs;
};

typedef QList<TLoadConnection> TLoadConnectionList;
typedef QListIterator<TLoadConnection> TLoadConnectionListIt;



TStoreType *createTypeQObject(TStoreType *parent, TLayoutWindow *lw) {
  return (new TStoreType(parent, lw));
}

QPixmap *pixmapQObject() {
  return new QPixmap((const char **)QObject_xpm);
}
  
bool isCreatableQObject() {
  return FALSE;
}

bool isBaseClassQObject() {
  return FALSE;
}


TStoreType::TStoreType(TStoreType *p, TLayoutWindow *lw): TObjectTree(NULL, p) {
	// set up the list of properties
	addProperties();
	addSignals();
	
	if (parent()) {
		QString foo = nextName(parent()->object(), name());
		object(new QObject(parent()->object(), foo));
	}
	else
		object(new QObject(lw, "QObject1"));

	fLayoutWindow = lw;
}



TStoreType::~TStoreType() {
}


bool TStoreType::isCreatable() {
  return FALSE;
}
 
bool TStoreType::isVisible() {
  return FALSE;
}

bool TStoreType::canItHaveChildren() {
  return FALSE;
}


// creates a list of all of the properties available to this type,
// including all of the properties from the parent
//
// There are 3 types of properties, ptText, ptList, and ptButton.
//   ptText is the most common and allows the user to type an arbitrary piece
//          of text (such as a name, or a number)
//   ptList allows the user to select from a list of options, usefull
//          for bools and enums
//   ptButton pops up a dialog with more questions, such as a file dialog which 
//          is used to select a bitmap file.
void TStoreType::addProperties() {
  TStoreProperty *sp = NULL;
  
  sp = new TStoreProperty("Name", ptText,
						  &QObjectSetNameFunc,
						  &QObjectGetNameFunc, 
						  0,
						  0, FALSE);
  CHECK_PTR(sp);
  fProperties.inSort(sp);

  sp = new TStoreProperty("Variable Name", ptText,
						  &QObjectSetVariableFunc,
						  &QObjectGetVariableFunc,
						  0,
						  0, TRUE);
  CHECK_PTR(sp);
  fProperties.inSort(sp);

}


void TStoreType::addSignals() {
  fSignalList.inSort(new TSignal("destroyed", ""));
}
  

// The name of this type
QString TStoreType::storeTypeName() {
  return "QObject";
}



// the headerfile for this type
QString TStoreType::headerFile() {
  return "qobject.h";
}


TObjectType TStoreType::type() {
  return otObject;
}



TStoreProperty *TStoreType::property(QString n) {
  TStoreProperty *i;
  for (i=getProperties()->first() ; i ; i=getProperties()->next())
    if (i->name==n)
      break;
  return (i);
}


TStorePropertyList *TStoreType::getProperties(void) { 
  return &fProperties; 
};

TSlotList *TStoreType::getSlots(void) { 
  return &fSlotList; 
};


TSignalList *TStoreType::getSignals(void) { 
  return &fSignalList; 
};




QString TStoreType::nextName(QObject *parent, QString type) {
  int count=1;
  QString nextname;
  bool finished = FALSE;

  QObject *root = parent;
  if (root)
    while (root->parent())
      root = root->parent();

  while (!finished) {
    nextname.sprintf("%s%d", (const char *) type, count++);
    if (root) {
      QObjectList *list = root->queryList(0, nextname, FALSE);
      if (list->isEmpty())
	finished = TRUE;
      delete list;
    }
    else
      finished = TRUE;
  }
  return nextname;
}  




// The declaration line for the header file we write out
void TStoreType::writeHFile(QTextStream &os, QString var) {
  if (var!="")
    os << "\tQObject *" << var << ";" << endl;
}



// The constructor line for the file we write out
void TStoreType::writeCFileConstructor(QTextStream &os, QString var, QString parent) {
  os << "\t" << var << " = new Object ( ";
  os << parent;
  os << ", \"" << object()->name() << "\");" << endl;
}


// Any properties that this type has, are written to the C file we generate
void TStoreType::writeCFileProperties(QTextStream &, QString ) {
  ;
}


// For certain types, such as bitmaps we need to store the data in a seperate file
void TStoreType::writeCFileData(QTextStream &, QString) {
  ;
}






void TStoreType::writeLayout(QTextStream &os, int tabs) {
	QString tabstr = TFileIO::setTabs(tabs);

	TFileIO::writeLayoutString(os, tabstr, "Name", object()->name() );
	if (variableName() != "")
		TFileIO::writeLayoutString(os, tabstr, "Variable Name", variableName() );

	// dump Signals
	TSignalList *slist = (TSignalList*)findData("CreatedSignals");
	if (slist) {
		TSignalListIt sit(*slist);
		for ( ; sit.current() ; ++sit)
			os << tabstr << "Signal { \"" 
			   << (const char *)sit.current()->name
			   << "\" \"" << (const char *)sit.current()->args
			   << "\" }" << endl;
	}

	// Dump Slots
	TSlotList *slotlist = (TSlotList*)findData("CreatedSlots");
	if (slotlist) {
		TSlotListIt sit(*slotlist);
		for ( ; sit.current() ; ++sit) {
			os << tabstr << "Slot { \"" << (const char *)sit.current()->name
			   << "\" \"" << (const char *)sit.current()->args
			   << "\" ";
			switch (sit.current()->privacy) {
			case spPublic:
				os << "\"public\" "; break;
			case spProtected:
				os << "\"protected\" "; break;
			case spPrivate:
				os << "\"private\" "; break;
			}

			if (sit.current()->isVirtual)
				os << "\"True\"";
			else
				os << "\"False\"";
			os << " }" << endl;
		}
	}

	// Dump Connections
	TLinkedConnectionsList *list = connections();
	if (list) {
		TLinkedConnectionsListIt it(*list);
		for ( ; it.current() ; ++it) {
			TLinkedConnections *connectionLink = it.current();
			TSignal *sourceSignal = connectionLink->source;

			TConnectionListIt cit(connectionLink->connections);
			for ( ; cit.current() ; ++cit ) {
				TConnection *connection = cit.current();

				// Dump the source name and args
				os << tabstr << "Connection { \""
				   << sourceSignal->name << "\" \""
				   << sourceSignal->args << "\" ";
			  
				// dump the object to which we are connected
				if (connection->dest == root())
					os << "\"this\"";
				else
					os << "\"" << connection->dest->object()->name() << "\"";

				// are we connecting to a signal or a slot
				if (ctSignal==connection->connection->type)
					os << " \"Signal\"";
				else
					os << " \"Slot\"";

				// Dump the name and args of the function we are connecting to
				os << " \""  
				   << (const char *)connection->connection->name 
				   << "\" \"" << (const char *)connection->connection->args
				   << "\" }" << endl;
			}
		}
	}
}


void TStoreType::readLayoutItem(QTextStream &is, QString propName) {
  if (propName=="Name") {
    QObjectSetNameFunc(TFileIO::readLayoutString(is));
    //cout << "Set name to: " << object()->name() << endl;
  }
  else if (propName=="Variable Name")
    variableName(TFileIO::readLayoutString(is));
  else if (propName=="Slot") {
	  QString sname = TFileIO::readDelimitedString(is);
	  QString sargs = TFileIO::readDelimitedString(is);

	  TSlot *slot = new TSlot(sname, sargs);

	  QString privacy = TFileIO::readDelimitedString(is);
	  QString isvirtual = TFileIO::readDelimitedString(is);
	  TFileIO::readLayoutUntil(is, '}');

	  if (slot->name.isNull())
		  slot->name = "";
	  if (slot->args.isNull())
		  slot->args = "";

	  if ("public"==privacy)
		  slot->privacy = spPublic;
	  else if ("protected"==privacy)
		  slot->privacy = spProtected;
	  else if ("private"==privacy)
		  slot->privacy = spPrivate;
	  else
		  slot->privacy = spPublic;

	  if ("True"==isvirtual)
		  slot->isVirtual = TRUE;
	  else
		  slot->isVirtual = FALSE;

	  TSlotList *slist = (TSlotList*)findData("CreatedSlots");
	  if (!slist) {
		  slist = new TSlotList;
		  addData("CreatedSlots", slist);
	  }
	  slist->append(slot);

	  //cout << "Read slot: " << slot->name << ", " << slot->args << endl;
  }
  else if (propName=="Signal") {
	  QString sname = TFileIO::readDelimitedString(is);
	  QString sargs = TFileIO::readDelimitedString(is);
	  TSignal *signal = new TSignal(sname, sargs);
	  TFileIO::readLayoutUntil(is, '}');

	  if (signal->name.isNull())
		  signal->name = "";
	  if (signal->args.isNull())
		  signal->args = "";

	  TSignalList *slist = (TSignalList*)findData("CreatedSignals");
	  if (!slist) {
		  slist = new TSignalList;
		  addData("CreatedSignals", slist);
	  }
	  slist->append(signal);

	  //cout << "Read signal: " << signal->name << ", " << signal->args << endl;
  }
  else if (propName=="Connection") {
	  TLoadConnection *conn = new TLoadConnection;
	  
	  conn->sourceName = TFileIO::readDelimitedString(is);
	  conn->sourceArgs = TFileIO::readDelimitedString(is);
	  conn->destObjectName = TFileIO::readDelimitedString(is);

	  QString destType = TFileIO::readDelimitedString(is);
	  if (0==stricmp("Signal", destType))
		  conn->destType = ctSignal;
	  else
		  conn->destType = ctSlot;

	  conn->destName = TFileIO::readDelimitedString(is);
	  conn->destArgs = TFileIO::readDelimitedString(is);
	  TFileIO::readLayoutUntil(is, '}');
	  
	  TLoadConnectionList *list = (TLoadConnectionList*) findData("LoadConnection");
	  if (!list) {
		  list = new TLoadConnectionList;
		  addData("LoadConnections", list);
	  }
	  list->append(conn);
  }
  else {
    // nothing else has handled this, so it must either be a dodgy
    // line, or an item that someone has written and forgotten to 
    // read back.
    TFileIO::readLayoutUntil(is, '}');
  }
}




// Copy the details from one object to the other
void TStoreType::copyDetails(TStoreType *oldObj) {
	//cout << "QObject::copyDetails" << endl;

  object()->setName(oldObj->object()->name());
  variableName(oldObj->variableName());

  //cout << "QObject::copyDetails" << endl;
}



void TStoreType::propertyChanged(const char *, 
				 const char *,
				 const char *) {
  TLayoutWindow *layout = (TLayoutWindow*) layoutWindow();
  layout->makeModified();
}

void TStoreType::postLayoutLoad() {
	//cout << "TStoreType::postLayoutLoad - " << object()->name() << endl;
	
	TLoadConnectionList *list = (TLoadConnectionList*)findData("LoadConnections");
	if (list) {
		TLoadConnectionListIt it(*list);
		for ( ; it.current() ; ++it) {
			TLoadConnection *lc = it.current();
			if (lc->sourceName.isNull())
				lc->sourceName = "";
			if (lc->sourceArgs.isNull())
				lc->sourceArgs = "";
			if (lc->destName.isNull())
				lc->destName = "";
			if (lc->destArgs.isNull())
				lc->destArgs = "";

			//cout << "Connection: " << lc->sourceName << ", " 
			//	 << lc->sourceArgs << ", " << lc->destObjectName << ", ";
			//if (lc->destType==ctSignal)
			//	cout << "Signal, ";
			//else
			//	cout << "Slot, ";
			//cout << lc->destName << ", " << lc->destArgs << endl;



			TLinkedConnections *conn = connections()->find(lc->sourceName, lc->sourceArgs);
			if (!conn) {
				//cout << "There is no linked connection for : " << lc->sourceName << endl;
				conn = new TLinkedConnections;
				conn->source = getSignals()->find(lc->sourceName, lc->sourceArgs);
				//cout << "  " << (void *)conn->source << endl;
				connections()->append(conn);
			}

			TConnection *c = new TConnection();
			if (lc->destObjectName=="this" || lc->destObjectName=="")
				c->dest = root();
			else
				c->dest = root()->findByObjName(lc->destObjectName);

			if (lc->destType==ctSignal) {
				c->connection = c->dest->getSignals()->find(lc->destName, lc->destArgs); 
				if (!c->connection) {
					TSignalList *slist = (TSignalList*)c->dest->findData("CreatedSignals");
					if (slist)
						c->connection = slist->find(lc->destName, lc->destArgs); 
				}
			}	
			else {
				c->connection = c->dest->getSlots()->find(lc->destName, lc->destArgs);
				if (!c->connection) {
					TSlotList *slist = (TSlotList*)c->dest->findData("CreatedSlots");
					if (slist)
						c->connection = slist->find(lc->destName, lc->destArgs); 
				}
			}	
			conn->connections.append(c);
			
		}
	}
}




/********************************************************************************
**
**  Property Functions
**
**  These are used by the addProperties function to perform the underlying functionality
**
********************************************************************************/


bool TStoreType::QObjectSetNameFunc(QString aValue) {
	QString oldName = object()->name();
  
	object()->setName(aValue);
	
	if (root() == this) {
		layoutWindow()->renameClass(oldName);
	}
	if (layoutWindow()->propertyWidget())
		layoutWindow()->propertyWidget()->updateObject(this, oldName);
	return TRUE;
}

QString TStoreType::QObjectGetNameFunc() {
  return object()->name();
}


bool TStoreType::QObjectSetVariableFunc(QString AValue) {
	if (this == &layoutWindow()->objectTree()) {
		return FALSE;
	}
	
	QString oldName = variableName();
	
	variableName(AValue);
	
	TLayoutWindow *layout = (TLayoutWindow*) layoutWindow();
	layout->changeVariableName(this, oldName);
	return TRUE;
}


QString TStoreType::QObjectGetVariableFunc() {
  return variableName();
}

