// Copyright (C) 1999 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of tdof.
// 
// The exception is that, if you link the tdof ixj library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the tdof ixj library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released by OST under the 
// name tdof.  If you copy code from other releases into a copy of
// tdof, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for tdof, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include <stdio.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include "ixjuser.h"
#include "ixj.h"

fd_set IXJPort::pending;
fd_set IXJPort::events;
int IXJPort::hiwater = 0;
IXJPort *IXJPort::first = NULL;
IXJPort *IXJPort::last = NULL;
Mutex IXJPort::update;
int IXJPort::iosync[2];
IXJService *IXJPort::callback = NULL;

IXJPort::IXJPort(char *fname)
{
	initial = true;
	offhook = ready = playing = recording = false;
	next = prev = NULL;
	termcode = termmask = 0;

	fd = open(fname, O_RDWR);
	if(fd < 0)
	{
		throw(this);
		return;
	}

	endTimer();
}

void IXJPort::Enable(void)
{
	dsptype = ioctl(fd, IXJCTL_DSP_TYPE);
	version = ioctl(fd, IXJCTL_DSP_VERSION);
	long flags;
 
	++update;	// lock before playing with pointers
	if(!first)
	{
		first = last = this;
		if(!hiwater)
		{
			FD_ZERO(&notify);
			FD_ZERO(&pending);
			FD_ZERO(&events);
			pipe(iosync);
			flags = fcntl(iosync[0], F_GETFL);
			flags |= O_NONBLOCK;
			fcntl(iosync[0], F_SETFL, flags); 
			hiwater = iosync[0] + 1;
			FD_SET(iosync[0], &pending);
		}
	}
	else		// always add to end of list
	{
		prev = last;
		last->next = this;
		last = this;
	}

	if(fd >= hiwater)
		hiwater = fd + 1;
	FD_SET(fd, &events);
	setCodec(IXJ_ULAW);
	--update;
	if(!callback)
	{
		callback = new IXJService;
		callback->Start();
	}
	else
		postUpdate();
}
		
void IXJPort::Disable(void)
{
	if(!FD_ISSET(fd, &events))	// if never enabled, skip
		return;

	setIdle();
	++update;
	if(!next && !prev)
	{
		first = last = NULL;
		if(callback)
		{
			delete callback;
			callback = NULL;
		}
		hiwater = iosync[0] + 1;
	}
	else
	{
		if(first == this)
			first = next;

		if(last == this)
			last = prev;

		if(next)
			next->prev = prev;

		if(prev)
			prev->next = next;
	}
	FD_CLR(fd, &events);
	FD_CLR(fd, &pending);
	FD_CLR(fd, &notify);
	close(fd);
	--update;
	if(callback)
		postUpdate();
}		

void IXJPort::setPending(bool set)
{
	++update;
	if(set)
		FD_SET(fd, &pending);
	else
		FD_CLR(fd, &pending);
	--update;
	if(callback)
		postUpdate();
}

void IXJPort::setNotify(void)
{
	++update;
	FD_SET(fd, &notify);
	--update;
	if(callback)
		postUpdate();
}

void IXJPort::setHook(bool mode)
{
	offhook = mode;
	if(offhook)
		ioctl(fd, IXJCTL_PSTN_SET_STATE, PSTN_OFF_HOOK);
	else
		ioctl(fd, IXJCTL_PSTN_SET_STATE, PSTN_ON_HOOK);
}

void IXJPort::setIdle(void)
{
	if(recording)
	{
		ioctl(fd, IXJCTL_REC_STOP);	
		recording = false;
	}

	if(playing)
	{
		ioctl(fd, IXJCTL_PLAY_STOP);
		playing = false;
	}
}

// if we are not within the callback thread, then we use the iosync
// pipe to wakeup callback since a timer or port availability has
// changed and we need to 'rescan'.
	
void IXJPort::postUpdate(void)
{
	char buf[1];

	if(!callback)
		return;

	if(callback->isThread())	// if in callback, no need
		return;

	buf[0] = 0;	// doesn't matter
	write(iosync[1], &buf, 1);
};


void IXJPort::RecordStop(void)
{
	if(recording)
	{
		ioctl(fd, IXJCTL_REC_STOP);
		recording = false;
	}
}

void IXJPort::PlayStop(void)
{
	if(playing)
	{
		ioctl(fd, IXJCTL_PLAY_STOP);
		playing = false;
	}
}

int IXJPort::RecordFrame(void *buf, size_t len)
{
	if(len > framesize)
		len = framesize;

	if(termcode)
	{
		RecordStop();
		return -1;
	}
	if(!recording)
	{
		ioctl(fd, IXJCTL_REC_START);
		recording = true;
	}
	return read(fd, buf, len);
}

int IXJPort::PlayFrame(void *buf, size_t len)
{
	if(len > framesize)
		len = framesize;

	if(termcode)
	{
		PlayStop();
		return -1;
	}

	if(!playing)
	{
		ioctl(fd, IXJCTL_PLAY_START);
		playing = true;
	}
	return write(fd, buf, len);
}

void IXJPort::endTimer(void)
{
	timer.tv_sec = timer.tv_usec = 0;
}

void IXJPort::setTimer(long msec)
{
	int secs = msec / 1000;
	int usecs = (msec % 1000) * 1000;

	gettimeofday(&timer, NULL);
	timer.tv_usec += usecs;
	if(timer.tv_usec > 1000000l)
	{
		++timer.tv_sec;
		timer.tv_usec %= 1000000l;
	}
	timer.tv_sec += secs;
	postUpdate();
}

long IXJPort::getTimer(void)
{
	struct timeval now;
	long diff;

	if(!timer.tv_sec)
		return -1l;

	gettimeofday(&now, NULL);
	diff = (timer.tv_sec - now.tv_sec) * 1000l;
	diff += (timer.tv_usec - now.tv_usec) / 1000l;
	if(diff < 0)
		return 0l;

	return diff;
}

unsigned long IXJPort::getTerminate(void)
{
	unsigned long ret = termcode;
	termcode = 0;
	return ret;
}

void IXJPort::setTerminate(unsigned long flag)
{
	termmask = flag;
	termcode = 0;
}

void IXJPort::ioExpired(void)
{
	if(!playing && !recording)
		return;

	if(!(termmask & IXJ_TERM_EXPIRED))
		return;

	if(termcode & IXJ_TERM_EXPIRED)
		return;

	termcode |= IXJ_TERM_EXPIRED;
	ioTerminate();
}

void IXJPort::ioEvent(void)
{
	IXJ_EXCEPTION event;
	IXJ_CID ixjcid;
	char digit;
	char cid[160];
	unsigned long mask;
	char *digptr = "0123456789*#";

	event.bytes = ioctl(fd, IXJCTL_EXCEPTION);
	
	if(event.bits.caller_id)
	{
		ioctl(fd, IXJCTL_CID, &ixjcid);
		sprintf(cid, "%s/%s %s:%s %s %s",
			ixjcid.month, ixjcid.day,
			ixjcid.hour, ixjcid.min,
			ixjcid.number, ixjcid.name);
		ioCallerId(cid);
	}

	if(event.bits.pstn_ring)
		ioRinging();

	if(event.bits.dtmf_ready)
	{
		digit = ioctl(fd, IXJCTL_GET_DTMF_ASCII);
		ioDigit(digit);
		mask = 1;
		while(*digptr)
		{
			if(*digptr == digit)
			{
				if(mask & termmask)
				{
					termcode |= mask;
					ioTerminate();
				}
				break;
			}
			++digptr;
			mask << 1;
		}
	}

	if(event.bits.pstn_wink)
		ioWinking();
}

ixjport_t IXJPort::getPortUsed(void)
{
	int port = ioctl(fd, IXJCTL_PORT, PORT_QUERY);
	switch(port)
	{
	case PORT_POTS:
		return IXJ_POTS_PORT;
	case PORT_PSTN:
		return IXJ_PSTN_PORT;
	case PORT_SPEAKER:
		return IXJ_SPEAKER_PORT;
	case PORT_HANDSET:
		return IXJ_HANDSET_PORT;
	}
	return IXJ_INVALID_PORT;
}

ixjport_t IXJPort::setPortUsed(ixjport_t port)
{
	int pt;

	switch(port)
	{
	case IXJ_POTS_PORT:
		pt = PORT_POTS;
		break;
	case IXJ_PSTN_PORT:
		pt = PORT_PSTN;
		break;
	case IXJ_SPEAKER_PORT:
		pt = PORT_SPEAKER;
		break;
	case IXJ_HANDSET_PORT:
		pt = PORT_HANDSET;
		break;
	default:
		return IXJ_INVALID_PORT;
	}
	ioctl(fd, IXJCTL_DSP_IDLE);
	ioctl(fd, IXJCTL_PORT, pt);
}

int IXJPort::setCodec(ixjcodec_t codec)
{
	int frame = 30;
	int coder, rtn;

	switch(codec)
	{
	case IXJ_G723_63:
		coder = G723_63;
		framesize = 24;
		break;
	case IXJ_G723_53:
		coder = G723_53;
		framesize = 20;
		break;
	case IXJ_TS85:
		coder = TS85;
		framesize = 32;
		break;
	case IXJ_TS41:
		coder = TS41;
		framesize = 16;
		break;
	case IXJ_G728:
		coder = G728;
		framesize = 96;
		break;
	case IXJ_G729:
		coder = G729;
		framesize = 10;
		frame = 10;
		break;
	case IXJ_ULAW:
		coder = ULAW;
		framesize = 240;
		break;
	case IXJ_ALAW:
		coder = ALAW;
		framesize = 240;
		break;
	case IXJ_LINEAR16:
		coder = LINEAR16;
		framesize = 480;
		break;
	case IXJ_LINEAR8:
		coder = LINEAR8;
		framesize = 240;
		break;
	case IXJ_WSS:
		coder = WSS;
		framesize = 240;
		break;
	}
	ioctl(fd, IXJCTL_DSP_IDLE);
	rtn = ioctl(fd, IXJCTL_FRAME, frame);
	if(rtn)
		return rtn;

	frametimer = (long)frame;
	rtn = ioctl(fd, IXJCTL_PLAY_CODEC, coder);
	if(rtn)
		return rtn;

	return ioctl(fd, IXJCTL_REC_CODEC, coder);
}
