// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your
// choice whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "config.h"
#include "thread.h"
#include "file.h"

DSO	*DSO::first = NULL;
DSO	*DSO::last = NULL;
Mutex	DSO::mutex;

void dynunload(void)
{
	while(DSO::last)
		delete DSO::last;
}

DSO::DSO(char *filename)
{
	next = prev = NULL;

	hImage = LoadLibrary(filename);
	if(!hImage)
	{
		err = "load failed";
		throw(this);
		return;
	}

	if(!last)
	{
		last = first = this;
		return;
	}
	
	++mutex;
	last->next = this;
	prev = last;
	last = this;
	--mutex;
}

DSO::~DSO()
{
	++mutex;
	if(hImage)
		FreeLibrary(hImage);

	if(first == this && last == this)
		first = last = NULL;

	if(!next && !prev)
	{
		--mutex;
		return;
	}

	if(prev)
		prev->next = next;

	if(next)
		next->prev = prev;

	if(first == this)
		first = next;

	if(last == this)
		last = prev;

	--mutex;
}

void *DSO::operator[](const char *sym)
{
	void *addr = GetProcAddress(hImage, sym);
	if(!addr)
		err = "symbol missing";

	return addr;
}

Dir::Dir(const char *fname)
{
	DWORD attr;

	name = NULL;
	hDir = INVALID_HANDLE_VALUE;
	strcpy(path, fname);
	attr = GetFileAttributes(path);
	if(attr == ~0l)
	{
		path[0] = 0;
		throw(this);
	}
	if(!(attr & FILE_ATTRIBUTE_DIRECTORY))
	{
		path[0] = 0;
		throw(this);
	}

	strcpy(path, "/*.*");
	hDir = FindFirstFile(path, &data);
	if(hDir != INVALID_HANDLE_VALUE)
		name = data.cFileName;
}

Dir::~Dir()
{
	if(hDir != INVALID_HANDLE_VALUE)
		FindClose(hDir);
}

char *Dir::getName(void)
{
	char *retname = name;
	
	if(retname)
	{
		name = NULL;
		if(FindNextFile(hDir, &data))
			name = data.cFileName;
	}
	return retname;
}

RandomFile::RandomFile()
{
	hFile = INVALID_HANDLE_VALUE;
	flags.thrown = flags.initial = flags.temp = false;
	flags.count = 0;
}

RandomFile::RandomFile(const RandomFile &rf)
{
	HANDLE pidHandle = GetCurrentProcess();
	HANDLE dupHandle;

	if(rf.hFile != INVALID_HANDLE_VALUE)
	{
		if(!DuplicateHandle(pidHandle, rf.hFile, pidHandle, &dupHandle, 0, FALSE, DUPLICATE_SAME_ACCESS))
			hFile = INVALID_HANDLE_VALUE;
		else
			hFile = dupHandle;
	}
	else
		hFile = INVALID_HANDLE_VALUE;

	flags = rf.flags;
	flags.count = 0;

	if(rf.filepath)
	{
		filepath = new char[strlen(rf.filepath) + 1];
		strcpy(filepath, rf.filepath);
	}
	else
		filepath = NULL;
}	

void RandomFile::Final(void)
{
	if(hFile != INVALID_HANDLE_VALUE)
	{
		CloseHandle(hFile);
		if(flags.temp)
			DeleteFile(filepath);
	}

	if(filepath)
	{
		delete[] filepath;
		filepath = NULL;
	}
	
	hFile = INVALID_HANDLE_VALUE;
	flags.initial = false;
}

bool RandomFile::Initial(void)
{
	bool init;
	fileattr_t access;

	if(hFile == INVALID_HANDLE_VALUE)
		return false;

	EnterMutex();
	init = flags.initial;
	flags.initial = false;
	if(!init)
	{
		LeaveMutex();
		return false;
	}

	access = Initialize();
	if(access == FILE_ATTR_INVALID)
	{
		CloseHandle(hFile);
		if(filepath)
			DeleteFile(filepath);
		hFile = INVALID_HANDLE_VALUE;
		Error(FILE_INIT_FAILED);
	}
	LeaveMutex();
	return init;
}	

fileerror_t RandomFile::Error(fileerror_t id, char *str)
{
	errid = id;
	errstr = str;
	if(!flags.thrown)
	{
		flags.thrown = true;
		throw(this);
	}
	return id;
}

LONG RandomFile::getCapacity(void)
{
	LONG eof, pos;
	if(!hFile)
		return 0;

	EnterMutex();
	pos = SetFilePointer(hFile, 0l, NULL, FILE_CURRENT);
	eof = SetFilePointer(hFile, 0l, NULL, FILE_END);
	SetFilePointer(hFile, pos, NULL, FILE_BEGIN);
	LeaveMutex();
	return eof;
}

bool RandomFile::operator!(void)
{
	return hFile == INVALID_HANDLE_VALUE;
}

ThreadFile::ThreadFile(const char *path) :
RandomFile()
{
	first = NULL;
	Open(path);
}

ThreadFile::~ThreadFile()
{
	Final();
	fcb_t *next;
	while(first)
	{
		next = first->next;
		delete first;
		first = next;
	}
}

fileerror_t ThreadFile::Open(const char *path)
{
	int len = strlen(path);
	if(hFile != INVALID_HANDLE_VALUE)
		Final();

	if(path != filepath)
	{
		filepath = new char[len + 1];
		strcpy(filepath, path);
	}

	flags.initial = false;
	hFile = CreateFile(filepath, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_RANDOM_ACCESS, NULL);
	if(hFile == INVALID_HANDLE_VALUE)
	{
		flags.initial = true;
		hFile = CreateFile(filepath, GENERIC_READ | GENERIC_WRITE, 0, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_RANDOM_ACCESS, NULL);
	}
	if(hFile == INVALID_HANDLE_VALUE)
		return FILE_OPEN_FAILED;

	return FILE_SUCCESS;
}		

fcb_t *ThreadFile::getFCB(void)
{
	fcb_t *fcb = (fcb_t *)state.getKey();

	if(!fcb)
	{
		fcb = new fcb_t;
		fcb->next = first;
		first = fcb;
		fcb->address = NULL;
		fcb->len = 0;
		fcb->pos = 0;
		state.setKey(fcb);
	}
	return fcb;
}

fileerror_t ThreadFile::Fetch(void *address, DWORD len, LONG pos)
{
	fcb_t *fcb = getFCB();
	DWORD count;

	if(hFile == INVALID_HANDLE_VALUE)
		return FILE_NOT_OPENED;

	if(address)
		fcb->address = address;

	if(len)
		fcb->len = len;

	if(pos != -1)
		fcb->pos = pos;

	EnterMutex();
	SetFilePointer(hFile, fcb->pos, NULL, FILE_BEGIN);
	if(!ReadFile(hFile, fcb->address, fcb->len, &count, NULL))
	{
		LeaveMutex();
		return FILE_READ_FAILURE;
	}
	LeaveMutex();
	if(count < fcb->len)
		return FILE_READ_INCOMPLETE;

	return FILE_SUCCESS;
}

fileerror_t ThreadFile::Update(void *address, DWORD len, LONG pos)
{
	fcb_t *fcb = getFCB();
	DWORD count;

	if(hFile == INVALID_HANDLE_VALUE)
		return FILE_NOT_OPENED;

	if(address)
		fcb->address = address;

	if(len)
		fcb->len = len;

	if(pos != -1)
		fcb->pos = pos;

	EnterMutex();
	SetFilePointer(hFile, fcb->pos, NULL, FILE_BEGIN);
	if(!WriteFile(hFile, fcb->address, fcb->len, &count, NULL))
	{
		LeaveMutex();
		return FILE_WRITE_FAILURE;
	}
	LeaveMutex();
	if(count < fcb->len)
		return FILE_WRITE_INCOMPLETE;

	return FILE_SUCCESS;
}

fileerror_t ThreadFile::Append(void *address, DWORD len)
{
	fcb_t *fcb = getFCB();
	DWORD count;

	if(hFile == INVALID_HANDLE_VALUE)
		return FILE_NOT_OPENED;

	if(address)
		fcb->address = address;

	if(len)
		fcb->len = len;

	EnterMutex();
	fcb->pos = SetFilePointer(hFile, 0l, NULL, FILE_END);
	if(!WriteFile(hFile, fcb->address, fcb->len, &count, NULL))
	{
		LeaveMutex();
		return FILE_WRITE_FAILURE;
	}
	LeaveMutex();
	if(count < fcb->len)
		return FILE_WRITE_INCOMPLETE;

	return FILE_SUCCESS;
}

off_t ThreadFile::getPosition(void)
{
	fcb_t *fcb = getFCB();

	return fcb->pos;
}

bool ThreadFile::operator++(void)
{
	LONG eof;
	fcb_t *fcb = getFCB();

	fcb->pos += fcb->len;
	EnterMutex();
	eof = SetFilePointer(hFile, 0l, NULL, FILE_END);
	LeaveMutex();

	if(fcb->pos >= eof)
	{
		fcb->pos = eof;
		return true;
	}
	return false;
}

bool ThreadFile::operator--(void)
{
	fcb_t *fcb = getFCB();

	fcb->pos -= fcb->len;
	if(fcb->pos <= 0)
	{
		fcb->pos = 0;
		return true;
	}
	return false;
}

SharedFile::SharedFile(const char *path) :
RandomFile()
{
	fcb.address = NULL;
	fcb.len = 0;
	fcb.pos = 0;
	Open(path);
}

SharedFile::SharedFile(const SharedFile &sh) :
RandomFile(sh)
{
}

SharedFile::~SharedFile()
{
	Final();
}

fileerror_t SharedFile::Open(const char *path)
{
	int len = strlen(path);
	if(hFile != INVALID_HANDLE_VALUE)
		Final();

	if(path != filepath)
	{
		filepath = new char[len + 1];
		strcpy(filepath, path);
	}

	flags.initial = false;
	hFile = CreateFile(filepath, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_RANDOM_ACCESS, NULL);
	if(hFile == INVALID_HANDLE_VALUE)
	{
		flags.initial = true;
		hFile = CreateFile(filepath, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_RANDOM_ACCESS, NULL);
	}
	if(hFile == INVALID_HANDLE_VALUE)
		return FILE_OPEN_FAILED;

	return FILE_SUCCESS;
}		

fileerror_t SharedFile::Fetch(void *address, DWORD len, LONG pos)
{
	DWORD count;
	OVERLAPPED over;

	if(hFile == INVALID_HANDLE_VALUE)
		return FILE_NOT_OPENED;

	EnterMutex();
	if(address)
		fcb.address = address;

	if(len)
		fcb.len = len;

	if(pos != -1)
		fcb.pos = pos;

	SetFilePointer(hFile, fcb.pos, NULL, FILE_BEGIN);
	over.hEvent = 0;
	over.Offset = fcb.pos;
	over.OffsetHigh = 0;
	LockFileEx(hFile, LOCKFILE_EXCLUSIVE_LOCK, 0, fcb.len, 0, &over);
	if(!ReadFile(hFile, fcb.address, fcb.len, &count, NULL))
	{
		LeaveMutex();
		return FILE_READ_FAILURE;
	}
	LeaveMutex();
	if(count < fcb.len)
		return FILE_READ_INCOMPLETE;

	return FILE_SUCCESS;
}

fileerror_t SharedFile::Update(void *address, DWORD len, LONG pos)
{
	OVERLAPPED over;
	DWORD count;

	if(hFile == INVALID_HANDLE_VALUE)
		return FILE_NOT_OPENED;

	over.hEvent = 0;
	EnterMutex();
	if(address)
		fcb.address = address;

	if(len)
		fcb.len = len;

	if(pos != -1)
		fcb.pos = pos;

	SetFilePointer(hFile, fcb.pos, NULL, FILE_BEGIN);
	over.hEvent = 0;
	over.Offset = pos;
	over.OffsetHigh = 0;
	if(!WriteFile(hFile, fcb.address, fcb.len, &count, NULL))
	{
		SetFilePointer(hFile, fcb.pos, NULL, FILE_CURRENT);
		UnlockFileEx(hFile, 0, len, 0, &over);
		LeaveMutex();
		return FILE_WRITE_FAILURE;
	}
	SetFilePointer(hFile, fcb.pos, NULL, FILE_CURRENT);
	UnlockFileEx(hFile, 0, len, 0, &over);
	LeaveMutex();
	if(count < fcb.len)
		return FILE_WRITE_INCOMPLETE;

	return FILE_SUCCESS;
}

fileerror_t SharedFile::Append(void *address, DWORD len)
{
	OVERLAPPED over;
	DWORD count;
	LONG eof;

	if(hFile == INVALID_HANDLE_VALUE)
		return FILE_NOT_OPENED;

	EnterMutex();
	if(address)
		fcb.address = address;

	if(len)
		fcb.len = len;

	fcb.pos = SetFilePointer(hFile, 0l, NULL, FILE_END);
	over.hEvent = 0;
	over.Offset = fcb.pos;
	over.OffsetHigh = 0;
	eof = fcb.pos;
	LockFileEx(hFile, LOCKFILE_EXCLUSIVE_LOCK, 0, 0x7fffffff, 0, &over);
	fcb.pos = SetFilePointer(hFile, 0l, NULL, FILE_END);
	if(!WriteFile(hFile, fcb.address, fcb.len, &count, NULL))
	{
		SetFilePointer(hFile, eof, NULL, FILE_CURRENT);
		UnlockFileEx(hFile, 0, 0x7fffffff, 0, &over);
		LeaveMutex();
		return FILE_WRITE_FAILURE;
	}
	SetFilePointer(hFile, eof, NULL, FILE_CURRENT);
	UnlockFileEx(hFile, 0, 0x7fffffff, 0, &over);
	LeaveMutex();
	if(count < fcb.len)
		return FILE_WRITE_INCOMPLETE;

	return FILE_SUCCESS;
}

off_t SharedFile::getPosition(void)
{
	return fcb.pos;
}

bool SharedFile::operator++(void)
{
	LONG eof;

	EnterMutex();
	fcb.pos += fcb.len;
	eof = SetFilePointer(hFile, 0l, NULL, FILE_END);

	if(fcb.pos >= eof)
	{
		fcb.pos = eof;
		LeaveMutex();
		return true;
	}
	LeaveMutex();
	return false;
}

bool SharedFile::operator--(void)
{
	EnterMutex();
	fcb.pos -= fcb.len;
	if(fcb.pos <= 0)
	{
		fcb.pos = 0;
		LeaveMutex();
		return true;
	}
	LeaveMutex();
	return false;
}

bool isDir(const char *path)
{
	DWORD attr;

	attr = GetFileAttributes(path);
	if(attr == ~0l)
		return false;

	if(attr & FILE_ATTRIBUTE_DIRECTORY)
		return true;

	return false;
}

bool isFile(const char *path)
{
	DWORD attr;

	attr = GetFileAttributes(path);
	if(attr == ~0l)
		return false;

	if(attr & FILE_ATTRIBUTE_DIRECTORY)
		return false;

	return true;
}

bool canAccess(const char *path)
{
	DWORD attr;

	attr = GetFileAttributes(path);
	if(attr == ~0l)
		return false;

	if(attr & FILE_ATTRIBUTE_SYSTEM)
		return false;

	if(attr & FILE_ATTRIBUTE_HIDDEN)
		return false;

	return true;
}

bool canModify(const char *path)
{
	DWORD attr;

	attr = GetFileAttributes(path);

	if(!canAccess(path))
		return false;

	if(attr & FILE_ATTRIBUTE_READONLY)
		return false;

	if(attr & (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_NORMAL))
		return true;

	return false;
}

/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
