
package org.mandas.docker.client.messages.mount;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link BindOptions}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableBindOptions.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableBindOptions implements BindOptions {
  private final @Nullable String propagation;

  private ImmutableBindOptions(@Nullable String propagation) {
    this.propagation = propagation;
  }

  /**
   * @return The value of the {@code propagation} attribute
   */
  @JsonProperty("Propagation")
  @Override
  public @Nullable String propagation() {
    return propagation;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link BindOptions#propagation() propagation} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for propagation (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableBindOptions withPropagation(@Nullable String value) {
    if (Objects.equals(this.propagation, value)) return this;
    return new ImmutableBindOptions(value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableBindOptions} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableBindOptions
        && equalsByValue((ImmutableBindOptions) another);
  }

  private boolean equalsByValue(ImmutableBindOptions another) {
    return Objects.equals(propagation, another.propagation);
  }

  /**
   * Computes a hash code from attributes: {@code propagation}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(propagation);
    return h;
  }

  /**
   * Prints the immutable value {@code BindOptions} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "BindOptions{"
        + "propagation=" + propagation
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link BindOptions} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable BindOptions instance
   */
  public static ImmutableBindOptions copyOf(BindOptions instance) {
    if (instance instanceof ImmutableBindOptions) {
      return (ImmutableBindOptions) instance;
    }
    return ImmutableBindOptions.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableBindOptions ImmutableBindOptions}.
   * <pre>
   * ImmutableBindOptions.builder()
   *    .propagation(String | null) // nullable {@link BindOptions#propagation() propagation}
   *    .build();
   * </pre>
   * @return A new ImmutableBindOptions builder
   */
  public static ImmutableBindOptions.Builder builder() {
    return new ImmutableBindOptions.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableBindOptions ImmutableBindOptions}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements BindOptions.Builder {
    private String propagation;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code BindOptions} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(BindOptions instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String propagationValue = instance.propagation();
      if (propagationValue != null) {
        propagation(propagationValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link BindOptions#propagation() propagation} attribute.
     * @param propagation The value for propagation (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Propagation")
    public final Builder propagation(@Nullable String propagation) {
      this.propagation = propagation;
      return this;
    }

    /**
     * Builds a new {@link ImmutableBindOptions ImmutableBindOptions}.
     * @return An immutable instance of BindOptions
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableBindOptions build() {
      return new ImmutableBindOptions(propagation);
    }
  }
}
