/**********************************************************************
 * This file is part of "Object Teams Development Tooling"-Software
 * 
 * Copyright 2004, 2010 Fraunhofer Gesellschaft, Munich, Germany,
 * for its Fraunhofer Institute and Computer Architecture and Software
 * Technology (FIRST), Berlin, Germany and Technical University Berlin,
 * Germany.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BonusGUI.java 23501 2010-02-08 18:27:55Z stephan $
 * 
 * Please visit http://www.eclipse.org/objectteams for updates and contact.
 * 
 * Contributors:
 * 	  Fraunhofer FIRST - Initial API and implementation
 * 	  Technical University Berlin - Initial API and implementation
 **********************************************************************/
package org.eclipse.objectteams.example.fbapplication;

import java.awt.Component;

import javax.swing.JOptionPane;

import org.eclipse.objectteams.example.flightbooking.model.Passenger;

/**
 * This team gives an unspecific, partially abstract implementation of a GUI
 * for a bonus programme. 
 * Please also see its nested team FlightBonusDialog
 * (since FlightBonusDialog is a role file, you may try pressing ctrl-o twice for navigation).
 * FlightBonusDialog already connects to FlightBonus.
 * However, no bindings to the flight booking system are given at this level.
 * @role FlightBonusDialog
 */
public team class BonusGUI {

	/** This is a reference to the active GUI. */
	View view = null;

	/**
	 * Make the GUI accessible and extend it by a registration question.
	 */    
    abstract protected class View  {

    	void registerView() {
    		BonusGUI.this.view = this;
    	}

    	protected boolean queryRegisterDialog() {
    		int choice = JOptionPane.showConfirmDialog(
			    				getComponent(), 
								"Register for Flight Bonus?", 
			    				"OT Bonus System", 
								JOptionPane.YES_NO_OPTION);
    		
    		return choice == JOptionPane.YES_OPTION;
    	}   

    	/**
    	 * Expected method: get an AWT component as parent for new windows.
    	 */
    	protected abstract Component getComponent();
    }
	
			
	/**
	 * This role is used as a hook into PassengerList.
	 */
    protected class Controller 
		when (BonusGUI.this.view != null) // don't receive callin triggers before proper initialization
	{
    	/**
		 * Given a passenger, ask if he/she should participate in the
		 * bonus programme.
		 */
		void queryRegisterForBonus (Passenger p) {
			if (BonusGUI.this.view.queryRegisterDialog()) 
				new FlightBonusDialog(new FlightBonus(p));
		};
    }    
    /**
     * A dialog for presenting information about collected bonus credits.
     *
     * This class is a nested team, i.e., it is a role of its enclosing team BonusGUI
     * and at the same time it is the team for the contained roles Collector and Message.
     * As a role it uses the option of role files, i.e., roles stored in their own file.
     * You may use the package declaration to navigate (F3) to the enclosing team BonusGUI.
     */
    protected team class FlightBonusDialog playedBy FlightBonus {

            /**
            * Message string to be placed in the dialog box
            */
            String message;	
            
            /**
            * Team/role constructor: Creates a new <code>FlightBonusDialog</code> object for the given 
            * <code>FlightBonus</code> object 
            */
            public FlightBonusDialog(FlightBonus fb) {
                    this.initializeMessage(0);
                    this.activate();
                    System.out.println("FBDialog ");
            }
            
            /**
                * Store old number of credits for the next message.
                * @param credits
                */
            void initializeMessage(int credits) {
                    this.message = new String("Collected credits in the past: "+credits+"\n");
            }
            
            /**
            *  When a subscriber is earning credits, the message string has to be updated.
            */
            protected abstract class Collector {
                    
                    /**
                        * Expected method: Returns the start string
                        */
                    public abstract String getStart();
                    
                    /**
                        * Expected method: Returns the destination string
                        */
                    public abstract String getDestination();
                    
                    /**
                        * Updates the message string when credits are calculated
                        */
                    callin int recordCredits() {
                            int credits = base.recordCredits();
                            FlightBonusDialog.this.message += "FlightSegment: \n";
                            FlightBonusDialog.this.message += "    "+this.getStart()+"-->"+this.getDestination()+"\n";
                            FlightBonusDialog.this.message += "    earning credit: "+credits+"\n";
                            return credits;
                    }
                    
            }
            
            /**
            *  When a subscriber is buying something, the earned credits are shown in a dialog box.
            */
            protected abstract class Message {
                    
                    abstract int getTotalCollectedCredits();
                    abstract String getName();
                    
                    /**
                        * Shows a dialog box with the bonus message
                        */
                    public void showBonusDialog() {
                            int currentCredits = this.getTotalCollectedCredits();
                            
                            String title = "Bonus message for Passenger "+this.getName(); 
                            FlightBonusDialog.this.message += new String ("Collected credits now: "+currentCredits);
                            
                            JOptionPane.showMessageDialog(
                                                            BonusGUI.this.view.getComponent(), 
                                                            FlightBonusDialog.this.message, 
                                                            title, 
                                                            JOptionPane.INFORMATION_MESSAGE);

                            // reinitialize for the next message:
                            FlightBonusDialog.this.initializeMessage(currentCredits);
                    }
            }
    }
}

