"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeWorkspaceEdits = exports.register = void 0;
const language_core_1 = require("@volar/language-core");
const cancellation_1 = require("../utils/cancellation");
const dedupe = require("../utils/dedupe");
const featureWorkers_1 = require("../utils/featureWorkers");
const transform_1 = require("../utils/transform");
function register(context) {
    return (uri, position, newName, token = cancellation_1.NoneCancellationToken) => {
        return (0, featureWorkers_1.languageFeatureWorker)(context, uri, () => ({ position, newName }), function* (map) {
            let _data;
            for (const mappedPosition of map.getGeneratedPositions(position, data => {
                _data = data;
                return (0, language_core_1.isRenameEnabled)(data);
            })) {
                yield {
                    position: mappedPosition,
                    newName: (0, language_core_1.resolveRenameNewName)(newName, _data),
                };
            }
            ;
        }, async (service, document, params) => {
            if (token.isCancellationRequested) {
                return;
            }
            const recursiveChecker = dedupe.createLocationSet();
            let result;
            await withLinkedCode(document, params.position, params.newName);
            return result;
            async function withLinkedCode(document, position, newName) {
                if (!service[1].provideRenameEdits) {
                    return;
                }
                if (recursiveChecker.has({ uri: document.uri, range: { start: position, end: position } })) {
                    return;
                }
                recursiveChecker.add({ uri: document.uri, range: { start: position, end: position } });
                const workspaceEdit = await service[1].provideRenameEdits(document, position, newName, token);
                if (!workspaceEdit) {
                    return;
                }
                if (!result) {
                    result = {};
                }
                if (workspaceEdit.changes) {
                    for (const editUri in workspaceEdit.changes) {
                        const textEdits = workspaceEdit.changes[editUri];
                        for (const textEdit of textEdits) {
                            let foundMirrorPosition = false;
                            recursiveChecker.add({ uri: editUri, range: { start: textEdit.range.start, end: textEdit.range.start } });
                            const decoded = context.decodeEmbeddedDocumentUri(editUri);
                            const sourceScript = decoded && context.language.scripts.get(decoded[0]);
                            const virtualCode = decoded && sourceScript?.generated?.embeddedCodes.get(decoded[1]);
                            const linkedCodeMap = virtualCode && sourceScript
                                ? context.documents.getLinkedCodeMap(virtualCode, sourceScript.id)
                                : undefined;
                            if (linkedCodeMap) {
                                for (const linkedPos of linkedCodeMap.getLinkedCodePositions(textEdit.range.start)) {
                                    if (recursiveChecker.has({ uri: linkedCodeMap.document.uri, range: { start: linkedPos, end: linkedPos } })) {
                                        continue;
                                    }
                                    foundMirrorPosition = true;
                                    await withLinkedCode(linkedCodeMap.document, linkedPos, newName);
                                }
                            }
                            if (!foundMirrorPosition) {
                                if (!result.changes) {
                                    result.changes = {};
                                }
                                if (!result.changes[editUri]) {
                                    result.changes[editUri] = [];
                                }
                                result.changes[editUri].push(textEdit);
                            }
                        }
                    }
                }
                if (workspaceEdit.changeAnnotations) {
                    for (const uri in workspaceEdit.changeAnnotations) {
                        if (!result.changeAnnotations) {
                            result.changeAnnotations = {};
                        }
                        result.changeAnnotations[uri] = workspaceEdit.changeAnnotations[uri];
                    }
                }
                if (workspaceEdit.documentChanges) {
                    if (!result.documentChanges) {
                        result.documentChanges = [];
                    }
                    result.documentChanges = result.documentChanges.concat(workspaceEdit.documentChanges);
                }
            }
        }, data => {
            return (0, transform_1.transformWorkspaceEdit)(data, context, 'rename');
        }, workspaceEdits => {
            const mainEdit = workspaceEdits[0];
            const otherEdits = workspaceEdits.slice(1);
            mergeWorkspaceEdits(mainEdit, ...otherEdits);
            if (mainEdit.changes) {
                for (const uri in mainEdit.changes) {
                    mainEdit.changes[uri] = dedupe.withTextEdits(mainEdit.changes[uri]);
                }
            }
            return workspaceEdits[0];
        });
    };
}
exports.register = register;
function mergeWorkspaceEdits(original, ...others) {
    for (const other of others) {
        for (const uri in other.changeAnnotations) {
            if (!original.changeAnnotations) {
                original.changeAnnotations = {};
            }
            original.changeAnnotations[uri] = other.changeAnnotations[uri];
        }
        for (const uri in other.changes) {
            if (!original.changes) {
                original.changes = {};
            }
            if (!original.changes[uri]) {
                original.changes[uri] = [];
            }
            const edits = other.changes[uri];
            original.changes[uri] = original.changes[uri].concat(edits);
        }
        if (other.documentChanges) {
            if (!original.documentChanges) {
                original.documentChanges = [];
            }
            for (const docChange of other.documentChanges) {
                (0, transform_1.pushEditToDocumentChanges)(original.documentChanges, docChange);
            }
        }
    }
}
exports.mergeWorkspaceEdits = mergeWorkspaceEdits;
//# sourceMappingURL=provideRenameEdits.js.map