/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is MacroTracker Bug Tracking System Source Code
 *
 * The Initial Developer of the Original Code is
 * R.J. Keller.
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Peter Bohan <neuralyzer@wowway.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */
package org.mozdev.MacroTracker.classes;

import java.io.Serializable;
import com.trfenv.rlk.Debug;

/**
 *Represents a user stored in the MacroTracker database. Information
 *stored in the class is the username of the user, the password
 *of the user, and the email of the user.
 *
 *Email is kept so that it can be displayed in the MacroTracker owner
 *field and other areas.
 *
 *This class also contains the permissions on what the user is allowed
 *to modify in the database.
 *
 *@version 0.1
 *@author R.J. Keller <rlk@trfenv.com>
 *@author Pete Bohan <neuralyzer@wowway.com>
 */
public final class User implements Serializable
{
	// Holds the users security permissions
	private boolean canconfirm;
	private boolean creategroups;
	private boolean editbugs;
	private boolean editcomponents;
	private boolean editkeywords;
	private boolean editusers;

	// Holds user's personal information
	private String mName;
	private String mEmail;
	private String mUsername;
	private String mPassword;

	/**
	 *Creates a new User with the user's name and email entered into the user.
	 *@param name The user's name
	 *@param email The user's email
	 */
	public User(String name, String email)
	{
		mName = name;
		mEmail = email;

		Debug.createObject(this);
	}

	/**
	 *This flag allows a user to do what you're doing right now: edit other
	 *users. This will allow those with the right to do so to remove
	 *administrator privileges from other users or grant them to themselves.
	 *Enable with care.
	 */
	public void setEditUsers(boolean ceu)
	{
		this.editusers = ceu;
	}

	/**
	 *Returns whether or not this user can edit other users settings.
	 *@return whether or not this user can edit other users settings.
	 *@see #setEditUsers
	 */
	public boolean canEditUsers()
	{
		return this.editusers;
	}

	/**
	 *If you use Bugzilla's keyword functionality, enabling this feature allows
	 *a user to create and destroy keywords. As always, the keywords for
	 *existing bugs containing the keyword the user wishes to destroy must be
	 *changed before Bugzilla will allow it to die.
	 */
	public void setEditKeywords(boolean cek)
	{
		this.editkeywords = cek;
	}

	/**
	 *Returns whether or not this user can edit Bugzilla keywords.
	 *@return whether or not this user can edit Bugzilla keywords.
	 *@see #setEditKeywords
	 */
	public boolean canEditKeywords()
	{
		return this.editkeywords;
	}

	/**
	 *This flag allows a user to create new products and components, as well
	 *as modify and destroy those that have no bugs associated with them. If a
	 *product or component has bugs associated with it, those bugs must be
	 *moved to a different product or component before Bugzilla will allow
	 *them to be destroyed.
	 */
	public void setEditComponents(boolean cec)
	{
		this.editcomponents = cec;
	}

	/**
	 *Returns whether or not this user can edit components.
	 *@return whether or not this user can edit components.
	 *@see #setEditComponents
	 */
	public boolean getEditComponents()
	{
		return this.editcomponents;
	}

	/**
	 *Unless a user has this bit set, they can only edit those bugs for which
	 *they are the assignee or the reporter. Even if this option is unchecked,
	 *users can still add comments to bugs.
	 */
	public void setEditBugs(boolean ceb)
	{
		this.editbugs = ceb;
	}

	/**
	 *Returns whether or not the current user can edit bugs.
	 *@return whether or not the current user can edit bugs.
	 *@see #setEditBugs
	 */
	public boolean canEditBugs()
	{
		return this.editbugs;
	}

	/**
	 *This option will allow a user to create and destroy groups in Bugzilla.
	 *@param crtgrp Whether or not the user can create or destroy groups.
	 */
	public void setCreateGroups(boolean crtgrp)
	{
		this.creategroups = crtgrp;
	}

	/**
	 *Returns whether or not this user can create groups.
	 *@return Whether or not this user can create groups.
	 *@see #setCreateGroups
	 */
	public boolean canCreateGroups()
	{
		return this.creategroups;
	}

	/**
	 *This field is only used if you have enabled the "unconfirmed" status. If
	 *you enable this for a user, that user can then move bugs from
	 *"Unconfirmed" to a "Confirmed" status (e.g.: "New" status).
	 *@param confirm Whether or not this user can confirm bugs.
	 */
	public void setCanConfirm(boolean confirm)
	{
		this.canconfirm = confirm;
	}

	/**
	 *Returns whether or not this user can confirm bugs.
	 *@return whether or not the user can confirm bugs.
	 *@see #setCanConfirm
	 */
	public boolean canConfirm()
	{
		return this.canconfirm;
	}

	/** Returns true if the user is equal to that of another user.
	 *@param userToCompare The user to compare to this user.
	 *@return Whether or not the user equals the user taken in.
	 */
	public boolean equals(User userToCompare)
	{
		if (userToCompare.getName() == mName &&
        	userToCompare.getPassword() == mPassword)
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	/**
	 *Returns the user's name.
	 */
	public String getName()
	{
		return mName;
	}

	/**
	 *Returns the user's email.
	 */
	public String getEmail()
	{
		return mEmail;
	}

	/**
	 *Returns the user's username.
	 */
	public String getUsername()
	{
		return mUsername;
	}

	/**
     *Returns the user's password.
	 */
	public String getPassword()
	{
		return mPassword;
	}
}
