/* The hash table structure in the HP-67 calculator. */

#ifndef HASHTABLE_H
#define HASHTABLE_H


/* Implementation of a separate chaining hash table for efficient
 * parsing and location of named memory. Acts on NUL-terminated
 * character strings. */


/* Interface is as follows:

   New hash is created, typically, by "new HashTable". This creates a
   hash table with 1021 elements. A single unsigned integer argument
   to the creation will produce a hash table with that number of
   elements. Hash tables can also be initialized from pre-existing
   tables, and can be copied.

   To add an element to the has table, call:
   hashinsert(char *key, DTYPE &value)

   Both fields are copied, so it is not necessary that the storage
   in them remain valid.

   To retrieve a pointer, call:
   getvalue(char *key)
   The return value is the "value" which was used in the insert
   function.

   To delete an element of the has table, call
   hashdelete(char *key)
   The return value is the "value" which was used in the insert
   function.

   The hash table is rehashed whenever the filling exceeds
   "REHASH_CRITFILL", currently two thirds. The rehashing can be
   forced by calling
   rehash()
   which increases the size of the has table by approximately a 
   factor of two, or
   rehash(unsigned int newsize)
   which changes the has table size to "newsize". Note that if the
   resulting table is overfilled, it will be automatically rehashed
   for a larger table size during the reconstruction of the hash
   table, so it is not possible, in practice, to produce a hash table
   which is overfilled and remains so.

 */



#include <string.h>
#include <stdio.h>

#include "calcfloat.h"

#include "hp67.h"


#define DEFAULT_HASHSIZE 1021

#define REHASH_CRITFILL (2.0 / 3.0)  /* Filling fraction before rehashing */


static unsigned int const newsizes[] = {2011, 4001, 8009, 16001,
				      32003, 64007, 128021 };  /* primes */
static unsigned int const numsizes = sizeof(newsizes) / sizeof(newsizes[0]);



/* For collisions, separate chaining */
template <class DTYPE>
struct hashcell {
  struct hashcell *next;
  char *keyval;
  DTYPE value;
};


template <class DTYPE>
class HashTable {

private:
  unsigned int tablesize;
  struct hashcell<DTYPE> *table;
  unsigned int numhashed;
  DTYPE const zeroval;
  struct hashcell<DTYPE> *recently[HASH_RECENT];

public:
  HashTable(void);
  HashTable(unsigned int mytablesize);
  HashTable(HashTable const &from);

  ~HashTable(void);

  HashTable const &operator= (HashTable const &from);

  void hashinsert(char const *key, DTYPE const &value); /* Puts into hash */
  DTYPE &getvalue(char const *key);  /* Returns value */
  DTYPE hashdelete(char const *key);  /* Returns old value */

  DTYPE &operator[] (char const *key);

  void rehash(void);
  void rehash(unsigned int tablesize);
  DTYPE *memelement(char const *key);
  void erase_memory(void);
  void writeoutput(FILE *ostream) const;
  DTYPE *recent_access(int i, char const **label) const;

private: 
  unsigned int computehash(char const *keyval) const;
  void gentable(unsigned int size);
  void copytable(HashTable const &from);
  void destroytable(HashTable &from);
  struct hashcell<DTYPE> *findcell(char const *key);
};


template <class DTYPE>
HashTable<DTYPE>::HashTable(void)
{
  HashTable::gentable((unsigned int) DEFAULT_HASHSIZE);
  for (int i = 0; i < HASH_RECENT; i++) recently[i] = NULL;
}


template <class DTYPE>
HashTable<DTYPE>::HashTable(unsigned int mytablesize)
{
  HashTable::gentable(mytablesize);
  for (int i = 0; i < HASH_RECENT; i++) recently[i] = NULL;
}


template <class DTYPE>
HashTable<DTYPE>::HashTable(HashTable const &from)
{
  HashTable::gentable(from.tablesize);
  HashTable::copytable(from);
  for (int i = 0; i < HASH_RECENT; i++) recently[i] = NULL;
}


template <class DTYPE>
HashTable<DTYPE>::~HashTable(void)
{
  HashTable::destroytable(*this);
}


template <class DTYPE>
DTYPE &HashTable<DTYPE>::operator[] (char const *key)
{
  int alreadyhere = -1;
  struct hashcell<DTYPE> *retval = findcell(key);

  for (int i = 0; i < HASH_RECENT; i++)
    if (recently[i] == retval) {
      alreadyhere = i;
      break;
    }
      
  if (alreadyhere != -1) {
    for (int i = alreadyhere; i > 0; i--)
      recently[i] = recently[i-1];

  } else {
    for (int i = HASH_RECENT - 1; i > 0; i--)
      recently[i] = recently[i-1];
  }
  return (recently[0] = retval)->value;
}


template <class DTYPE>
HashTable<DTYPE> const &HashTable<DTYPE>::operator= (HashTable const &from)
{
  if (this != &from) {
    HashTable::destroytable(*this);
    HashTable::gentable(from.tablesize);
    HashTable::copytable(from);
  }

  return *this;
}


template <class DTYPE>
void HashTable<DTYPE>::hashinsert(char const *key, DTYPE const &value)
{
  unsigned int hashedval;
  struct hashcell<DTYPE> *newcell;

  if (key == NULL) {
    fprintf(stderr, "Error. Called hashinsert() with a NULL key pointer.\n");
    return;
  }

  hashedval = computehash(key);
  if (table[hashedval].keyval != NULL) {
    newcell = new struct hashcell<DTYPE>;

    newcell->next = table[hashedval].next;
    table[hashedval].next = newcell;
  } else {
    newcell = &(table[hashedval]);
  }

  newcell->value.operator=( value);
  newcell->keyval = strdup(key);
  numhashed++;

  if (numhashed > REHASH_CRITFILL * tablesize &&
      tablesize < ::newsizes[numsizes-1])
    rehash();
}


template <class DTYPE>
inline DTYPE &HashTable<DTYPE>::getvalue(char const *key)
{
  return *memelement(key);
}


template <class DTYPE>
void HashTable<DTYPE>::writeoutput(FILE *ostream) const
{
  unsigned int i;
  struct hashcell<DTYPE> const *hptr;
  char buffer[1000];

  fprintf(ostream, "immed\n");
  for (i = 0; i < tablesize; i++) {
    hptr = &table[i];
    while (hptr != NULL && hptr->keyval != NULL) {
      if (hptr->value != 0) {
	hptr->value.printvalue(MAXDIGS - 1, SCI, buffer, 1000, NULL, NULL);
	fputs(buffer, ostream);
	if (strcmp(hptr->keyval, INDIRECT) == 0) {
	  fprintf(ostream, "\n%s\n%s\n", STOREINDIRECTSTR, CLEARXSTR);
	} else {
	  fprintf(ostream, "\n%s %s\n%s\n", STORESTR, hptr->keyval, CLEARXSTR);
	}
      }
      hptr = hptr->next;
    }
  }
}
  


template <class DTYPE>
DTYPE HashTable<DTYPE>::hashdelete(char const *key)
{
  struct hashcell *ptr;
  struct hashcell *lastp;
  struct hashcell *temp;
  unsigned int index;
  DTYPE retval;

  index = computehash(key);
  ptr = &table[index];
  if (strcmp(ptr->keyval, key) == 0) { /* Found it in one */
    free(ptr->keyval);
    retval = ptr->value;
    if (ptr->next != NULL)
      *ptr = *ptr->next;
      
    return retval;
  }

// If we make it here the value is hiding somewhere in the linked list

  while (ptr != NULL &&
	 strcmp(ptr->keyval, key) != 0) {
    lastp = ptr;
    ptr = ptr->next;
  }

  if (ptr == NULL) 
    return (DTYPE) 0.0;

  free(ptr->keyval);
  retval = ptr->value;
  if (ptr->next != NULL) {
    temp = ptr->next;
    *ptr = *temp;
    free(temp);
  } else {
    lastp->next = NULL;
    free(ptr);
  }

  this->numhashed--;
  return retval;
}


template <class DTYPE>
void HashTable<DTYPE>::rehash(void)
{
  unsigned int i;
  unsigned int newsize;

  newsize = 0;
  for (i = 0; i < numsizes; i++)
    if (::newsizes[i] > tablesize) {
      newsize = ::newsizes[i];
      break;
    }

  if (newsize == 0) {
    fprintf(stderr, "Warning. Unable to rehash.\n");
    return;
  }

  rehash(newsize);
}


template <class DTYPE>
void HashTable<DTYPE>::rehash(unsigned int newsize)
{
  HashTable *acopy;

  acopy = new HashTable(*this);
  destroytable(*this);
  gentable(newsize);
  copytable(*acopy);
  delete acopy;
}


template <class DTYPE>
inline unsigned int HashTable<DTYPE>::computehash(char const *key) const
{
  int i;
  unsigned int hash (0);
  
  /* This replacement algorithm provided by Dan Johnson
   * <rdj@wizard.net> after he discovered that the latest C++
   * installations don't provide the hashpjw() function. */
  for (i = 0; i < 3 && *key; i++, key++) {
    hash <<= 2;
    hash += *key;
  }
  return hash % tablesize;
}


template <class DTYPE>
void HashTable<DTYPE>::gentable(unsigned int size)
{
  unsigned int i;

  numhashed = 0;
  table = new struct hashcell<DTYPE> [size];
  tablesize = size;

  for (i = 0; i < tablesize; i++) {
    table[i].next = (struct hashcell<DTYPE> *) NULL;
    table[i].value = DTYPE(0);
    table[i].keyval = (char *) NULL;
  }
  for (i = 0; i < HASH_RECENT; i++)
    recently[i] = NULL;
}


template <class DTYPE>
void HashTable<DTYPE>::copytable(HashTable const &from)
{
  unsigned int i;
  struct hashcell<DTYPE> const *hptr;

  for (i = 0; i < from.tablesize; i++) {
    hptr = &from.table[i];
    while (hptr != NULL && hptr->keyval != NULL) {
      hashinsert(hptr->keyval, hptr->value);
      hptr = hptr->next;
    }
  }
}


template <class DTYPE>
void HashTable<DTYPE>::destroytable(HashTable &killme)
{
  unsigned int i;
  struct hashcell<DTYPE> *hptr;

  for (i = 0; i < killme.tablesize; i++)
    while (killme.table[i].next != NULL) {
      hptr = killme.table[i].next;
      free(hptr->keyval);
      killme.table[i].next = hptr->next;
      delete hptr;
    }

  delete [] killme.table;
  killme.table = NULL;
  killme.numhashed = 0;
  killme.tablesize = 0;
}


template <class DTYPE>
struct hashcell<DTYPE> *HashTable<DTYPE>::findcell(char const *key)
{
  struct hashcell<DTYPE> *ptr;

  ptr = &table[computehash(key)];
  while (ptr != NULL &&
	 ptr->keyval != NULL &&
	 strcmp(ptr->keyval, key) != 0)
    ptr = ptr->next;

  if (ptr == NULL || ptr->keyval == NULL) {
    hashinsert(key, (DTYPE)0.0);
    return findcell(key);
  }

  return ptr;
}


template <class DTYPE>
inline DTYPE *HashTable<DTYPE>::memelement(char const *key)
{
  return &(findcell(key)->value);
}


template <class DTYPE>
void HashTable<DTYPE>::erase_memory(void)
{
  destroytable(*this);
  gentable(newsizes[0]);
}

template <class DTYPE>
DTYPE *HashTable<DTYPE>::recent_access(int i, char const **label) const
{
  if (i >= HASH_RECENT ||
      recently[i] == NULL) return NULL;
  *label = recently[i]->keyval;
  return &recently[i]->value;
}


#endif  /* !HASHTABLE_H */

