/*************************************************************************
** funct-3.1      (command interpreter - funct)                          **
** funct.c : Operations on functions                                     **
**                                                                       **
** Copyright (C) 2003  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#include "interp.h"
#include "funct.h"


/*--------------------------------------------------------------------
    Evaluation of a real simple precision function at a point x.
    f_funct_eval(a, x) returns of a the evaluation at x.
--------------------------------------------------------------------*/
float
f_funct_eval(funct_f *a, float x)
{
    int             i;
    float           y;

    if (x <= a->x[1])
        return a->f[1];
    if (a->type == 0) {
        if (x >= a->x[a->nb])
            return a->f[a->nb];
        else {
            i = 1;
            while(x > a->x[i]) {
                i++;
            }
            return (a->f[i] * (x - a->x[i - 1]) + a->f[i - 1] * (a->x[i] - x))
                / (a->x[i] - a->x[i - 1]);
        }
    }
    else {
        if (x >= a->x[3])
            return a->f[a->nb];
        i = 1 + (x - a->x[1]) / a->x[2];
        y = a->x[1] + (i - 1) * a->x[2];
        return (a->f[i] * (y + a->x[2] - x) + a->f[i + 1] * (x - y))
            / a->x[2];
    }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Sum of two real simple precision functions.
    f_funct_add(a, b, c) fills the function c with the sum of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
f_funct_add(funct_f *a, funct_f *b, funct_f *c)
{
    int             i;
    float          *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        f_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = f_funct_eval(a, cx[i]) + f_funct_eval(b, cx[i]);
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = a->f[i] + b->f[i];
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Product of two real simple precision functions.
    f_funct_mul(a, b, c) fills the function c with the product of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
f_funct_mul(funct_f *a, funct_f *b, funct_f *c)
{
    int             i;
    float          *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        f_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = f_funct_eval(a, cx[i]) * f_funct_eval(b, cx[i]);
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = a->f[i] * b->f[i];
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Quotient of two real simple precision functions.
    f_funct_div(a, b, c) fills the function c with the quotient of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
f_funct_div(funct_f *a, funct_f *b, funct_f *c)
{
    int             i;
    float          *cx,
                    x;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        f_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */

        for (i = 1; i <= c->nb; i++) {
            x = f_funct_eval(b, cx[i]);
            if (x != 0.)
                c->f[i] = f_funct_eval(a, cx[i]) / x;
            else
                c->f[i] = 0.;
        }

        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++) {
            if (b->f[i] != 0)
                c->f[i] = a->f[i] / b->f[i];
            else
                c->f[i] = 0.;
        }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Fills the array ax[0] with the xrange of a. This array must not
    have been allocated before.
--------------------------------------------------------------------*/
void
f_fill(funct_f *a, float **ax)
{
    int             i,
                    j;
    float           dx;

    if (a->type == 0) {
        ax[0] = a->x;
    }
    else {
        ax[0] = float_alloc1(a->nb + 1);
        dx = a->x[2];
        j = a->nb;
        for (i = 1; i <= j; i++)
            ax[0][i] = a->x[1] + (i - 1) * dx;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Substraction of two real simple precision functions.
    f_funct_add(a, b, c) fills the function c with a-b.
    It is not assumed that these functions are defined on the same interval.
--------------------------------------------------------------------*/
void
f_funct_sub(funct_f *a, funct_f *b, funct_f *c)
{
    int             i;
    float          *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        f_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = f_funct_eval(a, cx[i]) - f_funct_eval(b, cx[i]);
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = a->f[i] - b->f[i];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of a real simple precision function by a number :
    f_funct_Rmul(t, a) fills b with t * a. It is not assumed that the two
    functions are defined on the same interval.
--------------------------------------------------------------------*/
void
f_funct_Rmul(float t, funct_f *a, funct_f *b)
{
    int             i;
    float          *bx;

    if (a->x != b->x) {
        bx = NULL;
        f_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = t * f_funct_eval(a, bx[i]);
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= a->nb; i++)
            b->f[i] = t * a->f[i];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the real simple precision function a with the real constant t.
--------------------------------------------------------------------*/
void
f_funct_const(float t, funct_f *a)
{
    int             i;

    for (i = 1; i <= a->nb; i++)
        a->f[i] = t;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Copy the real simple precision function a in b. It is not assumed
    that the two functions have the same xrange.
--------------------------------------------------------------------*/
void
f_funct_copy(funct_f *a, funct_f *b)
{
    int             i;
    float          *bx;

    if (a->x != b->x) {
        bx = NULL;
        f_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = f_funct_eval(a, bx[i]);
        if (b->type > 0)
            XFREE(bx);
    }
    else {
        for (i = 1; i <= b->nb; i++)
            b->f[i] = a->f[i];
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Frees the arrays which are members of a.
--------------------------------------------------------------------*/
void
f_funct_detruit(funct_f *a)
{
    if (a->x != NULL) {
        XFREE(a->x);
        a->x = NULL;
    }
    if (a->f != NULL) {
        XFREE(a->f);
        a->f = NULL;
    }
    if (a->nom != NULL) {
        free(a->nom);
        a->nom = NULL;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Evaluation of a real double precision function at a point x.
    d_funct_eval(a, x) returns the evaluation of a at x.
--------------------------------------------------------------------*/
double
d_funct_eval(funct_d *a, double x)
{
    int             i;
    double          y;

    if (x <= a->x[1])
        return a->f[1];
    if (a->type == 0) {
        if (x >= a->x[a->nb])
            return a->f[a->nb];
        else {
            i = 1;
            while(x > a->x[i]) {
                i++;
            }
            return (a->f[i] * (x - a->x[i - 1]) + a->f[i - 1] * (a->x[i] - x))
                / (a->x[i] - a->x[i - 1]);
        }
    }
    else {
        if (x >= a->x[3])
            return a->f[a->nb];
        i = 1 + (x - a->x[1]) / a->x[2];
        y = a->x[1] + (i - 1) * a->x[2];
        return (a->f[i] * (y + a->x[2] - x) + a->f[i + 1] * (x - y))
            / a->x[2];
    }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Sum of two real double precision functions.
    d_funct_add(a, b, c) fills the function c with the sum of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
d_funct_add(funct_d *a, funct_d *b, funct_d *c)
{
    int             i;
    double         *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        d_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = d_funct_eval(a, cx[i]) + d_funct_eval(b, cx[i]);
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = a->f[i] + b->f[i];
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Product of two real double precision functions.
    d_funct_mul(a, b, c) fills the function c with the product of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
d_funct_mul(funct_d *a, funct_d *b, funct_d *c)
{
    int             i;
    double         *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        d_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = d_funct_eval(a, cx[i]) * d_funct_eval(b, cx[i]);
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = a->f[i] * b->f[i];
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Quotient of two real double precision functions.
    d_funct_div(a, b, c) fills the function c with the quotient of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
d_funct_div(funct_d *a, funct_d *b, funct_d *c)
{
    int             i;
    double         *cx,
                    x;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        d_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */

        for (i = 1; i <= c->nb; i++) {
            x = d_funct_eval(b, cx[i]);
            if (x != 0.)
                c->f[i] = d_funct_eval(a, cx[i]) / x;
            else
                c->f[i] = 0.;
        }

        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++) {
            if (b->f[i] != 0)
                c->f[i] = a->f[i] / b->f[i];
            else
                c->f[i] = 0.;
        }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Fills the array ax[0] with the xrange of a. This array must not
    have been allocated before.
--------------------------------------------------------------------*/
void
d_fill(funct_d *a, double **ax)
{
    int             i,
                    j;
    double          dx;

    if (a->type == 0) {
        ax[0] = a->x;
    }
    else {
        ax[0] = double_alloc1(a->nb + 1);
        dx = a->x[2];
        j = a->nb;
        for (i = 1; i <= j; i++)
            ax[0][i] = a->x[1] + (i - 1) * dx;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Substraction of two real double precision functions.
    d_funct_add(a, b, c) fills the function c with a-b.
    It is not assumed that these functions are defined on the same interval.
--------------------------------------------------------------------*/
void
d_funct_sub(funct_d *a, funct_d *b, funct_d *c)
{
    int             i;
    double         *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        d_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = d_funct_eval(a, cx[i]) - d_funct_eval(b, cx[i]);
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = a->f[i] - b->f[i];
}
/*-----------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of a real double precision function by a number :
    d_funct_Rmul(t, a) fills b with t * a. It is not assumed that the two
    functions are defined on the same interval.
--------------------------------------------------------------------*/
void
d_funct_Rmul(double t, funct_d *a, funct_d *b)
{
    int             i;
    double         *bx;

    if (a->x != b->x) {
        bx = NULL;
        d_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = t * d_funct_eval(a, bx[i]);
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= a->nb; i++)
            b->f[i] = t * a->f[i];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the real double precision function a with the real constant t.
--------------------------------------------------------------------*/
void
d_funct_const(double t, funct_d *a)
{
    int             i;

    for (i = 1; i <= a->nb; i++)
        a->f[i] = t;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Copy the real double precision function a in b. It is not assumed
    that the two functions have the same xrange.
--------------------------------------------------------------------*/
void
d_funct_copy(funct_d *a, funct_d *b)
{
    int             i;
    double         *bx;

    bx = NULL;
    d_fill(b, &bx);   /* fills the array bx
                         with the x of the function b */
    for (i = 1; i <= b->nb; i++)
        b->f[i] = d_funct_eval(a, bx[i]);
    if (b->type > 0)
        XFREE(bx);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Frees the arrays which are members of a.
--------------------------------------------------------------------*/
void
d_funct_detruit(funct_d *a)
{
    if (a->x != NULL) {
        XFREE(a->x);
        a->x = NULL;
    }
    if (a->f != NULL) {
        XFREE(a->f);
        a->f = NULL;
    }
    if (a->nom != NULL) {
        free(a->nom);
        a->nom = NULL;
    }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Evaluation of a complex simple precision function at a point x.
    C_funct_eval(a, x) returns the evaluation of a at x.
--------------------------------------------------------------------*/
fcomplex
C_funct_eval(funct_C *a, float x)
{
    int             i;
    float           y;
    fcomplex        z;

    if (x <= a->x[1])
        return a->f[1];
    if (a->type == 0) {
        if (x >= a->x[a->nb])
            return a->f[a->nb];
        else {
            i = 1;
            while(x > a->x[i]) {
                i++;
            }
            y = a->x[i] - a->x[i - 1];
            z = Cadd(RCmul(x - a->x[i - 1], a->f[i]),
                RCmul(a->x[i] - x, a->f[i - 1]));
            return RCdiv(z, y);
        }
    }
    else {
        if (x >= a->x[3])
            return a->f[a->nb];
        i = 1 + (x - a->x[1]) / a->x[2];
        y = a->x[1] + (i - 1) * a->x[2];
        return RCmul(1. / a->x[2], Cadd(RCmul(y + a->x[2] - x, a->f[i]),
            RCmul(x - y, a->f[i + 1])));
    }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Sum of two complex simple precision functions.
    C_funct_add(a, b, c) fills the function c with the sum of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
C_funct_add(funct_C *a, funct_C *b, funct_C *c)
{
    int             i;
    float          *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        C_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = Cadd(C_funct_eval(a, cx[i]), C_funct_eval(b, cx[i]));
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = Cadd(a->f[i], b->f[i]);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Product of two complex simple precision functions.
    C_funct_mul(a, b, c) fills the function c with the product of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
C_funct_mul(funct_C *a, funct_C *b, funct_C *c)
{
    int             i;
    float          *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        C_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = Cmul(C_funct_eval(a, cx[i]), C_funct_eval(b, cx[i]));
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = Cmul(a->f[i], b->f[i]);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Quotient of two complex simple precision functions.
    C_funct_div(a, b, c) fills the function c with the quotient of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
C_funct_div(funct_C *a, funct_C *b, funct_C *c)
{
    int             i;
    float          *cx;
    fcomplex        z;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        C_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */

        for (i = 1; i <= c->nb; i++) {
            z = C_funct_eval(b, cx[i]);
            if (z.r != 0. || z.i != 0.)
                c->f[i] = Cdiv(C_funct_eval(a, cx[i]), z);
            else
                c->f[i] = Complex(0., 0.);
        }

        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++) {
            z = b->f[i];
            if (z.r != 0. || z.i != 0.)
                c->f[i] = Cdiv(a->f[i], z);
            else
                c->f[i] = Complex(0., 0.);
        }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Fills the array ax[0] with the xrange of a. This array must not
    have been allocated before.
--------------------------------------------------------------------*/
void
C_fill(funct_C *a, float **ax)
{
    int             i,
                    j;
    float           dx;

    if (a->type == 0) {
        ax[0] = a->x;
    }
    else {
        ax[0] = float_alloc1(a->nb + 1);
        dx = a->x[2];
        j = a->nb;
        for (i = 1; i <= j; i++)
            ax[0][i] = a->x[1] + (i - 1) * dx;
    }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Substraction of two complex simple precision functions.
    C_funct_add(a, b, c) fills the function c with a-b.
    It is not assumed that these functions are defined on the same interval.
--------------------------------------------------------------------*/
void
C_funct_sub(funct_C *a, funct_C *b, funct_C *c)
{
    int             i;
    float          *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        C_fill(c, &cx);   /* fills the array cx
                             with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = Csub(C_funct_eval(a, cx[i]), C_funct_eval(b, cx[i]));
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = Csub(a->f[i], b->f[i]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of a complex simple precision function by a complex number :
    C_funct_Rmul(t, a) fills b with t * a. It is not assumed that the two
    functions are defined on the same interval.
--------------------------------------------------------------------*/
void
C_funct_Rmul(fcomplex t, funct_C *a, funct_C *b)
{
    int             i;
    float          *bx;

    if (a->x != b->x) {
        bx = NULL;
        C_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = Cmul(t , C_funct_eval(a, bx[i]));
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= a->nb; i++)
            b->f[i] = Cmul(t, a->f[i]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the complex simple precision function a with the complex constant t.
--------------------------------------------------------------------*/
void
C_funct_const(fcomplex t, funct_C *a)
{
    int             i;

    for (i = 1; i <= a->nb; i++)
        a->f[i] = t;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Copy the complex simple precision function a in b. It is not assumed
    that the two functions have the same xrange.
--------------------------------------------------------------------*/
void
C_funct_copy(funct_C *a, funct_C *b)
{
    int             i;
    float          *bx;

    bx = NULL;
    C_fill(b, &bx);   /* fills the array bx
                         with the x of the function b */
    for (i = 1; i <= b->nb; i++)
        b->f[i] = C_funct_eval(a, bx[i]);
    if (b->type > 0)
        XFREE(bx);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Frees the arrays which are members of a.
--------------------------------------------------------------------*/
void
C_funct_detruit(funct_C *a)
{
    if (a->x != NULL) {
        XFREE(a->x);
        a->x = NULL;
    }
    if (a->f != NULL) {
        XFREE(a->f);
        a->f = NULL;
    }
    if (a->nom != NULL) {
        free(a->nom);
        a->nom = NULL;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Evaluation of a complex double precision function at a point x.
    dC_funct_eval(a, x) returns the evaluation of a at x.
--------------------------------------------------------------------*/
dcomplex
dC_funct_eval(funct_dC *a, double x)
{
    int             i;
    double          y;

    if (x <= a->x[1])
        return a->f[1];
    if (a->type == 0) {
        if (x >= a->x[a->nb])
            return a->f[a->nb];
        else {
            i = 1;
            while(x > a->x[i]) {
                i++;
            }
            return dRCdiv(dCadd(dRCmul(x - a->x[i - 1], a->f[i]),
                dRCmul(a->x[i] - x, a->f[i - 1])), a->x[i] - a->x[i - 1]);
        }
    }
    else {
        if (x >= a->x[3])
            return a->f[a->nb];
        i = 1 + (x - a->x[1]) / a->x[2];
        y = a->x[1] + (i - 1) * a->x[2];
        return dRCmul(1. / a->x[2], dCadd(dRCmul(y + a->x[2] - x, a->f[i]),
            dRCmul(x - y, a->f[i + 1])));
    }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Sum of two complex double precision functions.
    dC_funct_add(a, b, c) fills the function c with the sum of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
dC_funct_add(funct_dC *a, funct_dC *b, funct_dC *c)
{
    int             i;
    double         *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        dC_fill(c, &cx);   /* fills the array cx
                              with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = dCadd(dC_funct_eval(a, cx[i]), dC_funct_eval(b, cx[i]));
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = dCadd(a->f[i], b->f[i]);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Product of two complex double precision functions.
    dC_funct_mul(a, b, c) fills the function c with the product of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
dC_funct_mul(funct_dC *a, funct_dC *b, funct_dC *c)
{
    int             i;
    double         *cx;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        dC_fill(c, &cx);   /* fills the array cx
                              with the x of the function c */
        for (i = 1; i <= c->nb; i++)
            c->f[i] = dCmul(dC_funct_eval(a, cx[i]), dC_funct_eval(b, cx[i]));
        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = dCmul(a->f[i], b->f[i]);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Quotient of two complex double precision functions.
    dC_funct_div(a, b, c) fills the function c with the quotient of the
    functions a and b. It is not assumed that these functions are
    defined on the same interval.
--------------------------------------------------------------------*/
void
dC_funct_div(funct_dC *a, funct_dC *b, funct_dC *c)
{
    int             i;
    double         *cx;
    dcomplex        z;

    if (a->x != b->x || a->x != c->x) {
        cx = NULL;
        dC_fill(c, &cx);   /* fills the array cx
                              with the x of the function c */

        for (i = 1; i <= c->nb; i++) {
            z = dC_funct_eval(b, cx[i]);
            if (z.r != 0. || z.i != 0.)
                c->f[i] = dCdiv(dC_funct_eval(a, cx[i]), z);
            else
                c->f[i] = dComplex(0., 0.);
        }

        if (c->type > 0)
            XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++) {
            z = b->f[i];
            if (z.r != 0. || z.i != 0.)
                c->f[i] = dCdiv(a->f[i], z);
            else
                c->f[i] = dComplex(0., 0.);
        }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Fills the array ax[0] with the xrange of a. This array must not
    have been allocated before.
--------------------------------------------------------------------*/
void
dC_fill(funct_dC *a, double **ax)
{
    int             i,
                    j;
    double          dx;

    if (a->type == 0) {
        ax[0] = a->x;
    }
    else {
        ax[0] = double_alloc1(a->nb + 1);
        dx = a->x[2];
        j = a->nb;
        for (i = 1; i <= j; i++)
            ax[0][i] = a->x[1] + (i - 1) * dx;
    }
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Substraction of two complex double precision functions.
    dC_funct_add(a, b, c) fills the function c with a-b.
    It is not assumed that these functions are defined on the same interval.
--------------------------------------------------------------------*/
void
dC_funct_sub(funct_dC *a, funct_dC *b, funct_dC *c)
{
    int             i;
    double         *cx;

    if (a->x != b->x || a->x != c->x) {
    cx = NULL;
    dC_fill(c, &cx);   /* fills the array cx
                          with the x of the function c */
    for (i = 1; i <= c->nb; i++)
        c->f[i] = dCsub(dC_funct_eval(a, cx[i]), dC_funct_eval(b, cx[i]));
    if (c->type > 0)
        XFREE(cx);
    }
    else
        for (i = 1; i <= c->nb; i++)
            c->f[i] = dCsub(a->f[i], b->f[i]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of a complex simple precision function by a complex number :
    dC_funct_Rmul(t, a) fills b with t * a. It is not assumed that the two
    functions are defined on the same interval.
--------------------------------------------------------------------*/
void
dC_funct_Rmul(dcomplex t, funct_dC *a, funct_dC *b)
{
    int             i;
    double         *bx;

    if (a->x != b->x) {
        bx = NULL;
        dC_fill(b, &bx);   /* fills the array bx
                              with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = dCmul(t, dC_funct_eval(a, bx[i]));
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= a->nb; i++)
            b->f[i] = dCmul(t, a->f[i]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the complex double precision function a with the complex constant t.
--------------------------------------------------------------------*/
void
dC_funct_const(dcomplex t, funct_dC *a)
{
    int             i;

    for (i = 1; i <= a->nb; i++)
        a->f[i] = t;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Copy the complex double precision function a in b. It is not assumed
    that the two functions have the same xrange.
--------------------------------------------------------------------*/
void
dC_funct_copy(funct_dC *a, funct_dC *b)
{
    int             i;
    double         *bx;

    bx = NULL;
    dC_fill(b, &bx);   /* fills the array bx
                          with the x of the function b */
    for (i = 1; i <= b->nb; i++)
        b->f[i] = dC_funct_eval(a, bx[i]);
    if (b->type > 0)
        XFREE(bx);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Frees the arrays which are members of a.
--------------------------------------------------------------------*/
void
dC_funct_detruit(funct_dC *a)
{
    if (a->x != NULL) {
        XFREE(a->x);
        a->x = NULL;
    }
    if (a->f != NULL) {
        XFREE(a->f);
        a->f = NULL;
    }
    if (a->nom != NULL) {
        free(a->nom);
        a->nom = NULL;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    This function tests if the two arrays of simple precision numbers a and b
    contain the same values for indices between 1 and n. It returns 0 if
    the arrays are not the same, and 1 otherwise.
--------------------------------------------------------------------*/
int
test_f_equal_x(float *a, float *b, int n)
{
    int             i;

    for (i = 1; i <= n; i++) {
        if (a[i] != b[i])
            return 0;
    }

    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    This function tests if the two arrays of double precision numbers a and b
    contain the same values for indices between 1 and n. It returns 0 if
    the arrays are not the same, and 1 otherwise.
--------------------------------------------------------------------*/
int
test_d_equal_x(double *a, double *b, int n)
{
    int             i;

    for (i = 1; i <= n; i++) {
        if (a[i] != b[i])
            return 0;
    }

    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the real part of the complex simple precision function b with
    the real simple precision function a. It is not assumed that the
    two functions have the same xrange.
--------------------------------------------------------------------*/
void
conv_f_Cr(funct_f *a, funct_C *b)
{
    int             i;
    float          *bx;

    if (a->x != b->x) {
        bx = NULL;
        C_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i].r = f_funct_eval(a, bx[i]);
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= b->nb; i++)
            b->f[i].r = a->f[i];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the imaginary part of the complex simple precision function b with
    the real simple precision function a. It is not assumed that the
    two functions have the same xrange.
--------------------------------------------------------------------*/
void
conv_f_Ci(funct_f *a, funct_C *b)
{
    int             i;
    float          *bx;

    if (a->x != b->x) {
        bx = NULL;
        C_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i].i = f_funct_eval(a, bx[i]);
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= b->nb; i++)
            b->f[i].i = a->f[i];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the real part of the complex double precision function b with
    the real double precision function a. It is not assumed that the
    two functions have the same xrange.
--------------------------------------------------------------------*/
void
conv_d_dCr(funct_d *a, funct_dC *b)
{
    int             i;
    double         *bx;

    if (a->x != b->x) {
        bx = NULL;
        dC_fill(b, &bx);   /* fills the array bx
                              with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i].r = d_funct_eval(a, bx[i]);
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= b->nb; i++)
            b->f[i].r = a->f[i];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the imaginary part of the complex double precision function b with
    the real double precision function a. It is not assumed that the
    two functions have the same xrange.
--------------------------------------------------------------------*/
void
conv_d_dCi(funct_d *a, funct_dC *b)
{
    int             i;
    double         *bx;

    if (a->x != b->x) {
        bx = NULL;
        dC_fill(b, &bx);   /* fills the array bx
                              with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i].i = d_funct_eval(a, bx[i]);
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= b->nb; i++)
            b->f[i].i = a->f[i];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the real simple precision function b with the real part of the
    complex simple precision function a. It is not assumed that the two
    functions have the same xrange.
--------------------------------------------------------------------*/
void
reel_f(funct_C *a, funct_f *b)
{
    int             i;
    float          *bx;

    if (a->x != b->x) {
        bx = NULL;
        f_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = C_funct_eval(a, bx[i]).r;
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= b->nb; i++)
            b->f[i] = a->f[i].r;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the real simple precision function b with the imaginary part of the
    complex simple precision function a. It is not assumed that the two
    functions have the same xrange.
--------------------------------------------------------------------*/
void
imag_f(funct_C *a, funct_f *b)
{
    int             i;
    float          *bx;

    if (a->x != b->x) {
        bx = NULL;
        f_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = C_funct_eval(a, bx[i]).i;
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= b->nb; i++)
            b->f[i] = a->f[i].i;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the real double precision function b with the real part of the
    complex double precision function a. It is not assumed that the two
    functions have the same xrange.
--------------------------------------------------------------------*/
void
reel_d(funct_dC *a, funct_d *b)
{
    int             i;
    double         *bx;

    if (a->x != b->x) {
        bx = NULL;
        d_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = dC_funct_eval(a, bx[i]).r;
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= b->nb; i++)
            b->f[i] = a->f[i].r;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Fills the real double precision function b with the imaginary part of the
    complex double precision function a. It is not assumed that the two
    functions have the same xrange.
--------------------------------------------------------------------*/
void
imag_d(funct_dC *a, funct_d *b)
{
    int             i;
    double         *bx;

    if (a->x != b->x) {
        bx = NULL;
        d_fill(b, &bx);   /* fills the array bx
                             with the x of the function b */
        for (i = 1; i <= b->nb; i++)
            b->f[i] = dC_funct_eval(a, bx[i]).i;
        if (b->type > 0)
            XFREE(bx);
    }
    else
        for (i = 1; i <= b->nb; i++)
            b->f[i] = a->f[i].i;
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Computes the integral of the real simple precision function a and fills
    b with the result. It is assumed that the integral is zero at the first
    point x where a is defined.
--------------------------------------------------------------------*/
void
integ_funct_f(funct_f *a, funct_f *b)
{
    int             i;
    funct_f        *tmp;
    float           x;

    if (a->x != b->x) {
        tmp = (funct_f *) malloc((size_t) sizeof(funct_f));
        f_fill(a, &tmp->x);   /* fills the array tmp->x
                                 with the x of the function a */
        tmp->nb = a->nb;
        tmp->type = 0;
        tmp->f = float_alloc1(tmp->nb);
        tmp->f[1] = 0.;
        for (i = 2; i <= tmp->nb; i++)
            tmp->f[i] = tmp->f[i - 1] + (tmp->x[i] - tmp->x[i - 1]) *
                (a->f[i] + a->f[i - 1]) / 2.;
        f_funct_copy(tmp, b);
        if (a->type > 0)
            XFREE(tmp->x);
        XFREE(tmp->f);
        free(tmp);
    }
    else {
        b->f[1] = 0.;
        if (a->type == 0) {
            for (i = 2; i <= a->nb; i++)
                b->f[i] = b->f[i - 1] + (a->x[i] - a->x[i - 1]) *
                    (a->f[i] + a->f[i - 1]) / 2.;
        }
        else {
            x = a->x[2] / 2.;
            for (i = 2; i <= a->nb; i++)
                b->f[i] = b->f[i - 1] + x * (a->f[i] + a->f[i - 1]);
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the integral of the real double precision function a and fills
    b with the result. It is assumed that the integral is zero at the first
    point x where a is defined.
--------------------------------------------------------------------*/
void
integ_funct_d(funct_d *a, funct_d *b)
{
    int             i;
    funct_d        *tmp;
    double          x;

    if (a->x != b->x) {
        tmp = (funct_d *) malloc((size_t) sizeof(funct_d));
        d_fill(a, &tmp->x);   /* fills the array tmp->x
                                 with the x of the function a */
        tmp->nb = a->nb;
        tmp->type = 0;
        tmp->f = double_alloc1(tmp->nb);
        tmp->f[1] = 0.;
        for (i = 2; i <= tmp->nb; i++)
            tmp->f[i] = tmp->f[i - 1] + (tmp->x[i] - tmp->x[i - 1]) *
                (a->f[i] + a->f[i - 1]) / 2.;
        d_funct_copy(tmp, b);
        if (a->type > 0)
            XFREE(tmp->x);
        XFREE(tmp->f);
        free(tmp);
    }
    else {
        b->f[1] = 0.;
        if (a->type == 0) {
            for (i = 2; i <= a->nb; i++)
                b->f[i] = b->f[i - 1] + (a->x[i] - a->x[i - 1]) *
                    (a->f[i] + a->f[i - 1]) / 2.;
        }
        else {
            x = a->x[2] / 2.;
            for (i = 2; i <= a->nb; i++)
                b->f[i] = b->f[i - 1] + x * (a->f[i] + a->f[i - 1]);
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the integral of the complex simple precision function a and fills
    b with the result. It is assumed that the integral is zero at the first
    point x where a is defined.
--------------------------------------------------------------------*/
void
integ_funct_C(funct_C *a, funct_C *b)
{
    int             i;
    funct_C        *tmp;
    float           x;

    if (a->x != b->x) {
        tmp = (funct_C *) malloc((size_t) sizeof(funct_C));
        C_fill(a, &tmp->x);   /* fills the array tmp->x
                                 with the x of the function a */
        tmp->nb = a->nb;
        tmp->type = 0;
        tmp->f = fcomplex_alloc1(tmp->nb);
        tmp->f[1] = Complex(0., 0.);
        for (i = 2; i <= tmp->nb; i++)
            tmp->f[i] = Cadd(tmp->f[i - 1] ,
                RCmul((tmp->x[i] - tmp->x[i - 1]) / 2.,
                    Cadd(a->f[i], a->f[i - 1])));
        C_funct_copy(tmp, b);
        if (a->type > 0)
            XFREE(tmp->x);
        XFREE(tmp->f);
        free(tmp);
    }
    else {
        b->f[1] = Complex(0., 0.);
        if (a->type == 0) {
            for (i = 2; i <= a->nb; i++)
                b->f[i] = Cadd(b->f[i - 1], RCmul((a->x[i] - a->x[i - 1]) / 2.,
                    Cadd(a->f[i], a->f[i - 1])));
        }
        else {
            x = a->x[2] / 2.;
            for (i = 2; i <= a->nb; i++)
                b->f[i] = Cadd(b->f[i - 1], RCmul(x,
                    Cadd(a->f[i], a->f[i - 1])));
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the integral of the complex double precision function a and fills
    b with the result. It is assumed that the integral is zero at the first
    point x where a is defined.
--------------------------------------------------------------------*/
void
integ_funct_dC(funct_dC *a, funct_dC *b)
{
    int             i;
    funct_dC       *tmp;
    double          x;

    if (a->x != b->x) {
        tmp = (funct_dC *) malloc((size_t) sizeof(funct_dC));
        dC_fill(a, &tmp->x);   /* fills the array tmp->x
                                  with the x of the function a */
        tmp->nb = a->nb;
        tmp->type = 0;
        tmp->f = dcomplex_alloc1(tmp->nb);
        tmp->f[1] = dComplex(0., 0.);
        for (i = 2; i <= tmp->nb; i++)
            tmp->f[i] = dCadd(tmp->f[i - 1] ,
                dRCmul((tmp->x[i] - tmp->x[i - 1]) / 2.,
                    dCadd(a->f[i], a->f[i - 1])));
        dC_funct_copy(tmp, b);
        if (a->type > 0)
            XFREE(tmp->x);
        XFREE(tmp->f);
        free(tmp);
    }
    else {
        b->f[1] = dComplex(0., 0.);
        if (a->type == 0) {
            for (i = 2; i <= a->nb; i++)
                b->f[i] = dCadd(b->f[i - 1],
                    dRCmul((a->x[i] - a->x[i - 1]) / 2.,
                        dCadd(a->f[i], a->f[i - 1])));
        }
        else {
            x = a->x[2] / 2.;
            for (i = 2; i <= a->nb; i++)
                b->f[i] = dCadd(b->f[i - 1], dRCmul(x,
                    dCadd(a->f[i], a->f[i - 1])));
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
   Computes the derivative of the real simple precision function a and
   fills b with the result.
--------------------------------------------------------------------*/
void
diff_funct_f(funct_f *a, funct_f *b)
{
    int             i;
    funct_f        *tmp;
    float          *pt;

    if (test_correct_x_f(a) == 0)
        return;
    if (a->x != b->x) {
        tmp = (funct_f *) malloc((size_t) sizeof(funct_f));
        f_fill(a, &tmp->x);   /* fills the array tmp->x
                                 with the x of the function a */
        tmp->nb = a->nb;
        tmp->type = 0;
        tmp->f = float_alloc1(tmp->nb);
        pt = float_alloc1(tmp->nb);
        for (i = 1; i < tmp->nb; i++)
            pt[i] = (a->f[i + 1] - a->f[i]) / (tmp->x[i + 1] - tmp->x[i]);
        tmp->f[1] = pt[1];
        tmp->f[tmp->nb] = pt[tmp->nb - 1];
        for (i = 2; i < tmp->nb; i++)
            tmp->f[i] = (pt[i] + pt[i - 1]) / 2.;
        f_funct_copy(tmp, b);
        if (a->type > 0)
            XFREE(tmp->x);
        XFREE(tmp->f);
        XFREE(pt);
        free(tmp);
    }
    else {
        if (a->type == 0) {
            for (i = 1; i < a->nb; i++)
                b->f[i] = (a->f[i + 1] - a->f[i]) / (a->x[i + 1] - a->x[i]);
            b->f[a->nb] = b->f[a->nb - 1];
            for (i = a->nb - 1; i > 1; i--)
                b->f[i] = (b->f[i] + b->f[i - 1]) / 2.;
        }
        else {
            for (i = 1; i < a->nb; i++)
                b->f[i] = (a->f[i + 1] - a->f[i]) / a->x[2];
            b->f[a->nb] = b->f[a->nb - 1];
            for (i = a->nb - 1; i > 1; i--)
                b->f[i] = (b->f[i] + b->f[i - 1]) / 2.;
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
   Computes the derivative of the real double precision function a and
   fills b with the result.
--------------------------------------------------------------------*/
void
diff_funct_d(funct_d *a, funct_d *b)
{
    int             i;
    funct_d        *tmp;
    double         *pt;

    if (test_correct_x_d(a) == 0)
        return;
    if (a->x != b->x) {
        tmp = (funct_d *) malloc((size_t) sizeof(funct_d));
        d_fill(a, &tmp->x);   /* fills the array tmp->x
                                 with the x of the function a */
        tmp->nb = a->nb;
        tmp->type = 0;
        tmp->f = double_alloc1(tmp->nb);
        pt = double_alloc1(tmp->nb);
        for (i = 1; i < tmp->nb; i++)
            pt[i] = (a->f[i + 1] - a->f[i]) / (tmp->x[i + 1] - tmp->x[i]);
        tmp->f[1] = pt[1];
        tmp->f[tmp->nb] = pt[tmp->nb - 1];
        for (i = 2; i < tmp->nb; i++)
            tmp->f[i] = (pt[i] + pt[i - 1]) / 2.;
        d_funct_copy(tmp, b);
        if (a->type > 0)
            XFREE(tmp->x);
        XFREE(tmp->f);
        XFREE(pt);
        free(tmp);
    }
    else {
        if (a->type == 0) {
            for (i = 1; i < a->nb; i++)
                b->f[i] = (a->f[i + 1] - a->f[i]) / (a->x[i + 1] - a->x[i]);
            b->f[a->nb] = b->f[a->nb - 1];
            for (i = a->nb - 1; i > 1; i--)
                b->f[i] = (b->f[i] + b->f[i - 1]) / 2.;
        }
        else {
            for (i = 1; i < a->nb; i++)
                b->f[i] = (a->f[i + 1] - a->f[i]) / a->x[2];
            b->f[a->nb] = b->f[a->nb - 1];
            for (i = a->nb - 1; i > 1; i--)
                b->f[i] = (b->f[i] + b->f[i - 1]) / 2.;
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
   Computes the derivative of the complex simple precision function a and
   fills b with the result.
--------------------------------------------------------------------*/
void
diff_funct_C(funct_C *a, funct_C *b)
{
    int             i;
    funct_C        *tmp;
    fcomplex       *pt;

    if (test_correct_x_C(a) == 0)
        return;
    if (a->x != b->x) {
        tmp = (funct_C *) malloc((size_t) sizeof(funct_C));
        C_fill(a, &tmp->x);   /* fills the array tmp->x
                                 with the x of the function a */
        tmp->nb = a->nb;
        tmp->type = 0;
        tmp->f = fcomplex_alloc1(tmp->nb);
        pt = fcomplex_alloc1(tmp->nb);
        for (i = 1; i < tmp->nb; i++)
            pt[i] = RCdiv(Csub(a->f[i + 1], a->f[i]), a->x[i + 1] - a->x[i]);
        tmp->f[1] = pt[1];
        tmp->f[tmp->nb] = pt[tmp->nb - 1];
        for (i = 2; i < tmp->nb; i++)
            tmp->f[i] = RCdiv(Cadd(pt[i], pt[i - 1]), 2.);
        C_funct_copy(tmp, b);
        if (a->type > 0)
            XFREE(tmp->x);
        XFREE(tmp->f);
        XFREE(pt);
        free(tmp);
    }
    else {
        if (a->type == 0) {
            for (i = 1; i < a->nb; i++)
                b->f[i] = RCdiv(Csub(a->f[i + 1], a->f[i]),
                    a->x[i + 1] - a->x[i]);
            b->f[a->nb] = b->f[a->nb - 1];
            for (i = a->nb - 1; i > 1; i--)
                b->f[i] = RCdiv(Cadd(b->f[i], b->f[i - 1]), 2.);
        }
        else {
            for (i = 1; i < a->nb; i++)
                b->f[i] = RCdiv(Csub(a->f[i + 1], a->f[i]), a->x[2]);
            b->f[a->nb] = b->f[a->nb - 1];
            for (i = a->nb - 1; i > 1; i--)
                b->f[i] = RCdiv(Cadd(b->f[i], b->f[i - 1]), 2.);
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
   Computes the derivative of the complex double precision function a and
   fills b with the result.
--------------------------------------------------------------------*/
void
diff_funct_dC(funct_dC *a, funct_dC *b)
{
    int             i;
    funct_dC       *tmp;
    dcomplex       *pt;

    if (test_correct_x_dC(a) == 0)
        return;
    if (a->x != b->x) {
        tmp = (funct_dC *) malloc((size_t) sizeof(funct_dC));
        dC_fill(a, &tmp->x);   /* fills the array tmp->x
                                  with the x of the function a */
        tmp->nb = a->nb;
        tmp->type = 0;
        tmp->f = dcomplex_alloc1(tmp->nb);
        pt = dcomplex_alloc1(tmp->nb);
        for (i = 1; i < tmp->nb; i++)
            pt[i] = dRCdiv(dCsub(a->f[i + 1], a->f[i]), a->x[i + 1] - a->x[i]);
        tmp->f[1] = pt[1];
        tmp->f[tmp->nb] = pt[tmp->nb - 1];
        for (i = 2; i < tmp->nb; i++)
            tmp->f[i] = dRCdiv(dCadd(pt[i], pt[i - 1]), 2.);
        dC_funct_copy(tmp, b);
        if (a->type > 0)
            XFREE(tmp->x);
        XFREE(tmp->f);
        XFREE(pt);
        free(tmp);
    }
    else {
        if (a->type == 0) {
            for (i = 1; i < a->nb; i++)
                b->f[i] = dRCdiv(dCsub(a->f[i + 1], a->f[i]),
                    a->x[i + 1] - a->x[i]);
            b->f[a->nb] = b->f[a->nb - 1];
            for (i = a->nb - 1; i > 1; i--)
                b->f[i] = dRCdiv(dCadd(b->f[i], b->f[i - 1]), 2.);
        }
        else {
            for (i = 1; i < a->nb; i++)
                b->f[i] = dRCdiv(dCsub(a->f[i + 1], a->f[i]), a->x[2]);
            b->f[a->nb] = b->f[a->nb - 1];
            for (i = a->nb - 1; i > 1; i--)
                b->f[i] = dRCdiv(dCadd(b->f[i], b->f[i - 1]), 2.);
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the composition of the real simple precision functions a and
    b and puts the result in c : c = a o b .
--------------------------------------------------------------------*/
void
comp_funct_f(funct_f *a, funct_f *b, funct_f *c)
{
    int             i;
    float          *cx;

    f_fill(c, &cx);   /* fills the array cx
                         with the x of the function c */
    for (i = 1; i <= c->nb; i++)
        c->f[i] = f_funct_eval(a, f_funct_eval(b, cx[i]));
    if (c->type > 0)
        XFREE(cx);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the composition of the real double precision functions a and
    b and puts the result in c : c = a o b .
--------------------------------------------------------------------*/
void
comp_funct_d(funct_d *a, funct_d *b, funct_d *c)
{
    int             i;
    double         *cx;

    d_fill(c, &cx);   /* fills the array cx
                         with the x of the function c */
    for (i = 1; i <= c->nb; i++)
        c->f[i] = d_funct_eval(a, d_funct_eval(b, cx[i]));
    if (c->type > 0)
        XFREE(cx);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the composition of the real simple precision function b with
    the expression given by s (where 'x' is the variable) and puts the
    result in c. The expression s will be parsed by the expression
    evaluator. For example, to compute  a o sin , s will contain the
    expression sin(x).
--------------------------------------------------------------------*/
void
comp_functc_f(funct_f *b, char *s, funct_f *c, flow_data *flow_interp)
{
    int             i;
    float          *x_r;
    char            h[100];

    f_fill(c, &x_r);   /* fills the array x_r
                          with the x of the function c */

    for (i = 1; i <= c->nb; i++) {
        memset(h, 0, 100);
        sprintf(h, "x=%f", x_r[i]);
        convert_float(h, flow_interp);
        c->f[i] = f_funct_eval(b, convert_float(s, flow_interp));
    }

    if (c->type  > 0)
        XFREE(x_r);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the composition of the real double precision function b with
    the expression given by s (where 'x' is the variable) and puts the
    result in c. The expression s will be parsed by the expression
    evaluator. For example, to compute  a o sin , s will contain the
    expression sin(x).
--------------------------------------------------------------------*/
void
comp_functc_d(funct_d *b, char *s, funct_d *c, flow_data *flow_interp)
{
    int             i;
    double         *x_r;
    char            h[100];

    d_fill(c, &x_r);   /* fills the array x_r
                          with the x of the function c */

    for (i = 1; i <= c->nb; i++) {
        memset(h, 0, 100);
        sprintf(h, "x=%f", x_r[i]);
        convert_float(h, flow_interp);
        c->f[i] = d_funct_eval(b, convert_float(s, flow_interp));
    }

    if (c->type  > 0)
        XFREE(x_r);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the composition of the expression given by s (where 'x' is the
    variable) with the real simple precision function b and puts the
    result in c. The expression s will be parsed by the expression
    evaluator. For example, to compute  sin o a , s will contain the
    expression sin(x).
--------------------------------------------------------------------*/
void
comp_functb_f(char *s, funct_f *b, funct_f *c, flow_data *flow_interp)
{
    int             i;
    float          *x_r;
    char            h[100];

    f_fill(c, &x_r);   /* fills the array x_r
                          with the x of the function c */

    for (i = 1; i <= c->nb; i++) {
        memset(h, 0, 100);
        sprintf(h, "x=%f", f_funct_eval(b, x_r[i]));
        convert_float(h, flow_interp);
        c->f[i] = convert_float(s, flow_interp);
    }

    if (c->type  > 0)
        XFREE(x_r);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the composition of the expression given by s (where 'x' is the
    variable) with the real simple precision function b and puts the
    result in c. The expression s will be parsed by the expression
    evaluator. For example, to compute  sin o a , s will contain the
    expression sin(x).
--------------------------------------------------------------------*/
void
comp_functb_d(char *a, funct_d *b, funct_d *c, flow_data *flow_interp)
{
    int             i;
    double         *x_r;
    char            h[100];

    d_fill(c, &x_r);   /* fills the array x_r
                          with the x of the function c */

    for (i = 1; i <= c->nb; i++) {
        memset(h, 0, 100);
        sprintf(h, "x=%f", d_funct_eval(b, x_r[i]));
        convert_float(h, flow_interp);
        c->f[i] = convert_float(a, flow_interp);
    }

    if (c->type  > 0)
        XFREE(x_r);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the composition of the expression given by s (where 'x' is the
    variable) with the expression given by s2 and puts the
    result in c. The expressions s, s2 will be parsed by the expression
    evaluator. For example, to compute  sin o (x*x+1) , s will contain the
    expression sin(x), and s2 will contain x*x+1.
--------------------------------------------------------------------*/
void
comp_functd_f(char *s, char *s2, funct_f *c, flow_data *flow_interp)
{
    int             i;
    float          *x_r;
    char            h[100];

    f_fill(c, &x_r);   /* fills the array x_r
                          with the x of the function c */

    for (i = 1; i <= c->nb; i++) {
        memset(h, 0, 100);
        sprintf(h, "x=%f", x_r[i]);
        convert_float(h, flow_interp);
        memset(h, 0, 100);
        sprintf(h, "x=%f", convert_float(s2, flow_interp));
        convert_float(h, flow_interp);
        c->f[i] = convert_float(s, flow_interp);
    }

    if (c->type  > 0)
        XFREE(x_r);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the composition of the expression given by s (where 'x' is the
    variable) with the expression given by s2 and puts the
    result in c. The expressions s, s2 will be parsed by the expression
    evaluator. For example, to compute  sin o (x*x+1) , s will contain the
    expression sin(x), and s2 will contain x*x+1.
--------------------------------------------------------------------*/
void
comp_functd_d(char *s, char *s2, funct_d *c, flow_data *flow_interp)
{
    int             i;
    double         *x_r;
    char            h[100];

    d_fill(c, &x_r);   /* fills the array x_r
                          with the x of the function c */

    for (i = 1; i <= c->nb; i++) {
        memset(h, 0, 100);
        sprintf(h, "x=%f", x_r[i]);
        convert_float(h, flow_interp);
        memset(h, 0, 100);
        sprintf(h, "x=%f", convert_float(s2, flow_interp));
        convert_float(h, flow_interp);
        c->f[i] = convert_float(s, flow_interp);
    }

    if (c->type  > 0)
        XFREE(x_r);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the 'generalized inverse' of function a and puts the result
    in function b.
--------------------------------------------------------------------*/
void
inv_sup_funct_f(funct_f *a, funct_f *b)
{
    int             i,
                    ix,
                    j,
                    nb;
    float          *xr,
                   *xra,
                    y,
                    z;

    f_fill(b, &xr);
    f_fill(a, &xra);
    nb = a->nb;

    for (i = 1; i <= b->nb; i++) {
        y = xr[i];
        ix = -1;
        z = 0.;
        if (a->f[1] >= y) {
            ix = 1;
        }

        for (j = 2; j <= nb; j++) {
            if (ix == 1) {
                if (a->f[j] < y) {
                    z += ((y - a->f[j]) * xra[j - 1] + (a->f[j - 1] - y) *
                        xra[j]) / (a->f[j - 1] - a->f[j]) - xra[j - 1];
                    ix = -1;
                }
                else
                    z += xra[j] - xra[j - 1];
            }
            else {
                if (a->f[j] >= y) {
                    z += xra[j] - ((y - a->f[j]) * xra[j - 1] + (a->f[j - 1]
                        - y) * xra[j]) / (a->f[j - 1] - a->f[j]);
                    ix = 1;
                }
            }
        }

        b->f[i] = z;
    }

    if (b->type > 0)
        XFREE(xr);
    if (a->type > 0)
        XFREE(xra);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the 'generalized inverse' of function a and puts the result
    in function b.
--------------------------------------------------------------------*/
void
inv_sup_funct_d(funct_d *a, funct_d *b)
{
    int             i,
                    ix,
                    j,
                    nb;
    double         *xr,
                   *xra,
                    y,
                    z;

    d_fill(b, &xr);
    d_fill(a, &xra);
    nb = a->nb;

    for (i = 1; i <= b->nb; i++) {
        y = xr[i];
        ix = -1;
        z = 0.;
        if (a->f[1] >= y) {
            ix = 1;
        }

        for (j = 2; j <= nb; j++) {
            if (ix == 1) {
                if (a->f[j] < y) {
                    z += ((y - a->f[j]) * xra[j - 1] + (a->f[j - 1] - y) *
                        xra[j]) / (a->f[j - 1] - a->f[j]) - xra[j - 1];
                    ix = -1;
                }
                else
                    z += xra[j] - xra[j - 1];
            }
            else {
                if (a->f[j] >= y) {
                    z += xra[j] - ((y - a->f[j]) * xra[j - 1] + (a->f[j - 1]
                        - y) * xra[j]) / (a->f[j - 1] - a->f[j]);
                    ix = 1;
                }
            }
        }

        b->f[i] = z;
    }

    if (b->type > 0)
        XFREE(xr);
    if (a->type > 0)
        XFREE(xra);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
        Tests if the x-range of the real simple precision function a
        is correct
--------------------------------------------------------------------*/
int
test_correct_x_f(funct_f *a)
{
    int             i;

    if (a->type == 0) {
        for (i = 2; i <= a->nb; i++)
            if (a->x[i] - a->x[i - 1] <= 0.)
                return 0;
    }
    else {
        if (a->x[2] <= 0.)
            return 0;
    }
    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
        Tests if the x-range of the real double precision function a
        is correct
--------------------------------------------------------------------*/
int
test_correct_x_d(funct_d *a)
{
    int             i;

    if (a->type == 0) {
        for (i = 2; i <= a->nb; i++)
            if (a->x[i] - a->x[i - 1] <= 0.)
                return 0;
    }
    else {
        if (a->x[2] <= 0.)
            return 0;
    }
    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
        Tests if the x-range of the complex simple precision function a
        is correct
--------------------------------------------------------------------*/
int
test_correct_x_C(funct_C *a)
{
    int             i;

    if (a->type == 0) {
        for (i = 2; i <= a->nb; i++)
            if (a->x[i] - a->x[i - 1] <= 0.)
                return 0;
    }
    else {
        if (a->x[2] <= 0.)
            return 0;
    }
    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
        Tests if the x-range of the complex double precision function a
        is correct
--------------------------------------------------------------------*/
int
test_correct_x_dC(funct_dC *a)
{
    int             i;

    if (a->type == 0) {
        for (i = 2; i <= a->nb; i++)
            if (a->x[i] - a->x[i - 1] <= 0.)
                return 0;
    }
    else {
        if (a->x[2] <= 0.)
            return 0;
    }
    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the maximum of a real simple precision function.
--------------------------------------------------------------------*/
void
Max_funct_f(funct_f *a, float *xmax, float *fmax)
{
    int             i;
    float          *xr;

    f_fill(a, &xr);
    xmax[0] = xr[1];
    fmax[0] = a->f[1];

    for (i = 2; i <= a->nb; i++) {
        if (a->f[i] > fmax[0]) {
            xmax[0] = xr[i];
            fmax[0] = a->f[i];
        }
    }

    if (a->type > 0)
        XFREE(xr);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the maximum of a real double precision function.
--------------------------------------------------------------------*/
void
Max_funct_d(funct_d *a, double *xmax, double *fmax)
{
    int             i;
    double         *xr;

    d_fill(a, &xr);
    xmax[0] = xr[1];
    fmax[0] = a->f[1];

    for (i = 2; i <= a->nb; i++) {
        if (a->f[i] > fmax[0]) {
            xmax[0] = xr[i];
            fmax[0] = a->f[i];
        }
    }

    if (a->type > 0)
        XFREE(xr);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the minimum of a real simple precision function.
--------------------------------------------------------------------*/
void
Min_funct_f(funct_f *a, float *xmax, float *fmax)
{
    int             i;
    float          *xr;

    f_fill(a, &xr);
    xmax[0] = xr[1];
    fmax[0] = a->f[1];

    for (i = 2; i <= a->nb; i++) {
        if (a->f[i] < fmax[0]) {
            xmax[0] = xr[i];
            fmax[0] = a->f[i];
        }
    }

    if (a->type > 0)
        XFREE(xr);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the minimum of a real double precision function.
--------------------------------------------------------------------*/
void
Min_funct_d(funct_d *a, double *xmax, double *fmax)
{
    int             i;
    double         *xr;

    d_fill(a, &xr);
    xmax[0] = xr[1];
    fmax[0] = a->f[1];

    for (i = 2; i <= a->nb; i++) {
        if (a->f[i] < fmax[0]) {
            xmax[0] = xr[i];
            fmax[0] = a->f[i];
        }
    }

    if (a->type > 0)
        XFREE(xr);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the minimal modulus of a complex simple precision function.
--------------------------------------------------------------------*/
void
Min_funct_C(funct_C *a, float *xmax, fcomplex *fmax, float *mod)
{
    int             i;
    float           y,
                   *xr;

    C_fill(a, &xr);
    xmax[0] = xr[1];
    fmax[0] = a->f[1];
    mod[0] = Cabs(fmax[0]);

    for (i = 2; i <= a->nb; i++) {
        y = Cabs(a->f[i]);
        if (y < mod[0]) {
            xmax[0] = xr[i];
            fmax[0] = a->f[i];
            mod[0] = y;
        }
    }

    if (a->type > 0)
        XFREE(xr);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the minimal modulus of a complex double precision function.
--------------------------------------------------------------------*/
void
Min_funct_dC(funct_dC *a, double *xmax, dcomplex *fmax, double *mod)
{
    int             i;
    double          y,
                   *xr;

    dC_fill(a, &xr);
    xmax[0] = xr[1];
    fmax[0] = a->f[1];
    mod[0] = dCabs(fmax[0]);

    for (i = 2; i <= a->nb; i++) {
        y = dCabs(a->f[i]);
        if (y < mod[0]) {
            xmax[0] = xr[i];
            fmax[0] = a->f[i];
            mod[0] = y;
        }
    }

    if (a->type > 0)
        XFREE(xr);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the maximal modulus of a complex simple precision function.
--------------------------------------------------------------------*/
void
Max_funct_C(funct_C *a, float *xmax, fcomplex *fmax, float *mod)
{
    int             i;
    float           y,
                   *xr;

    C_fill(a, &xr);
    xmax[0] = xr[1];
    fmax[0] = a->f[1];
    mod[0] = Cabs(fmax[0]);

    for (i = 2; i <= a->nb; i++) {
        y = Cabs(a->f[i]);
        if (y > mod[0]) {
            xmax[0] = xr[i];
            fmax[0] = a->f[i];
            mod[0] = y;
        }
    }

    if (a->type > 0)
        XFREE(xr);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the maximal modulus of a complex double precision function.
--------------------------------------------------------------------*/
void
Max_funct_dC(funct_dC *a, double *xmax, dcomplex *fmax, double *mod)
{
    int             i;
    double          y,
                   *xr;

    dC_fill(a, &xr);
    xmax[0] = xr[1];
    fmax[0] = a->f[1];
    mod[0] = dCabs(fmax[0]);

    for (i = 2; i <= a->nb; i++) {
        y = dCabs(a->f[i]);
        if (y > mod[0]) {
            xmax[0] = xr[i];
            fmax[0] = a->f[i];
            mod[0] = y;
        }
    }

    if (a->type > 0)
        XFREE(xr);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Loads the data contained in the file s in the real SP function a
    (used by the command 'load_funct')
--------------------------------------------------------------------*/
int
load_funct_f(funct_f *a, FILE *s)
{
    float          *xr,
                   *xx,
                   *xF;
    int             i,
                    j,
                    nb,
                    n_size;

    f_fill(a, &xr);
    xx = (float *) malloc((size_t) 100 * sizeof(float));
    xF = (float *) malloc((size_t) 100 * sizeof(float));
    nb = 0;
    n_size = 100;

    while (fscanf(s, "%f  %f", &xx[nb], &xF[nb]) != EOF) {
        if (nb > 0) {
            if (xx[nb] <= xx[nb - 1]) {
                free(xx);
                free(xF);
                if (a->type > 0)
                        XFREE(xr);
                return 1;
            }
        }
        nb++;
        if (nb >= n_size) {
            n_size += 100;
            xx = (float *) realloc(xx, n_size * sizeof(float));
            xF = (float *) realloc(xF, n_size * sizeof(float));
        }
    }

    j = 0;
    nb--;

    for (i = 1; i <= a->nb; i++) {
        if (xr[i] <= xx[0]) {
            a->f[i] = xF[0];
        }
        else {
            if (xr[i] >= xx[nb]) {
                a->f[i] = xF[nb];
            }
            else {
                while (xx[j] < xr[i]) {
                    if (j >= nb - 1)
                        break;
                    j++;
                }
                a->f[i] = ((xr[i] - xx[j - 1]) * xF[j] + (xx[j] - xr[i]) *
                    xF[j - 1]) / (xx[j] - xx[j - 1]);
            }
        }
    }

    free(xx);
    free(xF);
    if (a->type > 0)
        XFREE(xr);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Loads the data contained in the file s in the real DP function a
    (used by the command 'load_funct')
--------------------------------------------------------------------*/
int
load_funct_d(funct_d *a, FILE *s)
{
    double         *xr,
                   *xx,
                   *xF;
    int             i,
                    j,
                    nb,
                    n_size;

    d_fill(a, &xr);
    xx = (double *) malloc((size_t) 100 * sizeof(double));
    xF = (double *) malloc((size_t) 100 * sizeof(double));
    nb = 0;
    n_size = 100;

    while (fscanf(s, "%lf  %lf", &xx[nb], &xF[nb]) != EOF) {
        if (nb > 0) {
            if (xx[nb] <= xx[nb - 1]) {
                free(xx);
                free(xF);
                if (a->type > 0)
                        XFREE(xr);
                return 1;
            }
        }
        nb++;
        if (nb >= n_size) {
            n_size += 100;
            xx = (double *) realloc(xx, n_size * sizeof(double));
            xF = (double *) realloc(xF, n_size * sizeof(double));
        }
    }

    j = 0;
    nb--;

    for (i = 1; i <= a->nb; i++) {
        if (xr[i] <= xx[0]) {
            a->f[i] = xF[0];
        }
        else {
            if (xr[i] >= xx[nb]) {
                a->f[i] = xF[nb];
            }
            else {
                while (xx[j] < xr[i]) {
                    if (j >= nb - 1)
                        break;
                    j++;
                }
                a->f[i] = ((xr[i] - xx[j - 1]) * xF[j] + (xx[j] - xr[i]) *
                    xF[j - 1]) / (xx[j] - xx[j - 1]);
            }
        }
    }

    free(xx);
    free(xF);
    if (a->type > 0)
        XFREE(xr);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Loads the data contained in the file s in the complex SP function a
    (used by the command 'load_funct')
--------------------------------------------------------------------*/
int
load_funct_C(funct_C *a, FILE *s)
{
    float          *xr,
                   *xx;
    fcomplex       *xF;
    int             i,
                    j,
                    nb,
                    n_size;

    C_fill(a, &xr);
    xx = (float *) malloc((size_t) 100 * sizeof(float));
    xF = (fcomplex *) malloc((size_t) 100 * sizeof(fcomplex));
    nb = 0;
    n_size = 100;

    while (fscanf(s, "%f  %f  %f", &xx[nb], &xF[nb].r, &xF[nb].i) != EOF) {
        if (nb > 0) {
            if (xx[nb] <= xx[nb - 1]) {
                free(xx);
                free(xF);
                if (a->type > 0)
                        XFREE(xr);
                return 1;
            }
        }
        nb++;
        if (nb >= n_size) {
            n_size += 100;
            xx = (float *) realloc(xx, n_size * sizeof(float));
            xF = (fcomplex *) realloc(xF, n_size * sizeof(fcomplex));
        }
    }

    j = 0;
    nb--;

    for (i = 1; i <= a->nb; i++) {
        if (xr[i] <= xx[0]) {
            a->f[i] = xF[0];
        }
        else {
            if (xr[i] >= xx[nb]) {
                a->f[i] = xF[nb];
            }
            else {
                while (xx[j] < xr[i]) {
                    if (j >= nb - 1)
                        break;
                    j++;
                }
                a->f[i].r = ((xr[i] - xx[j - 1]) * xF[j].r + (xx[j] - xr[i]) *
                    xF[j - 1].r) / (xx[j] - xx[j - 1]);
                a->f[i].i = ((xr[i] - xx[j - 1]) * xF[j].i + (xx[j] - xr[i]) *
                    xF[j - 1].i) / (xx[j] - xx[j - 1]);
            }
        }
    }

    free(xx);
    free(xF);
    if (a->type > 0)
        XFREE(xr);
   return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Loads the data contained in the file s in the complex DP function a
    (used by the command 'load_funct')
--------------------------------------------------------------------*/
int
load_funct_dC(funct_dC *a, FILE *s)
{
    double         *xr,
                   *xx;
    dcomplex       *xF;
    int             i,
                    j,
                    nb,
                    n_size;

    dC_fill(a, &xr);
    xx = (double *) malloc((size_t) 100 * sizeof(double));
    xF = (dcomplex *) malloc((size_t) 100 * sizeof(dcomplex));
    nb = 0;
    n_size = 100;

    while (fscanf(s, "%lf  %lf  %lf", &xx[nb], &xF[nb].r, &xF[nb].i) != EOF) {
        if (nb > 0) {
            if (xx[nb] <= xx[nb - 1]) {
                free(xx);
                free(xF);
                if (a->type > 0)
                        XFREE(xr);
                return 1;
            }
        }
        nb++;
        if (nb >= n_size) {
            n_size += 100;
            xx = (double *) realloc(xx, n_size * sizeof(double));
            xF = (dcomplex *) realloc(xF, n_size * sizeof(dcomplex));
        }
    }

    j = 0;
    nb--;

    for (i = 1; i <= a->nb; i++) {
        if (xr[i] <= xx[0]) {
            a->f[i] = xF[0];
        }
        else {
            if (xr[i] >= xx[nb]) {
                a->f[i] = xF[nb];
            }
            else {
                while (xx[j] < xr[i]) {
                    if (j >= nb - 1)
                        break;
                    j++;
                }
                a->f[i].r = ((xr[i] - xx[j - 1]) * xF[j].r + (xx[j] - xr[i]) *
                    xF[j - 1].r) / (xx[j] - xx[j - 1]);
                a->f[i].i = ((xr[i] - xx[j - 1]) * xF[j].i + (xx[j] - xr[i]) *
                    xF[j - 1].i) / (xx[j] - xx[j - 1]);
            }
        }
    }

    free(xx);
    free(xF);
    if (a->type > 0)
        XFREE(xr);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Reads the data contained in the file s to fill the arrays xr, xF
    (used by the command 'load_create_f ' to produce a x-range and
    a real SP function)
--------------------------------------------------------------------*/
void
load_data_f(FILE *s, float **xr, float **xF, int *nb)
{
    int             n_size;

    xr[0] = (float *) malloc((size_t) 100 * sizeof(float));
    xF[0] = (float *) malloc((size_t) 100 * sizeof(float));
    *nb = 0;
    n_size = 100;

    while (fscanf(s, "%f  %f", &xr[0][*nb], &xF[0][*nb]) != EOF) {
        if (*nb > 0) {
            if (xr[0][*nb] <= xr[0][*nb - 1]) {
                free(xr[0]);
                free(xF[0]);
               *nb = 0;
                return;
            }
        }
        (*nb)++;
        if (*nb >= n_size) {
            n_size += 100;
            xr[0] = (float *) realloc(xr, n_size * sizeof(float));
            xF[0] = (float *) realloc(xF, n_size * sizeof(float));
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Reads the data contained in the file s to fill the arrays xr, xF
    (used by the command 'load_create_d ' to produce a x-range and
    a real DP function)
--------------------------------------------------------------------*/
void
load_data_d(FILE *s, double **xr, double **xF, int *nb)
{
    int             n_size;

    xr[0] = (double *) malloc((size_t) 100 * sizeof(double));
    xF[0] = (double *) malloc((size_t) 100 * sizeof(double));
    *nb = 0;
    n_size = 100;

    while (fscanf(s, "%lf  %lf", &xr[0][*nb], &xF[0][*nb]) != EOF) {
        if (*nb > 0) {
            if (xr[0][*nb] <= xr[0][*nb - 1]) {
                free(xr[0]);
                free(xF[0]);
               *nb = 0;
                return;
            }
        }
        (*nb)++;
        if (*nb >= n_size) {
            n_size += 100;
            xr[0] = (double *) realloc(xr, n_size * sizeof(double));
            xF[0] = (double *) realloc(xF, n_size * sizeof(double));
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Reads the data contained in the file s to fill the arrays xr, xF
    (used by the command 'load_create_C ' to produce a x-range and
    a complex SP function)
--------------------------------------------------------------------*/
void
load_data_C(FILE *s, float **xr, fcomplex **xF, int *nb)
{
    int             n_size;

    xr[0] = (float *) malloc((size_t) 100 * sizeof(float));
    xF[0] = (fcomplex *) malloc((size_t) 100 * sizeof(fcomplex));
    *nb = 0;
    n_size = 100;

    while (fscanf(s, "%f  %f  %f", &xr[0][*nb], &xF[0][*nb].r, &xF[0][*nb].i)
        != EOF) {
        if (*nb > 0) {
            if (xr[0][*nb] <= xr[0][*nb - 1]) {
                free(xr[0]);
                free(xF[0]);
               *nb = 0;
                return;
            }
        }
        (*nb)++;
        if (*nb >= n_size) {
            n_size += 100;
            xr[0] = (float *) realloc(xr, n_size * sizeof(float));
            xF[0] = (fcomplex *) realloc(xF, n_size * sizeof(fcomplex));
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Reads the data contained in the file s to fill the arrays xr, xF
    (used by the command 'load_create_C ' to produce a x-range and
    a complex DP function)
--------------------------------------------------------------------*/
void
load_data_dC(FILE *s, double **xr, dcomplex **xF, int *nb)
{
    int             n_size;

    xr[0] = (double *) malloc((size_t) 100 * sizeof(double));
    xF[0] = (dcomplex *) malloc((size_t) 100 * sizeof(dcomplex));
    *nb = 0;
    n_size = 100;

    while (fscanf(s, "%lf  %lf  %lf", &xr[0][*nb], &xF[0][*nb].r, &xF[0][*nb].i)
        != EOF) {
        if (*nb > 0) {
            if (xr[0][*nb] <= xr[0][*nb - 1]) {
                free(xr[0]);
                free(xF[0]);
               *nb = 0;
                return;
            }
        }
        (*nb)++;
        if (*nb >= n_size) {
            n_size += 100;
            xr[0] = (double *) realloc(xr, n_size * sizeof(double));
            xF[0] = (dcomplex *) realloc(xF, n_size * sizeof(dcomplex));
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Performs the FFT of the real SP function a, on ifft points, and
    puts the result in function b (ifft = power of 2)
--------------------------------------------------------------------*/
void
XFFT_func_f(funct_f *a, funct_C *b, int ifft)
{
    float          *xr,
                    dx,
                    D,
                    x,
                    x0,
                    y,
                    t,
                    w,
                    df;
    double         *A;
    fcomplex       *B,
                    B0,
                    B1,
                    z0;
    int             i,
                    m;

    f_fill(a, &xr);
    x0 = xr[1];
    A = double_alloc1(ifft + 2);
    B = fcomplex_alloc1(ifft + 2);
    dx = (xr[a->nb] - xr[1]) / (double) ifft;
    for (i = 0; i < ifft; i++)
        A[i] = f_funct_eval(a, xr[1] + i * dx);
    if (a->type > 0)
        XFREE(xr);
    gsl_fft_real_radix2_transform(A, 1, ifft);
    m = ifft + 1;

    for (i = 0; i <= ifft / 2; i++) {
        B[i].r = A[i];
        B[i].i = - A[--m];
        B[ifft - i] = Conjg(B[i]);
    }

    C_fill(b, &xr);
    D = 2. * PI / dx;
    df = D / ifft;
    dx *= ifft;
    t = df * ifft / dx;

    for (i = 1; i <= b->nb; i++) {
        m = (int) floor(xr[i] / D);
        x = xr[i] - m * D;
        m = (int) floor(x / df);
        y = m * df;
        w = xr[i] - x + y;
        z0 = Complex(cos(x0 * w), sin(x0 * w));
        B0 = Cmul(B[m], z0);
        w += df;
        z0 = Complex(cos(x0 * w), sin(x0 * w));
        B1 = Cmul(B[m + 1], z0);
        b->f[i] = RCdiv(Cadd(RCmul(x - y, B1), RCmul(y + df - x, B0)), t);
    }

    if (b->type > 0)
        XFREE(xr);
    XFREE(A);
    XFREE(B);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Performs the FFT of the real DP function a, on ifft points, and
    puts the result in function b (ifft = power of 2)
--------------------------------------------------------------------*/
void
XFFT_func_d(funct_d *a, funct_dC *b, int ifft)
{
    double         *xr,
                    dx,
                    D,
                    x,
                    x0,
                    y,
                    t,
                    w,
                    df;
    double         *A;
    dcomplex       *B,
                    B0,
                    B1,
                    z0;
    int             i,
                    m;

    d_fill(a, &xr);
    x0 = xr[1];
    A = double_alloc1(ifft + 2);
    B = dcomplex_alloc1(ifft + 2);
    dx = (xr[a->nb] - xr[1]) / (double) ifft;
    for (i = 0; i < ifft; i++)
        A[i] = d_funct_eval(a, xr[1] + i * dx);
    if (a->type > 0)
        XFREE(xr);
    gsl_fft_real_radix2_transform(A, 1, ifft);
    m = ifft + 1;

    for (i = 0; i <= ifft / 2; i++) {
        B[i].r = A[i];
        B[i].i = - A[--m];
        B[ifft - i] = dConjg(B[i]);
    }

    dC_fill(b, &xr);
    D = 2. * PI / dx;
    df = D / ifft;
    dx *= ifft;
    t = df * ifft / dx;

    for (i = 1; i <= b->nb; i++) {
        m = (int) floor(xr[i] / D);
        x = xr[i] - m * D;
        m = (int) floor(x / df);
        y = m * df;
        w = xr[i] - x + y;
        z0 = dComplex(cos(x0 * w), sin(x0 * w));
        B0 = dCmul(B[m], z0);
        w += df;
        z0 = dComplex(cos(x0 * w), sin(x0 * w));
        B1 = dCmul(B[m + 1], z0);
        b->f[i] = dRCdiv(dCadd(dRCmul(x - y, B1), dRCmul(y + df - x, B0)), t);
    }

    if (b->type > 0)
        XFREE(xr);
    XFREE(A);
    XFREE(B);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Performs the FFT of the complex SP function a, on ifft points, and
    puts the result in function b (ifft = power of 2)
--------------------------------------------------------------------*/
void
XFFT_func_C(funct_C *a, funct_C *b, int ifft)
{
    float          *xr,
                    dx,
                    D,
                    x,
                    x0,
                    y,
                    t,
                    w,
                    df;
    double         *F;
    fcomplex       *A,
                   *B,
                    B0,
                    B1,
                    z,
                    z0,
                    zz;
    int             i,
                    m;

    C_fill(a, &xr);
    x0 = xr[1];
    A = fcomplex_alloc1(ifft + 2);
    F = double_alloc1(ifft + 2);
    B = fcomplex_alloc1(ifft + 2);
    dx = (xr[a->nb] - xr[1]) / (double) ifft;
    for (i = 0; i < ifft; i++)
        A[i] = C_funct_eval(a, xr[1] + i * dx);
    if (a->type > 0)
        XFREE(xr);
    for (i = 0; i < ifft; i++)
        F[i] = A[i].r;
    gsl_fft_real_radix2_transform(F, 1, ifft);
    m = ifft + 1;

    for (i = 0; i <= ifft / 2; i++) {
        B[i].r = F[i];
        B[i].i = - F[--m];
        B[ifft - i] = Conjg(B[i]);
    }

    for (i = 0; i < ifft; i++)
        F[i] = A[i].i;
    gsl_fft_real_radix2_transform(F, 1, ifft);
    m = ifft + 1;

    for (i = 0; i <= ifft / 2; i++) {
        z.r = F[i];
        z.i = - F[--m];
        zz = Conjg(z);
        B[i].r -= z.i;
        B[i].i += z.r;
        B[ifft - i].r -= zz.i;
        B[ifft - i].i += zz.r;
    }

    C_fill(b, &xr);
    D = 2. * PI / dx;
    df = D / ifft;
    dx *= ifft;
    t = df * ifft / dx;

    for (i = 1; i <= b->nb; i++) {
        m = (int) floor(xr[i] / D);
        x = xr[i] - m * D;
        m = (int) floor(x / df);
        y = m * df;
        w = xr[i] - x + y;
        z0 = Complex(cos(x0 * w), sin(x0 * w));
        B0 = Cmul(B[m], z0);
        w += df;
        z0 = Complex(cos(x0 * w), sin(x0 * w));
        B1 = Cmul(B[m + 1], z0);
        b->f[i] = RCdiv(Cadd(RCmul(x - y, B1), RCmul(y + df - x, B0)), t);
    }

    if (b->type > 0)
        XFREE(xr);
    XFREE(A);
    XFREE(B);
    XFREE(F);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Performs the FFT of the complex DP function a, on ifft points, and
    puts the result in function b (ifft = power of 2)
--------------------------------------------------------------------*/
void
XFFT_func_dC(funct_dC *a, funct_dC *b, int ifft)
{
    double         *xr,
                    dx,
                    D,
                    x,
                    x0,
                    y,
                    t,
                    w,
                    df,
                   *F;
    dcomplex       *A,
                   *B,
                    B0,
                    B1,
                    z,
                    z0,
                    zz;
    int             i,
                    m;

    dC_fill(a, &xr);
    x0 = xr[1];
    A = dcomplex_alloc1(ifft + 2);
    F = double_alloc1(ifft + 2);
    B = dcomplex_alloc1(ifft + 2);
    dx = (xr[a->nb] - xr[1]) / (double) ifft;
    for (i = 0; i < ifft; i++)
        A[i] = dC_funct_eval(a, xr[1] + i * dx);
    if (a->type > 0)
        XFREE(xr);
    for (i = 0; i < ifft; i++)
        F[i] = A[i].r;
    gsl_fft_real_radix2_transform(F, 1, ifft);
    m = ifft + 1;

    for (i = 0; i <= ifft / 2; i++) {
        B[i].r = F[i];
        B[i].i = - F[--m];
        B[ifft - i] = dConjg(B[i]);
    }

    for (i = 0; i < ifft; i++)
        F[i] = A[i].i;
    gsl_fft_real_radix2_transform(F, 1, ifft);
    m = ifft + 1;

    for (i = 0; i <= ifft / 2; i++) {
        z.r = F[i];
        z.i = - F[--m];
        zz = dConjg(z);
        B[i].r -= z.i;
        B[i].i += z.r;
        B[ifft - i].r -= zz.i;
        B[ifft - i].i += zz.r;
    }

    dC_fill(b, &xr);
    D = 2. * PI / dx;
    df = D / ifft;
    dx *= ifft;
    t = df * ifft / dx;

    for (i = 1; i <= b->nb; i++) {
        m = (int) floor(xr[i] / D);
        x = xr[i] - m * D;
        m = (int) floor(x / df);
        y = m * df;
        w = xr[i] - x + y;
        z0 = dComplex(cos(x0 * w), sin(x0 * w));
        B0 = dCmul(B[m], z0);
        w += df;
        z0 = dComplex(cos(x0 * w), sin(x0 * w));
        B1 = dCmul(B[m + 1], z0);
        b->f[i] = dRCdiv(dCadd(dRCmul(x - y, B1), dRCmul(y + df - x, B0)), t);
    }

    if (b->type > 0)
        XFREE(xr);
    XFREE(A);
    XFREE(B);
    XFREE(F);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Routine of convolution :
        Xshort : first set of data (ifft points, ifft = power of 2)
        Xlong : second set of data (n_long points, n_long >= ifft)
        res : where the result if put
    the routine uses FFTs on 2*ifft points
--------------------------------------------------------------------*/
void
_Xconvol(double *X_long, double *res, double *X_short, int ifft, int n_long)
{
    int             j,
                    j0,
                    k,
                    m,
                    ifft2,
                    inb,
                    iw2,
                    iw;
    double         *or1,
                   *oi1,
                   *f1,
                   *X_short_fft_r,
                   *X_short_fft_i,
                  **X_long_fft_r,
                   *X_longr2,
                   *X_longi2,
                  **X_long_fft_i;

    inb = n_long / ifft;
    if (n_long != inb * ifft)
        inb++;
    ifft2 = ifft;
    ifft *= 2;

    X_long_fft_r = double_alloc2(inb , ifft + 2);
    X_long_fft_i = double_alloc2(inb , ifft + 2);
    X_short_fft_r = double_alloc1(ifft + 2);
    X_short_fft_i = double_alloc1(ifft + 2);
    f1 = double_alloc1(2 * ifft + 4);
    iw = - ifft2;

    for (k = 0; k < inb; k++) {
        iw += ifft2;
        X_longr2 = X_long_fft_r[k];
        X_longi2 = X_long_fft_i[k];

        for (j = 0; j < ifft2; j++) {
            if (j + iw < n_long)
                f1[j] = X_long[j + iw] / ifft;
            else
                f1[j] = 0.;
        }
        for (j = ifft2; j <= ifft; j++)
            f1[j] = 0.;

        gsl_fft_real_radix2_transform(f1, 1, ifft);
        m = ifft + 1;

        for (j = 0; j < ifft2; j++) {
            X_longr2[j] = f1[j];
            X_longi2[j] = f1[--m];
        }
    }

    for (j = 0; j <= 2 * ifft; j++)
        f1[j] = 0.;

    for (j = 0; j < ifft2; j++)
        f1[j] = X_short[j];
    gsl_fft_real_radix2_transform(f1, 1, ifft);
    m = ifft + 1;

    for (j = 0; j < ifft2; j++) {
        X_short_fft_r[j] = f1[j];
        X_short_fft_i[j] = f1[--m];
    }

    iw2 = - ifft2;

    for (m = 0; m < inb; m++) {
        for (j = 0; j <= 2 * ifft; j++)
            f1[j] = 0.0;

        iw2 += ifft2;
        or1 = X_long_fft_r[m];
        oi1 = X_long_fft_i[m];
        iw = ifft + 1;

        for (j = 0; j < ifft2; j++) {
            f1[j] = X_short_fft_r[j] * or1[j] - X_short_fft_i[j] * oi1[j];
            f1[--iw] = X_short_fft_r[j] * oi1[j] + X_short_fft_i[j] * or1[j];
        }

        gsl_fft_halfcomplex_radix2_inverse(f1, 1, ifft);
        j0 = iw2;

        for (j = 0; j < ifft; j ++) {
            if (j0 < n_long + ifft2)
                res[j0++] += f1[j] * ifft;
        }
    }

    ifft /= 4;
    XFREE(f1);
    XFREE(X_long_fft_r);
    XFREE(X_short_fft_r);
    XFREE(X_long_fft_i);
    XFREE(X_short_fft_i);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Performs the convolution of the real SP function a with the real
    SP function b, and puts the result in function c, using FFts on
    2*ifft points (ifft = power of 2)
--------------------------------------------------------------------*/
void
Xconvol_f_f(funct_f *a, funct_f *b, funct_f *c, int ifft)
{
    float           dxa,
                    dxb,
                    dx,
                   *xrc;
    double         *X_long,
                   *res,
                   *Xres,
                   *X_short,
                    x;
    int             i,
                    j,
                    j0,
                    n,
                    n_long,
                    cas;

    if (a->type == 0)
        dxa = a->x[a->nb] - a->x[1];
    else
        dxa = a->x[3] - a->x[1];
    if (b->type == 0)
        dxb = b->x[b->nb] - b->x[1];
    else
        dxb = b->x[3] - b->x[1];
    cas = 0;
    if (dxa >= dxb) {
        dx = dxb / (ifft - 1.);
        if (dxa == dxb)
            n_long = ifft;
        else
            n_long = 1 + dxa / dx;
    }
    else {
        dx = dxa / (ifft - 1.);
        n_long = 1 + dxb / dx;
        cas = 1;
    }
    X_short = double_alloc1(ifft + 1);
    X_long = double_alloc1(n_long + 1);
    res = double_alloc1(n_long + ifft + 2);
    Xres = double_alloc1(n_long + ifft + 2);
    if (cas == 0) {
        for (i = 0; i < ifft; i++)
            X_short[i] = f_funct_eval(b, b->x[1] + i * dx);
        for (i = 0; i < n_long; i++)
            X_long[i] = f_funct_eval(a, a->x[1] + i * dx);
    }
    else {
        for (i = 0; i < ifft; i++)
            X_short[i] = f_funct_eval(a, a->x[1] + i * dx);
        for (i = 0; i < n_long; i++)
            X_long[i] = f_funct_eval(b, b->x[1] + i * dx);
    }
    _Xconvol(X_long, res, X_short, ifft, n_long);
    f_fill(c, &xrc);
    n = ifft + n_long - 1;
    for (i = 0; i < n; i++)
        Xres[i] = i * dx;

    for (i = 1; i <= c->nb; i++) {
        x = xrc[i] - a->x[1] - b->x[1];
        if (x <= 0 || x >= Xres[n])
            c->f[i] = 0.;
        j0 = n;

        for (j = 1; j < n; j++) {
            if (x < Xres[j]) {
                j0 = j;
                break;
            }
        }

        c->f[i] = (res[j0 - 1] * (Xres[j0] - x) + res[j0] * (x - Xres[j0 - 1]));

    }

    if (c->type > 0)
        XFREE(xrc);
    XFREE(X_long);
    XFREE(X_short);
    XFREE(res);
    XFREE(Xres);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Performs the convolution of the real DP function a with the real
    DP function b, and puts the result in function c, using FFts on
    2*ifft points (ifft = power of 2)
--------------------------------------------------------------------*/
void
Xconvol_d_d(funct_d *a, funct_d *b, funct_d *c, int ifft)
{
    double          dxa,
                    dxb,
                    dx,
                   *xrc,
                   *X_long,
                   *res,
                   *Xres,
                   *X_short,
                    x;
    int             i,
                    j,
                    j0,
                    n,
                    n_long,
                    cas;

    if (a->type == 0)
        dxa = a->x[a->nb] - a->x[1];
    else
        dxa = a->x[3] - a->x[1];
    if (b->type == 0)
        dxb = b->x[b->nb] - b->x[1];
    else
        dxb = b->x[3] - b->x[1];
    cas = 0;
    if (dxa >= dxb) {
        dx = dxb / (ifft - 1.);
        if (dxa == dxb)
            n_long = ifft;
        else
            n_long = 1 + dxa / dx;
    }
    else {
        dx = dxa / (ifft - 1.);
        n_long = 1 + dxb / dx;
        cas = 1;
    }
    X_short = double_alloc1(ifft + 1);
    X_long = double_alloc1(n_long + 1);
    res = double_alloc1(n_long + ifft + 2);
    Xres = double_alloc1(n_long + ifft + 2);
    if (cas == 0) {
        for (i = 0; i < ifft; i++)
            X_short[i] = d_funct_eval(b, b->x[1] + i * dx);
        for (i = 0; i < n_long; i++)
            X_long[i] = d_funct_eval(a, a->x[1] + i * dx);
    }
    else {
        for (i = 0; i < ifft; i++)
            X_short[i] = d_funct_eval(a, a->x[1] + i * dx);
        for (i = 0; i < n_long; i++)
            X_long[i] = d_funct_eval(b, b->x[1] + i * dx);
    }
    _Xconvol(X_long, res, X_short, ifft, n_long);
    d_fill(c, &xrc);
    n = ifft + n_long - 1;
    for (i = 0; i < n; i++)
        Xres[i] = i * dx;

    for (i = 1; i <= c->nb; i++) {
        x = xrc[i] - a->x[1] - b->x[1];
        if (x <= 0 || x >= Xres[n])
            c->f[i] = 0.;
        j0 = n;

        for (j = 1; j < n; j++) {
            if (x < Xres[j]) {
                j0 = j;
                break;
            }
        }

        c->f[i] = (res[j0 - 1] * (Xres[j0] - x) + res[j0] * (x - Xres[j0 - 1]));

    }

    if (c->type > 0)
        XFREE(xrc);
    XFREE(X_long);
    XFREE(X_short);
    XFREE(res);
    XFREE(Xres);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Performs the convolution of the real SP function a with the complex
    SP function b, and puts the result in function c, using FFts on
    2*ifft points (ifft = power of 2)
--------------------------------------------------------------------*/
void
Xconvol_f_C(funct_f *a, funct_C *b, funct_C *c, int ifft)
{
    Xconvol_C_f(b, a, c, ifft);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Performs the convolution of the complex SP function a with the real
    SP function b, and puts the result in function c, using FFts on
    2*ifft points (ifft = power of 2)
--------------------------------------------------------------------*/
void
Xconvol_C_f(funct_C *a, funct_f *b, funct_C *c, int ifft)
{
    int             i;
    funct_f         temp,
                    tmp;

    temp.type = a->type;
    temp.nb = a->nb;
    temp.x = a->x;
    temp.f = float_alloc1(temp.nb + 1);
    for (i = 0; i <= a->nb; i++)
        temp.f[i] = a->f[i].r;
    tmp.type = c->type;
    tmp.nb = c->nb;
    tmp.x = c->x;
    tmp.f = float_alloc1(tmp.nb + 1);
    Xconvol_f_f(&temp, b, &tmp, ifft);
    for (i = 0; i <= c->nb; i++)
        c->f[i].r = tmp.f[i];
    for (i = 0; i <= a->nb; i++)
        temp.f[i] = a->f[i].i;
    Xconvol_f_f(&temp, b, &tmp, ifft);
    for (i = 0; i <= c->nb; i++)
        c->f[i].i = tmp.f[i];
    XFREE(temp.f);
    XFREE(tmp.f);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Performs the convolution of the complex SP function a with the complex
    SP function b, and puts the result in function c, using FFts on
    2*ifft points (ifft = power of 2)
--------------------------------------------------------------------*/
void
Xconvol_C_C(funct_C *a, funct_C *b, funct_C *c, int ifft)
{
    int             i;
    funct_f         temp;
    funct_C         tmp;

    temp.type = b->type;
    temp.nb = b->nb;
    temp.x = b->x;
    temp.f = float_alloc1(temp.nb + 1);
    for (i = 0; i <= b->nb; i++)
        temp.f[i] = b->f[i].r;
    tmp.type = c->type;
    tmp.nb = c->nb;
    tmp.x = c->x;
    tmp.f = fcomplex_alloc1(tmp.nb + 1);
    Xconvol_C_f(a, &temp, &tmp, ifft);
    for (i = 0; i <= c->nb; i++)
        c->f[i] = tmp.f[i];
     for (i = 0; i <= b->nb; i++)
        temp.f[i] = b->f[i].i;
    Xconvol_C_f(a, &temp, &tmp, ifft);

    for (i = 0; i <= c->nb; i++) {
        c->f[i].r -= tmp.f[i].i;
        c->f[i].i += tmp.f[i].r;
    }

    XFREE(temp.f);
    XFREE(tmp.f);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Performs the convolution of the real DP function a with the complex
    DP function b, and puts the result in function c, using FFts on
    2*ifft points (ifft = power of 2)
--------------------------------------------------------------------*/
void
Xconvol_d_dC(funct_d *a, funct_dC *b, funct_dC *c, int ifft)
{
    Xconvol_dC_d(b, a, c, ifft);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Performs the convolution of the complex DP function a with the real
    DP function b, and puts the result in function c, using FFts on
    2*ifft points (ifft = power of 2)
--------------------------------------------------------------------*/
void
Xconvol_dC_d(funct_dC *a, funct_d *b, funct_dC *c, int ifft)
{
    int             i;
    funct_d         temp,
                    tmp;

    temp.type = a->type;
    temp.nb = a->nb;
    temp.x = a->x;
    temp.f = double_alloc1(temp.nb + 1);
    for (i = 0; i <= a->nb; i++)
        temp.f[i] = a->f[i].r;
    tmp.type = c->type;
    tmp.nb = c->nb;
    tmp.x = c->x;
    tmp.f = double_alloc1(tmp.nb + 1);
    Xconvol_d_d(&temp, b, &tmp, ifft);
    for (i = 0; i <= c->nb; i++)
        c->f[i].r = tmp.f[i];
    for (i = 0; i <= a->nb; i++)
        temp.f[i] = a->f[i].i;
    Xconvol_d_d(&temp, b, &tmp, ifft);
    for (i = 0; i <= c->nb; i++)
        c->f[i].i = tmp.f[i];
    XFREE(temp.f);
    XFREE(tmp.f);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Performs the convolution of the complex DP function a with the complex
    DP function b, and puts the result in function c, using FFts on
    2*ifft points (ifft = power of 2)
--------------------------------------------------------------------*/
void
Xconvol_dC_dC(funct_dC *a, funct_dC *b, funct_dC *c, int ifft)
{
    int             i;
    funct_d         temp;
    funct_dC        tmp;

    temp.type = b->type;
    temp.nb = b->nb;
    temp.x = b->x;
    temp.f = double_alloc1(temp.nb + 1);
    for (i = 0; i <= b->nb; i++)
        temp.f[i] = b->f[i].r;
    tmp.type = c->type;
    tmp.nb = c->nb;
    tmp.x = c->x;
    tmp.f = dcomplex_alloc1(tmp.nb + 1);
    Xconvol_dC_d(a, &temp, &tmp, ifft);
    for (i = 0; i <= c->nb; i++)
        c->f[i] = tmp.f[i];
     for (i = 0; i <= b->nb; i++)
        temp.f[i] = b->f[i].i;
    Xconvol_dC_d(a, &temp, &tmp, ifft);

    for (i = 0; i <= c->nb; i++) {
        c->f[i].r -= tmp.f[i].i;
        c->f[i].i += tmp.f[i].r;
    }

    XFREE(temp.f);
    XFREE(tmp.f);
}
/*------------------------------------------------------------------*/
