/*************************************************************************
** funct-3.1      (command interpreter - funct)                          **
** functcmd.c : Commands of manipulation of functions                    **
**                                                                       **
** Copyright (C) 2003  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#include "interp.h"
#include "funct.h"

funct_t F_COPY = {
    f_funct_copy,
    d_funct_copy,
    C_funct_copy,
    dC_funct_copy
};

funct_t F_INTEG = {
    integ_funct_f,
    integ_funct_d,
    integ_funct_C,
    integ_funct_dC
};

funct_t F_DIFF = {
    diff_funct_f,
    diff_funct_d,
    diff_funct_C,
    diff_funct_dC
};

funct2_t F_ADD = {
    f_funct_add,
    d_funct_add,
    C_funct_add,
    dC_funct_add
};

funct2_t F_SUB = {
    f_funct_sub,
    d_funct_sub,
    C_funct_sub,
    dC_funct_sub
};

funct2_t F_MUL = {
    f_funct_mul,
    d_funct_mul,
    C_funct_mul,
    dC_funct_mul
};

funct2_t F_DIV = {
    f_funct_div,
    d_funct_div,
    C_funct_div,
    dC_funct_div
};

extern  int  _NBFONC;




/*---------------------------------------------------------------------
        Command of definition of a real function in simple precision.
        Syntax :
            function_f f xr
        where 'f' is the name of the function and 'xr' the name of a
        "xrange" in simple precision.
---------------------------------------------------------------------*/
int
def_funct_f(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i0_b,
                    iw_b,
                   *d;
    funct_f        *a;
    float          *x_r;
    char          **e,
                   *k[4];
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj(argv[1], &i0, flow_interp);
    if (iw != 0) {
        error_mess(flow_interp, _FUNC_MESS);
        return 1;
    }
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("objdef");
    k[2] = ch_copy_int(_RFUNC_F - 1);
    k[3] = argv[1];
    i0 = obj_create(3, k + 1);
    if (i0 == -1) {
        error_mess(flow_interp, _FUNC_MESS);
        free(k[1]);
        free(k[2]);
        return 1;
    }
    free(k[1]);
    free(k[2]);
    iw_b = sketch_obj_restr(argv[2], &i0_b, _XRANGE_F, flow_interp);
    if (iw_b != _XRANGE_F) {
        error_mess(flow_interp, 1 + _FUNC_MESS);
        k[1] = ch_copy("destroy");
        k[2] = argv[1];
        detruit_obj(2, k + 1);
        free(k[1]);
        return 1;
    }
    a = (funct_f *) malloc((size_t) sizeof(funct_f));
    x_r = (float *) Obj[_XRANGE_F - 1][i0_b].adresse;
    d = Obj[_XRANGE_F - 1][i0_b].dim;
    a->type = x_r[0];
    if (a->type == 0)
        a->nb = d[0];
    else
        a->nb = x_r[7];
    a->x = x_r;
    a->f = float_alloc1(a->nb);
    a->nom = ch_copy(argv[2]);
    e = (char **) Obj[_RFUNC_F - 1][i0].adresse;
    e[0] = (char *) a;
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
        Command of definition of a real function in double precision.
        Syntax :
            function f xr
        where 'f' is the name of the function and 'xr' the name of a
        "xrange" in double precision.
---------------------------------------------------------------------*/
int
def_funct_d(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i0_b,
                    iw_b,
                   *d;
    funct_d        *a;
    double         *x_r;
    char          **e,
                   *k[4];
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj(argv[1], &i0, flow_interp);
    if (iw != 0) {
        error_mess(flow_interp, _FUNC_MESS);
        return 1;
    }
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("objdef");
    k[2] = ch_copy_int(_RFUNC_D - 1);
    k[3] = argv[1];
    i0 = obj_create(3, k + 1);
    if (i0 == -1) {
        error_mess(flow_interp, _FUNC_MESS);
        free(k[1]);
        free(k[2]);
        return 1;
    }
    free(k[1]);
    free(k[2]);
    iw_b = sketch_obj_restr(argv[2], &i0_b, _XRANGE_D, flow_interp);
    if (iw_b != _XRANGE_D) {
        error_mess(flow_interp, 1 + _FUNC_MESS);
        k[1] = ch_copy("destroy");
        k[2] = argv[1];
        detruit_obj(2, k + 1);
        free(k[1]);
        return 1;
    }
    a = (funct_d *) malloc((size_t) sizeof(funct_f));
    x_r = (double *) Obj[_XRANGE_D - 1][i0_b].adresse;
    d = Obj[_XRANGE_D - 1][i0_b].dim;
    a->type = x_r[0];
    if (a->type == 0)
        a->nb = d[0];
    else
        a->nb = x_r[7];
    a->x = x_r;
    a->f = double_alloc1(a->nb);
    a->nom = ch_copy(argv[2]);
    e = (char **) Obj[_RFUNC_D - 1][i0].adresse;
    e[0] = (char *) a;
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
        Command of definition of a complex function in simple precision.
        Syntax :
            function_fC f xr
        where 'f' is the name of the function and 'xr' the name of a
        "xrange" in simple precision.
---------------------------------------------------------------------*/
int
def_funct_C(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i0_b,
                    iw_b,
                   *d;
    funct_C        *a;
    float          *x_r;
    char          **e,
                   *k[4];
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj(argv[1], &i0, flow_interp);
    if (iw != 0) {
        error_mess(flow_interp, _FUNC_MESS);
        return 1;
    }
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("objdef");
    k[2] = ch_copy_int(_CFUNC_F - 1);
    k[3] = argv[1];
    i0 = obj_create(3, k + 1);
    if (i0 == -1) {
        error_mess(flow_interp, _FUNC_MESS);
        free(k[1]);
        free(k[2]);
        return 1;
    }
    free(k[1]);
    free(k[2]);
    iw_b = sketch_obj_restr(argv[2], &i0_b, _XRANGE_F, flow_interp);
    if (iw_b != _XRANGE_F) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        k[1] = ch_copy("destroy");
        k[2] = argv[1];
        detruit_obj(2, k + 1);
        free(k[1]);
        return 1;
    }
    a = (funct_C *) malloc((size_t) sizeof(funct_f));
    x_r = (float *) Obj[_XRANGE_F - 1][i0_b].adresse;
    d = Obj[_XRANGE_F - 1][i0_b].dim;
    a->type = x_r[0];
    if (a->type == 0)
        a->nb = d[0];
    else
        a->nb = x_r[7];
    a->x = x_r;
    a->f = fcomplex_alloc1(a->nb + 1);
    a->nom = ch_copy(argv[2]);
    e = (char **) Obj[_CFUNC_F - 1][i0].adresse;
    e[0] = (char *) a;
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
        Command of definition of a complex function in double precision.
        Syntax :
            function_C f xr
        where 'f' is the name of the function and 'xr' the name of a
        "xrange" in double precision.
---------------------------------------------------------------------*/
int
def_funct_dC(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i0_b,
                    iw_b,
                   *d;
    funct_dC       *a;
    double         *x_r;
    char          **e,
                   *k[4];
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj(argv[1], &i0, flow_interp);
    if (iw != 0) {
        error_mess(flow_interp, _FUNC_MESS);
        return 1;
    }
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("objdef");
    k[2] = ch_copy_int(_CFUNC_D - 1);
    k[3] = argv[1];
    i0 = obj_create(3, k + 1);
    if (i0 == -1) {
        error_mess(flow_interp, _FUNC_MESS);
        free(k[1]);
        free(k[2]);
        return 1;
    }
    free(k[1]);
    free(k[2]);
    iw_b = sketch_obj_restr(argv[2], &i0_b, _XRANGE_D, flow_interp);
    if (iw_b != _XRANGE_D) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        k[1] = ch_copy("destroy");
        k[2] = argv[1];
        detruit_obj(2, k + 1);
        free(k[1]);
        return 1;
    }
    a = (funct_dC *) malloc((size_t) sizeof(funct_f));
    x_r = (double *) Obj[_XRANGE_D - 1][i0_b].adresse;
    d = Obj[_XRANGE_D - 1][i0_b].dim;
    a->type = x_r[0];
    if (a->type == 0)
        a->nb = d[0];
    else
        a->nb = x_r[7];
    a->x = x_r;
    a->f = dcomplex_alloc1(a->nb);
    a->nom = ch_copy(argv[2]);
    e = (char **) Obj[_CFUNC_D - 1][i0].adresse;
    e[0] = (char *) a;
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Function that is called when a command of destruction of a function
    is given.
---------------------------------------------------------------------*/
int
detruit_funct(int iw, int i0)
{
    funct_f        *a_f;
    funct_d        *a_d;
    funct_C        *a_C;
    funct_dC       *a_dC;
    char          **c;

    c = (char **) Obj[iw + _XRANGE_F - 2][i0].adresse;
    if (iw == 3) {
        a_f = (funct_f *) c[0];
        if (a_f != NULL) {
            XFREE(a_f->f);
            free(a_f->nom);
            free(a_f);
        }
    }
    else {
        if (iw == 4) {
            a_d = (funct_d *) c[0];
            if (a_d != NULL) {
                XFREE(a_d->f);
                free(a_d->nom);
                free(a_d);
            }
        }
        else {
            if (iw == 5) {
                a_C = (funct_C *) c[0];
                if (a_C != NULL) {
                    XFREE(a_C->f);
                    free(a_C->nom);
                    free(a_C);
                 }
            }
            else {
                a_dC = (funct_dC *) c[0];
                if (a_dC != NULL) {
                    XFREE(a_dC->f);
                    free(a_dC->nom);
                    free(a_dC);
                }
            }
        }
    }
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
        Command of definition of a xrange in simple precision (xrange_f).
---------------------------------------------------------------------*/
int
def_xrange_f(int argc, char *argv[])
{
    return def_xrange_gen(argc, argv, 0);
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
        Command of definition of a xrange in double precision (xrange).
---------------------------------------------------------------------*/
int
def_xrange_d(int argc, char *argv[])
{
    return def_xrange_gen(argc, argv, 1);
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
---------------------------------------------------------------------*/
int
def_xrange_gen(int argc, char *argv[], int cas)
{
    int             i0,
                    iw,
                    nb,
                    type;
    char            h[100],
                   *k[4];
    float          *x_r;
    double         *x_rd;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj(argv[1], &i0, flow_interp);
    if (iw != 0) {
        error_mess(flow_interp, _FUNC_MESS);
        return 1;
    }
    if (argc < 3) {
        print(flow_interp, "%s", mess[_FUNC_MESS + 3]);
        read_int(&type, flow_interp);
        print(flow_interp, "%s", mess[_FUNC_MESS + 4]);
        read_int(&nb, flow_interp);
    }
    else {
        if (argc < 4) {
            error_mess(flow_interp, _FUNC_MESS + 6);
            return 1;
        }
        type = convert_int(argv[2], flow_interp);
        nb = convert_int(argv[3], flow_interp);
    }
    if (nb < 2 || type < 0 || type > 1) {
        error_mess(flow_interp, _FUNC_MESS + 2);
        return 1;
    }
    iw = 10;
    if (type == 0)
        iw = nb;
    memset(h, 0, 100);
    sprintf(h, "nbpoints_x=%d", iw);
    S_convert_int(h, flow_interp);
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("objdef");
    k[3] = argv[1];
    if (cas == 0) {
        k[2] = ch_copy_int(_XRANGE_F - 1);
        i0 = obj_create(3, k + 1);
        x_r = (float *) Obj[_XRANGE_F - 1][i0].adresse;
        x_r[0] = type;
        if (type > 0)
            x_r[7] = nb;
    }
    else {
        k[2] = ch_copy_int(_XRANGE_D - 1);
        i0 = obj_create(3, k + 1);
        x_rd = (double *) Obj[_XRANGE_D - 1][i0].adresse;
        x_rd[0] = type;
        if (type > 0)
            x_rd[7] = nb;
    }
    free(k[1]);
    free(k[2]);
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command of fixation of the values of a x-range in simple
    precision (fix_xrange_f).
---------------------------------------------------------------------*/
int
fix_xrange_f(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw,
                    type,
                   *d;
    float          *x_r;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _XRANGE_F, flow_interp);
    if (iw != _XRANGE_F) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        return 1;
    }
    x_r = (float *) Obj[_XRANGE_F - 1][i0].adresse;
    type = x_r[0];
    if (type == 0) {
        d = (int *) Obj[_XRANGE_F - 1][i0].dim;
        if (argc == 4) {
            iw = convert_int(argv[2], flow_interp);
            if (iw < 1 || iw > d[0]) {
                error_mess(flow_interp, _FUNC_MESS + 2);
                return 1;
            }
            x_r[iw] = convert_float(argv[3], flow_interp);
            return 0;
        }
        else {
            for (i = 1; i <= d[0]; i++)
                read_float(&x_r[i], flow_interp);
        }
    }
    else {
        if (argc < 4) {
            error_mess(flow_interp, _FUNC_MESS + 20);
            return 1;
        }
        x_r[1] = convert_float(argv[2], flow_interp);
        x_r[2] = convert_float(argv[3], flow_interp);
        x_r[3] = x_r[1] + (x_r[7] - 1.)* x_r[2];
    }
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command of fixation of the values of a x-range in double
    precision (fix_xrange).
---------------------------------------------------------------------*/
int
fix_xrange_d(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw,
                    type,
                   *d;
    double         *x_r;
    float           x;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _XRANGE_D, flow_interp);
    if (iw != _XRANGE_D) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        return 1;
    }
    x_r = (double *) Obj[_XRANGE_D - 1][i0].adresse;
    type = x_r[0];
    if (type == 0) {
        d = (int *) Obj[_XRANGE_D - 1][i0].dim;
        if (argc == 4) {
            iw = convert_int(argv[2], flow_interp);
            if (iw < 1 || iw > d[0]) {
                error_mess(flow_interp, _FUNC_MESS + 2);
                return 1;
            }
            x_r[iw] = convert_float(argv[3], flow_interp);
            return 0;
        }
        else {
            for (i = 1; i <= d[0]; i++) {
                read_float(&x, flow_interp);
                x_r[i] = x;
            }
        }
    }
    else {
        if (argc < 4) {
            error_mess(flow_interp, _FUNC_MESS + 20);
            return 1;
        }
        x_r[1] = convert_float(argv[2], flow_interp);
        x_r[2] = convert_float(argv[3], flow_interp);
        x_r[3] = x_r[1] + (x_r[7] - 1.)* x_r[2];
    }
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to set the value of a real function at a point
    (fix_func_R).
---------------------------------------------------------------------*/
int
fix_func_R(int argc, char *argv[])
{
    int             i0,
                    iw,
                    n,
                    nb;
    funct_f        *a_f;
    funct_d        *a_d;
    char          **c;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    a_f = NULL;
    a_d = NULL;
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw == 0)
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
    if (iw != _RFUNC_F && iw != _RFUNC_D) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _RFUNC_F) {
        a_f = (funct_f *) c[0];
        nb = a_f->nb;
    }
    else {
        a_d = (funct_d *) c[0];
        nb = a_d->nb;
    }
    n = convert_int(argv[2], flow_interp);
    if (n < 1 || n > nb) {
        error_mess(flow_interp, _FUNC_MESS + 8);
        return 1;
    }
    if (iw == _RFUNC_F)
        a_f->f[n] = convert_float(argv[3], flow_interp);
    else
        a_d->f[n] = convert_float(argv[3], flow_interp);

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to set the value of a complex function at a point
    (fix_func_C).
---------------------------------------------------------------------*/
int
fix_func_C(int argc, char *argv[])
{
    int             i0,
                    iw,
                    n,
                    nb;
    funct_C        *a_f;
    funct_dC       *a_d;
    char          **c;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    a_f = NULL;
    a_d = NULL;
    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
    if (iw != _CFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
        if (iw != _CFUNC_D) {
            error_mess(flow_interp, _FUNC_MESS + 7);
            return 1;
        }
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        a_f = (funct_C *) c[0];
        nb = a_f->nb;
    }
    else {
        a_d = (funct_dC *) c[0];
        nb = a_d->nb;
    }
    n = convert_int(argv[2], flow_interp);
    if (n < 1 || n > nb) {
        error_mess(flow_interp, _FUNC_MESS + 8);
        return 1;
    }
    if (iw == _CFUNC_F) {
        a_f->f[n].r = convert_float(argv[3], flow_interp);
        a_f->f[n].i = convert_float(argv[4], flow_interp);
    }
    else {
        a_d->f[n].r = convert_float(argv[3], flow_interp);
        a_d->f[n].i = convert_float(argv[4], flow_interp);
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to save a function (save_func).
---------------------------------------------------------------------*/
int
sauve_func(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i,
                    nb,
                    type;
    funct_f        *a_f;
    funct_d        *a_d;
    funct_C        *a_C;
    funct_dC       *a_dC;
    float          *x_r;
    double         *x_rd;
    FILE           *s;
    char          **c;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    a_f = NULL;
    a_d = NULL;
    a_C = NULL;
    a_dC = NULL;
    x_r = NULL;
    x_rd = NULL;
    iw = sketch_obj(argv[1], &i0, flow_interp);
    if (iw != _RFUNC_F && iw != _RFUNC_D
       && iw != _CFUNC_F && iw != _CFUNC_D) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        a_C = (funct_C *) c[0];
        nb = a_C->nb;
        type = a_C->type;
    }
    else {
        if (iw == _CFUNC_D) {
            a_dC = (funct_dC *) c[0];
            nb = a_dC->nb;
            type = a_dC->type;
        }
        else {
            if (iw == _RFUNC_F) {
                a_f = (funct_f *) c[0];
                nb = a_f->nb;
                type = a_f->type;
            }
            else {
                a_d = (funct_d *) c[0];
                nb = a_d->nb;
                type = a_d->type;
            }
        }
    }
    if (iw == _RFUNC_F || iw == _CFUNC_F) {
        x_r = float_alloc1(nb);
        if (type == 0) {
            if (iw == _RFUNC_F)
                for (i = 1; i <= nb; i++)
                    x_r[i] = a_f->x[i];
            else
                for (i = 1; i <= nb; i++)
                    x_r[i] = a_C->x[i];
        }
        else {
            if (iw == _RFUNC_F)
                for (i = 1; i <= nb; i++)
                    x_r[i] = a_f->x[1] + (i - 1) * a_f->x[2];
            else
                for (i = 1; i <= nb; i++)
                    x_r[i] = a_C->x[1] + (i - 1) * a_C->x[2];
        }
    }
    else {
        x_rd = double_alloc1(nb);
        if (type == 0) {
            if (iw == _RFUNC_D)
                for (i = 1; i <= nb; i++)
                    x_rd[i] = a_d->x[i];
            else
                for (i = 1; i <= nb; i++)
                    x_rd[i] = a_dC->x[i];
        }
        else {
            if (iw == _RFUNC_D)
                for (i = 1; i <= nb; i++)
                    x_rd[i] = a_d->x[1] + (i - 1) * a_d->x[2];
            else
                for (i = 1; i <= nb; i++)
                    x_rd[i] = a_dC->x[1] + (i - 1) * a_dC->x[2];
        }
    }
    s = Copen(result_rep, argv[2], "w");
    if (s == NULL) {
        error_mess(flow_interp, _FUNC_MESS + 9);
        if (x_r != NULL)
            XFREE(x_r);
        if (x_rd != NULL)
            XFREE(x_rd);
        return 1;
    }
    if (iw == _RFUNC_F)
        for (i = 1; i <= nb; i++)
            fprintf(s, "%f    %f\n", x_r[i], a_f->f[i]);
    else {
        if (iw == _RFUNC_D)
            for (i = 1; i <= nb; i++)
                fprintf(s, "%f    %f\n", x_rd[i], a_d->f[i]);
        else {
            if (iw == _CFUNC_F)
                for (i = 1; i <= nb; i++)
                    fprintf(s, "%f    %f    %f\n", x_r[i], a_C->f[i].r,
                        a_C->f[i].i);
            else
                for (i = 1; i <= nb; i++)
                    fprintf(s, "%f    %f    %f\n", x_rd[i], a_dC->f[i].r,
                        a_dC->f[i].i);

        }
    }

    if (x_r != NULL)
        XFREE(x_r);
    if (x_rd != NULL)
        XFREE(x_rd);
    fclose(s);
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
---------------------------------------------------------------------*/
int
exec_func2(int argc, char *argv[], funct2_t e)
{
    int             iw;
    char           **c1,
                   **c2,
                   **c3;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    c1 = NULL;
    c2 = NULL;
    c3 = NULL;
    if (fix_func(&c1, &c2, &c3, &iw, argv) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    if (iw == _RFUNC_F) {
        if (TX_f(c1[0]) * TX_f(c2[0]) * TX_f(c3[0]) == 1)
                    e.F_f((funct_f *) c1[0], (funct_f *) c2[0],
                        (funct_f *) c3[0]);
    }
    else {
        if (iw == _RFUNC_D) {
            if (TX_d(c1[0]) * TX_d(c2[0]) * TX_d(c3[0]) == 1)
                e.F_d((funct_d *) c1[0], (funct_d *) c2[0], (funct_d *) c3[0]);
        }
        else {
            if (iw == _CFUNC_F) {
                if (TX_C(c1[0]) * TX_C(c2[0]) * TX_C(c3[0]) == 1)
                    e.F_C((funct_C *) c1[0], (funct_C *) c2[0],
                        (funct_C *) c3[0]);
            }
            else {
                if (TX_dC(c1[0]) * TX_dC(c2[0]) * TX_dC(c3[0]) == 1)
                    e.F_dC((funct_dC *) c1[0], (funct_dC *) c2[0],
                        (funct_dC *) c3[0]);
            }
        }
    }
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to compute the sum of two functions (add_func).
---------------------------------------------------------------------*/
int
add_func(int argc, char *argv[])
{
    return exec_func2(argc, argv, F_ADD);
}
/*-------------------------------------------------------------------*/





/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
fix_func(char ***c1, char ***c2, char ***c3, int *iw, char *argv[])
{
    int             iw0,
                    iw1,
                    iw2,
                    i00,
                    i01,
                    i02;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw0 = sketch_obj_restr(argv[1], &i00, _RFUNC_F, flow_interp);
    if (iw0 != _RFUNC_F) {
        iw0 = sketch_obj_restr(argv[1], &i00, _RFUNC_D, flow_interp);
        if (iw0 != _RFUNC_D) {
            iw0 = sketch_obj_restr(argv[1], &i00, _CFUNC_F, flow_interp);
            if (iw0 != _CFUNC_F) {
                iw0 = sketch_obj_restr(argv[1], &i00, _CFUNC_D, flow_interp);
                if (iw0 != _CFUNC_D) {
                    error_mess(flow_interp, _FUNC_MESS + 7);
                    return 1;
                }
            }
        }
    }
    iw1 = sketch_obj_restr(argv[2], &i01, iw0, flow_interp);
    if (iw0 != iw1)
        return 1;
    iw2 = sketch_obj_restr(argv[3], &i02, iw0, flow_interp);
    if (iw0 != iw2)
        return 1;
    c1[0] = (char **) Obj[iw0 - 1][i00].adresse;
    c2[0] = (char **) Obj[iw0 - 1][i01].adresse;
    c3[0] = (char **) Obj[iw0 - 1][i02].adresse;
    iw[0] = iw0;
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to compute the difference of two functions (sub_func).
---------------------------------------------------------------------*/
int
sub_func(int argc, char *argv[])
{
    return exec_func2(argc, argv, F_SUB);
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to compute the product of two functions (mul_func).
---------------------------------------------------------------------*/
int
mul_func(int argc, char *argv[])
{
    return exec_func2(argc, argv, F_MUL);
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to compute the quotient of two functions (div_func).
---------------------------------------------------------------------*/
int
div_func(int argc, char *argv[])
{
    return exec_func2(argc, argv, F_DIV);
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to describe a function (desc_func).
---------------------------------------------------------------------*/
int
desc_func(int argc, char *argv[])
{
    int             iw,
                    i0,
                    iw0,
                    i00,
                    nb,
                    type;
    char          **c,
                   *nom_x,
                    h[100];
    funct_f        *a_f;
    funct_d        *a_d;
    funct_C        *a_C;
    funct_dC       *a_dC;
    double          xmin,
                    xmax;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj(argv[1], &i0, flow_interp);
    if (iw < _RFUNC_F || iw > _CFUNC_D) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;

    if (iw == _RFUNC_F) {
        print(flow_interp, "%s", mess[_FUNC_MESS + 10]);
        a_f = (funct_f *) c[0];
        nb = a_f->nb;
        type = a_f->type;
        nom_x = a_f->nom;
        xmin = (double) a_f->x[1];
        if (type == 1)
            xmax = (double) a_f->x[3];
        else
            xmax = (double) a_f->x[nb];
    }
    else {
        if (iw == _RFUNC_D) {
            print(flow_interp, "%s", mess[_FUNC_MESS + 11]);
            a_d = (funct_d *) c[0];
            nb = a_d->nb;
            type = a_d->type;
            nom_x = a_d->nom;
            xmin = a_d->x[1];
            if (type == 1)
                xmax = a_d->x[3];
            else
                xmax = a_d->x[nb];
        }
        else {
            if (iw == _CFUNC_F) {
                print(flow_interp, "%s", mess[_FUNC_MESS + 12]);
                a_C = (funct_C *) c[0];
                nb = a_C->nb;
                type = a_C->type;
                nom_x = a_C->nom;
                xmin = (double) a_C->x[1];
                if (type == 1)
                    xmax = (double) a_C->x[3];
                else
                    xmax = (double) a_C->x[nb];
            }
            else {
                print(flow_interp, "%s", mess[_FUNC_MESS + 13]);
                a_dC = (funct_dC *) c[0];
                nb = a_dC->nb;
                type = a_dC->type;
                nom_x = a_dC->nom;
                xmin = a_dC->x[1];
                if (type == 1)
                    xmax = a_dC->x[3];
                else
                    xmax = a_dC->x[nb];
            }
        }
    }
    iw0 = sketch_obj(nom_x, &i00, flow_interp);
    if (((iw == _RFUNC_F || iw == _CFUNC_F) && iw0 != _XRANGE_F) ||
        ((iw == _RFUNC_D || iw == _CFUNC_D) && iw0 != _XRANGE_D)) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        return 1;
    }
    print(flow_interp, "%s  %d\n", mess[_FUNC_MESS + 3], type);
    print(flow_interp, "%s  %d\n", mess[_FUNC_MESS + 4], nb);
    print(flow_interp, "%s  %s\n", mess[_FUNC_MESS + 14], nom_x);
    print(flow_interp, "%s  %f\n", mess[_FUNC_MESS + 21], xmin);
    print(flow_interp, "%s  %f\n", mess[_FUNC_MESS + 22], xmax);
    memset(h, 0, 100);
    sprintf(h, "xmin=%f", xmin);
    convert_float(h, flow_interp);
    memset(h, 0, 100);
    sprintf(h, "xmax=%f", xmax);
    convert_float(h, flow_interp);
    return 0;
}
/*-------------------------------------------------------------------*/





/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
exec_func(int argc, char *argv[], funct_t e)
{
    int             iw,
                    i0,
                    iw0,
                    i00;
    char          **c,
                  **c0;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
            if (iw != _CFUNC_F) {
                iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
                if (iw != _CFUNC_D) {
                    error_mess(flow_interp, _FUNC_MESS + 7);
                    return 1;
                }
            }
        }
    }
    iw0 = sketch_obj_restr(argv[2], &i00, iw, flow_interp);
    if (iw0 != iw) {
        error_mess(flow_interp, _FUNC_MESS + 15);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw == _RFUNC_F) {
        if (TX_f(c[0]) * TX_f(c0[0]) == 1)
            e.F_f((funct_f *) c[0], (funct_f *) c0[0]);
    }
    else {
        if (iw == _RFUNC_D) {
            if (TX_d(c[0]) * TX_d(c0[0]) == 1)
                e.F_d((funct_d *) c[0], (funct_d *) c0[0]);
        }
        else {
            if (iw == _CFUNC_F) {
                if (TX_C(c[0]) * TX_C(c0[0]) == 1)
                    e.F_C((funct_C *) c[0], (funct_C *) c0[0]);
            }
            else {
                if (TX_dC(c[0]) * TX_dC(c0[0]) == 1)
                    e.F_dC((funct_dC *) c[0], (funct_dC *) c0[0]);
            }
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to copy a function to another (copy_func).
---------------------------------------------------------------------*/
int
copy_func(int argc, char *argv[])
{
    return exec_func(argc, argv, F_COPY);
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to compute the product of a function with a number (rmul).
---------------------------------------------------------------------*/
int
Rmul_func(int argc, char *argv[])
{
    int             iw,
                    i0,
                    iw0,
                    i00,
                    ik;
    char          **c,
                  **c0;
    double          t,
                    ti,
                    tr;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
            if (iw != _CFUNC_F) {
                iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
                if (iw != _CFUNC_D) {
                    error_mess(flow_interp, _FUNC_MESS + 7);
                    return 1;
                }
            }
        }
    }
    if (iw > _RFUNC_D && argc < 5) {
        error_mess(flow_interp, _FUNC_MESS + 6);
        return 1;
    }
    t = 0.;
    ti = 0.;
    tr = 0.;
    if (iw < _CFUNC_F) {
        t = convert_float(argv[2], flow_interp);
        ik = 0;
    }
    else {
        tr = convert_float(argv[2], flow_interp);
        ti = convert_float(argv[3], flow_interp);
        ik = 1;
    }
    iw0 = sketch_obj_restr(argv[3 + ik], &i00, iw, flow_interp);
    if (iw0 != iw) {
        error_mess(flow_interp, 15);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw == _RFUNC_F) {
        if (TX_f(c[0]) * TX_f(c0[0]) == 1)
            f_funct_Rmul((float) t, (funct_f *) c[0], (funct_f *) c0[0]);
    }
    else {
        if (iw == _RFUNC_D) {
            if (TX_d(c[0]) * TX_d(c0[0]) == 1)
                d_funct_Rmul(t, (funct_d *) c[0], (funct_d *) c0[0]);
        }
        else {
            if (iw == _CFUNC_F) {
                if (TX_C(c[0]) * TX_C(c0[0]) == 1)
                    C_funct_Rmul(Complex((float) tr, (float) ti),
                        (funct_C *) c[0], (funct_C *) c0[0]);
            }
            else {
                if (TX_dC(c[0]) * TX_dC(c0[0]) == 1)
                    dC_funct_Rmul(dComplex(tr, ti), (funct_dC *) c[0],
                        (funct_dC *) c0[0]);
            }
        }
    }
    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to put the real or imaginary part of a complex function
    in a real one (real or imag).
---------------------------------------------------------------------*/
int
reel_imag_Cfunc(int argc, char *argv[])
{
    int             iw,
                    i0,
                    iw0,
                    i00;
    char          **c,
                  **c0;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
    if (iw != _CFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
        if (iw != _CFUNC_D) {
            error_mess(flow_interp, _FUNC_MESS + 7);
            return 1;
        }
    }
    iw0 = sketch_obj_restr(argv[2], &i00, iw - 2, flow_interp);
    if (iw0 != iw - 2) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw0 == _RFUNC_F) {
        if (TX_C(c[0]) * TX_f(c0[0]) == 1) {
            if (comp(argv[0], "real") == 1)
                reel_f((funct_C *) c[0], (funct_f *) c0[0]);
            else
                imag_f((funct_C *) c[0], (funct_f *) c0[0]);
        }
    }
    else {
        if (TX_dC(c[0]) * TX_d(c0[0]) == 1) {
            if (comp(argv[0], "real") == 1)
                reel_d((funct_dC *) c[0], (funct_d *) c0[0]);
            else
                imag_d((funct_dC *) c[0], (funct_d *) c0[0]);
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to put a real function in the real part or the imaginary
    part of a complex one (imag_fix or real_fix).
---------------------------------------------------------------------*/
int
reel_imag_fix_func(int argc, char *argv[])
{
    int             iw,
                    i0,
                    iw0,
                    i00;
    char          **c,
                  **c0;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            error_mess(flow_interp, _FUNC_MESS + 7);
            return 1;
        }
    }
    iw0 = sketch_obj_restr(argv[2], &i00, iw + 2, flow_interp);
    if (iw0 != iw + 2) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw == _RFUNC_F) {
        if (TX_f(c[0]) * TX_C(c0[0]) == 1) {
            if (comp(argv[0], "real_fix") == 1)
                conv_f_Cr((funct_f *) c[0], (funct_C *) c0[0]);
            else
                conv_f_Ci((funct_f *) c[0], (funct_C *) c0[0]);
        }
    }
    else {
        if (TX_d(c[0]) * TX_dC(c0[0]) == 1) {
            if (comp(argv[0], "real_fix") == 1)
                conv_d_dCr((funct_d *) c[0], (funct_dC *) c0[0]);
            else
                conv_d_dCi((funct_d *) c[0], (funct_dC *) c0[0]);
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command of integration of a function (integ)
---------------------------------------------------------------------*/
int
integ_func(int argc, char *argv[])
{
    return exec_func(argc, argv, F_INTEG);
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command of derivation of a function (diff)
---------------------------------------------------------------------*/
int
diff_func(int argc, char *argv[])
{
    return exec_func(argc, argv, F_DIFF);
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command of composition of 2 functions (comp_func)
---------------------------------------------------------------------*/
int
comp_func(int argc, char *argv[])
{
    int             iw,
                    i0,
                    iw0,
                    i00,
                    iw1,
                    i01,
                    iw2;
    char          **c,
                  **c0,
                  **c1;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F)
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
    if (iw > 0)
        iw0 = sketch_obj_restr(argv[2], &i00, iw, flow_interp);
    else {
        iw0 = sketch_obj_restr(argv[2], &i00, _RFUNC_F, flow_interp);
        if (iw0 != _RFUNC_F)
            iw0 = sketch_obj_restr(argv[2], &i00, _RFUNC_D, flow_interp);
    }
    if (iw > 0)
        iw2 = iw;
    else {
        if (iw0 > 0)
            iw2 = iw0;
        else
            iw2 = 0;
    }
    if (iw2 > 0) {
        iw1 = sketch_obj_restr(argv[3], &i01, iw2, flow_interp);
        if (iw1 != iw2) {
            error_mess(flow_interp, _FUNC_MESS + 7);
            return 1;
        }
    }
    else {
        iw1 = sketch_obj_restr(argv[3], &i01, _RFUNC_F, flow_interp);
        if (iw1 == _RFUNC_F) {
            iw1 = sketch_obj_restr(argv[3], &i01, _RFUNC_D, flow_interp);
            if (iw1 != _RFUNC_D) {
                error_mess(flow_interp, _FUNC_MESS + 7);
                return 1;
            }
        }
    }
    c = NULL;
    c0 = NULL;
    if (iw > 0)
        c = (char **) Obj[iw - 1][i0].adresse;
    if (iw0 > 0)
        c0 = (char **) Obj[iw0 - 1][i00].adresse;
    c1 = (char **) Obj[iw1 - 1][i01].adresse;
    if (iw == _RFUNC_F && iw0 == iw) {
        if (TX_f(c[0]) * TX_f(c0[0]) * TX_f(c1[0]) == 1)
            comp_funct_f((funct_f *) c[0], (funct_f *) c0[0],
                (funct_f *) c1[0]);
    }
    else {
        if (iw == _RFUNC_D && iw0 == iw) {
            if (TX_d(c[0]) * TX_d(c0[0]) * TX_d(c1[0]) == 1)
                comp_funct_d((funct_d *) c[0], (funct_d *) c0[0],
                    (funct_d *) c1[0]);
        }
        else {
            if (iw == 0 && iw0 == _RFUNC_F) {
                if (TX_f(c0[0]) * TX_f(c1[0]) == 1)
                    comp_functb_f(argv[1], (funct_f *) c0[0],
                        (funct_f *) c1[0], flow_interp);
            }
            else {
                if (iw == 0 && iw0 == _RFUNC_D) {
                    if (TX_d(c0[0]) * TX_d(c1[0]) == 1)
                    comp_functb_d(argv[1], (funct_d *) c0[0],
                        (funct_d *) c1[0], flow_interp);
                }
                else {
                    if (iw == _RFUNC_F && iw0 == 0) {
                        if (TX_f(c[0]) * TX_f(c1[0]) == 1)
                            comp_functc_f((funct_f *) c[0], argv[2],
                                (funct_f *) c1[0], flow_interp);
                    }
                    else {
                        if (iw == 0 && iw0 == _RFUNC_D) {
                            if (TX_d(c[0]) * TX_d(c1[0]) == 1)
                                comp_functc_d((funct_d *) c[0], argv[2],
                                    (funct_d *) c1[0], flow_interp);
                        }
                        else {
                            if (iw == 0 && iw0 == 0 && iw1 == _RFUNC_F) {
                                if (TX_f(c1[0]) == 1)
                                    comp_functd_f(argv[1], argv[2],
                                        (funct_f *) c1[0], flow_interp);
                            }
                            else {
                                if (TX_d(c1[0]) == 1)
                                    comp_functd_d(argv[1], argv[2],
                                        (funct_d *) c1[0], flow_interp);
                            }
                        }
                    }
                }
            }
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to fill a real function with a mathematical formula
    (fill_func).
---------------------------------------------------------------------*/
int
fill_func(int argc, char *argv[])
{
    int             iw,
                    i0,
                    i,
                    j,
                    j0;
    float          *x_r;
    double         *x_rd,
                    x;
    char          **c,
                    h[100];
    funct_f        *a_f;
    funct_d        *a_d;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            error_mess(flow_interp, _FUNC_MESS + 7);
            return 1;
        }
    }
    j0 = -1;

    for (j = 0; j < _NBFONC; j++) {
        if (comp(argv[2], Funcs[j].name) == 1) {
            if (Funcs[j].args == 1) {
                j0 = j;
                break;
            }
        }
    }

    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _RFUNC_F) {
        if (TX_f(c[0]) == 1) {
            a_f = (funct_f *) c[0];
            f_fill(a_f, &x_r);
            if (j0 >= 0) {
                for (i = 1; i <= a_f->nb; i++) {
                    x = (double) x_r[i];
                    a_f->f[i] = (float) Funcs[j0].func(&x);
                }
            }
            else {
                for (i = 1; i <= a_f->nb; i++) {
                    memset(h, 0, 100);
                    sprintf(h, "x=%f", x_r[i]);
                    convert_float(h, flow_interp);
                    a_f->f[i] = convert_float(argv[2], flow_interp);
                }
            }
            if (a_f->type > 0)
                XFREE(x_r);
        }
    }
    else {
        if (TX_d(c[0]) == 1) {
            a_d = (funct_d *) c[0];
            d_fill(a_d, &x_rd);
            if (j0 >= 0) {
                for (i = 1; i <= a_d->nb; i++)
                    a_d->f[i] = Funcs[j0].func(&x_rd[i]);
            }
            else {
                for (i = 1; i <= a_d->nb; i++) {
                    memset(h, 0, 100);
                    sprintf(h, "x=%f", x_rd[i]);
                    convert_float(h, flow_interp);
                    a_d->f[i] = convert_float(argv[2], flow_interp);
                }
            }
            if (a_d->type > 0)
                XFREE(x_rd);
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to give a constant value to a function (const_func).
---------------------------------------------------------------------*/
int
const_func(int argc, char *argv[])
{
    int             iw,
                    i0;
    char          **c;
    fcomplex        z;
    dcomplex        zd;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
        if (iw != _CFUNC_F)
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
        }
    }
    if (iw == 0) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _RFUNC_F) {
        if (TX_f(c[0]) == 1)
            f_funct_const(convert_float(argv[2], flow_interp),
                (funct_f *) c[0]);
    }
    else {
        if (iw == _RFUNC_D) {
            if (TX_d(c[0]) == 1)
                d_funct_const((double) convert_float(argv[2], flow_interp),
                    (funct_d *) c[0]);
        }
        else {
            if (iw == _CFUNC_F) {
                if (TX_C(c[0]) == 1) {
                    z = Complex(convert_float(argv[2], flow_interp),
                        convert_float(argv[3], flow_interp));
                    C_funct_const(z, (funct_C *) c[0]);
                }
            }
            else {
                if (TX_dC(c[0]) == 1) {
                    zd = dComplex((double) convert_float(argv[2], flow_interp),
                        (double) convert_float(argv[3], flow_interp));
                    dC_funct_const(zd, (funct_dC *) c[0]);
                }
            }
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
void
dest_prop_func(int typ, int i0, flow_data *flow_interp)
{
    int             i,
                    _nbfunc;
    char          **c;
    funct_f        *a_f;
    funct_d        *a_d;
    funct_C        *a_C;
    funct_dC       *a_dC;
    float          *xr;
    double         *xrd,
                    xx;
    char           *k[3];

    GetValue_Global_hidden("nb_funct", &xx);
    _nbfunc = (int) xx;
    if (typ > _XRANGE_F && typ < _CFUNC_D)
        detruit_funct(typ + 2 - _XRANGE_F, i0);
    if (typ == _FOUR_TR - 1)
        detruit_four_prec(typ + 1, i0);

    if (typ == _XRANGE_F - 1) {  /* when we destroy a x-range we must also
                                    destroy all the functions that use it  */
        xr = (float *) Obj[_XRANGE_F - 1][i0].adresse;
        k[0] = (char *) flow_interp;
        k[1] = ch_copy("destroy");

        for (i = 0; i < _nbfunc; i++) {
            if (Obj[_RFUNC_F - 1][i].occup == 1) {
                c = (char **) Obj[_RFUNC_F - 1][i].adresse;
                a_f = (funct_f *) c[0];
                if (a_f->x == xr) {
                    k[2] = ch_copy(Obj[_RFUNC_F - 1][i].nom_obj);
                    detruit_obj(2, k + 1);
                    free(k[2]);
                }
            }
        }

        for (i = 0; i < _nbfunc; i++) {
            if (Obj[_CFUNC_F - 1][i].occup == 1) {
                c = (char **) Obj[_CFUNC_F - 1][i].adresse;
                a_C = (funct_C *) c[0];
                if (a_C->x == xr) {
                    k[2] = ch_copy(Obj[_CFUNC_F - 1][i].nom_obj);
                    detruit_obj(2, k + 1);
                    free(k[2]);
                }
            }
        }
        free(k[1]);
    }
    if (typ == _XRANGE_D - 1) {  /* when we destroy a x-range we must also
                                    destroy all the functions that use it  */
        xrd = (double *) Obj[_XRANGE_D - 1][i0].adresse;
        k[0] = (char *) flow_interp;
        k[1] = ch_copy("destroy");

        for (i = 0; i < _nbfunc; i++) {
            if (Obj[_RFUNC_D - 1][i].occup == 1) {
                c = (char **) Obj[_RFUNC_D - 1][i].adresse;
                a_d = (funct_d *) c[0];
                if (a_d->x == xrd) {
                    k[2] = ch_copy(Obj[_RFUNC_D - 1][i].nom_obj);
                    detruit_obj(2, k + 1);
                    free(k[2]);
                }
            }
        }

        for (i = 0; i < _nbfunc; i++) {
            if (Obj[_CFUNC_D - 1][i].occup == 1) {
                c = (char **) Obj[_CFUNC_D - 1][i].adresse;
                a_dC = (funct_dC *) c[0];
                if (a_dC->x == xrd) {
                    k[2] = ch_copy(Obj[_CFUNC_D - 1][i].nom_obj);
                    detruit_obj(2, k + 1);
                    free(k[2]);
                }
            }
        }

        free(k[1]);
    }
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to print the value of a function at a point (val_func).
---------------------------------------------------------------------*/
int
val_func(int argc, char *argv[])
{
    int             iw,
                    i0;
    char          **c;
    float           x;
    double          xx,
                    y;
    fcomplex        z;
    dcomplex        zz;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    y = convert_float(argv[2], flow_interp);
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
            if (iw != _CFUNC_F)
                iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
        }
    }
    if (iw == 0) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        if (TX_C(c[0]) == 1) {
            z = C_funct_eval((funct_C *) c[0], (float) y);
            xx = z.r;
            SetValue("func_r", &xx, flow_interp);
            xx = z.i;
            SetValue("func_i", &xx, flow_interp);
            print(flow_interp, "%f + %f.I\n", z.r, z.i);
        }
    }
    else {
        if (iw == _CFUNC_D) {
            if (TX_dC(c[0]) == 1) {
                zz = dC_funct_eval((funct_dC *) c[0], y);
                SetValue("func_r", &zz.r, flow_interp);
                SetValue("func_i", &zz.i, flow_interp);
                print(flow_interp, "%f + %f.I\n", zz.r, zz.i);
            }
        }
        else {
            if (iw == _RFUNC_F) {
                if (TX_f(c[0]) == 1) {
                    x = f_funct_eval((funct_f *) c[0], (float) y);
                    xx = x;
                    SetValue("func", &xx, flow_interp);
                    print(flow_interp, "%f\n", x);
                }
            }
            else {
                if (TX_d(c[0]) == 1) {
                    xx = d_funct_eval((funct_d *) c[0], y);
                    SetValue("func", &xx, flow_interp);
                    print(flow_interp, "%f\n", xx);
                }
            }
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to compute the 'generalized inverse' of a real function
---------------------------------------------------------------------*/
int
inv_func_cmd(int argc, char *argv[])
{
    int             i0,
                    i1,
                    iw0,
                    iw1;
    funct_f        *a_f,
                   *b_f;
    funct_d        *a_d,
                   *b_d;
    char          **c;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw0 = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw0 != _RFUNC_F)
        iw0 = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
    if (iw0 == 0) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    iw1 = sketch_obj_restr(argv[2], &i1, iw0, flow_interp);
    if (iw0 != iw1) {
        error_mess(flow_interp, _FUNC_MESS + 15);
        return 1;
    }
    if (iw0 == _RFUNC_F) {
        c = (char **) Obj[iw0 - 1][i0].adresse;
        if (TX_f(c[0]) == 1) {
            a_f = (funct_f *) c[0];
            c = (char **) Obj[iw0 - 1][i1].adresse;
            if (TX_f(c[0]) == 1) {
                b_f = (funct_f *) c[0];
                inv_sup_funct_f(a_f, b_f);
            }
        }
    }
    else {
        c = (char **) Obj[iw0 - 1][i0].adresse;
        if (TX_d(c[0]) == 1) {
            a_d = (funct_d *) c[0];
            c = (char **) Obj[iw0 - 1][i1].adresse;
            if (TX_d(c[0]) == 1) {
                b_d = (funct_d *) c[0];
                inv_sup_funct_d(a_d, b_d);
            }
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to compute the maximum of a function
---------------------------------------------------------------------*/
int
Max_funct_cmd(int argc, char *argv[])
{
    int             iw,
                    i0;
    char          **c;
    float           xmax_f,
                    fmax_f,
                    Rmax_C;
    double          xmax_d,
                    fmax_d,
                    Rmax_dC;
    fcomplex        fmax_C;
    dcomplex        fmax_dC;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
            if (iw != _CFUNC_F)
                iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
        }
    }
    if (iw == 0) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        if (TX_C(c[0]) == 1) {
            Max_funct_C((funct_C *) c[0], &xmax_f, &fmax_C, &Rmax_C);
            xmax_d = xmax_f;
            SetValue("xmax_func" , &xmax_d, flow_interp);
            fmax_d = fmax_C.r;
            SetValue("fmax_func_r", &fmax_d, flow_interp);
            fmax_d = fmax_C.i;
            SetValue("fmax_func_i", &fmax_d, flow_interp);
            fmax_d = Rmax_C;
            SetValue("fmax_func_R", &fmax_d, flow_interp);
            print(flow_interp, "xmax = %f\n", xmax_f);
            print(flow_interp, "fmax = %f + %f.I\n", fmax_C.r, fmax_C.i);
            print(flow_interp, "Rmax = %f\n", Rmax_C);
        }
    }
    else {
        if (iw == _CFUNC_D) {
            if (TX_dC(c[0]) == 1) {
                Max_funct_dC((funct_dC *) c[0], &xmax_d, &fmax_dC, &Rmax_dC);
                SetValue("xmax_func" , &xmax_d, flow_interp);
                SetValue("fmax_func_r", &fmax_dC.r, flow_interp);
                SetValue("fmax_func_i", &fmax_dC.i, flow_interp);
                SetValue("fmax_func_R", &Rmax_dC, flow_interp);
                print(flow_interp, "xmax = %f\n", xmax_d);
                print(flow_interp, "fmax = %f + %f.I\n", fmax_dC.r, fmax_dC.i);
                print(flow_interp, "Rmax = %f\n", Rmax_dC);
            }
        }
        else {
            if (iw == _RFUNC_F) {
                if (TX_f(c[0]) == 1) {
                    Max_funct_f((funct_f *) c[0], &xmax_f, &fmax_f);
                    xmax_d = xmax_f;
                    SetValue("xmax_func" , &xmax_d, flow_interp);
                    fmax_d = fmax_f;
                    SetValue("fmax_func", &fmax_d, flow_interp);
                    print(flow_interp, "xmax = %f\n", xmax_f);
                    print(flow_interp, "fmax = %f\n", fmax_f);
                }
            }
            else {
                if (TX_d(c[0]) == 1) {
                    Max_funct_d((funct_d *) c[0], &xmax_d, &fmax_d);
                    SetValue("xmax_func" , &xmax_d, flow_interp);
                    SetValue("fmax_func", &fmax_d, flow_interp);
                    print(flow_interp, "xmax = %f\n", xmax_d);
                    print(flow_interp, "fmax = %f\n", fmax_d);
                }
            }
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/





/*---------------------------------------------------------------------
    Command used to compute the minimum of a function
---------------------------------------------------------------------*/
int
Min_funct_cmd(int argc, char *argv[])
{
    int             iw,
                    i0;
    char          **c;
    float           xmin_f,
                    fmin_f,
                    Rmin_C;
    double          xmin_d,
                    fmin_d,
                    Rmin_dC;
    fcomplex        fmin_C;
    dcomplex        fmin_dC;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
            if (iw != _CFUNC_F)
                iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
        }
    }
    if (iw == 0) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        if (TX_C(c[0]) == 1) {
            Min_funct_C((funct_C *) c[0], &xmin_f, &fmin_C, &Rmin_C);
            xmin_d = xmin_f;
            SetValue("xmin_func" , &xmin_d, flow_interp);
            fmin_d = fmin_C.r;
            SetValue("fmin_func_r", &fmin_d, flow_interp);
            fmin_d = fmin_C.i;
            SetValue("fmin_func_i", &fmin_d, flow_interp);
            fmin_d = Rmin_C;
            SetValue("fmin_func_R", &fmin_d, flow_interp);
            print(flow_interp, "xmin = %f\n", xmin_f);
            print(flow_interp, "fmin = %f + %f.I\n", fmin_C.r, fmin_C.i);
            print(flow_interp, "Rmin = %f\n", Rmin_C);
        }
    }
    else {
        if (iw == _CFUNC_D) {
            if (TX_dC(c[0]) == 1) {
                Min_funct_dC((funct_dC *) c[0], &xmin_d, &fmin_dC, &Rmin_dC);
                SetValue("xmin" , &xmin_d, flow_interp);
                SetValue("fmin_func_r", &fmin_dC.r, flow_interp);
                SetValue("fmin_func_i", &fmin_dC.i, flow_interp);
                SetValue("fmin_func_R", &Rmin_dC, flow_interp);
                print(flow_interp, "xmin = %f\n", xmin_d);
                print(flow_interp, "fmin = %f + %f.I\n", fmin_dC.r, fmin_dC.i);
                print(flow_interp, "Rmin = %f\n", Rmin_dC);
            }
        }
        else {
            if (iw == _RFUNC_F) {
                if (TX_f(c[0]) == 1) {
                    Min_funct_f((funct_f *) c[0], &xmin_f, &fmin_f);
                    xmin_d = xmin_f;
                    SetValue("xmin_func" , &xmin_d, flow_interp);
                    fmin_d = fmin_f;
                    SetValue("fmin_func", &fmin_d, flow_interp);
                    print(flow_interp, "xmin = %f\n", xmin_f);
                    print(flow_interp, "fmin = %f\n", fmin_f);
                }
            }
            else {
                if (TX_d(c[0]) == 1) {
                    Min_funct_d((funct_d *) c[0], &xmin_d, &fmin_d);
                    SetValue("xmin_func" , &xmin_d, flow_interp);
                    SetValue("fmin_func", &fmin_d, flow_interp);
                    print(flow_interp, "xmin = %f\n", xmin_d);
                    print(flow_interp, "fmin = %f\n", fmin_d);
                }
            }
        }
    }

    return 0;
}
/*-------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Function associated to the command 'load_funct'
--------------------------------------------------------------------*/
int
load_funct(int argc, char *argv[])
{
    int             i0,
                    iw;
    char          **c;
    FILE           *s;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
            if (iw != _CFUNC_F)
                iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
        }
    }
    if (iw == 0) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    s = Copen(result_rep, argv[2], "r");
    if (s == NULL) {
        error_mess(flow_interp, _FUNC_MESS + 9);
        return 1;
    }
    i0 = -1;

    if (iw == _RFUNC_F) {
        if (TX_f(c[0]) == 1) {
            i0 = load_funct_f((funct_f *) c[0], s);
        }
    }
    else {
        if (iw == _RFUNC_D) {
            if (TX_d(c[0]) == 1) {
                i0 = load_funct_d((funct_d *) c[0], s);
            }
        }
        else {
            if (iw == _CFUNC_F) {
                if (TX_C(c[0]) == 1) {
                    i0 = load_funct_C((funct_C *) c[0], s);
                }
            }
            else {
                if (iw == _CFUNC_D) {
                    if (TX_dC(c[0]) == 1) {
                        i0 = load_funct_dC((funct_dC *) c[0], s);
                    }
                }
            }
        }
    }
    fclose(s);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Function associated to the command 'load_create_f'
--------------------------------------------------------------------*/
int
load_create_f(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw,
                    nb;
    char          **c,
                   *k[5];
    FILE           *s;
    float          *xr,
                   *xF;
    funct_f        *a;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    xr = NULL;
    xF = NULL;
    s = Copen(result_rep, argv[3], "r");
    if (s == NULL) {
        error_mess(flow_interp, _FUNC_MESS + 9);
        return 1;
    }
    load_data_f(s, &xr, &xF, &nb);
    if (nb <= 1) {
        error_mess(flow_interp, 24);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("defxr_f");
    k[2] = argv[1];
    k[3] = ch_copy_int(0);
    k[4] = ch_copy_int(nb);
    if (def_xrange_f(4, k + 1) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        free(k[1]);
        free(k[3]);
        free(k[4]);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    free(k[1]);
    free(k[3]);
    free(k[4]);
    k[1] = ch_copy("defunc_f");
    k[2] = argv[2];
    k[3] = argv[1];
    if (def_funct_f(3, k + 1) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        free(k[1]);
        k[1] = ch_copy("destroy");
        k[2] = argv[1];
        detruit_obj(2, k + 1);
        free(k[1]);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    free(k[1]);
    iw = sketch_obj_restr(argv[2], &i0, _RFUNC_F, flow_interp);
    c = (char **) Obj[iw - 1][i0].adresse;
    a = (funct_f *) c[0];
    a->nb = nb;

    for (i = 0; i < nb; i++) {
        a->x[i + 1] = xr[i];
        a->f[i + 1] = xF[i];
    }

    free(xr);
    free(xF);
    fclose(s);
    return 0;
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Function associated to the command 'load_create_d'
--------------------------------------------------------------------*/
int
load_create_d(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw,
                    nb;
    char          **c,
                   *k[5];
    FILE           *s;
    double         *xr,
                   *xF;
    funct_d        *a;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    xr = NULL;
    xF = NULL;
    s = Copen(result_rep, argv[3], "r");
    if (s == NULL) {
        error_mess(flow_interp, _FUNC_MESS + 9);
        return 1;
    }
    load_data_d(s, &xr, &xF, &nb);
    if (nb <= 1) {
        error_mess(flow_interp, 24);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("defxr_d");
    k[2] = argv[1];
    k[3] = ch_copy_int(0);
    k[4] = ch_copy_int(nb);
    if (def_xrange_d(4, k + 1) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        free(k[1]);
        free(k[3]);
        free(k[4]);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    free(k[1]);
    free(k[3]);
    free(k[4]);
    k[1] = ch_copy("defunc_d");
    k[2] = argv[2];
    k[3] = argv[1];
    if (def_funct_d(3, k + 1) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        free(k[1]);
        k[1] = ch_copy("destroy");
        k[2] = argv[1];
        detruit_obj(2, k + 1);
        free(k[1]);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    free(k[1]);
    iw = sketch_obj_restr(argv[2], &i0, _RFUNC_D, flow_interp);
    c = (char **) Obj[iw - 1][i0].adresse;
    a = (funct_d *) c[0];
    a->nb = nb;

    for (i = 0; i < nb; i++) {
        a->x[i + 1] = xr[i];
        a->f[i + 1] = xF[i];
    }

    free(xr);
    free(xF);
    fclose(s);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Function associated to the command 'load_create_C'
--------------------------------------------------------------------*/
int
load_create_C(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw,
                    nb;
    char          **c,
                   *k[5];
    FILE           *s;
    float          *xr;
    fcomplex       *xF;
    funct_C        *a;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    xr = NULL;
    xF = NULL;
    s = Copen(result_rep, argv[3], "r");
    if (s == NULL) {
        error_mess(flow_interp, _FUNC_MESS + 9);
        return 1;
    }
    load_data_C(s, &xr, &xF, &nb);
    if (nb <= 1) {
        error_mess(flow_interp, 24);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("defxr_f");
    k[2] = argv[1];
    k[3] = ch_copy_int(0);
    k[4] = ch_copy_int(nb);
    if (def_xrange_f(4, k + 1) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        free(k[1]);
        free(k[3]);
        free(k[4]);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    free(k[1]);
    free(k[3]);
    free(k[4]);
    k[1] = ch_copy("defunc_C");
    k[2] = argv[2];
    k[3] = argv[1];
    if (def_funct_C(3, k + 1) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        free(k[1]);
        k[1] = ch_copy("destroy");
        k[2] = argv[1];
        detruit_obj(2, k + 1);
        free(k[1]);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    free(k[1]);
    iw = sketch_obj_restr(argv[2], &i0, _CFUNC_F, flow_interp);
    c = (char **) Obj[iw - 1][i0].adresse;
    a = (funct_C *) c[0];
    a->nb = nb;

    for (i = 0; i < nb; i++) {
        a->x[i + 1] = xr[i];
        a->f[i + 1] = xF[i];
    }

    free(xr);
    free(xF);
    fclose(s);
    return 0;
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Function associated to the command 'load_create_dC'
--------------------------------------------------------------------*/
int
load_create_dC(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw,
                    nb;
    char          **c,
                   *k[5];
    FILE           *s;
    double         *xr;
    dcomplex       *xF;
    funct_dC       *a;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    xr = NULL;
    xF = NULL;
    s = Copen(result_rep, argv[3], "r");
    if (s == NULL) {
        error_mess(flow_interp, _FUNC_MESS + 9);
        return 1;
    }
    load_data_dC(s, &xr, &xF, &nb);
    if (nb <= 1) {
        error_mess(flow_interp, 24);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    k[0] = (char *) flow_interp;
    k[1] = ch_copy("defxr_d");
    k[2] = argv[1];
    k[3] = ch_copy_int(0);
    k[4] = ch_copy_int(nb);
    if (def_xrange_d(4, k + 1) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 1);
        free(k[1]);
        free(k[3]);
        free(k[4]);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    free(k[1]);
    free(k[3]);
    free(k[4]);
    k[1] = ch_copy("defunc_dC");
    k[2] = argv[2];
    k[3] = argv[1];
    if (def_funct_dC(3, k + 1) == 1) {
        error_mess(flow_interp, _FUNC_MESS + 7);
        free(k[1]);
        k[1] = ch_copy("destroy");
        k[2] = argv[1];
        detruit_obj(2, k + 1);
        free(k[1]);
        free(xr);
        free(xF);
        fclose(s);
        return 1;
    }
    free(k[1]);
    iw = sketch_obj_restr(argv[2], &i0, _CFUNC_D, flow_interp);
    c = (char **) Obj[iw - 1][i0].adresse;
    a = (funct_dC *) c[0];
    a->nb = nb;

    for (i = 0; i < nb; i++) {
        a->x[i + 1] = xr[i];
        a->f[i + 1] = xF[i];
    }

    free(xr);
    free(xF);
    fclose(s);
    return 0;
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Function associated to the command 'fft'
--------------------------------------------------------------------*/
int
FFT_func(int argc, char *argv[])
{
    int             iw,
                    i0,
                    iw0,
                    i00,
                    ifft,
                    n;
    char          **c,
                  **c0;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    ifft = convert_int(argv[3], flow_interp);
    n = 1;
    while (n < ifft) {
        n *= 2;
    }
    if (n != ifft) {
        error_mess(flow_interp, _FUNC_MESS + 23);
        return 1;
    }
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
            if (iw != _CFUNC_F) {
                iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
                if (iw != _CFUNC_D) {
                    error_mess(flow_interp, _FUNC_MESS + 7);
                    return 1;
                }
            }
        }
    }
    iw0 = sketch_obj_restr(argv[2], &i00, iw + 2, flow_interp);
    if (iw0 != iw + 2) {
        iw0 = sketch_obj_restr(argv[2], &i00, iw, flow_interp);
        if (iw0 != iw) {
            error_mess(flow_interp, _FUNC_MESS + 7);
            return 1;
        }
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw == _RFUNC_F) {
        if (TX_f(c[0]) * TX_C(c0[0]) == 1)
            XFFT_func_f((funct_f *) c[0], (funct_C *) c0[0], ifft);
    }
    else {
        if (iw == _RFUNC_D) {
            if (TX_d(c[0]) * TX_dC(c0[0]) == 1)
                XFFT_func_d((funct_d *) c[0], (funct_dC *) c0[0], ifft);
        }
        else {
            if (iw == _CFUNC_F) {
                if (TX_C(c[0]) * TX_C(c0[0]) == 1)
                    XFFT_func_C((funct_C *) c[0], (funct_C *) c0[0], ifft);
            }
            else {
                if (iw == _CFUNC_D) {
                    if (TX_dC(c[0]) * TX_dC(c0[0]) == 1)
                        XFFT_func_dC((funct_dC *) c[0], (funct_dC *) c0[0],
                            ifft);
                }
            }
        }
    }

    return 0;
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    Function associated to the command 'convol'
--------------------------------------------------------------------*/
int
convol_func(int argc, char *argv[])
{
    int             iw,
                    i0,
                    iw0,
                    i00,
                    iw1,
                    i01,
                    ifft,
                    n;
    char          **c,
                  **c0,
                  **c1;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    ifft = convert_int(argv[4], flow_interp);
    n = 1;
    while (n < ifft) {
        n *= 2;
    }
    if (n != ifft) {
        error_mess(flow_interp, _FUNC_MESS + 25);
        return 1;
    }
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F, flow_interp);
    if (iw != _RFUNC_F) {
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D, flow_interp);
        if (iw != _RFUNC_D) {
            iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F, flow_interp);
            if (iw != _CFUNC_F) {
                iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D, flow_interp);
                if (iw != _CFUNC_D) {
                    error_mess(flow_interp, _FUNC_MESS + 7);
                    return 1;
                }
            }
        }
    }
    iw0 = sketch_obj_restr(argv[2], &i00, _RFUNC_F, flow_interp);
    if (iw0 != _RFUNC_F) {
        iw0 = sketch_obj_restr(argv[2], &i00, _RFUNC_D, flow_interp);
        if (iw0 != _RFUNC_D) {
            iw0 = sketch_obj_restr(argv[2], &i00, _CFUNC_F, flow_interp);
            if (iw0 != _CFUNC_F) {
                iw0 = sketch_obj_restr(argv[2], &i00, _CFUNC_D, flow_interp);
                if (iw0 != _CFUNC_D) {
                    error_mess(flow_interp, _FUNC_MESS + 7);
                    return 1;
                }
            }
        }
    }
    iw1 = sketch_obj_restr(argv[3], &i01, _RFUNC_F, flow_interp);
    if (iw1 != _RFUNC_F) {
        iw1 = sketch_obj_restr(argv[3], &i01, _RFUNC_D, flow_interp);
        if (iw1 != _RFUNC_D) {
            iw1 = sketch_obj_restr(argv[3], &i01, _CFUNC_F, flow_interp);
            if (iw1 != _CFUNC_F) {
                iw1 = sketch_obj_restr(argv[3], &i01, _CFUNC_D, flow_interp);
                if (iw1 != _CFUNC_D) {
                    error_mess(flow_interp, _FUNC_MESS + 7);
                    return 1;
                }
            }
        }
    }

    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    c1  = (char **) Obj[iw1 - 1][i01].adresse;

    if (iw == _RFUNC_F && iw0 == _RFUNC_F && iw1 == _RFUNC_F) {
        if (TX_f(c[0]) * TX_f(c0[0]) * TX_f(c1[0]) == 1)
            Xconvol_f_f((funct_f *) c[0], (funct_f *) c0[0],
                (funct_f *) c1[0], ifft);
        return 0;
    }
    if (iw == _RFUNC_D && iw0 == _RFUNC_D && iw1 == _RFUNC_D) {
        if (TX_d(c[0]) * TX_d(c0[0]) * TX_d(c1[0]) == 1)
            Xconvol_d_d((funct_d *) c[0], (funct_d *) c0[0],
                (funct_d *) c1[0], ifft);
        return 0;
    }
    if (iw == _RFUNC_F && iw0 == _CFUNC_F && iw1 == _CFUNC_F) {
        if (TX_f(c[0]) * TX_C(c0[0]) * TX_C(c1[0]) == 1)
            Xconvol_f_C((funct_f *) c[0], (funct_C *) c0[0],
                (funct_C *) c1[0], ifft);
        return 0;
    }
    if (iw == _CFUNC_F && iw0 == _RFUNC_F && iw1 == _CFUNC_F) {
        if (TX_C(c[0]) * TX_f(c0[0]) * TX_C(c1[0]) == 1)
           Xconvol_C_f((funct_C *) c[0], (funct_f *) c0[0],
                (funct_C *) c1[0], ifft);
        return 0;
    }
    if (iw == _CFUNC_F && iw0 == _CFUNC_F && iw1 == _CFUNC_F) {
        if (TX_C(c[0]) * TX_C(c0[0]) * TX_C(c1[0]) == 1)
            Xconvol_C_C((funct_C *) c[0], (funct_C *) c0[0],
                (funct_C *) c1[0], ifft);
        return 0;
    }
    if (iw == _RFUNC_D && iw0 == _CFUNC_D && iw1 == _CFUNC_D) {
        if (TX_d(c[0]) * TX_dC(c0[0]) * TX_dC(c1[0]) == 1)
            Xconvol_d_dC((funct_d *) c[0], (funct_dC *) c0[0],
                (funct_dC *) c1[0], ifft);
        return 0;
    }
    if (iw == _CFUNC_D && iw0 == _RFUNC_D && iw1 == _CFUNC_D) {
        if (TX_dC(c[0]) * TX_d(c0[0]) * TX_dC(c1[0]) == 1)
            Xconvol_dC_d((funct_dC *) c[0], (funct_d *) c0[0],
                (funct_dC *) c1[0], ifft);
        return 0;
    }
    if (iw == _CFUNC_D && iw0 == _CFUNC_D && iw1 == _CFUNC_D) {
        if (TX_dC(c[0]) * TX_dC(c0[0]) * TX_dC(c1[0]) == 1)
            Xconvol_dC_dC((funct_dC *) c[0], (funct_dC *) c0[0],
                (funct_dC *) c1[0], ifft);
        return 0;
    }

    error_mess(flow_interp, _FUNC_MESS + 7);
    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------------
    Function associated to the command 'def_four'.
--------------------------------------------------------------------------*/
int
def_four_prec(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i0_b,
                    iw_b,
                    ifft,
                    n;
    char          **e,
                  **c,
                   *k[4];
    float          *x_r;
    double         *x_rd;
    Tfourier       *Tr;
    CTfourier      *CTr;
    funct_f        *a_f;
    funct_d        *a_d;
    funct_C        *a_C;
    funct_dC       *a_dC;
    Transf         *Tra;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj(argv[1], &i0, flow_interp);
    if (iw != 0) {
        error_mess(flow_interp, _FUNC_MESS + 0);
        return 1;
    }
    ifft = convert_int(argv[3], flow_interp);
    if (ifft < 4) {
        error_mess(flow_interp, _FUNC_MESS + 2);
        return 1;
    }
    n = 1;
    while (n < ifft) {
        n *= 2;
    }
    if (n != ifft) {
        error_mess(flow_interp, _FUNC_MESS + 23);
        return 1;
    }

    Tra = (Transf *) malloc((size_t) sizeof(Transf));
    iw_b = sketch_obj_restr(argv[2], &i0_b, _RFUNC_F, flow_interp);
    if (iw_b != _RFUNC_F) {
        iw_b = sketch_obj_restr(argv[2], &i0_b, _RFUNC_D, flow_interp);
        if (iw_b != _RFUNC_D) {
            iw_b = sketch_obj_restr(argv[2], &i0_b, _CFUNC_F, flow_interp);
            if (iw_b != _CFUNC_F) {
                iw_b = sketch_obj_restr(argv[2], &i0_b, _CFUNC_D, flow_interp);
                if (iw_b != _CFUNC_D) {
                    error_mess(flow_interp, _FUNC_MESS + 7);
                    return 1;
                }
            }
        }
    }
    c = (char **) Obj[iw_b - 1][i0_b].adresse;

    if (iw_b == _RFUNC_F) {
        a_f = (funct_f *) c[0];
        f_fill(a_f, &x_r);
        if (test_xr_f(x_r, a_f->nb) == 1) {
            error_mess(flow_interp, _FUNC_MESS + 19);
            return 1;
        }
        Tr = fTf(&a_f->nb, x_r, a_f->f, ifft, 3);
        Tr->nom_func = ch_copy(a_f->nom);
        Tra->tr = (char *) Tr;
        Tra->type = 0;
        if (a_f->type > 0)
            XFREE(x_r);
    }
    else {
        if (iw_b == _RFUNC_D) {
            a_d = (funct_d *) c[0];
            d_fill(a_d, &x_rd);
            if (test_xr_d(x_rd, a_d->nb) == 1) {
                error_mess(flow_interp, _FUNC_MESS + 19);
                return 1;
            }
            Tr = Tf(&a_d->nb, x_rd, a_d->f, ifft, 3);
            Tr->nom_func = ch_copy(a_d->nom);
            Tra->tr = (char *) Tr;
            Tra->type = 0;
            if (a_d->type > 0)
                XFREE(x_rd);
        }
        else {
            if (iw_b == _CFUNC_F) {
                a_C = (funct_C *) c[0];
                C_fill(a_C, &x_r);
                if (test_xr_f(x_r, a_C->nb) == 1) {
                    error_mess(flow_interp, _FUNC_MESS + 19);
                    return 1;
                }
                CTr = CTf(&a_C->nb, x_r, a_C->f, ifft, 3);
                (CTr->Reel)->nom_func = ch_copy(a_C->nom);
                Tra->tr = (char *) CTr;
                Tra->type = 1;
                if (a_C->type > 0)
                    XFREE(x_r);
            }
            else {
                a_dC = (funct_dC *) c[0];
                dC_fill(a_dC, &x_rd);
                if (test_xr_d(x_rd, a_dC->nb) == 1) {
                    error_mess(flow_interp, _FUNC_MESS + 19);
                    return 1;
                }
                CTr = dCTf(&a_dC->nb, x_rd, a_dC->f, ifft, 3);
                (CTr->Reel)->nom_func = ch_copy(a_dC->nom);
                Tra->tr = (char *) CTr;
                Tra->type = 1;
                if (a_dC->type > 0)
                    XFREE(x_rd);
            }
        }
    }

    k[0] = (char *) flow_interp;
    k[1] = ch_copy("objdef");
    k[2] = ch_copy_int(_FOUR_TR - 1);
    k[3] = argv[1];
    i0 = obj_create(3, k + 1);
    if (i0 == -1) {
        error_mess(flow_interp, _FUNC_MESS);
        return 1;
    }
    e = (char **) Obj[_FOUR_TR - 1][i0].adresse;
    e[0] = (char *) Tra;
    free(k[1]);
    free(k[2]);
    return 0;
}
/*------------------------------------------------------------------------*/




/*--------------------------------------------------------------------------
    Function associated to the command 'trans_four'.
--------------------------------------------------------------------------*/
int
trans_four_prec(int argc, char *argv[])
{
    int             i,
                    iw,
                    i0,
                    iw0,
                    i00,
                    type;
    Transf         *Tra;
    Tfourier       *Tr;
    CTfourier      *CTr;
    char          **c,
                  **e;
    float          *x_r;
    double         *x_rd;
    funct_C        *a_C;
    funct_dC       *a_dC;
    flow_data      *flow_interp;

    INIT_FLOW(flow_interp);
    iw = sketch_obj_restr(argv[1], &i0, _FOUR_TR, flow_interp);
    if (iw != _FOUR_TR) {
        error_mess(flow_interp, _FUNC_MESS + 16);
        return 1;
    }
    e = (char **) Obj[iw - 1][i0].adresse;
    iw0 = sketch_obj_restr(argv[2], &i00, _CFUNC_F, flow_interp);
    if (iw0 != _CFUNC_F) {
        iw0 = sketch_obj_restr(argv[2], &i00, _CFUNC_D, flow_interp);
        if (iw0 != _CFUNC_D) {
            error_mess(flow_interp, _FUNC_MESS + 7);
            return 1;
        }
    }
    c = (char **) Obj[iw0 - 1][i00].adresse;
    Tra = (Transf *) e[0];
    type = Tra->type;

    if (iw0 == _CFUNC_F) {
        a_C = (funct_C *) c[0];
        C_fill(a_C, &x_r);
        if (type == 0) {
            Tr = (Tfourier *) Tra->tr;
            for (i = 1; i <= a_C->nb; i++)
                a_C->f[i] = fTrans(Tr, x_r[i]);
        }
        else {
            CTr = (CTfourier *) Tra->tr;
            for (i = 1; i <= a_C->nb; i++)
                a_C->f[i] = CTrans(CTr, x_r[i]);
        }
        if (a_C->type > 0)
            XFREE(x_r);
    }
    else {
        a_dC = (funct_dC *) c[0];
        dC_fill(a_dC, &x_rd);
        if (type == 0) {
            Tr = (Tfourier *) Tra->tr;
            for (i = 1; i <= a_dC->nb; i++)
                a_dC->f[i] = Trans(Tr, x_rd[i]);
        }
        else {
            CTr = (CTfourier *) Tra->tr;
            for (i = 1; i <= a_dC->nb; i++)
                a_dC->f[i] = dCTrans(CTr, x_rd[i]);
        }
        if (a_dC->type > 0)
            XFREE(x_rd);
    }
    return 0;
}
/*------------------------------------------------------------------------*/




/*--------------------------------------------------------------------------
    Function that is called when a command of destruction of a Fourier
    object is given.
--------------------------------------------------------------------------*/
void
detruit_four_prec(int iw, int i0)
{
    Transf         *Tra;
    char          **e;

    e = (char **) Obj[iw - 1][i0].adresse;
    Tra = (Transf *) e[0];
    if (Tra->type == 0) {
        Detruit((Tfourier *) Tra->tr);
    }
    else {
        CDetruit((CTfourier *) Tra->tr);
    }
    free(Tra);
}
/*------------------------------------------------------------------------*/




/*--------------------------------------------------------------------------
--------------------------------------------------------------------------*/
int
test_xr_f(float *x_r, int n)
{
    int             i;

    for (i = 1; i < n; i++)
        if (x_r[i + 1] - x_r[i] <= 0)
            return 1;
    return 0;
}
/*------------------------------------------------------------------------*/




/*--------------------------------------------------------------------------
--------------------------------------------------------------------------*/
int
test_xr_d(double *x_r, int n)
{
    int             i;

    for (i = 1; i < n; i++)
        if (x_r[i + 1] - x_r[i] <= 0)
            return 1;
    return 0;
}
/*------------------------------------------------------------------------*/
